<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ProjectSummaryTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);
    }

    protected function getRecords($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(
                route(
                    'projects.summary',
                    [
                        'project' => $this->project,
                        ...$data,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testRecordsListedWithInvalidFromAndToDate(): void
    {
        $names = [];

        for ($i = 0; $i < 3; $i++) {
            $driver = KeyDriver::factory()->create(['project_id' => $this->project]);
            $item = DriverItem::factory()->create(['key_driver_id' => $driver]);
            $names[$i] = $driver->name.' - '.$item->name;

            ItemRecord::factory()
                ->count(5)
                ->state(new Sequence(
                    ['date' => now()->startOfDay()],
                    ['date' => now()->startOfDay()->addDays()],
                    ['date' => now()->startOfDay()->addDays(2)],
                    ['date' => now()->startOfDay()->addDays(3)],
                    ['date' => now()->startOfDay()->addDays(4)],
                ))
                ->create(['driver_item_id' => $item->id]);
        }

        $response = $this->getRecords();

        $this->assertCount(3, $response->json('data'));

        foreach ($names as $name) {
            $this->assertCount(3, $response->json('data')[$name]);
            $this->assertCount(5, $response->json('data')[$name][0]['data']);
        }
    }

    public function testRecordsListedWithValidFromAndToDate(): void
    {
        $names = [];

        for ($i = 0; $i < 3; $i++) {
            $driver = KeyDriver::factory()->create(['project_id' => $this->project]);
            $item = DriverItem::factory()->create(['key_driver_id' => $driver]);
            $names[$i] = $driver->name.' - '.$item->name;

            ItemRecord::factory()
                ->count(5)
                ->state(new Sequence(
                    ['date' => now()->startOfDay()],
                    ['date' => now()->startOfDay()->addDays()],
                    ['date' => now()->startOfDay()->addDays(2)],
                    ['date' => now()->startOfDay()->addDays(3)],
                    ['date' => now()->startOfDay()->addDays(4)],
                ))
                ->create(['driver_item_id' => $item->id]);
        }

        $response = $this->getRecords([
            'from' => now()->addDays()->toDateString(),
            'to' => now()->addDays(3)->toDateString(),
        ]);

        $this->assertCount(3, $response->json('data'));

        foreach ($names as $name) {
            $this->assertCount(3, $response->json('data')[$name]);
            $this->assertCount(3, $response->json('data')[$name][0]['data']);
        }
    }

    public function testRecordsListedWithMinValue(): void
    {
        $names = [];

        for ($i = 0; $i < 3; $i++) {
            $driver = KeyDriver::factory()->create(['project_id' => $this->project]);
            $item = DriverItem::factory()->create(['key_driver_id' => $driver, 'range' => true]);
            $names[$i] = $driver->name.' - '.$item->name;

            ItemRecord::factory()
                ->count(5)
                ->state(new Sequence(
                    ['date' => now()->startOfDay()],
                    ['date' => now()->startOfDay()->addDays()],
                    ['date' => now()->startOfDay()->addDays(2)],
                    ['date' => now()->startOfDay()->addDays(3)],
                    ['date' => now()->startOfDay()->addDays(4)],
                ))
                ->create(['driver_item_id' => $item->id]);
        }

        $response = $this->getRecords([
            'from' => now()->addDays()->toDateString(),
            'to' => now()->addDays(3)->toDateString(),
        ]);

        $this->assertCount(3, $response->json('data'));

        foreach ($names as $name) {
            $this->assertCount(4, $response->json('data')[$name]);
            $this->assertCount(3, $response->json('data')[$name][0]['data']);
        }
    }
}
