<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\User;
use App\Notifications\ProjectDeleted;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ProjectDeleteTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'owner']);
    }

    protected function deleteProject(): TestResponse
    {
        return $this->actingAs($this->user)
            ->deleteJson(route('projects.update', ['project' => $this->project]));
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->deleteProject();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->deleteProject();

        $response->assertNotFound();
    }

    public function testProjectExistButUserNotAdminOrOwner(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->deleteProject();

        $response->assertForbidden();
    }

    public function testUnverifiedUserCannotDeleteProject(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->deleteProject();

        $response->assertForbidden();
    }

    public function testUserCannotDeleteProject(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->deleteProject();

        $response->assertForbidden();
    }

    public function testProjectWasDeleted(): void
    {
        Notification::fake();

        $users = User::factory()->count(3)->create();

        $this->project->users()->attach($users);

        $response = $this->deleteProject();

        $response->assertValid();

        Notification::assertSentTo($users, ProjectDeleted::class);

        $this->assertDatabaseMissing('projects', ['id' => $this->project->id]);
    }
}
