<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Auth\Notifications\ResetPassword;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class PasswordResetTest extends TestCase
{
    use RefreshDatabase;

    protected function resetPassword($data = []): TestResponse
    {
        return $this->postJson(route('password.reset'), $data);
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->resetPassword();

        $response->assertInvalid([
            'token' => __('validation.required', ['attribute' => 'token']),
            'email' => __('validation.required', ['attribute' => 'email']),
            'password' => __('validation.required', ['attribute' => 'password']),
        ]);
    }

    public function testEmailIsValid(): void
    {
        $response = $this->resetPassword(['email' => Str::random(10)]);

        $response->assertInvalid([
            'email' => __('validation.email', ['attribute' => 'email']),
        ]);
    }

    public function testPassowrdNotConfirmed(): void
    {
        $response = $this->resetPassword(['password' => Str::random(10)]);

        $response->assertInvalid([
            'password' => __('validation.confirmed', ['attribute' => 'password']),
        ]);
    }

    public function testPassowrdTooShort(): void
    {
        $response = $this->resetPassword(['password' => Str::random(7)]);

        $response->assertInvalid([
            'password' => __('validation.min.string', ['attribute' => 'password', 'min' => 8]),
        ]);
    }

    public function testUserInvalid(): void
    {
        $data = [
            'token' => Str::random(10),
            'email' => fake()->email(),
            'password' => 'password',
            'password_confirmation' => 'password',
        ];

        $response = $this->resetPassword($data);

        $response->assertJson(['message' => __(Password::INVALID_USER)]);
    }

    public function testTokenInvalid(): void
    {
        $user = User::factory()->create();

        $data = [
            'token' => Str::random(10),
            'email' => $user->email,
            'password' => 'password',
            'password_confirmation' => 'password',
        ];

        $response = $this->resetPassword($data);

        $response->assertJson(['message' => __(Password::INVALID_TOKEN)]);
    }

    public function testPasswordReset(): void
    {
        Notification::fake();

        $user = User::factory()->create();

        $this->postJson(route('password.forgot'), ['email' => $user->email]);

        Notification::assertSentTo(
            $user,
            ResetPassword::class,
            function ($notification) use ($user) {
                $response = $this->resetPassword([
                    'token' => $notification->token,
                    'email' => $user->email,
                    'password' => 'password',
                    'password_confirmation' => 'password',
                ]);

                $response->assertJson([
                    'message' => __(Password::PASSWORD_RESET),
                    'data' => [
                        'id' => $user->id,
                        'first_name' => $user->first_name,
                        'last_name' => $user->last_name,
                        'email' => $user->email,
                        'email_verified_at' => $user->email_verified_at->toJson(),
                        'role' => $user->role,
                    ],
                ]);

                $this->assertAuthenticated();

                $this->assertTrue(Hash::check('password', $user->fresh()->password));

                return true;
            }
        );
    }
}
