<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Auth\Notifications\ResetPassword;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class PasswordForgotTest extends TestCase
{
    use RefreshDatabase;

    protected function sendLink($data = []): TestResponse
    {
        return $this->postJson(route('password.forgot'), $data);
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->sendLink();

        $response->assertInvalid([
            'email' => __('validation.required', ['attribute' => 'email']),
        ]);
    }

    public function testEmailIsValid(): void
    {
        $response = $this->sendLink(['email' => Str::random(10)]);

        $response->assertInvalid([
            'email' => __('validation.email', ['attribute' => 'email']),
        ]);
    }

    public function testUserNotFound(): void
    {
        $response = $this->sendLink(['email' => fake()->email()]);

        $response->assertJson(['message' => __(Password::INVALID_USER)]);
    }

    public function testRouteThrottle(): void
    {
        $user = User::factory()->create();

        $this->sendLink(['email' => $user->email]);

        $response = $this->sendLink(['email' => $user->email]);

        $response->assertJson(['message' => __(Password::RESET_THROTTLED)]);
    }

    public function testResetLinkSent(): void
    {
        Notification::fake();

        $user = User::factory()->create();

        $response = $this->sendLink(['email' => $user->email]);

        $response->assertJson(['message' => __(Password::RESET_LINK_SENT)]);

        Notification::assertSentTo($user, ResetPassword::class);
    }
}
