<?php

namespace Tests\Feature;

use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class KeyDriverUpdateTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);
    }

    protected function updateDriver($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->putJson(
                route(
                    'projects.drivers.update',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->updateDriver();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->updateDriver();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->updateDriver();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->updateDriver();

        $response->assertNotFound();
    }

    public function testUnverifiedUserCannotUpdateDriver(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->updateDriver();

        $response->assertForbidden();
    }

    public function testUserCannotUpdateDriver(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->updateDriver();

        $response->assertForbidden();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->updateDriver();

        $response->assertInvalid([
            'name' => __('validation.required', ['attribute' => 'name']),
        ]);
    }

    public function testNameIsUnique(): void
    {
        $driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $response = $this->updateDriver(['name' => $driver->name]);

        $response->assertInvalid([
            'name' => __('validation.unique', ['attribute' => 'name']),
        ]);
    }

    public function testDriverWasUpdatedWithSameName(): void
    {
        $response = $this->updateDriver(['name' => $this->driver->name]);

        $response->assertValid();

        $this->assertDatabaseHas('key_drivers', [
            'id' => $this->driver->id,
            'project_id' => $this->project->id,
            'name' => $this->driver->name,
        ]);
    }

    public function testDriverWasUpdatedWithDifferentName(): void
    {
        $name = fake()->name();

        $response = $this->updateDriver(['name' => $name]);

        $response->assertValid();

        $this->assertDatabaseHas('key_drivers', [
            'id' => $this->driver->id,
            'project_id' => $this->project->id,
            'name' => $name,
        ]);
    }
}
