<?php

namespace Tests\Feature;

use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class KeyDriverStoreTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);
    }

    protected function storeDriver($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->postJson(
                route(
                    'projects.drivers.store',
                    [
                        'project' => $this->project,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->storeDriver();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->storeDriver();

        $response->assertNotFound();
    }

    public function testUnverifiedUserCannotStoreDriver(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->storeDriver();

        $response->assertForbidden();
    }

    public function testUserCannotStoreDriver(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->storeDriver();

        $response->assertForbidden();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->storeDriver();

        $response->assertInvalid([
            'name' => __('validation.required', ['attribute' => 'name']),
        ]);
    }

    public function testNameIsUnique(): void
    {
        $driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $response = $this->storeDriver(['name' => $driver->name]);

        $response->assertInvalid([
            'name' => __('validation.unique', ['attribute' => 'name']),
        ]);
    }

    public function testDriverWasSaved(): void
    {
        $name = fake()->name();

        $response = $this->storeDriver(['name' => $name]);

        $response->assertValid();

        $this->assertDatabaseHas('key_drivers', [
            'id' => $response->json('data.id'),
            'project_id' => $this->project->id,
            'name' => $name,
        ]);
    }
}
