<?php

namespace Tests\Feature;

use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class KeyDriverDeleteTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);
    }

    protected function deleteDriver($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->deleteJson(
                route(
                    'projects.drivers.delete',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->deleteDriver();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->deleteDriver();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->deleteDriver();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->deleteDriver();

        $response->assertNotFound();
    }

    public function testUnverifiedUserCannotDeleteDriver(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->deleteDriver();

        $response->assertForbidden();
    }

    public function testUserCannotDeleteDriver(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->deleteDriver();

        $response->assertForbidden();
    }

    public function testDriverWasDeleted(): void
    {
        $response = $this->deleteDriver();

        $response->assertValid();

        $this->assertDatabaseMissing('key_drivers', ['id' => $this->driver->id]);
    }
}
