<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ItemRecordsImportTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);
    }

    protected function storeRecords($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->postJson(
                route(
                    'projects.drivers.items.records.import',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->storeRecords();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->storeRecords();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->storeRecords();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->storeRecords();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->storeRecords();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherProject(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->storeRecords();

        $response->assertNotFound();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->storeRecords();

        $response->assertInvalid([
            'file' => __('validation.required', ['attribute' => 'file']),
        ]);
    }

    public function testFileExtensionNotCSV(): void
    {
        $response = $this->storeRecords();

        $response->assertInvalid([
            'file' => __('validation.required', ['attribute' => 'file']),
        ]);
    }

    public function testFileIsValid(): void
    {
        $response = $this->storeRecords(['file' => UploadedFile::fake()->image('test.png')]);

        $response->assertInvalid([
            'file' => __('validation.mimes', ['attribute' => 'file', 'values' => 'csv, txt']),
        ]);
    }

    public function testRowFieldsAreRequired(): void
    {
        $content = "Date,Max,Min,Type,Comments\n,,,,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'date' => 'The field date is required in line 2.',
            'max' => 'The field max is required in line 2.',
            'type' => 'The field type is required in line 2.',
        ]);
    }

    public function testDateFormatIsValid(): void
    {
        $invalidDate = Str::random(5);

        $content = "Date,Max,Min,Type,Comments\n$invalidDate,,,,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'date' => 'The date field must match the format Y-m-d in line 2.',
        ]);
    }

    public function testDateIsUnique(): void
    {
        $record = ItemRecord::factory()->create(['driver_item_id' => $this->item]);

        $date = $record->date->toDateString();

        $content = "Date,Max,Min,Type,Comments\n$date,,,,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'date' => 'The date has already been taken in line 2.',
        ]);
    }

    public function testMinOnlyRequiredWhenItemIsRange(): void
    {
        $this->item->update(['range' => true]);

        $content = "Date,Max,Min,Type,Comments\n,,,,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'min' => 'The field min is required in line 2.',
        ]);
    }

    public function testMinNotRequiredWhenItemIsNotRange(): void
    {
        $this->item->update(['range' => false]);

        $content = "Date,Max,Min,Type,Comments\n,,,,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertJsonMissingValidationErrors('min');
    }

    public function testMaxAndMinAreNumeric(): void
    {
        $this->item->update(['range' => true]);

        $min = Str::random(5);
        $max = Str::random(5);

        $content = "Date,Max,Min,Type,Comments\n,$max,$min,,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'max' => 'The max field must be a number in line 2.',
            'min' => 'The min field must be a number in line 2.',
        ]);
    }

    public function testMinIsLessThanMax(): void
    {
        $this->item->update(['range' => true]);

        $content = "Date,Max,Min,Type,Comments\n,10,15,normal,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'min' => 'The min field must be less than 10 in line 2.',
        ]);
    }

    public function testTypeIsValid(): void
    {
        $content = "Date,Max,Min,Type,Comments\n,10,15,".Str::random(5).',';

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'type' => 'The selected type is invalid in line 2.',
        ]);
    }

    public function testCommentsAreTooLarge(): void
    {
        $comments = fake()->text(1200);

        $content = "Date,Max,Min,Type,Comments\n,,,,$comments";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'comments' => 'The comments field must not be greater than 1000 characters in line 2.',
        ]);
    }

    public function testRecordsAreUniqueInFile(): void
    {
        $date = now()->toDateString();

        $content = 'Date,Max,Min,Type,Comments';
        $content .= "\n$date,10,5,normal,";
        $content .= "\n$date,10,5,normal,";

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertInvalid([
            'date' => 'The date field is not unique within the file.',
        ]);
    }

    public function testRecordsWereSaved(): void
    {
        $records = ItemRecord::factory()
            ->count(5)
            ->state(new Sequence(
                ['date' => now()->startOfDay()],
                ['date' => now()->startOfDay()->addDays()],
                ['date' => now()->startOfDay()->addDays(2)],
                ['date' => now()->startOfDay()->addDays(3)],
                ['date' => now()->startOfDay()->addDays(4)],
            ))
            ->make();

        $rows = ['Date,Max,Min,Type,Comments'];

        foreach ($records as $record) {
            $row = $record->date->toDateString().','.$record->max.',';
            $row .= is_null($record->min) ? '-,' : $record->min.',';
            $row .= $record->type.',';
            $row .= $record->comments;

            $rows[] = $row;
        }

        $content = implode("\n", $rows);

        $response = $this->storeRecords([
            'file' => UploadedFile::fake()->createWithContent('records.csv', $content),
        ]);

        $response->assertJson([
            'message' => $records->count().' records were successfully saved.',
        ]);

        foreach ($records as $record) {
            $this->assertDatabaseHas('item_records', [
                'driver_item_id' => $this->item->id,
                'date' => $record->date->toJson(),
            ]);
        }
    }
}
