<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class ItemRecordsExportTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);
    }

    protected function getRecords(): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(
                route(
                    'projects.drivers.items.records.export',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherProject(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testCsvWasDownloaded(): void
    {
        ItemRecord::factory()->count(10)->create(['driver_item_id' => $this->item->id]);

        $response = $this->getRecords();

        $fileName = Str::slug($this->item->name).'.csv';

        $response->assertHeader('Content-Type', 'text/csv; charset=UTF-8');
        $response->assertHeader('Content-Disposition', 'attachment; filename='.$fileName);

        $lines = str_getcsv($response->streamedContent(), "\n");

        $records = $this->item->records()->orderBy('date', 'desc')->get();

        for ($i = 1; $i < 10; $i++) {
            $data = explode(',', $lines[$i]);
            $this->assertTrue($records[$i]->date->toDateString() == $data[0]);
        }
    }
}
