<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class DriverItemUpdateTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);
    }

    protected function updateItem($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->putJson(
                route(
                    'projects.drivers.items.update',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                    ],
                ),
                $data,
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->updateItem();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->updateItem();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->updateItem();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->updateItem();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->updateItem();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherDriver(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->updateItem();

        $response->assertNotFound();
    }

    public function testUnverifiedUserCannotUpdateItem(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->updateItem();

        $response->assertForbidden();
    }

    public function testUserCannotUpdateItem(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->updateItem();

        $response->assertForbidden();
    }

    public function testFieldsAreRequired(): void
    {
        $response = $this->updateItem();

        $response->assertInvalid([
            'name' => __('validation.required', ['attribute' => 'name']),
            'range' => __('validation.required', ['attribute' => 'range']),
        ]);
    }

    public function testNameIsUnique(): void
    {
        $item = DriverItem::factory()->create(['key_driver_id' => $this->driver->id]);

        $response = $this->updateItem(['name' => $item->name]);

        $response->assertInvalid([
            'name' => __('validation.unique', ['attribute' => 'name']),
        ]);
    }

    public function testRangeIsBoolean(): void
    {
        $response = $this->updateItem(['range' => Str::random(10)]);

        $response->assertInvalid([
            'range' => __('validation.boolean', ['attribute' => 'range']),
        ]);
    }

    public function testMaxAndMinAreNumeric(): void
    {
        $response = $this->updateItem([
            'max' => Str::random(10),
            'min' => Str::random(10),
        ]);

        $response->assertInvalid([
            'max' => __('validation.numeric', ['attribute' => 'max']),
            'min' => __('validation.numeric', ['attribute' => 'min']),
        ]);
    }

    public function testMinIsLessThanMin(): void
    {
        $response = $this->updateItem([
            'max' => 10,
            'min' => 15,
        ]);

        $response->assertInvalid([
            'min' => __('validation.lt.numeric', ['attribute' => 'min', 'value' => 10]),
        ]);
    }

    public function testItemWasUpdatedWithSameName(): void
    {
        $oldName = $this->item->name;

        $item = DriverItem::factory()->make(['name' => $oldName]);

        $response = $this->updateItem($item->only(['name', 'range', 'max', 'min']));

        $response->assertValid();

        $this->assertDatabaseHas('driver_items', [
            'id' => $this->item->id,
            'name' => $oldName,
            'range' => $item->range,
            'max' => $item->max,
            'min' => $item->min,
        ]);
    }

    public function testItemWasUpdatedWithDifferentName(): void
    {
        $item = DriverItem::factory()->make();

        $response = $this->updateItem($item->only(['name', 'range', 'max', 'min']));

        $response->assertValid();

        $this->assertDatabaseHas('driver_items', [
            'id' => $this->item->id,
            'name' => $item->name,
            'range' => $item->range,
            'max' => $item->max,
            'min' => $item->min,
        ]);
    }
}
