<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class DriverItemShowTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);
    }

    protected function getItem(): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(
                route(
                    'projects.drivers.items.show',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->getItem();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->getItem();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->getItem();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->getItem();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->getItem();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherDriver(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->getItem();

        $response->assertNotFound();
    }

    public function testItemFound(): void
    {
        $response = $this->getItem();

        $response->assertJson([
            'data' => [
                'id' => $this->item->id,
                'name' => $this->item->name,
                'range' => $this->item->range,
                'max' => $this->item->max,
                'min' => $this->item->min,
            ],
        ]);
    }
}
