<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class DriverItemGraphicLineTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);
    }

    protected function getRecords($data = []): TestResponse
    {
        return $this->actingAs($this->user)
            ->getJson(
                route(
                    'projects.drivers.items.graphics.line',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                        ...$data,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherDriver(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->getRecords();

        $response->assertNotFound();
    }

    public function testRecordsListedWithInvalidFromAndToDate(): void
    {
        ItemRecord::factory()
            ->count(5)
            ->state(new Sequence(
                ['date' => now()->startOfDay()],
                ['date' => now()->startOfDay()->addDays()],
                ['date' => now()->startOfDay()->addDays(2)],
                ['date' => now()->startOfDay()->addDays(3)],
                ['date' => now()->startOfDay()->addDays(4)],
            ))
            ->create(['driver_item_id' => $this->item->id]);

        $response = $this->getRecords([
            'from' => Str::random(10),
            'to' => Str::random(10),
        ]);

        $this->assertCount(3, $response->json('data'));

        $this->assertCount(5, $response->json('data.0.data'));
    }

    public function testRecordsListedWithValidFromAndToDate(): void
    {
        ItemRecord::factory()
            ->count(5)
            ->state(new Sequence(
                ['date' => now()->startOfDay()],
                ['date' => now()->startOfDay()->addDays()],
                ['date' => now()->startOfDay()->addDays(2)],
                ['date' => now()->startOfDay()->addDays(3)],
                ['date' => now()->startOfDay()->addDays(4)],
            ))
            ->create(['driver_item_id' => $this->item->id]);

        $response = $this->getRecords([
            'from' => now()->addDays()->toDateString(),
            'to' => now()->addDays(3)->toDateString(),
        ]);

        $this->assertCount(3, $response->json('data'));

        $this->assertCount(3, $response->json('data.0.data'));
    }

    public function testRecordsListedWithMinValue(): void
    {
        $this->item->update(['range' => true]);

        ItemRecord::factory()
            ->count(5)
            ->state(new Sequence(
                ['date' => now()->startOfDay()],
                ['date' => now()->startOfDay()->addDays()],
                ['date' => now()->startOfDay()->addDays(2)],
                ['date' => now()->startOfDay()->addDays(3)],
                ['date' => now()->startOfDay()->addDays(4)],
            ))
            ->create(['driver_item_id' => $this->item->id]);

        $response = $this->getRecords([
            'from' => now()->addDays()->toDateString(),
            'to' => now()->addDays(3)->toDateString(),
        ]);

        $this->assertCount(4, $response->json('data'));

        $this->assertCount(3, $response->json('data.0.data'));
    }
}
