<?php

namespace Tests\Feature;

use App\Models\DriverItem;
use App\Models\KeyDriver;
use App\Models\Project;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Testing\TestResponse;
use Tests\TestCase;

class DriverItemDeleteTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected Project $project;

    protected KeyDriver $driver;

    protected DriverItem $item;

    public function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();

        $this->project = Project::factory()->create();

        $this->user->projects()->attach($this->project, ['role' => 'admin']);

        $this->driver = KeyDriver::factory()->create(['project_id' => $this->project]);

        $this->item = DriverItem::factory()->create(['key_driver_id' => $this->driver]);
    }

    protected function deleteItem(): TestResponse
    {
        return $this->actingAs($this->user)
            ->deleteJson(
                route(
                    'projects.drivers.items.delete',
                    [
                        'project' => $this->project,
                        'driver' => $this->driver,
                        'item' => $this->item,
                    ],
                ),
            );
    }

    public function testProjectNotFound(): void
    {
        $this->project->delete();

        $response = $this->deleteItem();

        $response->assertNotFound();
    }

    public function testProjectExistButUserIsNotCollaborator(): void
    {
        $this->user->projects()->detach($this->project);

        $response = $this->deleteItem();

        $response->assertNotFound();
    }

    public function testDriverNotFound(): void
    {
        $this->driver->delete();

        $response = $this->deleteItem();

        $response->assertNotFound();
    }

    public function testDriverExistButBelongsToAnotherProject(): void
    {
        $this->driver = KeyDriver::factory()->create();

        $response = $this->deleteItem();

        $response->assertNotFound();
    }

    public function testItemNotFound(): void
    {
        $this->item->delete();

        $response = $this->deleteItem();

        $response->assertNotFound();
    }

    public function testItemExistButBelongsToAnotherDriver(): void
    {
        $this->item = DriverItem::factory()->create();

        $response = $this->deleteItem();

        $response->assertNotFound();
    }

    public function testUnverifiedUserCannotDeleteItem(): void
    {
        $this->user->update(['email_verified_at' => null]);

        $response = $this->deleteItem();

        $response->assertForbidden();
    }

    public function testUserCannotDeleteItem(): void
    {
        $this->user->projects()->updateExistingPivot($this->project, ['role' => 'user']);

        $response = $this->deleteItem();

        $response->assertForbidden();
    }

    public function testItemWasDeleted(): void
    {
        $response = $this->deleteItem();

        $response->assertValid();

        $this->assertDatabaseMissing('driver_items', ['id' => $this->item->id]);
    }
}
