<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProjectStore;
use App\Http\Requests\ProjectUpdate;
use App\Http\Resources\ProjectResource;
use App\Models\Project;
use App\Models\User;
use App\Notifications\ProjectDeleted;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;

class ProjectController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(): AnonymousResourceCollection
    {
        /** @var User */
        $user = Auth::user();

        $projects = $user->projects()->latest()->paginate(12);

        return ProjectResource::collection($projects);
    }

    public function show(Project $project): ProjectResource
    {
        /** @var User */
        $user = Auth::user();

        $project = $user->projects()->find($project->id);

        return new ProjectResource($project);
    }

    public function store(ProjectStore $request): ProjectResource
    {
        $project = $request->createProject();

        return new ProjectResource($project);
    }

    public function update(Project $project, ProjectUpdate $request): ProjectResource
    {
        $project = $request->updateProject();

        return new ProjectResource($project);
    }

    public function delete(Project $project): ProjectResource
    {
        $user = $project->users()->find(Auth::id());

        abort_if(! $user->hasVerifiedEmail() || $user->pivot->role === 'user', 403, 'This action is unauthorized.');

        $users = $project->users()->where('id', '<>', Auth::id())->get();

        $project->delete();

        Notification::send($users, new ProjectDeleted($project));

        return new ProjectResource($project);
    }
}
