<?php

namespace App\Http\Controllers;

use App\Models\DriverItem;
use App\Models\ItemRecord;
use App\Models\KeyDriver;
use App\Models\Project;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class DriverItemGraphicController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function summary(Project $project, Request $request): JsonResponse
    {
        $project->load([
            'drivers.items.records' => function ($query) use ($request) {
                try {
                    $fromDate = Carbon::createFromFormat('Y-m-d', $request->get('from'))->toDateString();
                    $query->whereDate('date', '>=', $fromDate);
                } catch (Exception $e) {
                }

                try {
                    $toDate = Carbon::createFromFormat('Y-m-d', $request->get('to'))->toDateString();
                    $query->whereDate('date', '<=', $toDate);
                } catch (Exception $e) {
                }

                $query->orderBy('date', 'desc');
            },
        ]);

        $datasets = $project->drivers->reduce(function (array $carry, KeyDriver $driver) use ($request) {
            foreach ($driver->items as $item) {
                $data = [];

                if ($item->records->count() > 0) {
                    $data[] = [
                        'label' => $item->range ? 'Max Value' : 'Value',
                        'elements' => [
                            'line' => [
                                'backgroundColor' => '#14b8a6',
                                'borderColor' => '#5eead4',
                            ],
                        ],
                        'data' => $item->records->reduce(function (array $carry, ItemRecord $record) {
                            $carry[] = [
                                'x' => $record->date->toDateString(),
                                'y' => $record->max,
                                'type' => $record->type,
                                'comments' => $record->comments,
                                'attachments' => $record->attachments,
                            ];

                            return $carry;
                        }, []),
                    ];

                    if ($item->range) {
                        $data[] = [
                            'label' => 'Min value',
                            'elements' => [
                                'line' => [
                                    'backgroundColor' => '#f43f5e',
                                    'borderColor' => '#fda4af',
                                ],
                            ],
                            'data' => $item->records->reduce(function (array $carry, ItemRecord $record) {
                                $carry[] = [
                                    'x' => $record->date->toDateString(),
                                    'y' => $record->min,
                                    'type' => $record->type,
                                    'comments' => $record->comments,
                                    'attachments' => $record->attachments,
                                ];

                                return $carry;
                            }, []),
                        ];
                    }

                    if (! is_null($item->max)) {
                        $data[] = [
                            'elements' => [
                                'line' => [
                                    'borderColor' => '#a5b4fc',
                                    'borderDash' => [10, 5],
                                    'borderDashOffset' => 4,
                                ],
                                'point' => [
                                    'radius' => 0,
                                    'hoverRadius' => 0,
                                    'hitRadius' => 0,
                                ],
                            ],
                            'data' => [
                                [
                                    'x' => $request->get('from'),
                                    'y' => $item->max,
                                ],
                                [
                                    'x' => $request->get('to'),
                                    'y' => $item->max,
                                ],
                            ],
                        ];
                    }

                    if (! is_null($item->min)) {
                        $data[] = [
                            'elements' => [
                                'line' => [
                                    'borderColor' => '#f0abfc',
                                    'borderDash' => [10, 5],
                                    'borderDashOffset' => 4,
                                ],
                                'point' => [
                                    'radius' => 0,
                                    'hoverRadius' => 0,
                                    'hitRadius' => 0,
                                ],
                            ],
                            'data' => [
                                [
                                    'x' => $request->get('from'),
                                    'y' => $item->min,
                                ],
                                [
                                    'x' => $request->get('to'),
                                    'y' => $item->min,
                                ],
                            ],
                            'pointRadius' => 0,
                        ];
                    }
                }

                $carry[$driver->name.' - '.$item->name] = $data;
            }

            return $carry;
        }, []);

        return response()->json(['data' => $datasets]);
    }

    public function line(Project $project, KeyDriver $driver, DriverItem $item, Request $request): JsonResponse
    {
        $records = $item->records()
            ->whereDateBetween([
                'from' => $request->get('from'),
                'to' => $request->get('to'),
            ])
            ->orderBy('date', 'desc')
            ->get();

        $datasets = [];

        if ($records->count() > 0) {
            $datasets[] = [
                'label' => $item->range ? 'Max Value' : 'Value',
                'elements' => [
                    'line' => [
                        'backgroundColor' => '#14b8a6',
                        'borderColor' => '#5eead4',
                    ],
                ],
                'data' => $records->reduce(function (array $carry, ItemRecord $record) {
                    $carry[] = [
                        'x' => $record->date->toDateString(),
                        'y' => $record->max,
                        'type' => $record->type,
                        'comments' => $record->comments,
                        'attachments' => $record->attachments,
                    ];

                    return $carry;
                }, []),
            ];

            if ($item->range) {
                $datasets[] = [
                    'label' => 'Min value',
                    'elements' => [
                        'line' => [
                            'backgroundColor' => '#f43f5e',
                            'borderColor' => '#fda4af',
                        ],
                    ],
                    'data' => $records->reduce(function (array $carry, ItemRecord $record) {
                        $carry[] = [
                            'x' => $record->date->toDateString(),
                            'y' => $record->min,
                            'type' => $record->type,
                            'comments' => $record->comments,
                            'attachments' => $record->attachments,
                        ];

                        return $carry;
                    }, []),
                ];
            }

            if (! is_null($item->max)) {
                $datasets[] = [
                    'elements' => [
                        'line' => [
                            'borderColor' => '#bfdbfe',
                            'borderDash' => [10, 5],
                            'borderDashOffset' => 4,
                        ],
                        'point' => [
                            'radius' => 0,
                            'hoverRadius' => 0,
                            'hitRadius' => 0,
                        ],
                    ],
                    'data' => [
                        [
                            'x' => $request->get('from'),
                            'y' => $item->max,
                        ],
                        [
                            'x' => $request->get('to'),
                            'y' => $item->max,
                        ],
                    ],
                ];
            }

            if (! is_null($item->min)) {
                $datasets[] = [
                    'elements' => [
                        'line' => [
                            'borderColor' => '#fecaca',
                            'borderDash' => [10, 5],
                            'borderDashOffset' => 4,
                        ],
                        'point' => [
                            'radius' => 0,
                            'hoverRadius' => 0,
                            'hitRadius' => 0,
                        ],
                    ],
                    'data' => [
                        [
                            'x' => $request->get('from'),
                            'y' => $item->min,
                        ],
                        [
                            'x' => $request->get('to'),
                            'y' => $item->min,
                        ],
                    ],
                    'pointRadius' => 0,
                ];
            }
        }

        return response()->json(['data' => $datasets]);
    }
}
