%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code on Design of Experiments (DOE) for PIV uncertainty          % 
% quantification                                                          %
% Authors: Sagar Adatrao and Andrea Sciacchitano                          %
% Date: 10 December 2021                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Inputs:                                                                 %
%   - Number of design factors (Maximum 5)                                %
%   - Number of levels of each factor (2 or 3)                            %
%   - Number of blocks (Maximum 5 and 1 in case of no blocking)           %
%   - Number of replicates in each block (Minimum 1)                      %
%   - Run sequence with respective levels of the factors and blocks       %
%     [e.g. low (-1) and high (1) or low (1) and high (2)]                %
%   - Response variable measured for each run (e.g. mean velocity field)  %
% Outputs:                                                                %
%   - ANOVA table                                                         %
%   - Total and systematic uncertainties in the response variable         %
% Note:                                                                   %
%   - The analysis is limited for the factors with same number of levels  %
%     (either 2 or 3)                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clc; clear; close all;

%% Inputs

numFactors=4; % Number of factors (Maximum 5)
numLevels=2; % Number of levels of each factor (2 or 3)
numBlocks=2; % Number of blocks (Maximum 5 and 1 in case of no blocking)
numRepl=1; % Number of replicates in each block (Minimum 1)
numRuns=numBlocks*numRepl*(numLevels^numFactors); % Number of runs

doeMatrix(:,1)=[1;-1;1;1;-1;1;1;-1;-1;1;-1;1;1;-1;-1;-1;1;1;-1;-1;-1;1;...
    -1;-1;-1;1;1;1;-1;-1;1;1]; % Levels of A           
doeMatrix(:,2)=[-1;1;-1;1;-1;1;-1;-1;1;1;1;1;-1;-1;-1;1;1;-1;-1;-1;-1;1;...
    1;-1;1;1;-1;1;1;1;-1;-1]; % Levels of B
doeMatrix(:,3)=[1;1;1;-1;-1;-1;-1;1;1;1;-1;1;-1;-1;1;-1;1;1;-1;1;-1;-1;...
    1;1;-1;-1;-1;1;1;-1;1;-1]; % Levels of C 
doeMatrix(:,4)=[-1;-1;1;-1;-1;1;1;-1;1;1;1;-1;-1;1;1;-1;-1;1;-1;1;1;-1;...
    1;-1;-1;1;-1;1;-1;1;-1;1]; % Levels of D  
doeMatrix(:,5)=NaN(numRuns,1);
            % Levels of E => NaN as there is no factor E in this experiment
doeMatrix(:,6)=[1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;1;2;2;2;2;2;2;2;2;2;2;2;2;...
    2;2;2;2]; % Levels of block

uInf=10; % Free stream velocity [m/s]

datFileSmallWindow=('D:\data\B1_Run4\B00001.DAT'); 
                    % Data file of velocity field (higher spatial 
                    % resolution corresponding to smaller window size)
sizeMatrixSmallWindow=sizeOfMatrix(datFileSmallWindow); 
                    % Size of the velocity matrix
numColumnsSmallWindow=sizeMatrixSmallWindow(1); 
numRowsSmallWindow=sizeMatrixSmallWindow(2);
[xSmallWindow,ySmallWindow]=readCoordinates(datFileSmallWindow,...
    numRowsSmallWindow,numColumnsSmallWindow);

datFileBigWindow=('D:\data\B1_Run1\B00001.DAT'); 
                    % Data file of velocity field (lower spatial 
                    % resolution corresponding to bigger window size)
sizeMatrixBigWindow=sizeOfMatrix(datFileBigWindow); 
                    % Size of the velocity matrix
numColumnsBigWindow=sizeMatrixBigWindow(1); 
numRowsBigWindow=sizeMatrixBigWindow(2);
[xBigWindow,yBigWindow]=readCoordinates(datFileBigWindow,...
    numRowsBigWindow,numColumnsBigWindow);

run=1;
u=zeros(numRuns,numRowsBigWindow,numColumnsBigWindow); 
                    % Mean of X-velocity component  
Ruu=zeros(numRuns,numRowsBigWindow,numColumnsBigWindow); 
                    % Reynolds normal stress for X-velocity component

for i=1:numBlocks
    for j=1:(numRuns/numBlocks) % Reading data  
        datFileMean=['D:\data\B',num2str(i),'_Run',num2str(j),'\B00001.DAT']; 
                                                      % Mean
        datFileSD=['D:\data\B',num2str(i),'_Run',num2str(j),'\B00002.DAT']; 
                                                      % Standard deviation
        
        if doeMatrix(run,3)==-1 % smaller window
            [Vx,Vy]=readData(datFileMean,numRowsSmallWindow,numColumnsSmallWindow);
            [SDx,SDy]=readData(datFileSD,numRowsSmallWindow,numColumnsSmallWindow);
            u(run,:,:)=griddata(xSmallWindow,ySmallWindow,Vx,xBigWindow,yBigWindow); 
                        % Interpolation to convert to lower resolution data
            Rxx=SDx.^2;
            Ruu(run,:,:)=griddata(xSmallWindow,ySmallWindow,Rxx,xBigWindow,yBigWindow); 
        elseif doeMatrix(run,3)==1 % bigger window
            [Vx,Vy]=readData(datFileMean,numRowsBigWindow,numColumnsBigWindow);
            [SDx,SDy]=readData(datFileSD,numRowsBigWindow,numColumnsBigWindow);
            u(run,:,:)=Vx; % No interpolation
            Rxx=SDx.^2; 
            Ruu(run,:,:)=Rxx; % No interpolation          
        end
        
        run=run+1;
    end
end
                        
%% ANOVA calculations

x=xBigWindow;
y=yBigWindow;
numRows=numRowsBigWindow;
numColumns=numColumnsBigWindow;

anovaResults={zeros(numRows,numColumns)}; % ANOVA tables at all points 
avg=zeros(numRows,numColumns); % Average over all the runs 
U=zeros(numRows,numColumns); % Uncertainty in the average over all the runs
fA=zeros(numRows,numColumns); % F statistic for effect A 
fB=zeros(numRows,numColumns); % F statistic for effect B 
fC=zeros(numRows,numColumns); % F statistic for effect C  
fD=zeros(numRows,numColumns); % F statistic for effect D 
fE=zeros(numRows,numColumns); % F statistic for effect E 
fAB=zeros(numRows,numColumns); % F statistic for effect AB 
fAC=zeros(numRows,numColumns); % F statistic for effect AC 
fAD=zeros(numRows,numColumns); % F statistic for effect AD 
fAE=zeros(numRows,numColumns); % F statistic for effect AE 
fBC=zeros(numRows,numColumns); % F statistic for effect BC 
fBD=zeros(numRows,numColumns); % F statistic for effect BD 
fBE=zeros(numRows,numColumns); % F statistic for effect BE 
fCD=zeros(numRows,numColumns); % F statistic for effect CD 
fCE=zeros(numRows,numColumns); % F statistic for effect CE 
fDE=zeros(numRows,numColumns); % F statistic for effect DE 
fBlocks=zeros(numRows,numColumns); % F statistic for effect of blocks 

for r=1:numRows
    for c=1:numColumns
        doeMatrix(:,7)=u(:,r,c); 
                    % Response variable- mean of X-velocity component
%         doeMatrix(:,7)=Ruu(:,r,c); 
%                     % Response variable- Reynolds normal stress for X-velocity component
        
        if numLevels==2
            [anovaTable,avgGrand,UTotal]=anova_2k(numFactors,numBlocks,...
                numRepl,doeMatrix); % ANOVA calculations for 2 levels
        elseif numLevels==3
            [anovaTable,avgGrand,UTotal]=anova_3k(numFactors,numBlocks,...
                numRepl,doeMatrix); % ANOVA calculations for 3 levels
        end
        anovaResults{r,c}=anovaTable; % ANOVA tables at all points 
        avg(r,c)=avgGrand; % Average over all the runs 
        U(r,c)=UTotal; % Uncertainty in the average over all the runs
        
        for p=2:size(anovaTable,1)-2 % Matrices for F statistic
            if anovaTable{p,1}=="A"
                fA(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="B"
                fB(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="C"
                fC(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="D"
                fD(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="E"
                fE(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="AB"
                fAB(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="AC"
                fAC(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="AD"
                fAD(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="AE"
                fAE(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="BC"
                fBC(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="BD"
                fBD(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="BE"
                fBE(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="CD"
                fCD(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="CE"
                fCE(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="DE"
                fDE(r,c)=anovaTable{p,5};
            elseif anovaTable{p,1}=="Block"
                fBlocks(r,c)=anovaTable{p,5};  
            end
        end
         
    end
end
