#!/usr/bin/env python2
# -*- coding: utf-8 -*-
"""
Created on Sun Sep  1 16:39:40 2019

@author: alessandro
"""

import numpy as np
import requests
from bs4 import BeautifulSoup
import lxml.html as lh
from lxml import etree
import pandas as pd
import os
import openpyxl
import re
import time
import igraph as ig
import random
from collections import Counter
import matplotlib.pyplot as plt
plt.close('all')

def Haversine_distance(lat1,lon1,lat2,lon2,R):
    
    lat1 = lat1*np.pi/180;
    lon1 = lon1*np.pi/180;
    lat2 = lat2*np.pi/180;
    lon2 = lon2*np.pi/180;
    
    hav_distance = 2*R*np.arcsin(np.sqrt(np.sin((lat2-lat1)/2)*
                                         np.sin((lat2-lat1)/2)+np.cos(lat1)*
                                         np.cos(lat2)*np.sin((lon2-lon1)/2)*
                                         np.sin((lon2-lon1)/2)));
    
    return hav_distance;

R = 6378 # [km]

############################
### Loading all airports ###
############################
cwd         = os.getcwd()

wb          = openpyxl.load_workbook(os.path.join(cwd,'all_airports.xlsx'))
sheet       = wb['Airports']
eta         = sheet.max_row

airportCodeIATA  = []
airportCodeICAO  = []

print('Loading all airports...') 
for row in range(2, eta+1):
    thisairportCodeIATA     = sheet['A' + str(row)].value[0:3]
    thisairportCodeICAO     = sheet['B' + str(row)].value[0:4]
    airportCodeIATA.append(thisairportCodeIATA)
    airportCodeICAO.append(thisairportCodeICAO)
    
nodesNetwork            = []
latNodesNetwork         = []
lonNodesNetwork         = []
countryNodesNetwork     = []
tradeRegionNodesNetwork = []
continentNodesNetwork   = []
multiplierNodesNetwork  = []

for row in range(2, eta+1):
    thisNode        = sheet['A' + str(row)].value
    thistradeRegion = sheet['C' + str(row)].value
    thisContinent   = sheet['F' + str(row)].value
    
    if thistradeRegion[0:2] == 'US':
        thisCountry = 'US'
    elif thistradeRegion[0:2] == 'CN':
        thisCountry = 'CN'
    elif thistradeRegion[0:2] == 'CA':
        thisCountry = 'CA'
    else:
        thisCountry = thistradeRegion
    
    thisLat         = sheet['D' + str(row)].value
    thisLon         = sheet['E' + str(row)].value
    
    if thisContinent == 'EU' or thisContinent == 'NA':
        thisMultiplier = 1
    elif thisContinent == 'AS':
        if thisCountry in ['CN','HK','JP','KR','TW']:
            thisMultiplier = 1
        else:
            thisMultiplier = 3
    elif thisContinent == 'OC' or thisContinent == 'SA':
        thisMultiplier = 2
    else:
        thisMultiplier = 4
    
    nodesNetwork.append(thisNode)
    latNodesNetwork.append(thisLat)
    lonNodesNetwork.append(thisLon)
    countryNodesNetwork.append(thisCountry)
    tradeRegionNodesNetwork.append(thistradeRegion)
    continentNodesNetwork.append(thisContinent)
    multiplierNodesNetwork.append(thisMultiplier)
    
print('Loading all airports: done')
print('%%%%%%%%%%%%%%%%%')
print('')

# Get full list of (OD airport pairs,date) tuples
full_list           = os.listdir(os.path.join(cwd,'FedEx_Network'))
included_extensions = ['xlsx']
# Determine subset of airports that are part of the FedEx network
airportsInFedExNetwork = []
for i in range(0,len(airportCodeIATA)):
    
    this_airport        = airportCodeIATA[i]
    instances           = [fn for fn in full_list
                               if any(fn.endswith(ext) for ext in included_extensions) 
                               and (this_airport+'_' in fn or
                                    '_'+this_airport in fn)]
                               
    if len(instances) > 0:
        airportsInFedExNetwork.append(i)
       
# Generate new file with only airports part of the FedEx network, and re-number
# them so that they are sequential
airportsFedEx          = []
lon_airp_FedEx         = []
lat_airp_FedEx         = []
airportsFedEx_IATACode = []
cont           = 0
for i in range(0,len(airportCodeIATA)):
    if i in airportsInFedExNetwork:
        airportsFedEx.append([cont,airportCodeIATA[i],i])
        lon_airp_FedEx.append(lonNodesNetwork[i])
        lat_airp_FedEx.append(latNodesNetwork[i])
        airportsFedEx_IATACode.append(airportCodeIATA[i])
        cont += 1

# Save file to Excel
df = pd.DataFrame(np.array(airportsFedEx))
df.to_excel(os.path.join(cwd,'FedEx_Airports.xlsx'),
sheet_name='Sheet_name_1',header=False,index=False)

# Loading full freighter data
wb          = openpyxl.load_workbook(os.path.join(cwd,'FedEx_aircraft.xlsx'))
sheet       = wb['Sheet1']
eta         = sheet.max_row
freighterCode      = []
freighterMaxTonnes = []

print('Loading all freighters...') 
for row in range(2, eta+1):
    thisfreighterCode      = sheet['A' + str(row)].value
    thisfreighterMaxTonnes = sheet['B' + str(row)].value
    freighterCode.append(thisfreighterCode)
    freighterMaxTonnes.append(float(thisfreighterMaxTonnes))
print('Loading all freighters: done')
print('%%%%%%%%%%%%%%%%%')
print('')

Multiplier                      = 1
FedExSubnetworkEdgesInfo        = []
FedExSubnetworkEdgesIdx         = []
FedExSubnetworkEdgesCapacity    = []

# Associate each airport OD pair with the estimated AFT that is the
# weight of the edge connecting such OD pair
for i in range(0,len(airportsFedEx)):
    origin_airport = airportsFedEx[i][1]
    for j in range(0,len(airportsFedEx)): 
        if j!=i:
            destination_airport = airportsFedEx[j][1]
            instances  = [fn for fn in full_list
                         if any(fn.endswith(ext) for ext in included_extensions) 
                         and origin_airport+'_'+destination_airport in fn]
            
            if len(instances) > 0:
            
                thisODEdgeCargoFlow = 0
                for this_instance in instances:
                
                    thisOD              = os.path.join(cwd,'FedEx_Network',this_instance)
                        
                    
                    wb = openpyxl.load_workbook(thisOD)
                    sheet       = wb['Sheet_name_1']
                    eta         = sheet.max_row
                    
                    all_ac_types = []
                    # Determine most common aircraft type for this instance
                    for row in range(2, eta+1):
                        all_ac_types.append(sheet['C' + str(row)].value)
                    c = Counter(all_ac_types)
                    most_common_ac = c.most_common(1)[0][0]
                    if most_common_ac == 'N/A':
                        avgMaxTonnes = np.mean(freighterMaxTonnes)
                    else:
                        if len(np.where(np.array(freighterCode) == most_common_ac)[0])>0:
                            avgMaxTonnes = freighterMaxTonnes[np.where(np.array(freighterCode) == most_common_ac)[0][0]]
                    
                    
                    
                    for row in range(2, eta+1):
                        thisAircraftType = sheet['C' + str(row)].value
                        
                        if thisAircraftType in freighterCode:
                            idxAircraft          = np.where(np.array(freighterCode) == thisAircraftType)[0][0]
                            thisODEdgeCargoFlow += freighterMaxTonnes[idxAircraft]
                        else:
                            if thisAircraftType == 'N/A':
                                thisODEdgeCargoFlow += avgMaxTonnes
                
                if thisODEdgeCargoFlow>0:
                    FedExSubnetworkEdgesInfo.append([airportsFedEx[i][1]+'-'+airportsFedEx[j][1],i,j,thisODEdgeCargoFlow*Multiplier])
                    FedExSubnetworkEdgesIdx.append(airportsFedEx[i][1]+'-'+airportsFedEx[j][1])
                    FedExSubnetworkEdgesCapacity.append(thisODEdgeCargoFlow*Multiplier)

# Create graph representing FedEx network
G = ig.Graph(directed=True)

# Add nodes
for i in range(0,len(airportsFedEx)):
    idxAirport = airportsFedEx[i][0]
    G.add_vertex(name=airportsFedEx[i][1],airportCountry=countryNodesNetwork[airportsFedEx[idxAirport][2]],
                 latAirport=latNodesNetwork[airportsFedEx[idxAirport][2]],lonAirport=lonNodesNetwork[airportsFedEx[idxAirport][2]],idx=i) 
# Add edges
for i in range(0,int(len(FedExSubnetworkEdgesInfo))):    
    G.add_edge(source=int(FedExSubnetworkEdgesInfo[i][1]),target=int(FedExSubnetworkEdgesInfo[i][2]),
                      weight=FedExSubnetworkEdgesInfo[i][3],
                      latTailAirport=latNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][1])][2]],
                      lonTailAirport=lonNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][1])][2]],
                      latHeadAirport=latNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][2])][2]],
                      lonHeadAirport=lonNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][2])][2]],
                      haversineDistance = Haversine_distance(latNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][1])][2]],
                                                      lonNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][1])][2]],
                                                      latNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][2])][2]],
                                                      lonNodesNetwork[airportsFedEx[int(FedExSubnetworkEdgesInfo[i][2])][2]],R))

correctIndexing = []
for i in range(0,int(len(FedExSubnetworkEdgesInfo))):
    firstAirport     = FedExSubnetworkEdgesInfo[i][0][0:3]
    secondAirport    = FedExSubnetworkEdgesInfo[i][0][4:7]
    idxFirstAirport  = np.where(np.array(airportsFedEx_IATACode)==firstAirport)[0][0]
    idxSecondAirport = np.where(np.array(airportsFedEx_IATACode)==secondAirport)[0][0]
    correctIndexing.append([FedExSubnetworkEdgesInfo[i][0],idxFirstAirport,idxSecondAirport,FedExSubnetworkEdgesInfo[i][3],
                            G.es[i]['haversineDistance']])
    
df = pd.DataFrame(np.array(correctIndexing))
df.to_excel(os.path.join(cwd,'FedEx_Network.xlsx'),
sheet_name='Sheet_name_1')

CapacitySortedIdx   = sorted(range(len(FedExSubnetworkEdgesCapacity)), key=lambda k: 
                    FedExSubnetworkEdgesCapacity[k], reverse=True)

# Create dataframe with capacity sorted by descending order
capacity_sorted = []
for i in range(0,int(len(CapacitySortedIdx))):
    this_origin_airport_idx      = G.es[CapacitySortedIdx[i]].source
    this_destination_airport_idx = G.es[CapacitySortedIdx[i]].target
    this_origin_airport          = G.vs[this_origin_airport_idx]['name']
    this_destination_airport     = G.vs[this_destination_airport_idx]['name']
    this_capacity                = G.es[CapacitySortedIdx[i]]['weight']
    capacity_sorted.append([this_origin_airport_idx,this_destination_airport_idx,
                            this_origin_airport,this_destination_airport,
                            this_capacity])

df_capacity         = pd.DataFrame(np.array(capacity_sorted))
df_capacity.columns = ['idx_orig','idx_dest','orig_airp_code','dest_airp_code','capacity [tonnes]']
df_capacity.to_excel(os.path.join(cwd,'FedEx_Network_Capacity_Sorted.xlsx'),
sheet_name='Sheet_name_1',index=False)

# Define weights for betweenness based on capacity of each edge. Knowing the
# minimum and maximum capacity values, we define intervals such that the last
# interval (i.e., interval with highest capacity edges) has a weight of 1,
# the next one has a weight of 2, and so on so forth until the first interval
# which has a weight of n_intervals (it is just a heuristic to favor high
# capacity edges)


inDegreeFedExNetwork  = G.indegree()
outDegreeFedExNetwork = G.outdegree()

inDegreeSortedIdx   = sorted(range(len(inDegreeFedExNetwork)), key=lambda k: 
                    inDegreeFedExNetwork[k], reverse=True)
outDegreeSortedIdx  = sorted(range(len(outDegreeFedExNetwork)), key=lambda k: 
                    outDegreeFedExNetwork[k], reverse=True)

# Binary Directed Graph
adjMatrix            = np.array(G.get_adjacency().data)
adjMatrixSquare      = np.matmul(adjMatrix,adjMatrix)
adjMatPlusTranspose  = adjMatrix+np.transpose(adjMatrix)
adjMatPlusTranspose2 = np.matmul(adjMatPlusTranspose,adjMatPlusTranspose)
adjMatPlusTranspose3 = np.matmul(adjMatPlusTranspose2,adjMatPlusTranspose)
d_in      = []
d_out     = []
d_tot     = []
d_arrow   = []
for i in range(0,int(len(airportsFedEx))):
    d_in_i  = int(np.dot(np.transpose(adjMatrix[:,i]),np.ones([int(len(airportsFedEx)),1])))
    d_out_i = int(np.dot(np.transpose(adjMatrix[i,:]),np.ones([int(len(airportsFedEx)),1])))
    d_in.append(d_in_i)
    d_out.append(d_out_i)
    d_tot.append(d_in_i+d_out_i)
    d_arrow.append(adjMatrixSquare[i,i])
    
# Clustering coefficient for Binary Directed Network
CC_BDN = np.zeros([int(len(airportsFedEx)),1])

for i in range(0,int(len(airportsFedEx))):
    if (2.0*(d_tot[i]*(d_tot[i]-1.0)-2.0*d_arrow[i]))!=0:
        CC_BDN[i] = adjMatPlusTranspose3[i,i]/(2.0*(d_tot[i]*(d_tot[i]-1.0)-2.0*d_arrow[i]))
        
# Recap results 
nNodes             = len(G.vs)
nEdges             = len(G.es)
density            = 1.0*nEdges/(nNodes*(nNodes-1))
reciprocity        = G.reciprocity(ignore_loops=True,mode='ratio')
globalCC           = np.mean(CC_BDN)

giantComponent     = G.clusters().giant()
sizeGiantComponent = len(giantComponent.vs)

shortestPathMatrix = np.zeros([int(sizeGiantComponent),int(sizeGiantComponent)])
for i in range(0,int(sizeGiantComponent)):
    for j in range(0,int(sizeGiantComponent)):
        if j!=i:
            shortestPathMatrix[i][j]=giantComponent.shortest_paths_dijkstra(source=i,target=j,weights=None)[0][0] 

sumLowerTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(1,len(shortestPathMatrix)) for j in range(i)])           
sumUpperTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(len(shortestPathMatrix)) for j in range(i+1,len(shortestPathMatrix))])            
averageShortestPathLength = (sumLowerTriangularMatrix+sumUpperTriangularMatrix)/(sizeGiantComponent*(sizeGiantComponent-1))        
diamater                  = np.max(shortestPathMatrix)        

overallCapacity    = np.sum(np.array(G.es['weight']))

AFTK = 0
for i in range(0,int(len(G.es))):
    AFTK += G.es[i]['weight']*G.es[i]['haversineDistance']

# Compute degree of each airport
connDataFrame = []
for i in range(0,len(G.vs)):
    idxAll  = G.neighbors(i,mode='all')
    connDataFrame.append([int(i),int(len(idxAll))])

df       = pd.DataFrame(np.array(connDataFrame),columns = ['nodeID','deg'])
dfSorted = df.sort_values(by='deg',ascending=False)

print('Number of nodes: ' + str(nNodes))
print('Number of edges: ' + str(nEdges)) 
print('Mean degree: ' + str(np.mean(dfSorted.deg)))   
print('Density: ' + str(density))  
print('Reciprocity: ' + str(reciprocity))  
print('Global clustering coefficient: ' + str(globalCC))  
print('Size of giant component: ' + str(sizeGiantComponent)) 
print('Average shortest path of giant component: ' + str(averageShortestPathLength)) 
print('Diameter of giant component: ' + str(diamater))  
print('Overall capacity [tonnes]: ' + str(overallCapacity)) 
print('AFTK [tonnes*km]: ' + str(AFTK)) 

connDataFrameWithAirportCode = []
for i in range(0,len(G.vs)):
    idxAll  = G.neighbors(i,mode='all')
    connDataFrameWithAirportCode.append([i,G.vs[i]['name'],int(len(idxAll))])

dfAirpCode       = pd.DataFrame(np.array(connDataFrameWithAirportCode),columns = ['nodeID','airportCode','deg'])
dfAirpCode.deg   = pd.to_numeric(dfAirpCode.deg, errors='coerce')
dfAirpCodeSorted = dfAirpCode.sort_values(by='deg',ascending=False)
dfAirpCodeSorted = dfAirpCodeSorted.reset_index(drop=True)
dfAirpCodeSorted.to_excel(os.path.join(cwd,'FedEx_degree_Sorted.xlsx'),
sheet_name='Sheet_name_1',index=False)

degreeVec = []
for i in range(0,len(G.vs)):
    degreeVec.append(df['deg'][i])
    
degreeVec = np.array(degreeVec)
pK        = []
pCum      = []
kDegree   = []

for i in range(1,np.max(degreeVec)+1):
    kDegree.append(i)
    pK.append((len(np.where(degreeVec == i)[0]))*1.0/len(G.vs))
    thisP = (len(G.vs)-len(np.where(degreeVec < i)[0]))*1.0/len(G.vs)
    pCum.append(thisP)

#%%
color1 = (255.0/255,179.0/255,25.0/255)
axis_font  = {'fontname':'Arial', 'size':'14'}
text_font  = {'fontname':'Arial', 'size':'12'}

fig,ax = plt.subplots()   
hist, bins = np.histogram(degreeVec, bins=int(np.round(np.max(degreeVec)/2)))
width = 1 * (bins[1] - bins[0])
center = (bins[:-1] + bins[1:]) / 2
plt.bar(center, hist, align='center', width=width, color=color1)
# Plot IATA code of first 3 airports
for i in range(0,3):
    this_airp_code = dfAirpCodeSorted.loc[i]['airportCode']
    this_degree    = dfAirpCodeSorted.loc[i]['deg']
    plt.text((0.85+0.1*random.random())*this_degree,2,this_airp_code,**text_font)


ax.grid(True)
ax.set_xlabel(r'$k$',**axis_font)
ax.set_ylabel('Number of occurrences',**axis_font)
ax.set_xticks(np.arange(0,np.round(1.05*np.max(degreeVec)),5),minor=True)
plt.show()
plt.savefig('FedEx_deg_dist.png',dpi=600,bbox_inches='tight', 
              transparent=True,
              pad_inches=0.1) 

#%%

fig,ax = plt.subplots()
plt.plot(kDegree,pK,c='r',marker='o',markersize=4,linestyle='',label='P(=k)',zorder=3)
plt.plot(kDegree,pCum,c=color1,marker='o',markersize=4,linestyle='',label='P(>k)',zorder=2)
ax.set_xlabel(r'$k$',**axis_font)
ax.set_ylabel('P(>k)',**axis_font)
ax.grid(True)
handles, labels = ax.get_legend_handles_labels()
# sort both labels and handles by labels
labels, handles = zip(*sorted(zip(labels, handles), key=lambda t: t[0]))
ax.legend(handles, labels)
ax.legend(loc='upper right')
plt.show()
#plt.savefig('cumDistDegree_linear.png',dpi=1000,bbox_inches='tight', 
#               transparent=True,
#               pad_inches=0.1) 

# Saving cumulative degree distribution to file
cum_degree_dist = np.array(np.vstack([kDegree,pCum])).transpose()
df_cum_deg = pd.DataFrame(cum_degree_dist)
df_cum_deg.to_excel(os.path.join(cwd,'FedEx_cum_degree_dist.xlsx'),
sheet_name='Sheet_name_1',header=False,index=False)

    
fig,ax = plt.subplots()
plt.plot(kDegree,np.array(kDegree)**(-1.5),c='k',linestyle='-',label='P(=k)',zorder=4)
plt.plot(kDegree,pK,c='r',marker='o',markersize=4,linestyle='',label='P(=k)',zorder=3)
plt.plot(kDegree,pCum,c=color1,marker='o',markersize=4,linestyle='',label='P(>k)',zorder=2)
axis_font  = {'fontname':'Arial', 'size':'14'}
ax.set_xlabel(r'$k$',**axis_font)
ax.set_ylabel(r'$P( \geq k)$',**axis_font)
ax.set_xscale('log')
ax.set_yscale('log')
ax.grid(True)
handles, labels = ax.get_legend_handles_labels()
# sort both labels and handles by labels
labels, handles = zip(*sorted(zip(labels, handles), key=lambda t: t[0]))
ax.legend(handles, labels)
ax.legend(loc='lower left')
plt.show()
#plt.savefig('cumDistDegree_loglog.png',dpi=1000,bbox_inches='tight', 
#               transparent=True,
#               pad_inches=0.1)

#%%
#Detect communities in the FedEx network using the Infomap method
cl       = G.community_infomap(edge_weights='weight', vertex_weights=None, trials=100)
#cl       = G.community_infomap()
clusters = cl.membership

n_clusters = np.max(clusters)+1
for i in range(0,n_clusters):
    idx_this_cluster = np.where(np.array(clusters)==i)

    
from matplotlib.lines import Line2D
all_markers = list(Line2D.markers)

import colorsys
N          = 30
HSV_tuples = [(random.random(), random.random(), random.random()) for x in range(N)]
RGB_tuples = list(map(lambda x: colorsys.hsv_to_rgb(*x), HSV_tuples))

#############################
### Plot FedEx subnetwork ###
#############################
import cartopy
import cartopy.crs as ccrs
import matplotlib.pyplot as plt
import matplotlib.ticker as mticker
from cartopy.mpl.gridliner import LONGITUDE_FORMATTER, LATITUDE_FORMATTER

maxFlow                  = 0
undirectedEdges          = []
cargoFlowUndirectedEdges = []
for i in range(0,len(G.es)):
    tailNode          = G.get_edgelist()[i][0]
    headNode          = G.get_edgelist()[i][1]
    
    if [tailNode,headNode] in undirectedEdges or [headNode,tailNode] in undirectedEdges:
        pass
    else:
        
        
        thisEdgeTradeFlow = G.es[i]['weight']
        
        if G.get_eid(headNode,tailNode,directed=True,error=False) != -1:
            thisEdgeTradeFlow += G.es[G.get_eid(headNode,tailNode,directed=True,error=False)]['weight']
        
        if thisEdgeTradeFlow>maxFlow and tailNode<len(airportCodeIATA) and headNode<len(airportCodeIATA):
            maxFlow         = thisEdgeTradeFlow
            maxFlowTailNode = tailNode
            maxFlowHeadNode = headNode
        
        undirectedEdges.append([tailNode,headNode])
        cargoFlowUndirectedEdges.append(thisEdgeTradeFlow)
        


fig, ax = plt.subplots()
ax = plt.axes(projection=ccrs.Mercator())
ax.coastlines()

ax.add_feature(cartopy.feature.LAND,linewidth=0.1,facecolor='0.8',alpha=0.9)
ax.add_feature(cartopy.feature.OCEAN,facecolor=(242./255,242./255,242./255))
ax.add_feature(cartopy.feature.COASTLINE,linewidth=0.25)
ax.add_feature(cartopy.feature.BORDERS,linestyle='-',linewidth=0.1,edgecolor='k')
ax.set_extent([-180, 180, -50, 70],ccrs.PlateCarree())
gl = ax.gridlines(crs=ccrs.PlateCarree(central_longitude=0), draw_labels=True,
                  linewidth=0.2, color='black', alpha=0.5, linestyle='--')

gl.xlabels_top  = True
gl.ylabels_left = True
gl.xlines       = True
gl.xlocator = mticker.FixedLocator([-180,-135,-90,-45,0,45,90,135,180])
gl.xformatter = LONGITUDE_FORMATTER
gl.yformatter = LATITUDE_FORMATTER
gl.xlabel_style = {'size': 8, 'color': 'black'}
gl.ylabel_style = {'size': 8, 'color': 'black'}

for i in np.arange(n_clusters):
    # Find indices of airports for this community
    idx_this_cluster = np.where(np.array(clusters)==i)
    # All longitudes of this community
    lon_this_comm  = np.array(lon_airp_FedEx)[idx_this_cluster[0]]
    # All latitudes of this community
    lat_this_comm  = np.array(lat_airp_FedEx)[idx_this_cluster[0]]
    ax.scatter(lon_this_comm,lat_this_comm,40,
              color=RGB_tuples[i],marker=all_markers[i],
              transform=ccrs.Geodetic(),zorder=101
              )    
    
plt.show()

#%%
# Show results for top-10 airports according to different criteria
n_airports_to_show = 10
###############################
### Compute degree of each node
###############################
degree          = G.degree(mode='all',loops=False)
degreeSrt       = sorted(degree, reverse=True)
degreeIdxSorted = sorted(range(len(degree)), key=lambda k: degree[k], reverse=True)

for i in range(0,n_airports_to_show):
    print(airportsFedEx[degreeIdxSorted[i]][1])
    print(degree[degreeIdxSorted[i]])
    print('%%%')
print('')

#####################################            
### Compute strength of each node ###
#####################################    
strength          = G.strength(loops=False,weights='weight') 
strengthSrt       = sorted(strength, reverse=True)
strengthIdxSorted = sorted(range(len(strength)), key=lambda k: strength[k], reverse=True)

for i in range(0,n_airports_to_show):
    print(airportsFedEx[strengthIdxSorted[i]][1])
    print(strength[strengthIdxSorted[i]])
    print('%%%')
print('')

##################################
### Compute modal contribution ###
##################################    
Lapl              = G.laplacian(weights='weight')
v,w               = np.linalg.eig(Lapl)
Gamma             = np.matmul(np.transpose(Lapl),w)
modalConnectivity = []

for j in range(0,len(G.vs)):
    thisModalConn = 0
    for k in range(0,len(G.vs)):
        thisModalConn += np.abs(Gamma[j][k])
    modalConnectivity.append(thisModalConn)

modalConnSrt      = sorted(modalConnectivity, reverse=True)
idxModalConnSrt   = sorted(range(len(modalConnectivity)), key=lambda k: modalConnectivity[k], reverse=True)

for i in range(0,n_airports_to_show):
    print(airportsFedEx[idxModalConnSrt[i]][1])
    print(modalConnectivity[idxModalConnSrt[i]])
    print('%%%')
print('')
    
#################################
### Compute average shortest path
#################################
charPathMatrix    = np.zeros([int(len(G.vs)),int(len(G.vs))])
for i in range(0,int(len(G.vs))):
    for j in range(0,int(len(G.vs))):
        if j != i:
            c1                      = G.shortest_paths_dijkstra(source=i,target=j,weights=None)[0][0]  
            charPathMatrix[i][j]    = c1

numberOccurrencesThisSteps = np.zeros([int(np.max(charPathMatrix[charPathMatrix != np.inf])),1])
for i in range(0,int(len(G.vs))):
    for j in range(0,int(len(G.vs))):
        if j != i:
            if charPathMatrix[i,j] != np.inf:
                numberOccurrencesThisSteps[int(charPathMatrix[i,j]-1)] += 1

#%%
# Determine robustness of FedEx network according to different 
# node removal strategies. In sequence, they are:
# - degree, - strength, - betweenness centrality, - weighted betweenness centrality
                
# Compute giant component
G_gc                 = G.clusters().giant()
n_airports_gc        = len(G_gc.vs)
allAirports_gc       = list(np.arange(n_airports_gc))
G_edges_gc           = []
allAirports_IATACode = []

for i in range(0,len(G_gc.vs)):
    allAirports_IATACode.append(G_gc.vs[i]['name'])
for i in range(0,len(G_gc.es)):
    G_edges_gc.append([G_gc.es[i].source,G_gc.es[i].target,G_gc.vs[G_gc.es[i].source]['name'],
                       G_gc.vs[G_gc.es[i].target]['name'],G_gc.es[i]['weight']])
    

##############
### Degree ###
##############

qDegree                           = []
sizeGiantComponentDegree          = []
airportsDiscardedDegree           = [] 
airportsDiscarded_IATACode_Degree = []
char_path_length_degree_removal   = []
diameter_degree_removal           = [] 
average_degree_degree_removal     = []  
for cont in range(0,n_airports_gc):
    qDegree.append((cont+1.0)/n_airports_gc)
    
    airportsRemaining          = list(set(allAirports_gc) - set(airportsDiscardedDegree))
    airportsRemaining_IATACode = list(set(allAirports_IATACode) - set(airportsDiscarded_IATACode_Degree))
    
    GTemp = ig.Graph(directed=True)
    for i in range(0,n_airports_gc):
        if i in airportsRemaining:
            GTemp.add_vertex(name=str(i),code=allAirports_IATACode[i])
    for i in range(0,len(G_edges_gc)):
        if G_edges_gc[i][2] in airportsRemaining_IATACode and G_edges_gc[i][3] in airportsRemaining_IATACode:
            GTemp.add_edge(source=np.where(np.array(airportsRemaining)==G_edges_gc[i][0])[0][0],
                            target=np.where(np.array(airportsRemaining)==G_edges_gc[i][1])[0][0],
                            firstAirport=str(G_edges_gc[i][2]),secondAirport=str(G_edges_gc[i][3]),
                            weight=G_edges_gc[i][4])
    
    degreeNodes  = GTemp.degree()
    idxDegreeSrt = sorted(range(len(degreeNodes)), key=lambda k: degreeNodes[k], reverse=True)
    
    #print('Removing airport %s' %(GTemp.vs[idxDegreeSrt[0]]['code']))
    
    airportsDiscardedDegree.append(int(GTemp.vs[idxDegreeSrt[0]]['name']))
    airportsDiscarded_IATACode_Degree.append(GTemp.vs[idxDegreeSrt[0]]['code'])
    
    
    if len(GTemp.vs) == 0:
        sizeGiantComponentDegree.append(0)
        break
    else:
        giantComponent_degree = GTemp.clusters().giant()       
        sizeGiantComponentDegree.append(1.0*len(giantComponent_degree.vs)/n_airports_gc)
        
        sizeGiantComponent = len(giantComponent_degree.vs)
        
        if sizeGiantComponent >= 2:
        
            shortestPathMatrix = np.zeros([int(sizeGiantComponent),int(sizeGiantComponent)])
            for i in range(0,int(sizeGiantComponent)):
                for j in range(0,int(sizeGiantComponent)):
                    if j!=i:
                        shortestPathMatrix[i][j]=giantComponent.shortest_paths_dijkstra(source=i,target=j,weights=None)[0][0] 
            
            sumLowerTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(1,len(shortestPathMatrix)) for j in range(i)])           
            sumUpperTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(len(shortestPathMatrix)) for j in range(i+1,len(shortestPathMatrix))])            
            averageShortestPathLength = (sumLowerTriangularMatrix+sumUpperTriangularMatrix)/(sizeGiantComponent*(sizeGiantComponent-1))        
            diamater                  = np.max(shortestPathMatrix) 
            
            char_path_length_degree_removal.append(averageShortestPathLength)
            diameter_degree_removal.append(diamater)
            average_degree_degree_removal.append(np.mean(giantComponent_degree.degree()))
            
        else:
            
            char_path_length_degree_removal.append(0)
            diameter_degree_removal.append(0)
            average_degree_degree_removal.append(np.mean(giantComponent_degree.degree()))
            
        
        
        
qDegree.insert(0,0)
airportsDiscarded_IATACode_Degree.insert(0,'None')
sizeGiantComponentDegree.append(0)
char_path_length_degree_removal.append(0)
diameter_degree_removal.append(0)
average_degree_degree_removal.append(0)



degree_node_removal_FedEx = np.hstack([np.array(qDegree).reshape(-1,1),
                                       np.array(airportsDiscarded_IATACode_Degree).reshape(-1,1),
                                       np.array(sizeGiantComponentDegree).reshape(-1,1),
                                       np.array(char_path_length_degree_removal).reshape(-1,1),
                                       np.array(diameter_degree_removal).reshape(-1,1),
                                       np.array(average_degree_degree_removal).reshape(-1,1)])
df_deg_rem_FedEx          = pd.DataFrame(np.array(degree_node_removal_FedEx),columns = ['q','airportCode','size_gc','L','D','k_mean'])
df_deg_rem_FedEx.to_excel(os.path.join(cwd,'FedEx_degree_removal.xlsx'),
sheet_name='Sheet_name_1',index=False)

#%%

################
### Strength ###
################

q_strength                           = []
sizeGiantComponent_strength          = []
airportsDiscarded_strength           = [] 
airportsDiscarded_IATACode_strength  = []
char_path_length_strength_removal    = []
diameter_strength_removal            = [] 
average_degree_strength_removal      = []  
for cont in range(0,n_airports_gc):
    
    airportsRemaining          = list(set(allAirports_gc) - set(airportsDiscarded_strength))
    airportsRemaining_IATACode = list(set(allAirports_IATACode) - set(airportsDiscarded_IATACode_strength))
    
    GTemp = ig.Graph(directed=True)
    for i in range(0,n_airports_gc):
        if i in airportsRemaining:
            GTemp.add_vertex(name=str(i),code=allAirports_IATACode[i])
    for i in range(0,len(G_edges_gc)):
        if G_edges_gc[i][2] in airportsRemaining_IATACode and G_edges_gc[i][3] in airportsRemaining_IATACode:
            GTemp.add_edge(source=np.where(np.array(airportsRemaining)==G_edges_gc[i][0])[0][0],
                            target=np.where(np.array(airportsRemaining)==G_edges_gc[i][1])[0][0],
                            firstAirport=str(G_edges_gc[i][2]),secondAirport=str(G_edges_gc[i][3]),
                            weight=G_edges_gc[i][4])
    
    # All vertices are disconnected
    if len(GTemp.es) == 0:
        break
        
    else:
    
        strength_nodes   = GTemp.strength(loops=False,weights='weight')
        idx_strength_srt = sorted(range(len(strength_nodes)), key=lambda k: strength_nodes[k], reverse=True)
        
        #print('Removing airport %s' %(GTemp.vs[idx_strength_srt[0]]['code']))
        
        airportsDiscarded_strength.append(int(GTemp.vs[idx_strength_srt[0]]['name']))
        airportsDiscarded_IATACode_strength.append(GTemp.vs[idx_strength_srt[0]]['code'])
        
        
        if len(GTemp.vs) == 0:
            sizeGiantComponent_strength.append(0)
            break
        else:
            giantComponent_strength = GTemp.clusters().giant()       
            sizeGiantComponent_strength.append(1.0*len(giantComponent_strength.vs)/n_airports_gc)
            
            sizeGiantComponent = len(giantComponent_strength.vs)
            
            if sizeGiantComponent >= 2:
            
                shortestPathMatrix = np.zeros([int(sizeGiantComponent),int(sizeGiantComponent)])
                for i in range(0,int(sizeGiantComponent)):
                    for j in range(0,int(sizeGiantComponent)):
                        if j!=i:
                            shortestPathMatrix[i][j]=giantComponent.shortest_paths_dijkstra(source=i,target=j,weights=None)[0][0] 
                
                sumLowerTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(1,len(shortestPathMatrix)) for j in range(i)])           
                sumUpperTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(len(shortestPathMatrix)) for j in range(i+1,len(shortestPathMatrix))])            
                averageShortestPathLength = (sumLowerTriangularMatrix+sumUpperTriangularMatrix)/(sizeGiantComponent*(sizeGiantComponent-1))        
                diamater                  = np.max(shortestPathMatrix) 
                
                char_path_length_strength_removal.append(averageShortestPathLength)
                diameter_strength_removal.append(diamater)
                average_degree_strength_removal.append(np.mean(giantComponent_strength.degree()))
                
            else:
                
                char_path_length_strength_removal.append(0)
                diameter_strength_removal.append(0)
                average_degree_strength_removal.append(np.mean(giantComponent_strength.degree()))
                
    q_strength.append((cont+1.0)/n_airports_gc)
            
        
        
        
q_strength.insert(0,0)
airportsDiscarded_IATACode_strength.insert(0,'None')
sizeGiantComponent_strength.append(0)
char_path_length_strength_removal.append(0)
diameter_strength_removal.append(0)
average_degree_strength_removal.append(0)



strength_node_removal_FedEx = np.hstack([np.array(q_strength).reshape(-1,1),
                                       np.array(airportsDiscarded_IATACode_strength).reshape(-1,1),
                                       np.array(sizeGiantComponent_strength).reshape(-1,1),
                                       np.array(char_path_length_strength_removal).reshape(-1,1),
                                       np.array(diameter_strength_removal).reshape(-1,1),
                                       np.array(average_degree_strength_removal).reshape(-1,1)])
df_strength_rem_FedEx          = pd.DataFrame(np.array(strength_node_removal_FedEx),columns = ['q','airportCode','size_gc','L','D','k_mean'])
df_strength_rem_FedEx.to_excel(os.path.join(cwd,'FedEx_strength_removal.xlsx'),
sheet_name='Sheet_name_1',index=False)

#%%

#########################################
### Unweighted betweenness centrality ###
#########################################

q_betw                           = []
sizeGiantComponent_betw          = []
airportsDiscarded_betw           = [] 
airportsDiscarded_IATACode_betw  = []
char_path_length_betw_removal    = []
diameter_betw_removal            = [] 
average_degree_betw_removal      = []  
for cont in range(0,n_airports_gc):
    
    airportsRemaining          = list(set(allAirports_gc) - set(airportsDiscarded_betw))
    airportsRemaining_IATACode = list(set(allAirports_IATACode) - set(airportsDiscarded_IATACode_betw))
    
    GTemp = ig.Graph(directed=True)
    for i in range(0,n_airports_gc):
        if i in airportsRemaining:
            GTemp.add_vertex(name=str(i),code=allAirports_IATACode[i])
    for i in range(0,len(G_edges_gc)):
        if G_edges_gc[i][2] in airportsRemaining_IATACode and G_edges_gc[i][3] in airportsRemaining_IATACode:
            GTemp.add_edge(source=np.where(np.array(airportsRemaining)==G_edges_gc[i][0])[0][0],
                            target=np.where(np.array(airportsRemaining)==G_edges_gc[i][1])[0][0],
                            firstAirport=str(G_edges_gc[i][2]),secondAirport=str(G_edges_gc[i][3]),
                            weight=G_edges_gc[i][4])
    
    # All vertices are disconnected
    if len(GTemp.es) == 0:
        break
        
    else:
    
        betw_nodes   = list(np.array(GTemp.betweenness(directed=True,weights=None))/((len(GTemp.vs)-1)*(len(GTemp.vs)-1)))
        idx_betw_srt = sorted(range(len(betw_nodes)), key=lambda k: betw_nodes[k], reverse=True)
        
        #print('Removing airport %s' %(GTemp.vs[idx_betw_srt[0]]['code']))
        
        airportsDiscarded_betw.append(int(GTemp.vs[idx_betw_srt[0]]['name']))
        airportsDiscarded_IATACode_betw.append(GTemp.vs[idx_betw_srt[0]]['code'])
        
        
        if len(GTemp.vs) == 0:
            sizeGiantComponent_betw.append(0)
            break
        else:
            giantComponent_betw = GTemp.clusters().giant()       
            sizeGiantComponent_betw.append(1.0*len(giantComponent_betw.vs)/n_airports_gc)
            
            sizeGiantComponent = len(giantComponent_betw.vs)
            
            if sizeGiantComponent >= 2:
            
                shortestPathMatrix = np.zeros([int(sizeGiantComponent),int(sizeGiantComponent)])
                for i in range(0,int(sizeGiantComponent)):
                    for j in range(0,int(sizeGiantComponent)):
                        if j!=i:
                            shortestPathMatrix[i][j]=giantComponent.shortest_paths_dijkstra(source=i,target=j,weights=None)[0][0] 
                
                sumLowerTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(1,len(shortestPathMatrix)) for j in range(i)])           
                sumUpperTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(len(shortestPathMatrix)) for j in range(i+1,len(shortestPathMatrix))])            
                averageShortestPathLength = (sumLowerTriangularMatrix+sumUpperTriangularMatrix)/(sizeGiantComponent*(sizeGiantComponent-1))        
                diamater                  = np.max(shortestPathMatrix) 
                
                char_path_length_betw_removal.append(averageShortestPathLength)
                diameter_betw_removal.append(diamater)
                average_degree_betw_removal.append(np.mean(giantComponent_betw.degree()))
                
            else:
                
                char_path_length_betw_removal.append(0)
                diameter_betw_removal.append(0)
                average_degree_betw_removal.append(np.mean(giantComponent_betw.degree()))
                
    q_betw.append((cont+1.0)/n_airports_gc)
            
        
        
        
q_betw.insert(0,0)
airportsDiscarded_IATACode_betw.insert(0,'None')
sizeGiantComponent_betw.append(0)
char_path_length_betw_removal.append(0)
diameter_betw_removal.append(0)
average_degree_betw_removal.append(0)



betw_node_removal_FedEx = np.hstack([np.array(q_betw).reshape(-1,1),
                                       np.array(airportsDiscarded_IATACode_betw).reshape(-1,1),
                                       np.array(sizeGiantComponent_betw).reshape(-1,1),
                                       np.array(char_path_length_betw_removal).reshape(-1,1),
                                       np.array(diameter_betw_removal).reshape(-1,1),
                                       np.array(average_degree_betw_removal).reshape(-1,1)])
df_betw_rem_FedEx          = pd.DataFrame(np.array(betw_node_removal_FedEx),columns = ['q','airportCode','size_gc','L','D','k_mean'])
df_betw_rem_FedEx.to_excel(os.path.join(cwd,'FedEx_betw_removal.xlsx'),
sheet_name='Sheet_name_1',index=False)

#%%

#######################################
### Weighted betweenness centrality ###
#######################################

max_capacity    = capacity_sorted[0][4]
min_capacity    = capacity_sorted[-1][4]

heur_weight_min_cap = 5
heur_weight_max_cap = 1
heur_weights = []
for i in range(0,int(len(G.es))):
    this_capacity    = G.es[i]['weight']
    this_heur_weight = heur_weight_min_cap+(heur_weight_max_cap-heur_weight_min_cap)/(max_capacity-min_capacity)*this_capacity
    G.es[i]['heur_weight'] = this_heur_weight
    heur_weights.append(this_heur_weight)
    
# Compute giant component
G_gc                 = G.clusters().giant()
n_airports_gc        = len(G_gc.vs)
allAirports_gc       = list(np.arange(n_airports_gc))
G_edges_gc           = []
allAirports_IATACode = []

for i in range(0,len(G_gc.vs)):
    allAirports_IATACode.append(G_gc.vs[i]['name'])
for i in range(0,len(G_gc.es)):
    G_edges_gc.append([G_gc.es[i].source,G_gc.es[i].target,G_gc.vs[G_gc.es[i].source]['name'],
                       G_gc.vs[G_gc.es[i].target]['name'],G_gc.es[i]['weight'],G_gc.es[i]['heur_weight']])    
    
    

q_betw                           = []
sizeGiantComponent_betw          = []
airportsDiscarded_betw           = [] 
airportsDiscarded_IATACode_betw  = []
char_path_length_betw_removal    = []
diameter_betw_removal            = [] 
average_degree_betw_removal      = []  
for cont in range(0,n_airports_gc):
    
    airportsRemaining          = list(set(allAirports_gc) - set(airportsDiscarded_betw))
    airportsRemaining_IATACode = list(set(allAirports_IATACode) - set(airportsDiscarded_IATACode_betw))
    
    GTemp = ig.Graph(directed=True)
    for i in range(0,n_airports_gc):
        if i in airportsRemaining:
            GTemp.add_vertex(name=str(i),code=allAirports_IATACode[i])
    for i in range(0,len(G_edges_gc)):
        if G_edges_gc[i][2] in airportsRemaining_IATACode and G_edges_gc[i][3] in airportsRemaining_IATACode:
            GTemp.add_edge(source=np.where(np.array(airportsRemaining)==G_edges_gc[i][0])[0][0],
                            target=np.where(np.array(airportsRemaining)==G_edges_gc[i][1])[0][0],
                            firstAirport=str(G_edges_gc[i][2]),secondAirport=str(G_edges_gc[i][3]),
                            weight=G_edges_gc[i][4],heur_weight=G_edges_gc[i][5])
    
    # All vertices are disconnected
    if len(GTemp.es) == 0:
        break
        
    else:
    
        betw_nodes   = list(np.array(GTemp.betweenness(directed=True,weights='heur_weight'))/((len(GTemp.vs)-1)*(len(GTemp.vs)-1)))
        idx_betw_srt = sorted(range(len(betw_nodes)), key=lambda k: betw_nodes[k], reverse=True)
        
        #print('Removing airport %s' %(GTemp.vs[idx_betw_srt[0]]['code']))
        
        airportsDiscarded_betw.append(int(GTemp.vs[idx_betw_srt[0]]['name']))
        airportsDiscarded_IATACode_betw.append(GTemp.vs[idx_betw_srt[0]]['code'])
        
        
        if len(GTemp.vs) == 0:
            sizeGiantComponent_betw.append(0)
            break
        else:
            giantComponent_betw = GTemp.clusters().giant()       
            sizeGiantComponent_betw.append(1.0*len(giantComponent_betw.vs)/n_airports_gc)
            
            sizeGiantComponent = len(giantComponent_betw.vs)
            
            if sizeGiantComponent >= 2:
            
                shortestPathMatrix = np.zeros([int(sizeGiantComponent),int(sizeGiantComponent)])
                for i in range(0,int(sizeGiantComponent)):
                    for j in range(0,int(sizeGiantComponent)):
                        if j!=i:
                            shortestPathMatrix[i][j]=giantComponent.shortest_paths_dijkstra(source=i,target=j,weights=None)[0][0] 
                
                sumLowerTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(1,len(shortestPathMatrix)) for j in range(i)])           
                sumUpperTriangularMatrix  = sum([shortestPathMatrix[i][j] for i in range(len(shortestPathMatrix)) for j in range(i+1,len(shortestPathMatrix))])            
                averageShortestPathLength = (sumLowerTriangularMatrix+sumUpperTriangularMatrix)/(sizeGiantComponent*(sizeGiantComponent-1))        
                diamater                  = np.max(shortestPathMatrix) 
                
                char_path_length_betw_removal.append(averageShortestPathLength)
                diameter_betw_removal.append(diamater)
                average_degree_betw_removal.append(np.mean(giantComponent_betw.degree()))
                
            else:
                
                char_path_length_betw_removal.append(0)
                diameter_betw_removal.append(0)
                average_degree_betw_removal.append(np.mean(giantComponent_betw.degree()))
                
    q_betw.append((cont+1.0)/n_airports_gc)
            
        
        
        
q_betw.insert(0,0)
airportsDiscarded_IATACode_betw.insert(0,'None')
sizeGiantComponent_betw.append(0)
char_path_length_betw_removal.append(0)
diameter_betw_removal.append(0)
average_degree_betw_removal.append(0)



betw_node_removal_FedEx = np.hstack([np.array(q_betw).reshape(-1,1),
                                       np.array(airportsDiscarded_IATACode_betw).reshape(-1,1),
                                       np.array(sizeGiantComponent_betw).reshape(-1,1),
                                       np.array(char_path_length_betw_removal).reshape(-1,1),
                                       np.array(diameter_betw_removal).reshape(-1,1),
                                       np.array(average_degree_betw_removal).reshape(-1,1)])
df_betw_rem_FedEx          = pd.DataFrame(np.array(betw_node_removal_FedEx),columns = ['q','airportCode','size_gc','L','D','k_mean'])
df_betw_rem_FedEx.to_excel(os.path.join(cwd,'FedEx_betw_weighted_removal.xlsx'),
sheet_name='Sheet_name_1',index=False)

#%%
# Plot flows between major airports using chord diagrams
    
#####################
### chord diagram ###
#####################
import matplotlib.pyplot as plt
from matplotlib.path import Path
import matplotlib.patches as patches

import numpy as np

LW = 0.3

def polar2xy(r, theta):
    return np.array([r*np.cos(theta), r*np.sin(theta)])

def hex2rgb(c):
    return tuple(int(c[i:i+2], 16)/256.0 for i in (1, 3 ,5))

def IdeogramArc(start=0, end=60, radius=1.0, width=0.2, ax=None, color=(1,0,0)):
    # start, end should be in [0, 360)
    if start > end:
        start, end = end, start
    start *= np.pi/180.
    end *= np.pi/180.
    # optimal distance to the control points
    # https://stackoverflow.com/questions/1734745/how-to-create-circle-with-b%C3%A9zier-curves
    opt = 4./3. * np.tan((end-start)/ 4.) * radius
    inner = radius*(1-width)
    verts = [
        polar2xy(radius, start),
        polar2xy(radius, start) + polar2xy(opt, start+0.5*np.pi),
        polar2xy(radius, end) + polar2xy(opt, end-0.5*np.pi),
        polar2xy(radius, end),
        polar2xy(inner, end),
        polar2xy(inner, end) + polar2xy(opt*(1-width), end-0.5*np.pi),
        polar2xy(inner, start) + polar2xy(opt*(1-width), start+0.5*np.pi),
        polar2xy(inner, start),
        polar2xy(radius, start),
        ]

    codes = [Path.MOVETO,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.LINETO,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CLOSEPOLY,
             ]

    if ax == None:
        return verts, codes
    else:
        path = Path(verts, codes)
        patch = patches.PathPatch(path, facecolor=color+(0.5,), edgecolor=color+(0.4,), lw=LW)
        ax.add_patch(patch)


def ChordArc(start1=0, end1=60, start2=180, end2=240, radius=1.0, chordwidth=0.7, ax=None, color=(1,0,0)):
    # start, end should be in [0, 360)
    if start1 > end1:
        start1, end1 = end1, start1
    if start2 > end2:
        start2, end2 = end2, start2
    start1 *= np.pi/180.
    end1 *= np.pi/180.
    start2 *= np.pi/180.
    end2 *= np.pi/180.
    opt1 = 4./3. * np.tan((end1-start1)/ 4.) * radius
    opt2 = 4./3. * np.tan((end2-start2)/ 4.) * radius
    rchord = radius * (1-chordwidth)
    verts = [
        polar2xy(radius, start1),
        polar2xy(radius, start1) + polar2xy(opt1, start1+0.5*np.pi),
        polar2xy(radius, end1) + polar2xy(opt1, end1-0.5*np.pi),
        polar2xy(radius, end1),
        polar2xy(rchord, end1),
        polar2xy(rchord, start2),
        polar2xy(radius, start2),
        polar2xy(radius, start2) + polar2xy(opt2, start2+0.5*np.pi),
        polar2xy(radius, end2) + polar2xy(opt2, end2-0.5*np.pi),
        polar2xy(radius, end2),
        polar2xy(rchord, end2),
        polar2xy(rchord, start1),
        polar2xy(radius, start1),
        ]

    codes = [Path.MOVETO,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             ]

    if ax == None:
        return verts, codes
    else:
        path = Path(verts, codes)
        patch = patches.PathPatch(path, facecolor=color+(0.5,), edgecolor=color+(0.4,), lw=LW)
        ax.add_patch(patch)

def selfChordArc(start=0, end=60, radius=1.0, chordwidth=0.7, ax=None, color=(1,0,0)):
    # start, end should be in [0, 360)
    if start > end:
        start, end = end, start
    start *= np.pi/180.
    end *= np.pi/180.
    opt = 4./3. * np.tan((end-start)/ 4.) * radius
    rchord = radius * (1-chordwidth)
    verts = [
        polar2xy(radius, start),
        polar2xy(radius, start) + polar2xy(opt, start+0.5*np.pi),
        polar2xy(radius, end) + polar2xy(opt, end-0.5*np.pi),
        polar2xy(radius, end),
        polar2xy(rchord, end),
        polar2xy(rchord, start),
        polar2xy(radius, start),
        ]

    codes = [Path.MOVETO,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             Path.CURVE4,
             ]

    if ax == None:
        return verts, codes
    else:
        path = Path(verts, codes)
        patch = patches.PathPatch(path, facecolor=color+(0.5,), edgecolor=color+(0.4,), lw=LW)
        ax.add_patch(patch)

def chordDiagram(X, ax, colors=None, width=0.1, pad=2, chordwidth=0.7):
    """Plot a chord diagram
    Parameters
    ----------
    X :
        flux data, X[i, j] is the flux from i to j
    ax :
        matplotlib `axes` to show the plot
    colors : optional
        user defined colors in rgb format. Use function hex2rgb() to convert hex color to rgb color. Default: d3.js category10
    width : optional
        width/thickness of the ideogram arc
    pad : optional
        gap pad between two neighboring ideogram arcs, unit: degree, default: 2 degree
    chordwidth : optional
        position of the control points for the chords, controlling the shape of the chords
    """
    # X[i, j]:  i -> j
    x = X.sum(axis = 1) # sum over rows
    ax.set_xlim(-1.1, 1.1)
    ax.set_ylim(-1.1, 1.1)

    if colors is None:
    # use d3.js category10 https://github.com/d3/d3-3.x-api-reference/blob/master/Ordinal-Scales.md#category10
        #colors = ['#1f77b4', '#ff7f0e', '#2ca02c', '#d62728', '#9467bd',
        #          '#8c564b', '#e377c2', '#7f7f7f', '#bcbd22', '#17becf']
        colors = ['#CD6839','#4169E1','#4682B4','#2E8B57','#228B22','#CD5C5C','#191970','#FFA500']
        if len(x) > 10:
            print('x is too large! Use x smaller than 10')
        colors = [hex2rgb(colors[i]) for i in range(len(x))]

    # find position for each start and end
    y = x/np.sum(x).astype(float) * (360 - pad*len(x))

    pos = {}
    arc = []
    nodePos = []
    start = 0
    for i in range(len(x)):
        end = start + y[i]
        arc.append((start, end))
        angle = 0.5*(start+end)
        #print(start, end, angle)
        if -30 <= angle <= 210:
            angle -= 90
        else:
            angle -= 270
        nodePos.append(tuple(polar2xy(1.1, 0.5*(start+end)*np.pi/180.)) + (angle,))
        z = (X[i, :]/x[i].astype(float)) * (end - start)
        ids = np.argsort(z)
        z0 = start
        for j in ids:
            pos[(i, j)] = (z0, z0+z[j])
            z0 += z[j]
        start = end + pad

    for i in range(len(x)):
        start, end = arc[i]
        IdeogramArc(start=start, end=end, radius=1.0, ax=ax, color=colors[i], width=width)
        start, end = pos[(i,i)]
        selfChordArc(start, end, radius=1.-width, color=colors[i], chordwidth=chordwidth*0.7, ax=ax)
        for j in range(i):
            this_color = colors[i]
            if X[i, j] > X[j, i]:
                this_color = colors[j]
            start1, end1 = pos[(i,j)]
            start2, end2 = pos[(j,i)]
            ChordArc(start1, end1, start2, end2,
                     radius=1.-width, color=this_color, chordwidth=chordwidth, ax=ax)

    #print(nodePos)
    return nodePos

##################################################
### Plotting a chord plot with some major airports   
##################################################

airports_to_consider = ['ANC','CDG','EWR','HKG','KIX','LAX','MEM','MIA']

flux = np.zeros([len(airports_to_consider),len(airports_to_consider)])

for i in range(0,len(airports_to_consider)):
    for j in range(0,len(airports_to_consider)):
        if len(df_capacity.loc[(df_capacity.orig_airp_code == airports_to_consider[i]) 
                               & (df_capacity.dest_airp_code == airports_to_consider[j])].index) > 0:
            flux[i][j] = float(df_capacity.loc[df_capacity.loc[(df_capacity.orig_airp_code == airports_to_consider[i]) 
                               & (df_capacity.dest_airp_code == airports_to_consider[j])].index[0]]['capacity [tonnes]'])

fig = plt.figure(figsize=(6,6))
ax = plt.axes([0,0,1,1])

#nodePos = chordDiagram(flux, ax, colors=[hex2rgb(x) for x in ['#666666', '#66ff66', '#ff6666', '#6666ff']])
nodePos = chordDiagram(flux, ax)
ax.axis('off')
prop = dict(fontsize=16*0.8, ha='center', va='center')
nodes = airports_to_consider
for i in range(0,len(airports_to_consider)):
    ax.text(nodePos[i][0], nodePos[i][1], nodes[i]+': '+'%.1f' % (np.sum(flux[i,:])/1000) + ' kt', rotation=nodePos[i][2], **prop)

plt.savefig("FedEx_flows.png", dpi=600,
        transparent=True,
        bbox_inches='tight', pad_inches=0.02)

#%%

#############################
### Plot FedEx subnetwork ###
#############################
import cartopy
import cartopy.crs as ccrs
import matplotlib.pyplot as plt
import matplotlib.ticker as mticker
from cartopy.mpl.gridliner import LONGITUDE_FORMATTER, LATITUDE_FORMATTER

maxFlow                  = 0
undirectedEdges          = []
cargoFlowUndirectedEdges = []
for i in range(0,len(G.es)):
    tailNode          = G.get_edgelist()[i][0]
    headNode          = G.get_edgelist()[i][1]
    
    if [tailNode,headNode] in undirectedEdges or [headNode,tailNode] in undirectedEdges:
        pass
    else:
        
        
        thisEdgeTradeFlow = G.es[i]['weight']
        
        if G.get_eid(headNode,tailNode,directed=True,error=False) != -1:
            thisEdgeTradeFlow += G.es[G.get_eid(headNode,tailNode,directed=True,error=False)]['weight']
        
        if thisEdgeTradeFlow>maxFlow and tailNode<len(airportCodeIATA) and headNode<len(airportCodeIATA):
            maxFlow         = thisEdgeTradeFlow
            maxFlowTailNode = tailNode
            maxFlowHeadNode = headNode
        
        undirectedEdges.append([tailNode,headNode])
        cargoFlowUndirectedEdges.append(thisEdgeTradeFlow)

fig, ax = plt.subplots()
ax = plt.axes(projection=ccrs.Mercator())
ax.coastlines()

ax.add_feature(cartopy.feature.LAND,linewidth=0.1,facecolor='0.8',alpha=0.9)
ax.add_feature(cartopy.feature.OCEAN,facecolor=(242./255,242./255,242./255))
ax.add_feature(cartopy.feature.COASTLINE,linewidth=0.25)
ax.add_feature(cartopy.feature.BORDERS,linestyle='-',linewidth=0.1,edgecolor='k')
ax.set_extent([-180, 180, -50, 70],ccrs.PlateCarree())
gl = ax.gridlines(crs=ccrs.PlateCarree(central_longitude=0), draw_labels=True,
                  linewidth=0.2, color='black', alpha=0.5, linestyle='--')

gl.xlabels_top = True
gl.ylabels_left = True
gl.xlines = True
gl.xlocator = mticker.FixedLocator([-180,-135,-90,-45,0,45,90,135,180])
gl.xformatter = LONGITUDE_FORMATTER
gl.yformatter = LATITUDE_FORMATTER
gl.xlabel_style = {'size': 8, 'color': 'black'}
gl.ylabel_style = {'size': 8, 'color': 'black'}

plt.scatter(lon_airp_FedEx,lat_airp_FedEx,1.5,
          color='red',marker='>',
          transform=ccrs.Geodetic(),zorder=101
          )

maxLineWidthNew = 2

for i in range(0,len(undirectedEdges)):
    thisEdgeTradeFlow = cargoFlowUndirectedEdges[i]
    if thisEdgeTradeFlow >= 0:
        tailNode          = undirectedEdges[i][0]
        headNode          = undirectedEdges[i][1]
        
        if tailNode<len(airportCodeIATA) and headNode<len(airportCodeIATA):
            
            
            tailNodeLat       = G.vs[tailNode]['latAirport']
            tailNodeLon       = G.vs[tailNode]['lonAirport']
            headNodeLat       = G.vs[headNode]['latAirport']
            headNodeLon       = G.vs[headNode]['lonAirport']
            
            thisLineWidth     = thisEdgeTradeFlow/maxFlow*maxLineWidthNew
            
            plt.plot([tailNodeLon, headNodeLon], [tailNodeLat, headNodeLat],
                      color=color1, linewidth=thisLineWidth,
                      transform=ccrs.Geodetic(),zorder=100
                      ) 

cargoFlowLegend = 50000

plt.plot([-170, -160], [-35, -35],
                      color=color1, linewidth=cargoFlowLegend/maxFlow*maxLineWidthNew,
                      transform=ccrs.Geodetic(),
                      )

plt.text(-156,-37.8, '= '+ str(cargoFlowLegend) +' ton',
          horizontalalignment='left',
          transform=ccrs.Geodetic())


plt.show() 
fig.savefig('FedEx_network.png', format='png', dpi=400, bbox_inches='tight',
             transparent=True,pad_inches=0.02)

#%%
max_capacity    = capacity_sorted[0][4]
min_capacity    = capacity_sorted[-1][4]


heur_weight_min_cap = 5
heur_weight_max_cap = 1
heur_weights = []
for i in range(0,int(len(G.es))):
    this_capacity    = G.es[i]['weight']
    this_heur_weight = heur_weight_min_cap+(heur_weight_max_cap-heur_weight_min_cap)/(max_capacity-min_capacity)*this_capacity
    G.es[i]['heur_weight'] = this_heur_weight
    heur_weights.append(this_heur_weight)
    
# Compute giant component
G_gc                 = G.clusters().giant()
n_airports_gc        = len(G_gc.vs)
allAirports_gc       = list(np.arange(n_airports_gc))
G_edges_gc           = []
allAirports_IATACode = []


for i in range(0,len(G_gc.vs)):
    allAirports_IATACode.append(G_gc.vs[i]['name'])
for i in range(0,len(G_gc.es)):
    G_edges_gc.append([G_gc.es[i].source,G_gc.es[i].target,G_gc.vs[G_gc.es[i].source]['name'],
                       G_gc.vs[G_gc.es[i].target]['name'],G_gc.es[i]['weight'],G_gc.es[i]['heur_weight']])
    
airportsDiscarded_betw           = [] 
airportsDiscarded_IATACode_betw  = []

airportsRemaining          = list(set(allAirports_gc) - set(airportsDiscarded_betw))
airportsRemaining_IATACode = list(set(allAirports_IATACode) - set(airportsDiscarded_IATACode_betw))    
    
GTemp = ig.Graph(directed=True)
for i in range(0,n_airports_gc):
    if i in airportsRemaining:
        GTemp.add_vertex(name=str(i),code=allAirports_IATACode[i])
for i in range(0,len(G_edges_gc)):
    if G_edges_gc[i][2] in airportsRemaining_IATACode and G_edges_gc[i][3] in airportsRemaining_IATACode:
        GTemp.add_edge(source=np.where(np.array(airportsRemaining)==G_edges_gc[i][0])[0][0],
                        target=np.where(np.array(airportsRemaining)==G_edges_gc[i][1])[0][0],
                        firstAirport=str(G_edges_gc[i][2]),secondAirport=str(G_edges_gc[i][3]),
                        weight=G_edges_gc[i][4],heur_weight=G_edges_gc[i][5])

betw_nodes   = list(np.array(GTemp.betweenness(directed=True,weights='heur_weight'))/((len(GTemp.vs)-1)*(len(GTemp.vs)-1)))
idx_betw_srt = sorted(range(len(betw_nodes)), key=lambda k: betw_nodes[k], reverse=True)

betw_nodes_2   = list(np.array(GTemp.betweenness(directed=True,weights=None))/((len(GTemp.vs)-1)*(len(GTemp.vs)-1)))
idx_betw_srt_2 = sorted(range(len(betw_nodes_2)), key=lambda k: betw_nodes_2[k], reverse=True)








