### Set working directory
{
  setwd("D:/Documents/papers/Elke Wenting/final model v5 by Jasper/") # change accordingly to save output at different location
}

### Install (first time use only) and load packages
{
  packages = c("deSolve", "data.table", "parallel", "doSNOW", "ggplot2")
  
  for(i in seq_along(packages)) {
    if(!require(packages[i], character.only = TRUE)) {
      install.packages(packages[i])
    }
    library(packages[i], character.only = TRUE)
  }
  
  rm(packages, i)
}

### Load functions
{
  # Wrapper function around ode()
  odeFun = function(y, times, func, parms, ...) {
    require(deSolve)
    
    y_initial = y
    
    rootfun = function(t, x, p) {
      return(min(x[y_initial!=0]))
    }
    
    states = ode(y = y,
                 times = times,
                 func = func,
                 parms = parms,
                 rootfun = rootfun,
                 ...)
    
    attr(states, "model") = func
    attr(states, "initials") = y
    attr(states, "parameters") = parms
    
    return(states)
  }
  
  # Wrapper function around ode() for phase plane, performed in parallel on CPU cores
  odePhaseFun = function(y, func, parms, vrbls = c("V","D","B","S","W")[1:2],
                         times = seq(0,100,1/12),
                         ...) {
    require(deSolve)
    require(data.table)
    require(parallel)
    require(doSNOW)
    
    if(length(vrbls)!=2) {
      stop("make sure to choose exactly 2 variables")
    }
    
    cores = (detectCores()-1)
    if(cores==0) {
      cores = 1
    }
    
    cl = makePSOCKcluster(cores, outfile="")
    registerDoSNOW(cl)
    
    pb = txtProgressBar(min=0, max=1, initial=0, char="-", style=3)
    progress = function(n) setTxtProgressBar(pb, n/(nrow(y)))
    opts = list(progress = progress)
    
    dat = foreach(i=1:nrow(y),
                  .packages=c("deSolve","data.table"),
                  .options.snow = opts) %dopar% {
                    
                    yy = as.numeric(y[i,])
                    names(yy) = colnames(y)
                    
                    y_initial = yy
                    
                    rootfun = function(t, x, p) {
                      return(min(x[y_initial!=0]))
                    }
                    
                    output = ode(y = yy,
                                 times = times,
                                 func = func,
                                 parms = parms,
                                 rootfun = rootfun,
                                 ...)
                    
                    output = as.data.table(output)[,c("time",vrbls), with = FALSE]
                    output[,i:=i]
                    output$startx = output[time==0,][[2]]
                    output$starty = output[time==0,][[3]]
                    output$endx = output[nrow(output),][[2]]
                    output$endy = output[nrow(output),][[3]]
                    output$dxy = sum(sqrt(diff(output[[2]])^2 +
                                            diff(output[[3]])^2))
                    
                    setTxtProgressBar(pb, i/(nrow(y)))
                    return(output)
                  }
    stopCluster(cl)
    
    dat = rbindlist(dat)
    
    class(dat) = c("phasePlane",class(dat))
    
    return(dat)
  }
  
  # Wrapper function around ode() for sensitivity analysis, performed in parallel on CPU cores
  odeSensFun = function(y, times, func, parmsFun, namedSeqs, ...) {
    require(deSolve)
    require(parallel)
    require(doSNOW)
    
    cores = (detectCores()-1)
    if(cores==0) {
      cores = 1
    }
    
    cl = makePSOCKcluster(cores, outfile="")
    registerDoSNOW(cl)
    
    pb = txtProgressBar(min=0, max=1, initial=0, char="-", style=3)
    progress = function(n) setTxtProgressBar(pb, n/(nrow(namedSeqs)))
    opts = list(progress = progress)
    
    sensitivity = foreach(i=1:nrow(namedSeqs),
                          .packages=c("deSolve"),
                          .options.snow = opts,
                          .export = c("odeFun")) %dopar% {
                            
                            index = rep(FALSE, times = nrow(namedSeqs))
                            index[i] = TRUE
                            
                            output = odeFun(y = y,
                                            times = times,
                                            func = func,
                                            parms = do.call(parmsFun, subset(namedSeqs, subset = index)),
                                            ...)
                            
                            setTxtProgressBar(pb, i/(nrow(namedSeqs)))
                            return(output)
                          }
    stopCluster(cl)
    
    attr(sensitivity, "sensitivity parameters") = namedSeqs
    class(sensitivity) = c(class(sensitivity), "deSolveList", "sensitivityAnalysis")
    
    return(sensitivity)
  }
  
  # Changed settings for printing output to console
  print.deSolve = function(x) {
    require(data.table)
    
    print(str(x))
    print(as.data.table(x))
    warning("only printed the structure of the deSolve object")
  }
  
  print.deSolveList = function(x) {
    require(data.table)
    
    print(str(x[[1]]))
    print(as.data.table(x[[1]]))
    warning("only printed the structure of the first deSolve object in the list")
  }
  
  # Changed settings for plotting output
  plot.deSolve = function(x,
                          vrbls = c("V","D","B","S","W"),
                          savefile = FALSE,
                          plotpx_x = 5000,
                          plotpx_y = 5000) {
    require(data.table)
    require(ggplot2)
    
    x = as.data.table(x)
    xx = list()
    for(i in seq_along(vrbls)) {
      xx[[i]] = x[,c("time",vrbls[i]), with = FALSE]
      xx[[i]][, var := vrbls[i]]
      colnames(xx[[i]])[2] = "value"
    }
    xx = rbindlist(xx)
    xx[, var := factor(var, levels = vrbls)]
    
    pp = ggplot(xx,
                aes(x = time,
                    y = value,
                    group = var)) +
      geom_line() +
      facet_wrap(. ~ var,
                 scales = "free_y") + 
      expand_limits(y = 0)
    
    if(savefile!=FALSE) {
      ggsave(paste0("plots/",savefile,".pdf"),
             pp,
             "pdf",
             width = plotpx_x,
             height = plotpx_y,
             units = "px")
    }
    
    return(pp)
  }
  
  plot.phasePlane = function(dat,
                             background = TRUE,
                             trajectory = TRUE,
                             startvalues = TRUE,
                             endvalues = TRUE,
                             savefile = FALSE,
                             plotpx_x = 5000,
                             plotpx_y = 5000) {
    require(data.table)
    require(ggplot2)
    
    ppp = ggplot(dat)
    
    if(background) {
      ppp = ppp + 
        geom_raster(aes(x = startx,
                        y = starty,
                        fill = -dxy)) +
        guides(fill = "none")
    }
    
    if(trajectory) {
      ppp = ppp +
        geom_path(aes(x = get(colnames(dat)[2]),
                      y = get(colnames(dat)[3]),
                      group = i))
    }
    
    if(startvalues) {
      ppp = ppp +
        geom_point(aes(x = startx,
                       y = starty))
    }
    
    if(endvalues) {
      ppp = ppp +
        geom_point(aes(x = endx,
                       y = endy),
                   colour = "red",
                   shape = 18,
                   size = 3)
    }
    
    ppp = ppp +
      labs(x = colnames(dat)[2],
           y = colnames(dat)[3])
    
    if(savefile!=FALSE) {
      ggsave(paste0("plots/",savefile,".pdf"),
             ppp,
             "pdf",
             width = plotpx_x,
             height = plotpx_y,
             units = "px")
    }
    
    return(ppp)
  }
  
  plot.sensitivityAnalysis = function(dat,
                                      type = c("timeEvolution","endState")[1],
                                      vrbl = c("V","D","B","S","W")[1],
                                      pars = "all",
                                      setparvalues = "median",
                                      colasfact = FALSE,
                                      savefile = FALSE,
                                      plotpx_x = 5000,
                                      plotpx_y = 5000) {
    require(deSolve)
    require(data.table)
    require(ggplot2)
    
    if(length(vrbl)!=1) {
      stop("make sure to choose 1 vrbl")
    }
    
    namedSeqs = attr(dat, "sensitivity parameters")
    if(all(pars=="all") |
       all(c(colnames(namedSeqs) %in% pars,
             pars %in% colnames(namedSeqs)))) {
      
      if(ncol(namedSeqs)>4) {
        stop("make sure to choose a maximum of 4 pars")
      }
      if(pars!="all") {
        namedSeqs = namedSeqs[,pars]
      }
      xx = copy(dat)
      for(i in seq_along(xx)) {
        xx[[i]] = cbind(xx[[i]][,c("time",vrbl)],
                        as.data.table(namedSeqs[i,,
                                                drop = FALSE]))
      }
    } else {
      if(length(pars)>4) {
        stop("make sure to choose a maximum of 4 pars")
      }
      if(all(setparvalues=="median")) {
        namedSeqs_sel = 
          apply(namedSeqs[,!(colnames(namedSeqs) %in% pars),
                          drop = FALSE],
                2,
                function(x){sort(unique(x), decreasing = TRUE)[which.min(abs(median(unique(x))-sort(unique(x), decreasing = TRUE)))]})
      } else {
        if(ncol(namedSeqs)!=(length(pars)+length(setparvalues))) {
          stop("make sure to set fixed values for all pars you don't want to plot")
        } else if(!all(names(setparvalues) %in% colnames(namedSeqs)[!(colnames(namedSeqs) %in% pars)])) {
          stop("make sure to set fixed values for all pars you don't want to plot")
        }
        namedSeqs_sel = setparvalues
      }
      
      for(i in seq_along(namedSeqs_sel)) {
        namedSeqs = namedSeqs[namedSeqs[,names(namedSeqs_sel)[i]]==namedSeqs_sel[[i]],]
      }
      xx = copy(dat[as.integer(rownames(namedSeqs))])
      namedSeqs = namedSeqs[,pars,
                            drop = FALSE]
      for(i in seq_along(xx)) {
        xx[[i]] = cbind(xx[[i]][,c("time",vrbl)],
                        as.data.table(namedSeqs[i,,
                                                drop = FALSE]))
      }
    }
    
    if(ncol(namedSeqs)==1) {
      if(type=="timeEvolution") {
        xx = rbindlist(xx)
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = get(colnames(xx)[3]),
                          col = as.factor(get(colnames(xx)[3])))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3]) +
            expand_limits(y = 0)
        } else {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = get(colnames(xx)[3]),
                          col = get(colnames(xx)[3]))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3]) +
            expand_limits(y = 0)
        }
      } else if(type=="endState") {
        for(i in seq_along(xx)) {
          xx[[i]] = xx[[i]][nrow(xx[[i]]),]
        }
        xx = rbindlist(xx)
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[2]))) +
            geom_line() +
            geom_point() +
            labs(x = colnames(xx)[3],
                 y = vrbl) +
            expand_limits(y = 0)
        } else {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[2]))) +
            geom_line() +
            geom_point() +
            labs(x = colnames(xx)[3],
                 y = vrbl) +
            expand_limits(y = 0)
        }
      }
    } else if(ncol(namedSeqs)==2) {
      if(type=="timeEvolution") {
        xx = rbindlist(xx)
        xx[[4]] = as.factor(xx[[4]])
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = interaction(get(colnames(xx)[3]),
                                              get(colnames(xx)[4])),
                          col = as.factor(get(colnames(xx)[3])),
                          linetype = get(colnames(xx)[4]))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3],
                 linetype = colnames(xx)[4]) +
            expand_limits(y = 0)
        } else {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = interaction(get(colnames(xx)[3]),
                                              get(colnames(xx)[4])),
                          col = get(colnames(xx)[3]),
                          linetype = get(colnames(xx)[4]))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3],
                 linetype = colnames(xx)[4]) +
            expand_limits(y = 0)
        }
      } else if(type=="endState") {
        for(i in seq_along(xx)) {
          xx[[i]] = xx[[i]][nrow(xx[[i]]),]
        }
        xx = rbindlist(xx)
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[4]),
                          fill = get(colnames(xx)[2]))) +
            geom_raster() +
            labs(x = colnames(xx)[3],
                 y = colnames(xx)[4],
                 fill = colnames(xx)[2])
        } else {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[4]),
                          fill = get(colnames(xx)[2]))) +
            geom_raster() +
            labs(x = colnames(xx)[3],
                 y = colnames(xx)[4],
                 fill = colnames(xx)[2])
        }
      }
    } else if(ncol(namedSeqs)==3) {
      if(type=="timeEvolution") {
        xx = rbindlist(xx)
        xx[[4]] = as.factor(xx[[4]])
        xx[[5]] = as.factor(xx[[5]])
        plotLabeller = function(variable,value) {
          return(paste(colnames(xx)[5],"=",value))
        }
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = interaction(get(colnames(xx)[3]),
                                              get(colnames(xx)[4]),
                                              get(colnames(xx)[5])),
                          col = as.factor(get(colnames(xx)[3])),
                          linetype = get(colnames(xx)[4]))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3],
                 linetype = colnames(xx)[4]) +
            facet_wrap(get(colnames(xx)[5]) ~ .,
                       labeller = plotLabeller) +
            expand_limits(y = 0)
        } else {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = interaction(get(colnames(xx)[3]),
                                              get(colnames(xx)[4]),
                                              get(colnames(xx)[5])),
                          col = get(colnames(xx)[3]),
                          linetype = get(colnames(xx)[4]))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3],
                 linetype = colnames(xx)[4]) +
            facet_wrap(get(colnames(xx)[5]) ~ .,
                       labeller = plotLabeller) +
            expand_limits(y = 0)
        }
      } else if(type=="endState") {
        for(i in seq_along(xx)) {
          xx[[i]] = xx[[i]][nrow(xx[[i]]),]
        }
        xx = rbindlist(xx)
        xx[[5]] = as.factor(xx[[5]])
        plotLabeller = function(variable,value) {
          return(paste(colnames(xx)[5],"=",value))
        }
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[4]),
                          fill = get(colnames(xx)[2]))) +
            geom_raster() +
            labs(x = colnames(xx)[3],
                 y = colnames(xx)[4],
                 fill = colnames(xx)[2]) +
            facet_wrap(get(colnames(xx)[5]) ~ .,
                       labeller = plotLabeller)
        } else {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[4]),
                          fill = get(colnames(xx)[2]))) +
            geom_raster() +
            labs(x = colnames(xx)[3],
                 y = colnames(xx)[4],
                 fill = colnames(xx)[2]) +
            facet_wrap(get(colnames(xx)[5]) ~ .,
                       labeller = plotLabeller)
        }
      }
    } else if(ncol(namedSeqs)==4) {
      if(type=="timeEvolution") {
        xx = rbindlist(xx)
        xx[[4]] = as.factor(xx[[4]])
        xx[[5]] = as.factor(xx[[5]])
        xx[[6]] = as.factor(xx[[6]])
        plotLabeller = function(variable,value) {
          if(variable=="get(colnames(xx)[5])") {
            return(paste(colnames(xx)[5],"=",value))
          } else {
            return(paste(colnames(xx)[6],"=",value))
          }
        }
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = interaction(get(colnames(xx)[3]),
                                              get(colnames(xx)[4]),
                                              get(colnames(xx)[5])),
                          col = as.factor(get(colnames(xx)[3])),
                          linetype = get(colnames(xx)[4]))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3],
                 linetype = colnames(xx)[4]) +
            facet_grid(get(colnames(xx)[6]) ~ get(colnames(xx)[5]),
                       labeller = plotLabeller) +
            expand_limits(y = 0)
        } else {
          pp = ggplot(xx,
                      aes(x = time,
                          y = get(colnames(xx)[2]),
                          group = interaction(get(colnames(xx)[3]),
                                              get(colnames(xx)[4]),
                                              get(colnames(xx)[5])),
                          col = get(colnames(xx)[3]),
                          linetype = get(colnames(xx)[4]))) +
            geom_line() +
            labs(y = vrbl,
                 colour = colnames(xx)[3],
                 linetype = colnames(xx)[4]) +
            facet_grid(get(colnames(xx)[6]) ~ get(colnames(xx)[5]),
                       labeller = plotLabeller) +
            expand_limits(y = 0)
        }
      } else if(type=="endState") {
        for(i in seq_along(xx)) {
          xx[[i]] = xx[[i]][nrow(xx[[i]]),]
        }
        xx = rbindlist(xx)
        xx[[5]] = as.factor(xx[[5]])
        xx[[6]] = as.factor(xx[[6]])
        plotLabeller = function(variable,value) {
          if(variable=="get(colnames(xx)[5])") {
            return(paste(colnames(xx)[5],"=",value))
          } else {
            return(paste(colnames(xx)[6],"=",value))
          }
        }
        
        if(colasfact) {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[4]),
                          fill = get(colnames(xx)[2]))) +
            geom_raster() +
            labs(x = colnames(xx)[3],
                 y = colnames(xx)[4],
                 fill = colnames(xx)[2]) +
            facet_grid(get(colnames(xx)[6]) ~ get(colnames(xx)[5]),
                       labeller = plotLabeller)
        } else {
          pp = ggplot(xx,
                      aes(x = get(colnames(xx)[3]),
                          y = get(colnames(xx)[4]),
                          fill = get(colnames(xx)[2]))) +
            geom_raster() +
            labs(x = colnames(xx)[3],
                 y = colnames(xx)[4],
                 fill = colnames(xx)[2]) +
            facet_grid(get(colnames(xx)[6]) ~ get(colnames(xx)[5]),
                       labeller = plotLabeller)
        }
      }
    }
    
    if(savefile!=FALSE) {
      ggsave(paste0("plots/",savefile,".pdf"),
             pp,
             "pdf",
             width = plotpx_x,
             height = plotpx_y,
             units = "px")
    }
    
    return(pp)
  }
  
}

### Set up simulation model
{
  pp = function(Time,
                State,
                Param) {
    with(as.list(c(State, Param)),{
      
      # first auxiliaries 
      Vgrowth = R0 * V * (1-(V / k0))
      
      # second auxiliaries (1)
      AxwResponse = (Axw * (D+B)) / (Bxw + (D+B))
      DpredW = ifelse(
        (D<=0) & (B<=0),
        0,
        AxwResponse * (D^2 / (D^2 + B^2)) * W
      )
      
      killedDleft = x * DpredW
      WgrowthD = Cxw * (1-x) * DpredW 
      
      # second auxiliaries (2)
      BpredW = ifelse(
        (D<=0) & (B<=0),
        0,
        AxwResponse * (B^2 / (D^2 + B^2)) * W
      )
      
      killedBleft = x * BpredW
      WgrowthB = Cxw * (1-x) * BpredW 
      
      # second auxiliaries (3)
      deathW = deltaW * W
      
      # third auxiliaries (1)
      AvdResponse = (Avd * V) / (Bxx + V)
      VconsD = AvdResponse * D
      DgrowthV = Cvd * VconsD 
      
      deathD = ifelse(
        D<=0,
        0,
        deltaD * exp(-wpr * (D^2 / (D^2 + B^2)) * (W / D)) * D
      )
      
      huntingD = ifelse(
        D<=aimpopD,
        0,
        (1-(aimpopD / D))
      ) * pHuntD * (D-aimpopD)
      
      huntedDleft = huntingD * pHDleft
      
      # fourth auxiliaries (1)
      carrionD = deathD + huntedDleft + killedDleft
      
      carrionFracBavail = ifelse(
        B<=0,
        1,
        carrionD / (Axb * B)
      )
      carrionFracBcons = (z * carrionFracBavail) / (Bz + carrionFracBavail)
      AxbCorrected = Axb * (((Cvb*(1-za) + Cdb*za)) / 
                              (Cvb*(1-carrionFracBcons) + Cdb*carrionFracBcons))
      AxbResponse = (AxbCorrected * (V*(1-carrionFracBcons) + carrionD*carrionFracBcons)) / 
        (Bxx + (V*(1-carrionFracBcons) + carrionD*carrionFracBcons))
      
      # fourth auxiliaries (2)
      VconsB = AxbResponse * (1-carrionFracBcons) * B 
      BgrowthV = Cvb * VconsB  
      
      deathB = ifelse(
        B<=0,
        0,
        deltaB * exp(-wpr * (B^2 / (D^2 + B^2)) * (W / B)) * B
      ) 
      
      huntingB = ifelse(
        B<=aimpopB,
        0,
        (1-(aimpopB / B))
      ) * pHuntB * (B-aimpopB)
      
      huntedBleft = huntingB * pHBleft
      
      # fourth auxiliaries (3)
      DscavB = AxbResponse * carrionFracBcons * B
      BgrowthD = Cdb * DscavB 
      
      # fifth auxiliaries (1)
      carrionB = deathB + huntedBleft + killedBleft
      carrionDS = carrionD - DscavB
      AxsResponse = (Axs * (V+carrionDS+carrionB)) / (Bxx + (V+carrionDS+carrionB))
      DscavS = min(AxsResponse * S, carrionDS)
      SgrowthD = Cxs * DscavS 
      
      # fifth auxiliaries (2)
      BscavS = min(AxsResponse * S, carrionB)
      SgrowthB = Cxs * BscavS
      
      # fifth auxiliaries (3)
      VconsS = AxsResponse * V/(V+carrionDS+carrionB) * S
      SgrowthV = Cvs * VconsS
      deathS = deltaS * S
      
      # sixth auxiliaries 
      huntingbagD = huntingD - huntedDleft
      huntingbagB = huntingB - huntedBleft
      
      # seventh auxiliaries
      huntkillleftfrac_01transf = 1-exp(-(huntedDleft+huntedBleft)/(killedDleft+killedBleft))
      carrionDBfrac = carrionD/carrionB
      carrionDB = carrionD+carrionB
      Vcarriongrowthfrac_01transf = 1-exp(-SgrowthV/(SgrowthD+SgrowthB))
      
      # equations
      V = Vgrowth - VconsD - VconsB - VconsS
      D = DgrowthV - DpredW - huntingD - deathD
      B = BgrowthV + BgrowthD - BpredW - huntingB - deathB
      S = SgrowthV + SgrowthD + SgrowthB - deathS 
      W = WgrowthD + WgrowthB - deathW
      
      # output
      return(list(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ),
      Vgrowth = Vgrowth,
      AvdResponse = AvdResponse,
      VconsD = VconsD,
      DgrowthV = DgrowthV,
      deathD = deathD,
      VconsB = VconsB,
      deathB = deathB,
      carrionD = carrionD,
      carrionFracBavail = carrionFracBavail,
      carrionFracBcons = carrionFracBcons,
      AxbCorrected = AxbCorrected,
      AxbResponse = AxbResponse,
      carrionDS = carrionDS,
      BgrowthV = BgrowthV,
      DscavB = DscavB,
      BgrowthD = BgrowthD,
      huntingD = huntingD,
      huntedDleft = huntedDleft,
      huntingbagD = huntingbagD,
      huntingB = huntingB,
      huntedBleft = huntedBleft,
      carrionB = carrionB,
      huntingbagB = huntingbagB,
      deathS = deathS,
      DscavS = DscavS,
      BscavS = BscavS,
      VconsS = VconsS,
      AxsResponse = AxsResponse,
      SgrowthV = SgrowthV,
      SgrowthD = SgrowthD,
      SgrowthB = SgrowthB,
      deathW = deathW,
      AxwResponse = AxwResponse,
      DpredW = DpredW,
      killedDleft = killedDleft,
      WgrowthD = WgrowthD,
      BpredW = BpredW,
      killedBleft = killedBleft,
      WgrowthB = WgrowthB,
      huntkillleftfrac_01transf = huntkillleftfrac_01transf,
      carrionDBfrac = carrionDBfrac,
      carrionDB = carrionDB,
      Vcarriongrowthfrac_01transf = Vcarriongrowthfrac_01transf
      ))
    })
  }
}

### Sensitivity analysis
{
  ## V
  {
    pp_init = function(V = 1,
                       S = 0, #NA
                       B = 0, #NA
                       D = 0, #NA
                       W = 0 #NA
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    pp_param = function(R0 = 4,
                        k0 = 10,
                        Avd = 23.5, #NA
                        Bxx = 10, #NA
                        Cvd = 0.017, #NA
                        deltaD = 0.125, #NA 
                        Axb = 20.15, #NA
                        Cvb = 0.055, #NA
                        deltaB = 0.08, #NA
                        z = 0.2, #NA
                        za = 0.16, #NA
                        Bz = 0.1, #NA
                        Cdb = 0.069, #NA
                        wpr = 10, #NA
                        aimpopD = 1, #NA
                        pHuntD = 0, #NA
                        pHDleft = 2/3, #NA
                        aimpopB = 1, #NA
                        pHuntB = 0, #NA
                        pHBleft = 2/3, #NA
                        deltaS = 0.2, #NA
                        Cxs = 0.054, #NA
                        deltaW = 0.07, #NA
                        Axw = 96.6, #NA
                        Bxw = 1, #NA
                        x = 0.32, #NA
                        Cxw = 0.038, #NA
                        Axs = 20, #NA
                        Cvs = 0.036 #NA
    ) {
      return(c(
        R0 = R0,
        k0 = k0,
        Avd = Avd,
        Bxx = Bxx,
        Cvd = Cvd,
        deltaD = deltaD,
        Axb = Axb,
        Cvb = Cvb,
        deltaB = deltaB,
        z = z,
        za = za,
        Bz = Bz,
        Cdb = Cdb,
        wpr = wpr,
        aimpopD = aimpopD,
        pHuntD = pHuntD,
        pHDleft = pHDleft,
        aimpopB = aimpopB,
        pHuntB = pHuntB,
        pHBleft = pHBleft,
        deltaS = deltaS,
        Cxs = Cxs,
        deltaW = deltaW,
        Axw = Axw,
        Bxw = Bxw,
        x = x,
        Cxw = Cxw,
        Axs = Axs,
        Cvs = Cvs
      ))
    }
    
    sim = odeFun(y = pp_init(),
                 times = seq(from=0, to=10, by=1),
                 func = pp,
                 parms = pp_param())
    
    plot(sim, vrbls = c("V"), savefile = "V")
    
    plot(sim, vrbls = c("V","Vgrowth"), savefile = "V_auxpar")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=10, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                     to = 1.25,
                                                                     length.out = 5),
                                         k0 = pp_param()[["k0"]]*seq(from = .75,
                                                                     to = 1.25,
                                                                     length.out = 5)))
    
    plot(sens_sim, vrbl = "V", savefile = "V_sens_V")
    
    sens_full_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=10, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                     to = 1.25,
                                                                     length.out = 25),
                                         k0 = pp_param()[["k0"]]*seq(from = .75,
                                                                     to = 1.25,
                                                                     length.out = 25)))
    
    plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "V_sensend_V")
  }
  
  ## V + D
  {
    # No hunting
    {
      pp_init = function(V = 1,
                         S = 0, #NA
                         B = 0, #NA
                         D = 1,
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.017,
                          deltaD = 0.125,
                          Axb = 20.15, #NA
                          Cvb = 0.055, #NA
                          deltaB = 0.08, #NA
                          z = 0.2, #NA
                          za = 0.16, #NA
                          Bz = 0.1, #NA
                          Cdb = 0.069, #NA
                          wpr = 10, #NA
                          aimpopD = 1, #NA
                          pHuntD = 0, #NA
                          pHDleft = 2/3, #NA
                          aimpopB = 1, #NA
                          pHuntB = 0, #NA
                          pHBleft = 2/3, #NA
                          deltaS = 0.2, #NA
                          Cxs = 0.054, #NA
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20, #NA
                          Cvs = 0.036 #NA
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=100, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D"), savefile = "VD")
      ##### NOTE: density of D versus V is high, but this is not unlike Focardi et al. 2017
      
      plot(sim, vrbls = c("V","D",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD"), 
           savefile = "VD_auxpar")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=100, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5),
                                           Avd = pp_param()[["Avd"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5),
                                           Bxx = pp_param()[["Bxx"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvd = pp_param()[["Cvd"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      
      plot(sens_sim, vrbl = "V", savefile = "VD_sens_V")
      
      plot(sens_sim, vrbl = "D", savefile = "VD_sens_D")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 25),
                                           Avd = pp_param()[["Avd"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 25),
                                           Bxx = pp_param()[["Bxx"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvd = pp_param()[["Cvd"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VD_sensend_V")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", savefile = "VD_sensend_D")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,10,length.out=10),
                                              S = 0, #NA
                                              B = 0, #NA
                                              D = seq(.5,5,length.out=10),
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(),
                              vrbls = c("V","D"),
                              times = seq(0,100,1/12))
      
      plot(phase_sim, savefile = "VD_phase_VD")
    }
    
    # Hunting
    {
      pp_init = function(V = 1,
                         S = 0, #NA
                         B = 0, #NA
                         D = 1,
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.017,
                          deltaD = 0.125,
                          Axb = 20.15, #NA
                          Cvb = 0.055, #NA
                          deltaB = 0.08, #NA
                          z = 0.2, #NA
                          za = 0.16, #NA
                          Bz = 0.1, #NA
                          Cdb = 0.069, #NA
                          wpr = 10, #NA
                          aimpopD = 1,
                          pHuntD = .75,
                          pHDleft = 2/3, #NA
                          aimpopB = 1, #NA
                          pHuntB = 0, #NA
                          pHBleft = 2/3, #NA
                          deltaS = 0.2, #NA
                          Cxs = 0.054, #NA
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20, #NA
                          Cvs = 0.036 #NA
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=100, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D"), savefile = "VD_dH")
      
      plot(sim, vrbls = c("V","D",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD",
                          "huntingD"), 
           savefile = "VD_dH_auxpar")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=100, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(aimpopD = pp_param()[["aimpopD"]]*seq(from = .75,
                                                                                 to = 1.25,
                                                                                 length.out = 5),
                                           pHuntD = pp_param()[["pHuntD"]]*seq(from = .75,
                                                                               to = 1.25,
                                                                               length.out = 5),
                                           Avd = pp_param()[["Avd"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvd = pp_param()[["Cvd"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      
      plot(sens_sim, vrbl = "V", savefile = "VD_dH_sens_V")
      
      plot(sens_sim, vrbl = "D", savefile = "VD_dH_sens_D")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(aimpopD = pp_param()[["aimpopD"]]*seq(from = .75,
                                                                                 to = 1.25,
                                                                                 length.out = 25),
                                           pHuntD = pp_param()[["pHuntD"]]*seq(from = .75,
                                                                               to = 1.25,
                                                                               length.out = 25),
                                           Avd = pp_param()[["Avd"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvd = pp_param()[["Cvd"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VD_dH_sensend_V")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", savefile = "VD_dH_sensend_D")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,10,length.out=10),
                                              S = 0, #NA
                                              B = 0, #NA
                                              D = seq(.5,5,length.out=10),
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(),
                              vrbls = c("V","D"),
                              times = seq(0,100,1/12))
      
      plot(phase_sim, savefile = "VD_dH_phase_VD")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,10,length.out=10),
                                              S = 0, #NA
                                              B = 0, #NA
                                              D = seq(.5,5,length.out=10),
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(aimpopD = .25,
                                               pHuntD = .99),
                              vrbls = c("V","D"),
                              times = seq(0,100,1/12))
      
      plot(phase_sim, savefile = "VD_dH_phase_VD_.25aimpopD.99pHuntD")
      ##### NOTE: also very high hunting pressure (low population and high culling efficiency) doesn't lead to extinction
    }
  }
  
  ## V + B
  {
    # No hunting
    {
      pp_init = function(V = 1,
                         S = 0, #NA
                         B = 1,
                         D = 0, #NA
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5, #NA
                          Bxx = 10,
                          Cvd = 0.017, #NA
                          deltaD = 0.125, #NA
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2, #NA
                          za = 0.16, #NA
                          Bz = 0.1, #NA
                          Cdb = 0.069, #NA
                          wpr = 10, #NA
                          aimpopD = 1, #NA
                          pHuntD = 0, #NA
                          pHDleft = 2/3, #NA
                          aimpopB = 1, #NA
                          pHuntB = 0, #NA
                          pHBleft = 2/3, #NA
                          deltaS = 0.2, #NA
                          Cxs = 0.054, #NA
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20, #NA
                          Cvs = 0.036 #NA
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","B"), savefile = "VB")
      ##### NOTE: fluctuates more and takes longer to stabilize than with only deer, probably due to large growth rate of boar
      
      plot(sim, vrbls = c("V","B",
                          "Vgrowth","AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB"), 
           savefile = "VB_auxpar")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5),
                                           Axb = pp_param()[["Axb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Bxx = pp_param()[["Bxx"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvb = pp_param()[["Cvb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      
      plot(sens_sim, vrbl = "V", savefile = "VB_sens_V")
      
      plot(sens_sim, vrbl = "B", savefile = "VB_sens_B")
      ##### NOTE: apparently higher Bxx values reduce the time of the fluctuations
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 25),
                                           Axb = pp_param()[["Axb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 25),
                                           Bxx = pp_param()[["Bxx"]]*seq(from = 1,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvb = pp_param()[["Cvb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      ##### NOTE: reduced range of Bxx here compared to previous to avoid having end states that are not stabilized
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VB_sensend_V")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", savefile = "VB_sensend_B")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,8,length.out=10),
                                              S = 0, #NA
                                              B = seq(.5,4,length.out=10),
                                              D = 0, #NA
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(),
                              vrbls = c("V","B"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VB_phase_VB")
    }
    
    # Hunting
    {
      pp_init = function(V = 1,
                         S = 0, #NA
                         B = 1,
                         D = 0, #NA
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5, #NA
                          Bxx = 10,
                          Cvd = 0.017, #NA
                          deltaD = 0.125, #NA
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2, #NA
                          za = 0.16, #NA
                          Bz = 0.1, #NA
                          Cdb = 0.069, #NA
                          wpr = 10, #NA
                          aimpopD = 1, #NA
                          pHuntD = 0, #NA
                          pHDleft = 2/3, #NA
                          aimpopB = 1,
                          pHuntB = .75,
                          pHBleft = 2/3, #NA
                          deltaS = 0.2, #NA
                          Cxs = 0.054, #NA
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20, #NA
                          Cvs = 0.036 #NA
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=100, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","B"), savefile = "VB_bH")
      
      plot(sim, vrbls = c("V","B",
                          "Vgrowth","AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB",
                          "huntingB"), 
           savefile = "VB_bH_auxpar")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=100, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(aimpopB = pp_param()[["aimpopB"]]*seq(from = .75,
                                                                                 to = 1.25,
                                                                                 length.out = 5),
                                           pHuntB = pp_param()[["pHuntB"]]*seq(from = .75,
                                                                               to = 1.25,
                                                                               length.out = 5),
                                           Axb = pp_param()[["Axb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvb = pp_param()[["Cvb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      
      plot(sens_sim, vrbl = "V", savefile = "VB_bH_sens_V")
      
      plot(sens_sim, vrbl = "B", savefile = "VB_bH_sens_B")
      ##### NOTE: hunting prevents oscillation
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(aimpopB = pp_param()[["aimpopB"]]*seq(from = .75,
                                                                                 to = 1.25,
                                                                                 length.out = 25),
                                           pHuntB = pp_param()[["pHuntB"]]*seq(from = .75,
                                                                               to = 1.25,
                                                                               length.out = 25),
                                           Axb = pp_param()[["Axb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5),
                                           Cvb = pp_param()[["Cvb"]]*seq(from = .75,
                                                                         to = 1.25,
                                                                         length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VB_bH_sensend_V")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", savefile = "VB_bH_sensend_B")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,8,length.out=10),
                                              S = 0, #NA
                                              B = seq(.5,4,length.out=10),
                                              D = 0, #NA
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(),
                              vrbls = c("V","B"),
                              times = seq(0,100,1/12))
      
      plot(phase_sim, savefile = "VB_bH_phase_VB")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,8,length.out=10),
                                              S = 0, #NA
                                              B = seq(.5,4,length.out=10),
                                              D = 0, #NA
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(aimpopB = .25,
                                               pHuntB = .99),
                              vrbls = c("V","B"),
                              times = seq(0,100,1/12))
      
      plot(phase_sim, savefile = "VB_bH_phase_VB_.25aimpopB.99pHuntB")
      ##### NOTE: also very high hunting pressure (low population and high culling efficiency) doesn't lead to extinction
    }
  }
  
  ## V + S
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 0, #NA
                       D = 0, #NA
                       W = 0 #NA
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    pp_param = function(R0 = 4,
                        k0 = 10,
                        Avd = 23.5, #NA
                        Bxx = 10,
                        Cvd = 0.017, #NA
                        deltaD = 0.125, #NA
                        Axb = 20.15, #NA
                        Cvb = 0.055, #NA
                        deltaB = 0.08, #NA
                        z = 0.2, #NA
                        za = 0.16, #NA
                        Bz = 0.1, #NA
                        Cdb = 0.069, #NA
                        wpr = 10, #NA
                        aimpopD = 1, #NA
                        pHuntD = 0, #NA
                        pHDleft = 1, #NA
                        aimpopB = 1, #NA
                        pHuntB = 0, #NA
                        pHBleft = 1, #NA
                        deltaS = 0.2,
                        Cxs = 0.054, #NA
                        deltaW = 0.07, #NA
                        Axw = 96.6, #NA
                        Bxw = 1, #NA
                        x = 0.32, #NA
                        Cxw = 0.038, #NA
                        Axs = 20,
                        Cvs = 0.036
    ) {
      return(c(
        R0 = R0,
        k0 = k0,
        Avd = Avd,
        Bxx = Bxx,
        Cvd = Cvd,
        deltaD = deltaD,
        Axb = Axb,
        Cvb = Cvb,
        deltaB = deltaB,
        z = z,
        za = za,
        Bz = Bz,
        Cdb = Cdb,
        wpr = wpr,
        aimpopD = aimpopD,
        pHuntD = pHuntD,
        pHDleft = pHDleft,
        aimpopB = aimpopB,
        pHuntB = pHuntB,
        pHBleft = pHBleft,
        deltaS = deltaS,
        Cxs = Cxs,
        deltaW = deltaW,
        Axw = Axw,
        Bxw = Bxw,
        x = x,
        Cxw = Cxw,
        Axs = Axs,
        Cvs = Cvs
      ))
    }
    
    sim = odeFun(y = pp_init(),
                 times = seq(from=0, to=100, by=1),
                 func = pp,
                 parms = pp_param())
    
    plot(sim, vrbls = c("V","S"), savefile = "VS")
    
    plot(sim, vrbls = c("V","S",
                        "Vgrowth","AxsResponse",
                        "VconsS","SgrowthV","deathS"), 
         savefile = "VS_auxpar")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=100, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                     to = 1.25,
                                                                     length.out = 5),
                                         Axs = pp_param()[["Axs"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5),
                                         Bxx = pp_param()[["Bxx"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5),
                                         Cvs = pp_param()[["Cvs"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5)))
    
    plot(sens_sim, vrbl = "V", savefile = "VS_sens_V")
    
    plot(sens_sim, vrbl = "S", savefile = "VS_sens_S")
    
    sens_full_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=1000, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(R0 = pp_param()[["R0"]]*seq(from = .75,
                                                                     to = 1.25,
                                                                     length.out = 25),
                                         Axs = pp_param()[["Axs"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 25),
                                         Bxx = pp_param()[["Bxx"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5),
                                         Cvs = pp_param()[["Cvs"]]*seq(from = .75,
                                                                       to = 1.25,
                                                                       length.out = 5)))
    
    plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VS_sensend_V")
    
    plot(sens_full_sim, vrbl = "S", type = "endState", savefile = "VS_sensend_S")
    
    phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,10,length.out=10),
                                            S = seq(.5,5,length.out=10),
                                            B = 0, #NA
                                            D = 0, #NA
                                            W = 0), #NA
                            func = pp,
                            parms = pp_param(),
                            vrbls = c("V","S"),
                            times = seq(0,100,1/12))
    
    plot(phase_sim, savefile = "VS_phase_VS")
  }
  
  ## V + D + B
  {
    # No hunting
    {
      pp_init = function(V = 1,
                         S = 0, #NA
                         B = 1,
                         D = 1,
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.017,
                          deltaD = 0.125,
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2,
                          za = 0.16,
                          Bz = 0.1,
                          Cdb = 0.069,
                          wpr = 10, #NA
                          aimpopD = 1, #NA
                          pHuntD = 0, #NA
                          pHDleft = 2/3, #NA
                          aimpopB = 1, #NA
                          pHuntB = 0, #NA
                          pHBleft = 2/3, #NA
                          deltaS = 0.2, #NA
                          Cxs = 0.054, #NA
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20, #NA
                          Cvs = 0.036 #NA
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D","B"), savefile = "VDB")
      ##### NOTE: D dies out, while B lives on. However, this is similar to Focardi et al. In their model the boar got outcompeted by deer because they were more efficient in exploiting vegetation. Only with the presence of predators that had a species-specific preference, could boar and deer coexist. With our parameter values the boar are more efficient, so it makes sense that deer die out in this model.
      
      plot(sim, vrbls = c("V","D","B",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD",
                          "AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB"), 
           savefile = "VDB_auxpar")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Avd = pp_param()[["Avd"]]*seq(from = .5,
                                                                       to = 1.5,
                                                                       length.out = 5),
                                           Cvd = pp_param()[["Cvd"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Axb = pp_param()[["Axb"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Cvb = pp_param()[["Cvb"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5)))
      ##### NOTE: the range of sensitivity analysis is increased here compared to previous runs, to showcase that deer can also outcompete boar with different parameter values
      
      plot(sens_sim, vrbl = "V", savefile = "VDB_sens_V")
      
      plot(sens_sim, vrbl = "D", savefile = "VDB_sens_D")
      
      plot(sens_sim, vrbl = "B", savefile = "VDB_sens_B")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Avd = pp_param()[["Avd"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           Cvd = pp_param()[["Cvd"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           Axb = pp_param()[["Axb"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Cvb = pp_param()[["Cvb"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VDB_sensend_V")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", savefile = "VDB_sensend_D")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", savefile = "VDB_sensend_B")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,8,length.out=10),
                                              S = 0, #NA
                                              B = seq(.5,4,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of original model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(),
                              vrbls = c("V","B"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDB_phase_VB")
    }
    
    # Hunting
    {
      pp_init = function(V = 1,
                         S = 0, #NA
                         B = 1,
                         D = 1,
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.017,
                          deltaD = 0.125,
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2,
                          za = 0.16,
                          Bz = 0.1,
                          Cdb = 0.069,
                          wpr = 10, #NA
                          aimpopD = 1,
                          pHuntD = .75,
                          pHDleft = 2/3,
                          aimpopB = 1,
                          pHuntB = .75,
                          pHBleft = 2/3, #NA
                          deltaS = 0.2, #NA
                          Cxs = 0.054, #NA
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20, #NA
                          Cvs = 0.036 #NA
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D","B"), savefile = "VDB_dbH")
      
      plot(sim, vrbls = c("V","D","B",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD",
                          "AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB",
                          "huntingD","huntingB",
                          "huntedDleft","carrionD",
                          "carrionFracBavail","carrionFracBcons",
                          "DscavB","BgrowthD"), 
           savefile = "VDB_dbH_auxpar")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(aimpopD = pp_param()[["aimpopD"]]*seq(from = .5,
                                                                                 to = 1.5,
                                                                                 length.out = 5),
                                           pHuntD = pp_param()[["pHuntD"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5),
                                           aimpopB = pp_param()[["aimpopB"]]*seq(from = .5,
                                                                                 to = 1.5,
                                                                                 length.out = 5),
                                           pHuntB = pp_param()[["pHuntB"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5)))
      ##### NOTE: the range of sensitivity analysis is increased here compared to previous runs, to showcase that deer and boar can coexist when both species have a different hunting regime
      
      plot(sens_sim, vrbl = "V", savefile = "VDB_dbH_sens_V")
      
      plot(sens_sim, vrbl = "D", savefile = "VDB_dbH_sens_D")
      
      plot(sens_sim, vrbl = "B", savefile = "VDB_dbH_sens_B")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(aimpopD = pp_param()[["aimpopD"]]*seq(from = .5,
                                                                                 to = 1.5,
                                                                                 length.out = 25),
                                           pHuntD = pp_param()[["pHuntD"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 25),
                                           aimpopB = pp_param()[["aimpopB"]]*seq(from = .5,
                                                                                 to = 1.5,
                                                                                 length.out = 5),
                                           pHuntB = pp_param()[["pHuntB"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VDB_dbH_sensend_V")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", savefile = "VDB_dbH_sensend_D")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", savefile = "VDB_dbH_sensend_B")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,8,length.out=10),
                                              S = 0, #NA
                                              B = seq(.5,4,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of original model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(),
                              vrbls = c("V","B"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDB_dbH_phase_VB")
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parms = pp_param(aimpopB = .5))
      
      plot(sim, vrbls = c("V","D","B"), savefile = "VDB_dbH_.5aimpopB")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model with .5 aimpopB
                                              S = 0, #NA
                                              B = seq(.2,2,length.out=10),
                                              D = seq(.1,1,length.out=10),
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(aimpopB = .5),
                              vrbls = c("D","B"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDB_dbH_phase_DB_.5aimpopB")
      ##### NOTE: with lower target population for boar, then deer and boar always stabilize to same equilibrium
    }
  }
  
  ## V + D + B + S
  {
    # No hunting
    {
      pp_init = function(V = 1,
                         S = 1,
                         B = 1,
                         D = 1,
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.017,
                          deltaD = 0.125,
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2,
                          za = 0.16,
                          Bz = 0.1,
                          Cdb = 0.069,
                          wpr = 10, #NA
                          aimpopD = 1, #NA
                          pHuntD = 0, #NA
                          pHDleft = 2/3, #NA
                          aimpopB = 1, #NA
                          pHuntB = 0, #NA
                          pHBleft = 2/3, #NA
                          deltaS = 0.2,
                          Cxs = 0.054,
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20,
                          Cvs = 0.036
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D","B","S"), savefile = "VDBS")
      
      plot(sim, vrbls = c("V","D","B","S",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD",
                          "AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB",
                          "DscavB","BgrowthD",
                          "AxsResponse",
                          "VconsS","SgrowthV","deathS",
                          "carrionD","carrionFracBavail",
                          "carrionFracBcons",
                          "carrionB","carrionDS",
                          "DscavS","SgrowthD",
                          "BscavS","SgrowthB"), 
           savefile = "VDBS_auxpar")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Axs = pp_param()[["Axs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Cxs = pp_param()[["Cxs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Cvs = pp_param()[["Cvs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           deltaS = pp_param()[["deltaS"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5)))
      ##### NOTE: the range of sensitivity analysis is increased here compared to previous runs, to showcase that scavengers can only survive together with boar with lower Axs+deltaS and higher Cxs values
      
      plot(sens_sim, vrbl = "V", savefile = "VDBS_sens_V")
      
      plot(sens_sim, vrbl = "D", savefile = "VDBS_sens_D")
      
      plot(sens_sim, vrbl = "B", savefile = "VDBS_sens_B")
      
      plot(sens_sim, vrbl = "S", savefile = "VDBS_sens_S")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Axs = pp_param()[["Axs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           Cxs = pp_param()[["Cxs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           Cvs = pp_param()[["Cvs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           deltaS = pp_param()[["deltaS"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VDBS_sensend_V")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", savefile = "VDBS_sensend_D")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", savefile = "VDBS_sensend_B")
      
      plot(sens_full_sim, vrbl = "S", type = "endState", savefile = "VDBS_sensend_S")
      ##### NOTE: the values for D, B, and S in these figures should only be interpreted for cases when vegetation did not reach zero (as the simulation terminates as soon as one state reaches zero)
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parms = pp_param(deltaS = .1, Axs = 10, Cxs = .075))
      
      plot(sim, vrbls = c("V","D","B","S"), savefile = "VDBS_.1deltaS10Axs.075Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,8,length.out=10),
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = seq(.5,4,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(deltaS = .1, Axs = 10, Cxs = .075),
                              vrbls = c("V","B"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_phase_VB_.1deltaS10Axs.075Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,5,length.out=10),
                                              S = seq(.1,1,length.out=10),
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(deltaS = .1, Axs = 10, Cxs = .075),
                              vrbls = c("V","S"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_phase_VS_.1deltaS10Axs.075Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = seq(.1,1,length.out=10),
                                              B = seq(.5,4,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(deltaS = .1, Axs = 10, Cxs = .075),
                              vrbls = c("B","S"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_phase_BS_.1deltaS10Axs.075Cxs")
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parms = pp_param(Cxs = .1))
      
      plot(sim, vrbls = c("V","D","B","S"), savefile = "VDBS_.1Cxs")
      ##### NOTE: only a much higher Cxs value is also enough to have scavengers and boar coexist
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,8,length.out=10),
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = seq(.5,4,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cxs = .1),
                              vrbls = c("V","B"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_phase_VB_.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,5,length.out=10),
                                              S = seq(.05,.5,length.out=10),
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cxs = .1),
                              vrbls = c("V","S"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_phase_VS_.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = seq(.05,.5,length.out=10),
                                              B = seq(.5,4,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cxs = .1),
                              vrbls = c("B","S"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_phase_BS_.1Cxs")
    }
    
    # Hunting
    {
      pp_init = function(V = 1,
                         S = 1,
                         B = 1,
                         D = 1,
                         W = 0 #NA
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.017,
                          deltaD = 0.125,
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2,
                          za = 0.16,
                          Bz = 0.1,
                          Cdb = 0.069,
                          wpr = 10, #NA
                          aimpopD = 1,
                          pHuntD = .75,
                          pHDleft = 2/3,
                          aimpopB = 1,
                          pHuntB = .75,
                          pHBleft = 2/3,
                          deltaS = 0.2,
                          Cxs = 0.1, # updated from 0.054 based on previous sensitivity analysis
                          deltaW = 0.07, #NA
                          Axw = 96.6, #NA
                          Bxw = 1, #NA
                          x = 0.32, #NA
                          Cxw = 0.038, #NA
                          Axs = 20,
                          Cvs = 0.036
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D","B","S"), savefile = "VDBS_dbH_.1Cxs")
      
      plot(sim, vrbls = c("V","D","B","S",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD",
                          "AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB",
                          "DscavB","BgrowthD",
                          "AxsResponse",
                          "VconsS","SgrowthV","deathS",
                          "carrionD","carrionFracBavail",
                          "carrionFracBcons",
                          "carrionB","carrionDS",
                          "DscavS","SgrowthD",
                          "BscavS","SgrowthB",
                          "huntingD","huntingB",
                          "huntedDleft","huntedBleft"), 
           savefile = "VDBS_dbH_auxpar_.1Cxs")
      
      sens_sim =
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=250, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Cvd = pp_param()[["Cvd"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Cvs = pp_param()[["Cvs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           aimpopB = pp_param()[["aimpopB"]]*seq(from = .5,
                                                                                 to = 1.5,
                                                                                 length.out = 5),
                                           pHuntB = pp_param()[["pHuntB"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5)))
      ##### NOTE: the range of sensitivity analysis is changed here compared to previous runs, to showcase that deer can only survive together with boar and scavenger when both Cvd is higher relative to Cvs and when boar is under stricter hunting regime -> so increase value of Cvd in overall model
      
      plot(sens_sim, vrbl = "V", savefile = "VDBS_dbH_sens_V_.1Cxs")
      
      plot(sens_sim, vrbl = "D", savefile = "VDBS_dbH_sens_D_.1Cxs")
      
      plot(sens_sim, vrbl = "B", savefile = "VDBS_dbH_sens_B_.1Cxs")
      
      plot(sens_sim, vrbl = "S", savefile = "VDBS_dbH_sens_S_.1Cxs")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Cvd = pp_param()[["Cvd"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           Cvs = pp_param()[["Cvs"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           aimpopB = pp_param()[["aimpopB"]]*seq(from = .5,
                                                                                 to = 1.5,
                                                                                 length.out = 5),
                                           pHuntB = pp_param()[["pHuntB"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VDBS_dbH_sensend_V_.1Cxs")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", savefile = "VDBS_dbH_sensend_D_.1Cxs")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", savefile = "VDBS_dbH_sensend_B_.1Cxs")
      
      plot(sens_full_sim, vrbl = "S", type = "endState", savefile = "VDBS_dbH_sensend_S_.1Cxs")
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parms = pp_param(Cvd = .025, aimpopB = .5))

      plot(sim, vrbls = c("V","D","B","S"), savefile = "VDBS_dbH_.5aimpopB.025Cvd.1Cxs")

      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = seq(.2,2,length.out=10),
                                              D = seq(.1,1,length.out=10),
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cvd = .025, aimpopB = .5),
                              vrbls = c("D","B"),
                              times = seq(0,500,1/12))

      plot(phase_sim, savefile = "VDBS_dbH_phase_DB_.5aimpopB.025Cvd.1Cxs")

      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = seq(.1,1,length.out=10),
                                              B = seq(.2,2,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cvd = .025, aimpopB = .5),
                              vrbls = c("S","B"),
                              times = seq(0,500,1/12))

      plot(phase_sim, savefile = "VDBS_dbH_phase_SB_.5aimpopB.025Cvd.1Cxs")

      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = seq(.1,1,length.out=10),
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = seq(.1,1,length.out=10),
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cvd = .025, aimpopB = .5),
                              vrbls = c("S","D"),
                              times = seq(0,500,1/12))

      plot(phase_sim, savefile = "VDBS_dbH_phase_SD_.5aimpopB.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,5,length.out=10),
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = seq(.2,2,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cvd = .025, aimpopB = .5),
                              vrbls = c("V","B"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_dbH_phase_VB_.5aimpopB.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,5,length.out=10),
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = seq(.1,1,length.out=10),
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cvd = .025, aimpopB = .5),
                              vrbls = c("V","D"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_dbH_phase_VD_.5aimpopB.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,5,length.out=10),
                                              S = seq(.1,1,length.out=10),
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = 0), #NA
                              func = pp,
                              parms = pp_param(Cvd = .025, aimpopB = .5),
                              vrbls = c("V","S"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBS_dbH_phase_VS_.5aimpopB.025Cvd.1Cxs")
    }
  }
  
  ## V + D + B + S + W
  {
    # No hunting
    {
      pp_init = function(V = 1,
                         S = 1,
                         B = 1,
                         D = 1,
                         W = .01
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.025, # updated from 0.017 based on previous sensitivity analysis
                          deltaD = 0.125,
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2,
                          za = 0.16,
                          Bz = 0.1,
                          Cdb = 0.069,
                          wpr = 10,
                          aimpopD = 1, #NA
                          pHuntD = 0, #NA
                          pHDleft = 2/3, #NA
                          aimpopB = 1, #NA
                          pHuntB = 0, #NA
                          pHBleft = 2/3, #NA
                          deltaS = 0.2,
                          Cxs = 0.1, # updated from 0.054 based on previous sensitivity analysis
                          deltaW = 0.07,
                          Axw = 96.6,
                          Bxw = 1,
                          x = 0.32,
                          Cxw = 0.038,
                          Axs = 20,
                          Cvs = 0.036
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D","B","S","W"), savefile = "VDBSW_.025Cvd.1Cxs")
      ##### NOTE: wolves consume too much with these parameter values
      
      plot(sim, vrbls = c("V","D","B","S","W",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD",
                          "AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB",
                          "DscavB","BgrowthD",
                          "AxsResponse",
                          "VconsS","SgrowthV","deathS",
                          "carrionD","carrionFracBavail",
                          "carrionFracBcons",
                          "carrionB","carrionDS",
                          "DscavS","SgrowthD",
                          "BscavS","SgrowthB",
                          "AxwResponse","DpredW",
                          "killedDleft","WgrowthD",
                          "BpredW","killedBleft",
                          "WgrowthB","deathW"), 
           savefile = "VDBSW_auxpar_.025Cvd.1Cxs")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Axw = pp_param()[["Axw"]]*seq(from = .1,
                                                                         to = 1,
                                                                         length.out = 5),
                                           Bxw = pp_param()[["Bxw"]]*seq(from = 1,
                                                                         to = 10,
                                                                         length.out = 5),
                                           Cxw = pp_param()[["Cxw"]]*seq(from = .1,
                                                                         to = 1,
                                                                         length.out = 5),
                                           deltaW = pp_param()[["deltaW"]]*seq(from = 1,
                                                                               to = 10,
                                                                               length.out = 5)))
      
      plot(sens_sim, vrbl = "V", savefile = "VDBSW_sens_V_.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "D", savefile = "VDBSW_sens_D_.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "B", savefile = "VDBSW_sens_B_.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "S", savefile = "VDBSW_sens_S_.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "W", savefile = "VDBSW_sens_W_.025Cvd.1Cxs")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Axw = pp_param()[["Axw"]]*seq(from = .1,
                                                                         to = 1,
                                                                         length.out = 25),
                                           Bxw = pp_param()[["Bxw"]]*seq(from = 1,
                                                                         to = 10,
                                                                         length.out = 25),
                                           Cxw = pp_param()[["Cxw"]]*seq(from = .1,
                                                                         to = 1,
                                                                         length.out = 5),
                                           deltaW = pp_param()[["deltaW"]]*seq(from = 1,
                                                                               to = 10,
                                                                               length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", savefile = "VDBSW_sensend_V_.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", savefile = "VDBSW_sensend_D_.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", savefile = "VDBSW_sensend_B_.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "S", type = "endState", savefile = "VDBSW_sensend_S_.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "W", type = "endState", savefile = "VDBSW_sensend_W_.025Cvd.1Cxs")
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parms = pp_param(Bxw = 10, deltaW = .2))
      
      plot(sim, vrbls = c("V","D","B","S","W"),
           savefile = "VDBSW_10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,5,length.out=10),
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("V","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_phase_VW_10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = seq(.25,2.5,length.out=10),
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("S","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_phase_SW_10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = seq(.25,2.5,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("B","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_phase_BW_10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = seq(.15,1.5,length.out=10),
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("D","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_phase_DW_10Bxw.2deltaW.025Cvd.1Cxs")
    }
    
    # Hunting
    {
      pp_init = function(V = 1,
                         S = 1,
                         B = 1,
                         D = 1,
                         W = .01
      ) {
        return(c(
          V = V,
          S = S,
          B = B,
          D = D,
          W = W
        ))
      }
      
      pp_param = function(R0 = 4,
                          k0 = 10,
                          Avd = 23.5,
                          Bxx = 10,
                          Cvd = 0.025, # updated from 0.017 based on previous sensitivity analysis
                          deltaD = 0.125,
                          Axb = 20.15,
                          Cvb = 0.055,
                          deltaB = 0.08,
                          z = 0.2,
                          za = 0.16,
                          Bz = 0.1,
                          Cdb = 0.069,
                          wpr = 10,
                          aimpopD = .5, # updated from 1 based on previous sensitivity analysis
                          pHuntD = .75,
                          pHDleft = 2/3,
                          aimpopB = .5, # updated from 1 based on previous sensitivity analysis
                          pHuntB = .75,
                          pHBleft = 2/3,
                          deltaS = 0.2,
                          Cxs = 0.1, # updated from 0.054 based on previous sensitivity analysis
                          deltaW = 0.2, # updated from .07 based on previous sensitivity analysis
                          Axw = 96.6,
                          Bxw = 10, # updated from 1 based on previous sensitivity analysis
                          x = 0.32,
                          Cxw = 0.038,
                          Axs = 20,
                          Cvs = 0.036
      ) {
        return(c(
          R0 = R0,
          k0 = k0,
          Avd = Avd,
          Bxx = Bxx,
          Cvd = Cvd,
          deltaD = deltaD,
          Axb = Axb,
          Cvb = Cvb,
          deltaB = deltaB,
          z = z,
          za = za,
          Bz = Bz,
          Cdb = Cdb,
          wpr = wpr,
          aimpopD = aimpopD,
          pHuntD = pHuntD,
          pHDleft = pHDleft,
          aimpopB = aimpopB,
          pHuntB = pHuntB,
          pHBleft = pHBleft,
          deltaS = deltaS,
          Cxs = Cxs,
          deltaW = deltaW,
          Axw = Axw,
          Bxw = Bxw,
          x = x,
          Cxw = Cxw,
          Axs = Axs,
          Cvs = Cvs
        ))
      }
      
      sim = odeFun(y = pp_init(),
                   times = seq(from=0, to=250, by=1),
                   func = pp,
                   parms = pp_param())
      
      plot(sim, vrbls = c("V","D","B","S","W"),
           savefile = "VDBSW_dbH_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sim, vrbls = c("V","D","B","S","W",
                          "Vgrowth","AvdResponse",
                          "VconsD","DgrowthV","deathD",
                          "AxbCorrected","AxbResponse",
                          "VconsB","BgrowthV","deathB",
                          "DscavB","BgrowthD",
                          "AxsResponse",
                          "VconsS","SgrowthV","deathS",
                          "carrionD","carrionFracBavail",
                          "carrionFracBcons",
                          "carrionB","carrionDS",
                          "DscavS","SgrowthD",
                          "BscavS","SgrowthB",
                          "AxwResponse","DpredW",
                          "killedDleft","WgrowthD",
                          "BpredW","killedBleft",
                          "WgrowthB","deathW",
                          "huntingD","huntingB",
                          "huntedDleft","huntedBleft",
                          "huntingbagD","huntingbagB"), 
           savefile = "VDBSW_dbH_auxpar_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      sens_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=500, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Axw = pp_param()[["Axw"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Bxw = pp_param()[["Bxw"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           Cxw = pp_param()[["Cxw"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           deltaW = pp_param()[["deltaW"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5)))
      
      plot(sens_sim, vrbl = "V", savefile = "VDBSW_dbH_sens_V_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "D", savefile = "VDBSW_dbH_sens_D_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "B", savefile = "VDBSW_dbH_sens_B_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "S", savefile = "VDBSW_dbH_sens_S_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_sim, vrbl = "W", savefile = "VDBSW_dbH_sens_W_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      sens_full_sim = 
        odeSensFun(y = pp_init(),
                   times = seq(from=0, to=1000, by=1),
                   func = pp,
                   parmsFun = pp_param,
                   namedSeqs = expand.grid(Axw = pp_param()[["Axw"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           Bxw = pp_param()[["Bxw"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 25),
                                           Cxw = pp_param()[["Cxw"]]*seq(from = .5,
                                                                         to = 1.5,
                                                                         length.out = 5),
                                           deltaW = pp_param()[["deltaW"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5)))
      
      plot(sens_full_sim, vrbl = "V", type = "endState", 
           savefile = "VDBSW_dbH_sensend_V_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "D", type = "endState", 
           savefile = "VDBSW_dbH_sensend_D_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "B", type = "endState", 
           savefile = "VDBSW_dbH_sensend_B_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "S", type = "endState", 
           savefile = "VDBSW_dbH_sensend_S_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      plot(sens_full_sim, vrbl = "W", type = "endState", 
           savefile = "VDBSW_dbH_sensend_W_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = seq(.5,5,length.out=10),
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("V","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_dbH_phase_VW_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = seq(.25,2.5,length.out=10),
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("S","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_dbH_phase_SW_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = seq(.25,2.5,length.out=10),
                                              D = round(sim[nrow(sim),][["D"]],3), # final equilibrium value of model
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("B","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_dbH_phase_BW_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
      
      phase_sim = odePhaseFun(y = expand.grid(V = round(sim[nrow(sim),][["V"]],3), # final equilibrium value of model
                                              S = round(sim[nrow(sim),][["S"]],3), # final equilibrium value of model
                                              B = round(sim[nrow(sim),][["B"]],3), # final equilibrium value of model
                                              D = seq(.15,1.5,length.out=10),
                                              W = seq(.005,.05,length.out=10)),
                              func = pp,
                              parms = pp_param(Bxw = 10, deltaW = .2),
                              vrbls = c("D","W"),
                              times = seq(0,500,1/12))
      
      plot(phase_sim, savefile = "VDBSW_dbH_phase_DW_.5aimpopD.5aimpopB10Bxw.2deltaW.025Cvd.1Cxs")
    }
  }
}

### Hunting effect
{
  pp_param = function(R0 = 4,
                      k0 = 10,
                      Avd = 23.5,
                      Bxx = 10,
                      Cvd = 0.025, # updated from 0.017 based on previous sensitivity analysis
                      deltaD = 0.125,
                      Axb = 20.15,
                      Cvb = 0.055,
                      deltaB = 0.08,
                      z = 0.2,
                      za = 0.16,
                      Bz = 0.1,
                      Cdb = 0.069,
                      wpr = 10,
                      aimpopD = .5, # updated from 1 based on previous sensitivity analysis
                      pHuntD = .75,
                      pHDleft = 2/3,
                      aimpopB = .5, # updated from 1 based on previous sensitivity analysis
                      pHuntB = .75,
                      pHBleft = 2/3,
                      deltaS = 0.2,
                      Cxs = 0.1, # updated from 0.054 based on previous sensitivity analysis
                      deltaW = 0.2, # updated from .07 based on previous sensitivity analysis
                      Axw = 96.6,
                      Bxw = 10, # updated from 1 based on previous sensitivity analysis
                      x = 0.32,
                      Cxw = 0.038,
                      Axs = 20,
                      Cvs = 0.036
  ) {
    return(c(
      R0 = R0,
      k0 = k0,
      Avd = Avd,
      Bxx = Bxx,
      Cvd = Cvd,
      deltaD = deltaD,
      Axb = Axb,
      Cvb = Cvb,
      deltaB = deltaB,
      z = z,
      za = za,
      Bz = Bz,
      Cdb = Cdb,
      wpr = wpr,
      aimpopD = aimpopD,
      pHuntD = pHuntD,
      pHDleft = pHDleft,
      aimpopB = aimpopB,
      pHuntB = pHuntB,
      pHBleft = pHBleft,
      deltaS = deltaS,
      Cxs = Cxs,
      deltaW = deltaW,
      Axw = Axw,
      Bxw = Bxw,
      x = x,
      Cxw = Cxw,
      Axs = Axs,
      Cvs = Cvs
    ))
  }
  
  ## With boar, without wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 1,
                       D = 1,
                       W = 0 #NA
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=500, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopD = pp_param()[["aimpopD"]]*seq(from = .5,
                                                                       to = 1.5,
                                                                       length.out = 5),
                                         aimpopB = pp_param()[["aimpopB"]]*seq(from = .5,
                                                                       to = 1.5,
                                                                       length.out = 5),
                                         pHuntD = pp_param()[["pHuntD"]]*seq(from = .5,
                                                                       to = 1.5,
                                                                       length.out = 5),
                                         pHuntB = pp_param()[["pHuntB"]]*seq(from = .5,
                                                                             to = 1.5,
                                                                             length.out = 5)))
    
    plot(sens_sim, vrbl = "V", savefile = "hunting analysis/sens_aimpop_pHunt_withboar_nowolves_V")
    
    plot(sens_sim, vrbl = "D", savefile = "hunting analysis/sens_aimpop_pHunt_withboar_nowolves_D")
    
    plot(sens_sim, vrbl = "B", savefile = "hunting analysis/sens_aimpop_pHunt_withboar_nowolves_B")
    
    plot(sens_sim, vrbl = "S", savefile = "hunting analysis/sens_aimpop_pHunt_withboar_nowolves_S")
    ##### NOTE: pHunt parameters don't change the qualitative patterns, so it's better to stick to one value for these and influence the realized population control only through aimpop parameters
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=500, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopD = pp_param()[["aimpopD"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5),
                                         aimpopB = pp_param()[["aimpopB"]]*seq(from = .5,
                                                                               to = 1.5,
                                                                               length.out = 5),
                                         pHDleft = pp_param()[["pHDleft"]]*seq(from = .5,
                                                                             to = 1.5,
                                                                             length.out = 5),
                                         pHBleft = pp_param()[["pHBleft"]]*seq(from = .5,
                                                                             to = 1.5,
                                                                             length.out = 5)))
    
    plot(sens_sim, vrbl = "V", savefile = "hunting analysis/sens_aimpop_pHleft_withboar_nowolves_V")
    
    plot(sens_sim, vrbl = "D", savefile = "hunting analysis/sens_aimpop_pHleft_withboar_nowolves_D")
    
    plot(sens_sim, vrbl = "B", savefile = "hunting analysis/sens_aimpop_pHleft_withboar_nowolves_B")
    
    plot(sens_sim, vrbl = "S", savefile = "hunting analysis/sens_aimpop_pHleft_withboar_nowolves_S")
    ##### NOTE: clear patterns, now continue with manually picked values for the actual analysis below
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopD = seq(from = .2, to = 1, by = .2),
                                         aimpopB = seq(from = .2, to = 1, by = .2),
                                         pHDleft = seq(from = 0, to = 1, by = .25),
                                         pHBleft = seq(from = 0, to = 1, by = .25)))
    
    plot(sens_sim, vrbl = "V", savefile = "hunting analysis/sens_aimpop_withboar_nowolves_V")
    
    plot(sens_sim, vrbl = "D", savefile = "hunting analysis/sens_aimpop_withboar_nowolves_D")
    
    plot(sens_sim, vrbl = "B", savefile = "hunting analysis/sens_aimpop_withboar_nowolves_B")
    
    plot(sens_sim, vrbl = "S", savefile = "hunting analysis/sens_aimpop_withboar_nowolves_S")
    ##### NOTE: switching panels with color/linetype will probably make the plots clearer
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .25),
                                         pHBleft = seq(from = 0, to = 1, by = .25),
                                         aimpopD = seq(from = .2, to = 1, by = .2),
                                         aimpopB = seq(from = .2, to = 1, by = .2)))
    
    plot(sens_sim, vrbl = "V", savefile = "hunting analysis/final/sens_withboar_nowolves_V")
    
    plot(sens_sim, vrbl = "D", savefile = "hunting analysis/final/sens_withboar_nowolves_D")
    
    plot(sens_sim, vrbl = "B", savefile = "hunting analysis/final/sens_withboar_nowolves_B")
    
    plot(sens_sim, vrbl = "S", savefile = "hunting analysis/final/sens_withboar_nowolves_S")
    
    sens_full_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=1000, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopD = seq(from = 0, to = 3, by = .1),
                                         aimpopB = seq(from = 0, to = 3, by = .1),
                                         pHDleft = seq(from = 0, to = 1, by = .25),
                                         pHBleft = seq(from = 0, to = 1, by = .25)))
    
    plot(sens_full_sim, vrbl = "V", type = "endState", 
         savefile = "hunting analysis/sensend_aimpop_withboar_nowolves_V")
    
    plot(sens_full_sim, vrbl = "D", type = "endState", 
         savefile = "hunting analysis/sensend_aimpop_withboar_nowolves_D")
    
    plot(sens_full_sim, vrbl = "B", type = "endState", 
         savefile = "hunting analysis/sensend_aimpop_withboar_nowolves_B")
    
    plot(sens_full_sim, vrbl = "S", type = "endState", 
         savefile = "hunting analysis/sensend_aimpop_withboar_nowolves_S")
    ##### NOTE: also for end state switching panels will probably make the plots clearer
    
    sens_full_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=1000, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .05),
                                         pHBleft = seq(from = 0, to = 1, by = .05),
                                         aimpopD = seq(from = .2, to = 1, by = .2),
                                         aimpopB = seq(from = .2, to = 1, by = .2)))
    
    plot(sens_full_sim, vrbl = "V", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_nowolves_V")
    
    plot(sens_full_sim, vrbl = "D", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_nowolves_D")
    
    plot(sens_full_sim, vrbl = "B", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_nowolves_B")
    
    plot(sens_full_sim, vrbl = "S", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_nowolves_S")
  }
  
  ## With boar, with wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 1,
                       D = 1,
                       W = .01
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .25),
                                         pHBleft = seq(from = 0, to = 1, by = .25),
                                         aimpopD = seq(from = .2, to = 1, by = .2),
                                         aimpopB = seq(from = .2, to = 1, by = .2)))
    
    plot(sens_sim, vrbl = "V", savefile = "hunting analysis/final/sens_withboar_withwolves_V")
    
    plot(sens_sim, vrbl = "D", savefile = "hunting analysis/final/sens_withboar_withwolves_D")
    
    plot(sens_sim, vrbl = "B", savefile = "hunting analysis/final/sens_withboar_withwolves_B")
    
    plot(sens_sim, vrbl = "S", savefile = "hunting analysis/final/sens_withboar_withwolves_S")
    
    plot(sens_sim, vrbl = "W", savefile = "hunting analysis/final/sens_withboar_withwolves_W")
    ##### NOTE: wolves can take over role of hunters to sustain scavengers
    
    sens_full_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=1000, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .05),
                                         pHBleft = seq(from = 0, to = 1, by = .05),
                                         aimpopD = seq(from = .2, to = 1, by = .2),
                                         aimpopB = seq(from = .2, to = 1, by = .2)))
    
    plot(sens_full_sim, vrbl = "V", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_withwolves_V")
    
    plot(sens_full_sim, vrbl = "D", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_withwolves_D")
    
    plot(sens_full_sim, vrbl = "B", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_withwolves_B")
    
    plot(sens_full_sim, vrbl = "S", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_withwolves_S")
    
    plot(sens_full_sim, vrbl = "W", type = "endState", 
         savefile = "hunting analysis/final/sensend_withboar_withwolves_W")
  }
  
  ## Without boar, with wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 0, #NA
                       D = 1,
                       W = .01
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .25),
                                         aimpopD = seq(from = .2, to = 1, by = .2))
      )
    
    plot(sens_sim, vrbl = "V", savefile = "hunting analysis/final/sens_noboar_withwolves_V")
    
    plot(sens_sim, vrbl = "D", savefile = "hunting analysis/final/sens_noboar_withwolves_D")
    
    plot(sens_sim, vrbl = "S", savefile = "hunting analysis/final/sens_noboar_withwolves_S")
    
    plot(sens_sim, vrbl = "W", savefile = "hunting analysis/final/sens_noboar_withwolves_W")
    
    sens_full_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=1000, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .05),
                                         aimpopD = seq(from = .2, to = 1, by = .04))
      )
    
    plot(sens_full_sim, vrbl = "V", type = "endState", 
         savefile = "hunting analysis/final/sensend_noboar_withwolves_V")
    
    plot(sens_full_sim, vrbl = "D", type = "endState", 
         savefile = "hunting analysis/final/sensend_noboar_withwolves_D")
    
    plot(sens_full_sim, vrbl = "S", type = "endState", 
         savefile = "hunting analysis/final/sensend_noboar_withwolves_S")
    
    plot(sens_full_sim, vrbl = "W", type = "endState", 
         savefile = "hunting analysis/final/sensend_noboar_withwolves_W")
  }
  
  ## Without boar, without wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 0, #NA
                       D = 1,
                       W = 0 #NA
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .25),
                                         aimpopD = seq(from = .2, to = 1, by = .2))
      )
    
    plot(sens_sim, vrbl = "V", savefile = "hunting analysis/final/sens_noboar_nowolves_V")
    
    plot(sens_sim, vrbl = "D", savefile = "hunting analysis/final/sens_noboar_nowolves_D")
    
    plot(sens_sim, vrbl = "S", savefile = "hunting analysis/final/sens_noboar_nowolves_S")
    
    sens_full_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=1000, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(pHDleft = seq(from = 0, to = 1, by = .05),
                                         aimpopD = seq(from = .2, to = 1, by = .04))
      )
    
    plot(sens_full_sim, vrbl = "V", type = "endState", 
         savefile = "hunting analysis/final/sensend_noboar_nowolves_V")
    
    plot(sens_full_sim, vrbl = "D", type = "endState", 
         savefile = "hunting analysis/final/sensend_noboar_nowolves_D")
    
    plot(sens_full_sim, vrbl = "S", type = "endState", 
         savefile = "hunting analysis/final/sensend_noboar_nowolves_S")
  }
}

### Manuscript figures
{
  ## Set pars
  {
    pp_param = function(R0 = 4,
                        k0 = 10,
                        Avd = 23.5,
                        Bxx = 10,
                        Cvd = 0.025, # updated from 0.017 based on previous sensitivity analysis
                        deltaD = 0.125,
                        Axb = 20.15,
                        Cvb = 0.055,
                        deltaB = 0.08,
                        z = 0.2,
                        za = 0.16,
                        Bz = 0.1,
                        Cdb = 0.069,
                        wpr = 10,
                        aimpopD = .5, # updated from 1 based on previous sensitivity analysis &  set to "medium" value for manuscript figure
                        pHuntD = .5, # set to "medium" value for manuscript figure
                        pHDleft = .5, # set to "medium" value for manuscript figure
                        aimpopB = .5, # updated from 1 based on previous sensitivity analysis &  set to "medium" value for manuscript figure
                        pHuntB = .5, # set to "medium" value for manuscript figure
                        pHBleft = .5, # set to "medium" value for manuscript figure
                        deltaS = 0.2,
                        Cxs = 0.1, # updated from 0.054 based on previous sensitivity analysis
                        deltaW = 0.2, # updated from .07 based on previous sensitivity analysis
                        Axw = 96.6,
                        Bxw = 10, # updated from 1 based on previous sensitivity analysis
                        x = 0.32,
                        Cxw = 0.038,
                        Axs = 20,
                        Cvs = 0.036
    ) {
      return(c(
        R0 = R0,
        k0 = k0,
        Avd = Avd,
        Bxx = Bxx,
        Cvd = Cvd,
        deltaD = deltaD,
        Axb = Axb,
        Cvb = Cvb,
        deltaB = deltaB,
        z = z,
        za = za,
        Bz = Bz,
        Cdb = Cdb,
        wpr = wpr,
        aimpopD = aimpopD,
        pHuntD = pHuntD,
        pHDleft = pHDleft,
        aimpopB = aimpopB,
        pHuntB = pHuntB,
        pHBleft = pHBleft,
        deltaS = deltaS,
        Cxs = Cxs,
        deltaW = deltaW,
        Axw = Axw,
        Bxw = Bxw,
        x = x,
        Cxw = Cxw,
        Axs = Axs,
        Cvs = Cvs
      ))
    }
  }
  
  ## Without boar, without wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 0, #NA
                       D = 1,
                       W = 0 #NA
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHuntD = seq(from = 0, to = 1, by = .25),
                                        pHuntB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_nowolves_S_pHunt")
    plot(sens_sim, vrbl = "D")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(aimpopD = seq(from = 0, to = 1, by = .25),
                                        aimpopB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_nowolves_S_aimpop")
    plot(sens_sim, vrbl = "D")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHDleft = seq(from = 0, to = 1, by = .25),
                                        pHBleft = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_nowolves_S_pHleft")
    plot(sens_sim, vrbl = "D")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopD = seq(from = 0, to = 1, by = .25),
                                         pHDleft = seq(from = 0, to = 1, by = .25),
                                         pHuntD = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_nowolves_S_allhuntingvars")
    plot(sens_sim, vrbl = "D")
  }
  
  ## Without boar, with wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 0, #NA
                       D = 1,
                       W = 0.01
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHuntD = seq(from = 0, to = .5, by = .1),
                                        pHuntB = seq(from = 0, to = .5, by = .1))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_withwolves_S_pHunt")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "W")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(aimpopD = seq(from = 0, to = 1, by = .25),
                                        aimpopB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_withwolves_S_aimpop")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "W")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHDleft = seq(from = 0, to = 1, by = .25),
                                        pHBleft = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_withwolves_S_pHleft")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "W")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopD = seq(from = 0, to = 1, by = .25),
                                         pHDleft = seq(from = 0, to = 1, by = .25),
                                         pHuntD = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/noboar_withwolves_S_allhuntingvars")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "W")
  }
  
  ## With boar, without wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 1,
                       D = 1,
                       W = 0 #NA
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHuntD = seq(from = 0, to = 1, by = .25),
                                        pHuntB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_nowolves_S_pHunt")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "B")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(aimpopD = seq(from = 0, to = 1, by = .25),
                                        aimpopB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_nowolves_S_aimpop")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "B")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHDleft = seq(from = 0, to = 1, by = .25),
                                        pHBleft = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_nowolves_S_pHleft")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "B")
    plot(sens_sim, vrbl = "VconsS")
    plot(sens_sim, vrbl = "BscavS")
    plot(sens_sim, vrbl = "DscavS")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopB = seq(from = 0, to = 1, by = .25),
                                         pHBleft = seq(from = 0, to = 1, by = .25),
                                         pHuntB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_nowolves_S_allhuntingvars")
  }
  
  ## With boar, without wolves
  {
    pp_init = function(V = 1,
                       S = 1,
                       B = 1,
                       D = 1,
                       W = 0.01
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHuntD = seq(from = 0, to = 1, by = .25),
                                        pHuntB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_withwolves_S_pHunt")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "B")
    plot(sens_sim, vrbl = "W")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(aimpopD = seq(from = 0, to = 1, by = .25),
                                        aimpopB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_withwolves_S_aimpop")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "B")
    plot(sens_sim, vrbl = "W")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = data.frame(pHDleft = seq(from = 0, to = 1, by = .25),
                                        pHBleft = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_withwolves_S_pHleft")
    plot(sens_sim, vrbl = "D")
    plot(sens_sim, vrbl = "B")
    plot(sens_sim, vrbl = "W")
    plot(sens_sim, vrbl = "killedBleft")
    plot(sens_sim, vrbl = "killedDleft")
    plot(sens_sim, vrbl = "huntedBleft")
    plot(sens_sim, vrbl = "huntedDleft")
    plot(sens_sim, vrbl = "VconsS")
    plot(sens_sim, vrbl = "BscavS")
    plot(sens_sim, vrbl = "DscavS")
    
    sens_sim = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpopB = seq(from = 0, to = 1, by = .25),
                                         pHBleft = seq(from = 0, to = 1, by = .25),
                                         pHuntB = seq(from = 0, to = 1, by = .25))
      )
    
    plot(sens_sim, vrbl = "S", savefile = "manuscript/withboar_withwolves_S_allhuntingvars")
  }
  
  ## Final subfigures
  {
    pp_param = function(R0 = 4,
                        k0 = 10,
                        Avd = 23.5,
                        Bxx = 10,
                        Cvd = 0.025, # updated from 0.017 based on previous sensitivity analysis
                        deltaD = 0.125,
                        Axb = 20.15,
                        Cvb = 0.055,
                        deltaB = 0.08,
                        z = 0.2,
                        za = 0.16,
                        Bz = 0.1,
                        Cdb = 0.069,
                        wpr = 10,
                        aimpopD = .5, # updated from 1 based on previous sensitivity analysis &  set to "medium" value for manuscript figure
                        pHuntD = .5, # set to "medium" value for manuscript figure
                        pHDleft = .5, # set to "medium" value for manuscript figure
                        aimpopB = .5, # updated from 1 based on previous sensitivity analysis &  set to "medium" value for manuscript figure
                        pHuntB = .5, # set to "medium" value for manuscript figure
                        pHBleft = .5, # set to "medium" value for manuscript figure
                        deltaS = 0.2,
                        Cxs = 0.1, # updated from 0.054 based on previous sensitivity analysis
                        deltaW = 0.2, # updated from .07 based on previous sensitivity analysis
                        Axw = 96.6,
                        Bxw = 10, # updated from 1 based on previous sensitivity analysis
                        x = 0.32,
                        Cxw = 0.038,
                        Axs = 20,
                        Cvs = 0.036,
                        aimpop = NA_real_, # added to set the 2 D and B hunting pars simultaneously
                        pHunt = NA_real_, # added to set the 2 D and B hunting pars simultaneously
                        pHleft = NA_real_ # added to set the 2 D and B hunting pars simultaneously
    ) {
      return(c(
        R0 = R0,
        k0 = k0,
        Avd = Avd,
        Bxx = Bxx,
        Cvd = Cvd,
        deltaD = deltaD,
        Axb = Axb,
        Cvb = Cvb,
        deltaB = deltaB,
        z = z,
        za = za,
        Bz = Bz,
        Cdb = Cdb,
        wpr = wpr,
        aimpopD = ifelse(is.na(aimpop),aimpopD,aimpop),
        pHuntD = ifelse(is.na(pHunt),pHuntD,pHunt),
        pHDleft = ifelse(is.na(pHleft),pHDleft,pHleft),
        aimpopB = ifelse(is.na(aimpop),aimpopB,aimpop),
        pHuntB = ifelse(is.na(pHunt),pHuntB,pHunt),
        pHBleft = ifelse(is.na(pHleft),pHBleft,pHleft),
        deltaS = deltaS,
        Cxs = Cxs,
        deltaW = deltaW,
        Axw = Axw,
        Bxw = Bxw,
        x = x,
        Cxw = Cxw,
        Axs = Axs,
        Cvs = Cvs,
        aimpop = aimpop,
        pHunt = pHunt,
        pHleft = pHleft
      ))
    }
    
    pp_init = function(V = 1,
                       S = 1,
                       B = 1,
                       D = 1,
                       W = 0.01
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    withboar_withwolves = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5))
      )
    
    plot(withboar_withwolves, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_V")
    plot(withboar_withwolves, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_S")
    plot(withboar_withwolves, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_D")
    plot(withboar_withwolves, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_B")
    plot(withboar_withwolves, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_W")
    plot(withboar_withwolves, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_huntkillleftfrac_01transf")
    plot(withboar_withwolves, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_carrionDBfrac")
    plot(withboar_withwolves, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/withboar_withwolves_carrionDB")
    
    withboar_withwolves_cvs = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5),
                                         Cvs = seq(from = .026, to = .046, by = .01))
      )
    
    plot(withboar_withwolves_cvs, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_V")
    plot(withboar_withwolves_cvs, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_S")
    plot(withboar_withwolves_cvs, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_D")
    plot(withboar_withwolves_cvs, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_B")
    plot(withboar_withwolves_cvs, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_W")
    plot(withboar_withwolves_cvs, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_huntkillleftfrac_01transf")
    plot(withboar_withwolves_cvs, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_carrionDBfrac")
    plot(withboar_withwolves_cvs, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_withwolves_cvs_carrionDB")
    
    pp_init = function(V = 1,
                       S = 1,
                       B = 0,
                       D = 1,
                       W = 0.01
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    noboar_withwolves = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5))
      )
    
    plot(noboar_withwolves, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_V")
    plot(noboar_withwolves, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_S")
    plot(noboar_withwolves, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_D")
    plot(noboar_withwolves, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_B")
    plot(noboar_withwolves, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_W")
    plot(noboar_withwolves, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_huntkillleftfrac_01transf")
    plot(noboar_withwolves, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_carrionDBfrac")
    plot(noboar_withwolves, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/noboar_withwolves_carrionDB")
    
    noboar_withwolves_cvs = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5),
                                         Cvs = seq(from = .026, to = .046, by = .01))
      )
    
    plot(noboar_withwolves_cvs, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_V")
    plot(noboar_withwolves_cvs, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_S")
    plot(noboar_withwolves_cvs, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_D")
    plot(noboar_withwolves_cvs, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_B")
    plot(noboar_withwolves_cvs, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_W")
    plot(noboar_withwolves_cvs, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_huntkillleftfrac_01transf")
    plot(noboar_withwolves_cvs, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_carrionDBfrac")
    plot(noboar_withwolves_cvs, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_withwolves_cvs_carrionDB")
    
    pp_init = function(V = 1,
                       S = 1,
                       B = 1,
                       D = 1,
                       W = 0
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    withboar_nowolves = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5))
      )
    
    plot(withboar_nowolves, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_V")
    plot(withboar_nowolves, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_S")
    plot(withboar_nowolves, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_D")
    plot(withboar_nowolves, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_B")
    plot(withboar_nowolves, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_W")
    plot(withboar_nowolves, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_huntkillleftfrac_01transf")
    plot(withboar_nowolves, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_carrionDBfrac")
    plot(withboar_nowolves, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/withboar_nowolves_carrionDB")
    
    withboar_nowolves_cvs = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5),
                                         Cvs = seq(from = .026, to = .046, by = .01))
      )
    
    plot(withboar_nowolves_cvs, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_V")
    plot(withboar_nowolves_cvs, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_S")
    plot(withboar_nowolves_cvs, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_D")
    plot(withboar_nowolves_cvs, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_B")
    plot(withboar_nowolves_cvs, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_W")
    plot(withboar_nowolves_cvs, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_huntkillleftfrac_01transf")
    plot(withboar_nowolves_cvs, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_carrionDBfrac")
    plot(withboar_nowolves_cvs, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/withboar_nowolves_cvs_carrionDB")
    
    pp_init = function(V = 1,
                       S = 1,
                       B = 0,
                       D = 1,
                       W = 0
    ) {
      return(c(
        V = V,
        S = S,
        B = B,
        D = D,
        W = W
      ))
    }
    
    noboar_nowolves = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5))
      )
    
    plot(noboar_nowolves, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_V")
    plot(noboar_nowolves, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_S")
    plot(noboar_nowolves, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_D")
    plot(noboar_nowolves, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_B")
    plot(noboar_nowolves, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_W")
    plot(noboar_nowolves, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_huntkillleftfrac_01transf")
    plot(noboar_nowolves, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_carrionDBfrac")
    plot(noboar_nowolves, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/noboar_nowolves_carrionDB")
    
    noboar_nowolves_cvs = 
      odeSensFun(y = pp_init(),
                 times = seq(from=0, to=250, by=1),
                 func = pp,
                 parmsFun = pp_param,
                 namedSeqs = expand.grid(aimpop = seq(from = 0, to = 1, by = .5),
                                         pHleft = seq(from = 0, to = 1, by = .5),
                                         Cvs = seq(from = .026, to = .046, by = .01))
      )
    
    plot(noboar_nowolves_cvs, vrbl = "V", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_V")
    plot(noboar_nowolves_cvs, vrbl = "S", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_S")
    plot(noboar_nowolves_cvs, vrbl = "D", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_D")
    plot(noboar_nowolves_cvs, vrbl = "B", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_B")
    plot(noboar_nowolves_cvs, vrbl = "W", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_W")
    plot(noboar_nowolves_cvs, vrbl = "huntkillleftfrac_01transf", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_huntkillleftfrac_01transf")
    plot(noboar_nowolves_cvs, vrbl = "carrionDBfrac", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_carrionDBfrac")
    plot(noboar_nowolves_cvs, vrbl = "carrionDB", colasfact = TRUE, savefile = "manuscript/final/cvs_sens/noboar_nowolves_cvs_carrionDB")
  }
  
  ## Combined final subfigures
  {
    all_figs_dat = c(noboar_nowolves_cvs,
                     noboar_withwolves_cvs,
                     withboar_nowolves_cvs,
                     withboar_withwolves_cvs)
    
    comb_figs_dat = list()
    
    for(i in seq_along(all_figs_dat)) {
      comb_figs_dat[[i]] = as.data.table(all_figs_dat[[i]])
      
      comb_figs_dat[[i]][, `:=` (
        init_B = ifelse(attr(all_figs_dat[[i]],"initials")[["B"]]==0,
                        FALSE,
                        TRUE),
        init_W = ifelse(attr(all_figs_dat[[i]],"initials")[["W"]]==0,
                        FALSE,
                        TRUE),
        par_aimpop = attr(all_figs_dat[[i]],"parameters")[["aimpop"]],
        par_pHleft = attr(all_figs_dat[[i]],"parameters")[["pHleft"]],
        par_Cvs = attr(all_figs_dat[[i]],"parameters")[["Cvs"]]
        )]
    }
    
    comb_figs_dat = rbindlist(comb_figs_dat)
    setkeyv(comb_figs_dat,
            c("init_B",
              "init_W",
              "par_aimpop",
              "par_pHleft",
              "par_Cvs",
              "time"))
    setcolorder(comb_figs_dat)
    
    if(FALSE) {
      saveRDS(comb_figs_dat,
              "comb_figs_dat.RDS")
    } # only run if comb_figs_dat.RDS can be overwritten
    
    if(FALSE) {
      comb_figs_dat = readRDS("comb_figs_dat.RDS")
    } # only run if comb_figs_dat needs to be loaded from disk
    
    plotLabeller = function(variable,value) {
      if(variable=="init_B") {
        return(paste("Boar presence","=",value))
      } else {
        return(paste("Wolf presence","=",value))
      }
    }
    
    comb_figs_dat[, `:=` (par_aimpop = as.factor(par_aimpop),
                          par_pHleft = as.factor(par_pHleft),
                          par_Cvs = as.factor(par_Cvs))]
    
    cvs = c(.026,.036,.046)
    
    for(i in seq_along(cvs)) {
      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = V,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "Vegetation biomass density",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$V))) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/V",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = S,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "Scavenger biomass density",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$S))) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/S",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = D,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "Deer biomass density",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$D))) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/D",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = B,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "Boar biomass density",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$B))) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/B",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = W,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "Wolf biomass density",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$W))) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/W",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = huntkillleftfrac_01transf,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "1 - e^(-(Carrion biomass density from hunting) / (Carrion biomass density from predation))",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,1)) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/huntkillleftfrac_01transf",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i] & init_W==TRUE,],
                  aes(x = time,
                      y = huntkillleftfrac_01transf,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "1 - e^(-(Carrion biomass density from hunting) / (Carrion biomass density from predation))",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(. ~ init_B,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,1)) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/huntkillleftfrac_01transf_1col",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 4000,
             height = 2000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = carrionDBfrac,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "(Carrion deer biomass density) / (Carrion boar biomass density)",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$carrionDBfrac))) +
        theme(legend.position = "top")

      ggsave(paste0("plots/","manuscript/final/combined/carrionDBfrac",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")
      
      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i] & init_B==TRUE,],
                  aes(x = time,
                      y = carrionDBfrac,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "(Carrion deer biomass density) / (Carrion boar biomass density)",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(. ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$carrionDBfrac))) +
        theme(legend.position = "top")
      
      ggsave(paste0("plots/","manuscript/final/combined/carrionDBfrac_1col",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 4000,
             height = 2000,
             units = "px")

      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = carrionDB,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "Carrion deer + boar biomass density",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$carrionDB))) +
        theme(legend.position = "top")
      
      ggsave(paste0("plots/","manuscript/final/combined/carrionDB",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")
      
      pp = ggplot(comb_figs_dat[par_Cvs==cvs[i],],
                  aes(x = time,
                      y = Vcarriongrowthfrac_01transf,
                      group = interaction(par_aimpop,
                                          par_pHleft,
                                          init_B,
                                          init_W),
                      col = par_aimpop,
                      linetype = par_pHleft)) +
        geom_line() +
        labs(x = "Time (y)",
             y = "1 - e^(-(Scavenger growth from vegetation) / (Scavenger growth from carrion))",
             colour = "Hunters' target biomass density for deer and boar population",
             linetype = "Fraction of carrion left by hunters") +
        facet_grid(init_B ~ init_W,
                   labeller = plotLabeller) +
        expand_limits(y = c(0,max(comb_figs_dat$Vcarriongrowthfrac_01transf))) +
        theme(legend.position = "top")
      
      ggsave(paste0("plots/","manuscript/final/combined/Vcarriongrowthfrac_01transf",
                    ifelse(cvs[i]==.036,
                           "",
                           ifelse(cvs[i]==.026,
                                  "_lowCvs",
                                  "_highCvs")),
                    ".pdf"),
             pp,
             "pdf",
             width = 3000,
             height = 3000,
             units = "px")
    }
  }
}
