# Raquel Viula, 2022
# "Discomfort Glare in Classrooms: an Investigation in Space"
# PhD Thesis, TU Delft

# glare model parameter estimation for different positions in the classroom


#### load required libraries ####
library("data.table")
library("GA")
library("dplyr")
# to increase the number of decimals in the parameter calculations
library(tibble)
options(pillar.sigfig = 6)


# identify working directory (of R project)
setwd("~/glare-spatial-model")
wd <- getwd()
#
#### GA optimisation outputs  ####
#
# THE LINES BELOW SHOULD BE COMMENTED TO AVOID ACCIDENTALLY DELETING ALL RESULTS STORED FROM PREVIOUS RUNS

glare_parameter_optimisation_results <- data.frame(
  # optimisation context
  position=character(), parameter=character(), glare=character()
  # optimisation input
  c1_lbound=double(), c1_ubound=double(), c2_lbound=double(), c2_ubound=double(), c3_lbound=double(), c3_ubound=double(), c4_lbound=double(), c4_ubound=double(),
  c1_suggest=double(), c2_suggest=double(), c3_suggest=double(), c4_suggest=double(),
  # optimisation parameters
  popsize=integer(), maxiterations=integer(), iter_stop=integer(), seed=integer(),
  # optimisation results
  iterations=integer(), fitness_val=double(), c1_solution=double(), c2_solution=double(), c3_solution=double(), c4_solution=double(),
  stringsAsFactors=FALSE
)
# write the headers
write.csv(glare_parameter_optimisation_results,file=paste0(wd,"/outputs/ch9/model_r2_optimisation_results_log.csv"),row.names=F)

glare_parameter_optimisation_population <- data.frame(
  # optimisation context
  position=character(), parameter=character(), glare=character(),
  # population parameters
  c1_solution=double(), c2_solution=double(), c3_solution=double(), c4_solution=double(),
  # population fitness
  fitness_val=double(),
  stringsAsFactors=FALSE
)
# write the headers
write.csv(glare_parameter_optimisation_population,file=paste0(wd,"/outputs/ch9/model_r2_optimisation_population_log.csv"),row.names=F)

# END SAFETY COMMENT


#
#### calculation routines ####
#

# function: calculate DGP and DGP with log10(E_v)
dgp_coeff <- function(E_v1,E_v2,l_s,omega_s,posindx,c1,c2,c3,c4){
  dgp <- (c1*10^-5*E_v1)+coalesce((c2*10^-2*log10(1+sum((l_s^2*omega_s)/(E_v2^c3*posindx^2)))),0)+c4
  return(dgp)
}
calc_dgp <- function(df,c1=5.87,c2=9.18,c3=1.87,c4=0.16){
  df %>%
    group_by(year, subject, position) %>%
    dplyr::summarise(dgp_calc = first(dgp_coeff(E_v,E_v,l_s,omega_s,posindx,c1,c2,c3,c4)),.groups="keep")
}
calc_dgp_log_ev <- function(df,c1=5.87,c2=9.18,c3=1.87,c4=0.16){
  df %>%
    group_by(year, subject, position) %>%
    dplyr::summarise(dgp_log_ev = first(dgp_coeff(log10(E_v),E_v,l_s,omega_s,posindx,c1,c2,c3,c4)),.groups="keep")
}


# function: calculate UGP
ugp_coeff <- function(lum_backg,l_s,omega_s,posindx,c1,c2){
  ugp <- abs(c1)*coalesce(log10((abs(c2)/lum_backg)*sum((l_s^2*omega_s)/coalesce(posindx^2,0))),0)
  return(ugp)
}
calc_ugp <- function(df,c1=0.26,c2=0.25){
  df %>%
    group_by(year, subject, position) %>%
    dplyr::summarise(ugp_calc = first(ugp_coeff(lum_backg,l_s,omega_s,posindx,c1,c2)),.groups="keep")
}

# function: calculate bins
calc_bins <- function(df,nbin,parameter){
  ranked_values = rank(df[,parameter],ties.method='first')
  df$bin = as.integer(cut(ranked_values,quantile(ranked_values, c(0:nbin)/nbin),include.lowest=T))    
  return(df[,c("year","subject","position",parameter,"bin")])
}

# function: join parameters and glare vote data
join_data <- function(df_parameters, df_glare){
  df_join = merge(df_glare[,c(1:5,32:33)],df_parameters,by=c('year','subject','position'),all = FALSE)
  return(df_join)
}

# function: calculate glare
calc_glare <- function(df,ind,ind_glare){
  df_agg = aggregate(df[,ind], by=list(df$bin), FUN=mean)
  colnames(df_agg)[1:2] = c('bin','ind_mean')
  df_agg$ind_sd = aggregate(df, by=list(df$bin), FUN=sd)$bin
  df_agg$size = aggregate(df, by=list(df$bin), FUN=length)$bin
  df_agg$pc_glare = aggregate(df[,ind_glare], by=list(df$bin), FUN=sum)$x/df_agg$size
  return(df_agg)
}

# function: calculate regression R2
calc_r2_fit <- function(df_agg){
  # test negative corr
  val_cor = cor(df_agg$ind_mean,df_agg$pc_glare)
  if(val_cor < 0){
    val_r2 = 0
  } else {
    lm_df_agg = lm(pc_glare ~ ind_mean, data=df_agg)
    # test cooks distance
    val_cooksd = cooks.distance(lm_df_agg)
    if (length(which(val_cooksd > 1)) == 0){
      val_r2 = summary(lm_df_agg)[[8]]      
    } else { #if any value exceeds the cook's distance
      val_r2 = 0
    }
  }
  return(coalesce(val_r2,0))
}


#
#### optimisation parameters ####
#
optim_parameters = c("dgp_calc","dgp_log_ev","ugp_calc")
names(optim_parameters) = c("DGP mean","DGP log Ev mean","ugp mean")
optim_glare = c("disturbing","any_glare")
names(optim_glare) = c("% disturbing glare","% any glare")
optim_positions = c("wall", "window", "full")
names(optim_positions) = c("Wall positions","Window positions","Room")

#
#### optimisation fitness functions ####
#

# function: DGP fitness (c1, c2, c3, c4)
fitness_r2_dgp <- function(glare_data, vote_data, glare, c1, c2, c3, c4, bins=10){
  # calculate the parameter
  df_dgp = calc_dgp(glare_data,c1,c2,c3,c4)
  # calculate bins
  df_bins = calc_bins(df_dgp,bins,'dgp_calc')
  # join glare and vote data
  df_join = join_data(df_bins,vote_data)
  # aggregate by bin
  df_glare = calc_glare(df_join,'dgp_calc',glare)
  # calculate fitness
  r2 = calc_r2_fit(df_glare)
  return(r2)
}

# function: DGP log E_v fitness (c1, c2, c3, c4)
fitness_r2_dgp_log_ev <- function(glare_data, vote_data, glare, c1, c2, c3, c4, bins=10){
  # calculate the parameter
  df_dgp = calc_dgp_log_ev(glare_data,c1,c2,c3,c4)
  # calculate bins
  df_bins = calc_bins(df_dgp,bins,'dgp_log_ev')
  # join glare and vote data
  df_join = join_data(df_bins,vote_data)
  # aggregate by bin
  df_glare = calc_glare(df_join,'dgp_log_ev',glare)
  # calculate fitness
  r2 = calc_r2_fit(df_glare)
  return(r2)
}

# function: UGP fitness (c1, c2)
fitness_r2_ugp <- function(glare_data, vote_data, glare, c1, c2, bins=10){
  # calculate the parameter
  df_ugp = calc_ugp(glare_data,c1,c2)
  # calculate bins
  df_bins = calc_bins(df_ugp,bins,'ugp_calc')
  # join glare and vote data
  df_join = join_data(df_bins,vote_data)
  # aggregate by bin
  df_glare = calc_glare(df_join,'ugp_calc',glare)
  # calculate fitness
  r2 = calc_r2_fit(df_glare)
  return(r2)
}

#
#### GA optimisation functions ####
#

# DGP optimisation
GA_r2_dgp <- function(glare_data,position_df,position,glare_ind,
  # define parameters ranges
  c1_lbound=0, c1_ubound=20,
  c2_lbound=0, c2_ubound=20,
  c3_lbound=0, c3_ubound=20,
  c4_lbound=0, c4_ubound=20,
  bins=10,
  # suggest population member
  c1_suggest=5.87,c2_suggest=9.18,
  c3_suggest=1.87,c4_suggest=0.16,
  # optimisation settings
  popsize=50,maxiterations=1000,
  iter_stop=100,my_seed=12345)
{
  # run the GA algorithm
  GA_dgp <- ga(type = "real-valued", 
           fitness =  function(x) fitness_r2_dgp(glare_data,position_df,glare_ind,x[1], x[2], x[3], x[4], bins),
           lower = c(c1_lbound, c2_lbound, c3_lbound, c4_lbound), 
           upper = c(c1_ubound, c2_ubound, c3_ubound, c4_ubound), 
           suggestions = as.matrix(data.frame(c1_suggest,c2_suggest,c3_suggest,c4_suggest), ncol = 4),
           popSize = popsize, pcrossover = 0.8, pmutation = 0.1, keepBest = TRUE,
           maxiter = maxiterations, run = iter_stop, seed = my_seed, 
           parallel = TRUE, monitor = FALSE)
  # write data results
  GA_r2_results(GA_dgp,"dgp_calc",position,glare_ind,my_seed)
  # write best population results
  GA_r2_population(GA_dgp,"dgp_calc",position,glare_ind)
  # plot results
  GA_r2_plot(GA_dgp,"dgp_calc",position,glare_ind)
  return(GA_dgp)
}

# DGP log e_v optimisation
GA_r2_dgp_log_ev <- function(glare_data,position_df,position,glare_ind,
       # define parameters ranges
       c1_lbound=0, c1_ubound=20,
       c2_lbound=0, c2_ubound=20,
       c3_lbound=0, c3_ubound=20,
       c4_lbound=0, c4_ubound=20,
       bins=10,
       # suggest population member
       c1_suggest=5.87,c2_suggest=9.18,
       c3_suggest=1.87,c4_suggest=0.16,
       # optimisation settings
       popsize=50,maxiterations=1000,
       iter_stop=100,my_seed=12345)
{
  # run the GA algorithm
  GA_dgp <- ga(type = "real-valued", 
               fitness =  function(x) fitness_r2_dgp_log_ev(glare_data,position_df,glare_ind,x[1], x[2], x[3], x[4], bins),
               lower = c(c1_lbound, c2_lbound, c3_lbound, c4_lbound), 
               upper = c(c1_ubound, c2_ubound, c3_ubound, c4_ubound), 
               suggestions = as.matrix(data.frame(c1_suggest,c2_suggest,c3_suggest,c4_suggest), ncol = 4),
               popSize = popsize, pcrossover = 0.8, pmutation = 0.1, keepBest = TRUE,
               maxiter = maxiterations, run = iter_stop, seed = my_seed, 
               parallel = TRUE, monitor = FALSE)
  # write data results
  GA_r2_results(GA_dgp,"dgp_log_ev",position,glare_ind,my_seed)
  # write best population results
  GA_r2_population(GA_dgp,"dgp_log_ev",position,glare_ind)
  # plot results
  GA_r2_plot(GA_dgp,"dgp_log_ev",position,glare_ind)
  return(GA_dgp)
}

# UGP optimisation
GA_r2_ugp <- function(glare_data,position_df,position,glare_ind,
          # define parameters ranges
          c1_lbound=0.0001, c1_ubound=20,
          c2_lbound=0.0001, c2_ubound=20,
          bins=10,
          # suggest population member
          c1_suggest=0.26,c2_suggest=0.25,
          # optimisation settings
          popsize=50,maxiterations=1000,
          iter_stop=100,my_seed=12345)
{
  # run the GA algorithm
  GA_ugp <- ga(type = "real-valued", 
               fitness =  function(x) fitness_r2_ugp(glare_data,position_df,glare_ind,x[1], x[2], bins),
               lower = c(c1_lbound, c2_lbound), upper = c(c1_ubound, c2_ubound), 
               suggestions = as.matrix(data.frame(c1_suggest,c2_suggest), ncol = 2),
               popSize = popsize, pcrossover = 0.8, pmutation = 0.1, keepBest = TRUE,
               maxiter = maxiterations, run = iter_stop, seed = my_seed, 
               parallel = TRUE, monitor = FALSE)
  # write data results
  GA_r2_results(GA_ugp,"ugp_calc",position,glare_ind,my_seed)
  # write best population results
  GA_r2_population(GA_ugp,"ugp_calc",position,glare_ind)
  # plot results
  GA_r2_plot(GA_ugp,"ugp_calc",position,glare_ind)
  return(GA_ugp)
}

#
##### Plot GA results ####
#
GA_r2_plot <- function(ga_results,parameter,position,glare_ind){
  plot(0,0, type="n", xlab = "", ylab = "fitness", xlim = c(1,ga_results@iter), ylim = c(0,ga_results@fitnessValue))
  lines(c(1:ga_results@iter),ga_results@summary[,1],type="l",lwd=1.5,lty=1, col="green")
  lines(c(1:ga_results@iter),ga_results@summary[,2],type="l",lwd=1,lty=1, col="blue")
  if(parameter=='ugp_calc'){
    title(
      main=paste0(names(which(optim_parameters==parameter))," ",position," optimisation for  % ",glare_ind),
      sub=paste0(ga_results@iter," iterations, R2 = ",round(ga_results@fitnessValue,3),
                 " (c1=",round(ga_results@solution[1,1],3),", c2=",round(ga_results@solution[1,2],3),")"
      )
    )    
  } else {
    title(
      main=paste0(names(which(optim_parameters==parameter))," ",position," optimisation for  % ",glare_ind),
      sub=paste0(ga_results@iter," iterations, R2 = ",round(ga_results@fitnessValue,3),
                " (c1=",round(ga_results@solution[1,1],3),", c2=",round(ga_results@solution[1,2],3),
                ", c3=",round(ga_results@solution[1,3],3),", c4=",round(ga_results@solution[1,4],3),")"
                )
      )
  }  
}

#
#### Write GA results ####
#
GA_r2_results <- function(ga_results,parameter,position,glare_ind,my_seed){
  if(parameter=='ugp_calc'){
    df_data = data.frame(position, parameter = names(which(optim_parameters==parameter)), names(which(optim_glare==glare_ind)),
                         ga_results@lower[1],ga_results@upper[1],ga_results@lower[2],ga_results@upper[2], 0, 0, 0, 0,
                         ga_results@suggestions[1],ga_results@suggestions[2],0,0,
                         ga_results@popSize,ga_results@maxiter,ga_results@run,my_seed,
                         ga_results@iter,ga_results@fitnessValue,
                         ga_results@solution[1,1],ga_results@solution[1,2],0,0)
  } else {
    df_data = data.frame(position, parameter = names(which(optim_parameters==parameter)), names(which(optim_glare==glare_ind)),
                         ga_results@lower[1],ga_results@upper[1],ga_results@lower[2],ga_results@upper[2],ga_results@lower[3], ga_results@upper[3],ga_results@lower[4],ga_results@upper[4],
                         ga_results@suggestions[1],ga_results@suggestions[2],ga_results@suggestions[3],ga_results@suggestions[4],
                         ga_results@popSize,ga_results@maxiter,ga_results@run,my_seed,
                         ga_results@iter,ga_results@fitnessValue,
                         ga_results@solution[1,1],ga_results@solution[1,2],ga_results@solution[1,3],ga_results@solution[1,4])    
  }
  write.table(df_data,file=paste0(wd,"/outputs/ch9/model_r2_optimisation_results_log.csv"),sep = ",",append=T,col.names=F)  
}

#
#### Write GA population ####
#
GA_r2_population <- function(ga_results,parameter,position,glare_ind){
  top_cases = lapply(ga_results@bestSol,head,1) #get only the first case when there are ties
  best_pop = data.frame(matrix(unlist(top_cases), ncol = max(lengths(top_cases)), byrow = TRUE))
  if(parameter=='ugp_calc'){
    df_data = data.frame(position, parameter = names(which(optim_parameters==parameter)), names(which(optim_glare==glare_ind)),
                         best_pop[,1],best_pop[,2],0,0,ga_results@summary[,1])
  } else {
    df_data = data.frame(position, parameter = names(which(optim_parameters==parameter)), names(which(optim_glare==glare_ind)),
                         best_pop[,1],best_pop[,2],best_pop[,3],best_pop[,4],
                         ga_results@summary[,1])    
  }
  write.table(df_data %>% distinct(),file=paste0(wd,"/outputs/ch9/model_r2_optimisation_population_log.csv"),sep = ",",append=T,col.names=F)  
}


#
##### GA optimisation runs ####
#
### using modified parameters
optim_dgp <- GA_r2_dgp(wall_glare_data, wall_vote_data,'wall','disturbing',
     # define parameters ranges
     c1_lbound=0, c1_ubound=15,
     c2_lbound=0, c2_ubound=20,
     c3_lbound=0, c3_ubound=5,
     c4_lbound=0, c4_ubound=0,
     # suggest population member
     c1_suggest=5.87,c2_suggest=9.18,
     c3_suggest=1.87,c4_suggest=0,
     # optimisation settings
     popsize=50,
     maxiterations=10000,
     iter_stop=10000,
     my_seed=12345
)

optim_dgp_log_ev <- GA_r2_dgp_log_ev(wall_glare_data, wall_vote_data,'wall','disturbing',
       # define parameters ranges
       c1_lbound=0, c1_ubound=15,
       c2_lbound=0, c2_ubound=20,
       c3_lbound=0, c3_ubound=5,
       c4_lbound=0, c4_ubound=0,
       # suggest population member
       c1_suggest=5.87,c2_suggest=9.18,
       c3_suggest=1.87,c4_suggest=0,
       # optimisation settings
       popsize=50,
       maxiterations=10000,
       iter_stop=10000,
       my_seed=12345
)


optim_ugp <- GA_r2_ugp(wall_glare_data, wall_vote_data,'wall','disturbing',
         # define parameters ranges
         c1_lbound=0.26, c1_ubound=0.26,
         c2_lbound=0, c2_ubound=0.15,
         # optimisation settings
         popsize=50,
         maxiterations=10000,
         iter_stop=10000,
         my_seed=12345
)

optim_dgp <- GA_r2_dgp(wall_glare_data, wall_vote_data,'wall','any_glare',
        # define parameters ranges
        c1_lbound=0, c1_ubound=15,
        c2_lbound=0, c2_ubound=20,
        c3_lbound=0, c3_ubound=5,
        c4_lbound=0, c4_ubound=0,
        # suggest population member
        c1_suggest=5.87,c2_suggest=9.18,
        c3_suggest=1.87,c4_suggest=0,
        # optimisation settings
        popsize=50,
        maxiterations=10000,
        iter_stop=10000,
        my_seed=12345
)

optim_dgp_log_ev <- GA_r2_dgp_log_ev(wall_glare_data, wall_vote_data,'wall','any_glare',
        # define parameters ranges
        c1_lbound=0, c1_ubound=15,
        c2_lbound=0, c2_ubound=20,
        c3_lbound=0, c3_ubound=5,
        c4_lbound=0, c4_ubound=0,
        # suggest population member
        c1_suggest=5.87,c2_suggest=9.18,
        c3_suggest=1.87,c4_suggest=0,
        # optimisation settings
        popsize=50,
        maxiterations=10000,
        iter_stop=10000,
        my_seed=12345
)

optim_ugp <- GA_r2_ugp(wall_glare_data, wall_vote_data,'wall','any_glare',
        # define parameters ranges
        c1_lbound=0.26, c1_ubound=0.26,
        c2_lbound=0, c2_ubound=0.15,
        # optimisation settings
        popsize=50,
        maxiterations=10000,
        iter_stop=10000,
        my_seed=12345
)

optim_dgp <- GA_r2_dgp(window_glare_data, window_vote_data,'window','disturbing',
        # define parameters ranges
        c1_lbound=0, c1_ubound=15,
        c2_lbound=0, c2_ubound=20,
        c3_lbound=0, c3_ubound=5,
        c4_lbound=0, c4_ubound=0,
        # suggest population member
        c1_suggest=5.87,c2_suggest=9.18,
        c3_suggest=1.87,c4_suggest=0,
        # optimisation settings
        popsize=50,
        maxiterations=10000,
        iter_stop=10000,
        my_seed=12345
)

optim_dgp_log_ev <- GA_r2_dgp_log_ev(window_glare_data, window_vote_data,'window','disturbing',
        # define parameters ranges
        c1_lbound=0, c1_ubound=15,
        c2_lbound=0, c2_ubound=20,
        c3_lbound=0, c3_ubound=5,
        c4_lbound=0, c4_ubound=0,
        # suggest population member
        c1_suggest=5.87,c2_suggest=9.18,
        c3_suggest=1.87,c4_suggest=0,
        # optimisation settings
        popsize=50,
        maxiterations=10000,
        iter_stop=10000,
        my_seed=12345
)

optim_ugp <- GA_r2_ugp(window_glare_data, window_vote_data,'window','disturbing',
      # define parameters ranges
      c1_lbound=0.26, c1_ubound=0.26,
      c2_lbound=0, c2_ubound=0.15,
      # optimisation settings
      popsize=50,
      maxiterations=10000,
      iter_stop=10000,
      my_seed=12345
)

optim_dgp <- GA_r2_dgp(window_glare_data, window_vote_data,'window','any_glare',
      # define parameters ranges
      c1_lbound=0, c1_ubound=15,
      c2_lbound=0, c2_ubound=20,
      c3_lbound=0, c3_ubound=5,
      c4_lbound=0, c4_ubound=0,
      # suggest population member
      c1_suggest=5.87,c2_suggest=9.18,
      c3_suggest=1.87,c4_suggest=0,
      # optimisation settings
      popsize=50,
      maxiterations=10000,
      iter_stop=10000,
      my_seed=12345
)

optim_dgp_log_ev <- GA_r2_dgp_log_ev(window_glare_data, window_vote_data,'window','any_glare',
      # define parameters ranges
      c1_lbound=0, c1_ubound=15,
      c2_lbound=0, c2_ubound=20,
      c3_lbound=0, c3_ubound=5,
      c4_lbound=0, c4_ubound=0,
      # suggest population member
      c1_suggest=5.87,c2_suggest=9.18,
      c3_suggest=1.87,c4_suggest=0,
      # optimisation settings
      popsize=50,
      maxiterations=10000,
      iter_stop=10000,
      my_seed=12345
)

optim_ugp <- GA_r2_ugp(window_glare_data, window_vote_data,'window','any_glare',
      # define parameters ranges
      c1_lbound=0.26, c1_ubound=0.26,
      c2_lbound=0, c2_ubound=0.15,
      # optimisation settings
      popsize=50,
      maxiterations=10000,
      iter_stop=10000,
      my_seed=12345
)


#### Calculate optimised population model ####
#
# read in population results file
population_results <- read.csv(paste0(wd,"/outputs/ch9/model_r2_optimisation_population_log.csv"),header = TRUE)
#initiate vectors for extra results
intercept_vals <- c()
slope_vals <- c()
# loop to calculate extra linear model results for best population
for(row in 1:nrow(population_results)){
  # extract information for analysis
  bins <- 10
  c1 <- population_results[row,"c1_solution"]
  c2 <- population_results[row,"c2_solution"]
  c3 <- population_results[row,"c3_solution"]
  c4 <- population_results[row,"c4_solution"]
  if(population_results[row,"position"] == 'window'){
    input_data <- window_glare_data
  }else{
    input_data <- wall_glare_data
  }
  if(population_results[row,"glare"] == "% disturbing glare"){
    glare <- "disturbing"    
  }else{
    glare <- "any_glare"
  }
  # calculate the parameter
  if(population_results[row,"parameter"] == 'DGP mean'){
    parameter <- calc_dgp(input_data,c1,c2,c3,c4)
    column <- 'dgp_calc'
  }else if(population_results[row,"parameter"] == 'ugp mean'){
    parameter <- calc_ugp(input_data,c1,c2)
    column <- 'ugp_calc'
  }else if(population_results[row,"parameter"] == 'DGP log Ev mean'){
    parameter <- calc_dgp_log_ev(input_data,c1,c2,c3,c4)
    column <- 'dgp_log_ev'
  }
  # calculate bins
  df_bins = calc_bins(parameter,bins,column)
  # join glare and vote data
  df_join = join_data(df_bins,vote_data)
  # aggregate by bin
  df_glare = calc_glare(df_join,column,glare)
  # calculate fitness
  lm_df_agg = lm(pc_glare ~ ind_mean, data=df_glare)
  intercept_vals = c(intercept_vals,coef(summary(lm_df_agg))[[1]])
  slope_vals = c(slope_vals,coef(summary(lm_df_agg))[[2]])
}
# append new values to columns
population_results$slope <- slope_vals
population_results$intercept <- intercept_vals
write.table(population_results,file=paste0(wd,"/outputs/ch9/model_r2_optimisation_population_log.csv"),sep = ",",append=F,col.names=T,row.names=F)

