# Raquel Viula, 2022
# "Discomfort Glare in Classrooms: an Investigation in Space"
# PhD Thesis, TU Delft

# glare model final results r2 for alt solution (c4=0)


#### load required libraries ####
library("data.table")
library("dplyr")
# to increase the number of decimals in the parameter calculations
library(tibble)
options(pillar.sigfig = 6)

# identify working directory (of R project)
setwd("~/glare-spatial-model")
wd <- getwd()


# read in optimisation solutions file ("hand-made")
optimisation_solutions <- read.csv(paste0(wd,"/outputs/ch9/model_r2_optimisation_solution_alt.csv"),header = TRUE)
optimisation_solutions[,1]<-as.character(optimisation_solutions[,1])
optimisation_solutions[,2]<-as.character(optimisation_solutions[,2])
optimisation_solutions[,3]<-as.character(optimisation_solutions[,3])


#
#### Calculate r2 for standard optimised equations ####
#
# results dataframe for the new optimised equation (includes intercept and slope)
optimised_regression_results = data.frame(
  position=character(),
  metric=character(),
  glare=character(),
  c1=double(),
  c2=double(),
  c3=double(),
  c4=double(),
  bins=integer(),
  val_sd_mean=double(),
  val_sd_max=double(),
  val_r2=double(),
  val_adjr2=double(),
  val_p=double(),
  se_intercept=double(),
  se_slope=double(),
  val_intercept=double(),
  val_slope=double(),
  val_rmse=double(),
  stringsAsFactors=FALSE
)
# write the headers
write.csv(optimised_regression_results,file=paste0(wd,"/outputs/ch9/optimised_glare_regression_alt.csv"),row.names=F)

# vote bins dataframe
optimised_bins_data = data.frame(
  position=character(),
  metric=character(),
  glare=character(),
  bin=integer(),
  ind_mean=double(),
  ind_sd=double(),
  size=double(),
  pc_glare=double(),
  stringsAsFactors=FALSE
)
# write the headers
write.csv(optimised_bins_data,file=paste0(wd,"/outputs/ch9/optimised_glare_bins_alt.csv"),row.names=F)
#

# loop to calculate extra linear model results for best population
for(row in 1:nrow(optimisation_solutions)){
  # extract information for analysis
  c1 <- deframe(optimisation_solutions[row,"c1_solution"])
  c2 <- deframe(optimisation_solutions[row,"c2_solution"])
  c3 <- deframe(optimisation_solutions[row,"c3_solution"])
  c4 <- deframe(optimisation_solutions[row,"c4_solution"])
  position <- deframe(optimisation_solutions[row,"position"])
  #
  bins <- 10
  if(position == 'window'){
    input_data <- window_glare_data
  }else{
    input_data <- wall_glare_data  
  }
  glare <- optimisation_solutions[row,"glare"]
  # calculate the metric
  if(optimisation_solutions[row,"parameter"] == 'DGP'){
    parameter <- calc_dgp(input_data,c1,c2,c3,c4)
    column <- 'dgp_calc'
  }else if(optimisation_solutions[row,"parameter"] == 'UGP'){
    parameter <- calc_ugp(input_data,c1,c2)
    column <- 'ugp_calc'
  }else{
    parameter <- calc_dgp_log_ev(input_data,c1,c2,c3,c4)
    column <- 'dgp_log_ev'
  }
  # calculate bins
  df_bins = calc_bins(parameter,bins,column)
  # join glare and vote data
  df_join = join_data(df_bins,vote_data_1)
  # aggregate by bin
  df_glare = calc_glare(df_join,column,glare)
  # calculate fitness
  lm_df_agg = lm(pc_glare ~ ind_mean, data=df_glare)
  # extract values
  val_r2 = summary(lm_df_agg)[[8]]
  val_adjr2 = summary(lm_df_agg)[[9]]
  val_p = coef(summary(lm_df_agg))[[8]]
  val_intercept = coef(summary(lm_df_agg))[[1]]
  val_slope = coef(summary(lm_df_agg))[[2]]
  val_interc_std_err = coef(summary(lm_df_agg))[[3]]
  val_slope_std_err = coef(summary(lm_df_agg))[[4]]
  val_sd_mean = mean(df_glare$ind_sd)
  val_sd_max = max(df_glare$ind_sd)
  val_rmse = sqrt(mean(lm_df_agg$residuals^2))

  # draw a plot with interval bars
  plot(df_glare$ind_mean, df_glare$pc_glare, col="red", pch = 10, cex = 1, ylim=c(0, 1), xlab = column, 
       ylab = names(which(glare_inds==glare)),
       main = bquote(.(position) ~ .(bins) ~ " bins: " ~ R^2 == .(round(val_r2,5)) ~ ", p-value: " ~ .(round(val_p,5))))
  axis(side=2, at=seq(0, 1, by=0.2))
  segments(df_glare$ind_mean-(df_glare$ind_sd/2), df_glare$pc_glare, df_glare$ind_mean+(df_glare$ind_sd/2), df_glare$pc_glare)
  lines(df_glare$ind_mean,predict(lm_df_agg), lty=2 )
  
  # write the model results
  res <- data.frame(position,metric=column,glare=names(which(glare_inds==glare)),c1,c2,c3,c4,bins,val_sd_mean,val_sd_max,val_r2,val_adjr2,val_p,val_interc_std_err,val_slope_std_err,val_intercept,val_slope,val_rmse)
  optimised_regression_results <- rbind(optimised_regression_results,res)
  write.table(res,file=paste0(wd,"/outputs/ch9/optimised_glare_regression_alt.csv"),sep = ",",append=T,col.names=F,row.names=F)
  # write data results
  df_data = data.frame(position = position, metric = column, glare = names(which(glare_inds==glare)), df_glare)
  write.table(df_data,file=paste0(wd,"/outputs/ch9/optimised_glare_bins_alt.csv"),sep = ",",append=T,col.names=F,row.names=F)
}


#
#### Calculate metrics using standard optimised equations ####
#
# store calculation in data table
calculation_results <- vote_data[,c(1:5,27,29,31)]
setDT(calculation_results)

for(row in 1:nrow(optimised_regression_results)){
  # extract information for analysis
  c1 <- deframe(optimised_regression_results[row,"c1"])
  c2 <- deframe(optimised_regression_results[row,"c2"])
  c3 <- deframe(optimised_regression_results[row,"c3"])
  c4 <- deframe(optimised_regression_results[row,"c4"])
  position <- deframe(optimised_regression_results[row,"position"])
  #
  bins <- 10
  if(position == 'window'){
    input_data <- window_glare_data
  }else{
    input_data <- wall_glare_data
  }
  if(optimised_regression_results[row,"glare"] == "% disturbing glare"){
    glare <- "disturbing"    
  }else{
    glare <- "any_glare"    
  }
  # calculate the metric
  if(optimised_regression_results[row,"metric"] == 'dgp_calc'){
    metric <- calc_dgp(input_data,c1,c2,c3,c4)
    column <- 'dgp_optim'
    result_col <- quote(dgp_calc)
  }else if(optimised_regression_results[row,"metric"] == 'ugp_calc'){
    metric <- calc_ugp(input_data,c1,c2)
    column <- 'ugp_optim'
    result_col <- quote(ugp_calc)
  }else{
    metric <- calc_dgp_log_ev(input_data,c1,c2,c3,c4)
    column <- 'dgp_log_ev_optim'
    result_col <- quote(dgp_log_ev)
  }
  # append results column
  setDT(metric)
  optim_col <- paste(column,glare,sep="_")
  calculation_results[metric, on = .(year, subject, position), eval(optim_col) := metric[,eval(result_col)]]
}

# write the results
write.csv(calculation_results,file=paste0(wd,"/outputs/ch9/optimised_glare_calculation_alt.csv"),row.names=F)

rm(bins, c1,c2,c3,c4,position,input_data,glare,metric,column,result_col,optim_col)


#
#### Calculate adaptation and contrast for the original and optimised equations ####
#
# adaptation calculation functions
dgp_adaptation <- function(E_v1,c1){
  adaptation <- (c1*10^-5*E_v1)
  return(adaptation)
}
dgp_contrast <- function(E_v2,l_s,omega_s,posindx,c2,c3){
  contrast <- coalesce((c2*10^-2*log10(1+sum((l_s^2*omega_s)/(E_v2^c3*posindx^2)))),0)
  return(contrast)
}
calc_dgp_terms <- function(df,c1=5.87,c2=9.18,c3=1.87){
  df %>%
    group_by(year, subject, position) %>%
    dplyr::summarise(dgp_adaptation = first(dgp_adaptation(E_v,c1)),
                     dgp_contrast = first(dgp_contrast(E_v,l_s,omega_s,posindx,c2,c3)),
                    .groups="keep")
}
calc_dgp_log_ev_terms <- function(df,c1=5.87,c2=9.18,c3=1.87){
  df %>%
    group_by(year, subject, position) %>%
    dplyr::summarise(dgp_log_ev_adaptation = first(dgp_adaptation(log10(E_v),c1)),
                     dgp_log_ev_contrast = first(dgp_contrast(E_v,l_s,omega_s,posindx,c2,c3)),
                     .groups="keep")
}

# prepare calculation data table
terms_calculation_results <- vote_data[,c(1:5)]
setDT(terms_calculation_results)

# Calculate glare terms
metric <- calc_dgp_terms(glare_data)
# append results column
setDT(metric)
optim_col <- colnames(metric[,4])
terms_calculation_results[metric, on = .(year, subject, position), eval(optim_col) := metric[,4]]
optim_col <- colnames(metric[,5])
terms_calculation_results[metric, on = .(year, subject, position), eval(optim_col) := metric[,5]]
#
metric <- calc_dgp_log_ev_terms(glare_data)
# append results column
setDT(metric)
optim_col <- colnames(metric[,4])
terms_calculation_results[metric, on = .(year, subject, position), eval(optim_col) := metric[,4]]
optim_col <- colnames(metric[,5])
terms_calculation_results[metric, on = .(year, subject, position), eval(optim_col) := metric[,5]]

# Next calculate the optimised glare terms
for(row in 1:nrow(optimised_regression_results)){
  # extract information for analysis
  c1 <- deframe(optimised_regression_results[row,"c1"])
  c2 <- deframe(optimised_regression_results[row,"c2"])
  c3 <- deframe(optimised_regression_results[row,"c3"])
  position <- deframe(optimised_regression_results[row,"position"])
  #
  bins <- 10
  if(position == 'window'){
    input_data <- window_glare_data
  }else{
    input_data <- wall_glare_data
  }
  if(optimised_regression_results[row,"glare"] == "% disturbing glare"){
    glare <- "disturbing"    
  }else{
    glare <- "any_glare"    
  }
  # calculate the metric
  if(optimised_regression_results[row,"metric"] == 'dgp_calc'){
    metric <- calc_dgp_terms(input_data,c1,c2,c3)
  }else{
    metric <- calc_dgp_log_ev_terms(input_data,c1,c2,c3)
  }
  # append results column
  setDT(metric)
  optim_col <- paste(colnames(metric[,4]),"optim",glare,sep="_")
  terms_calculation_results[metric, on = .(year, subject, position), eval(optim_col) := metric[,4]]

  optim_col <- paste(colnames(metric[,5]),"optim",glare,sep="_")
  terms_calculation_results[metric, on = .(year, subject, position), eval(optim_col) := metric[,5]]
}

# write the results
write.csv(terms_calculation_results,file=paste0(wd,"/outputs/ch9/calculated_glare_terms_alt.csv"),row.names=F)


rm(bins, c1,c2,c3,position,input_data,glare,metric,optim_col)


#
#### Correlation between adaptation and contrast of different equations ####
#
cor_data <- data.frame(terms_calculation_results)
# results dataframe
correlation_terms_results = data.frame(
  position=character(),
  metric=character(),
  glare=character(),
  val_cor=double(),
  val_r2=double(),
  val_intercept=double(),
  val_slope=double(),
  stringsAsFactors=FALSE
)
# write the headers
write.csv(correlation_terms_results,file=paste0(wd,"/outputs/ch9/correlation_glare_terms_alt.csv"),row.names=F)
#
correlation_terms_plots <- function(df,x,y,position,metric,glare){
  val_cor = cor(df[,x],df[,y])
  lm_df = lm(df[,y] ~ df[,x])
  val_r2 = summary(lm_df)[[8]]
  val_intercept = coef(lm_df)[[1]]
  val_slope = coef(lm_df)[[2]]

  # make plot
  plot(df[,x],df[,y], col="red", pch = 10, cex = 0.5, xlab = x, ylab = y,
       main = paste(position,metric,glare,"r:",round(val_cor,3),"inter:",round(val_intercept,3),"slope:",round(val_slope,3),sep=" "))
  lines(df[,x],predict(lm_df), lty=2 )

  # write results
  df_data = data.frame(position, metric, glare, val_cor, val_r2, val_intercept, val_slope)
  write.table(df_data,file=paste0(wd,"/outputs/ch9/correlation_glare_terms_alt.csv"),sep = ",",append=T,col.names=F)
}


# wall data
correlation_terms_plots(cor_data[cor_data$position %in% c("3","4"),],"dgp_adaptation","dgp_contrast","wall","dgp","original")
correlation_terms_plots(cor_data[cor_data$position %in% c("3","4"),],"dgp_log_ev_adaptation","dgp_log_ev_contrast","wall","dgp_log_ev","original")

correlation_terms_plots(cor_data[cor_data$position %in% c("3","4"),],"dgp_adaptation_optim_disturbing","dgp_contrast_optim_disturbing","wall","dgp","disturbing")
correlation_terms_plots(cor_data[cor_data$position %in% c("3","4"),],"dgp_log_ev_adaptation_optim_disturbing","dgp_log_ev_contrast_optim_disturbing","wall","dgp_log_ev","disturbing")

correlation_terms_plots(cor_data[cor_data$position %in% c("3","4"),],"dgp_adaptation_optim_any_glare","dgp_contrast_optim_any_glare","wall","dgp","any glare")
correlation_terms_plots(cor_data[cor_data$position %in% c("3","4"),],"dgp_log_ev_adaptation_optim_any_glare","dgp_log_ev_contrast_optim_any_glare","wall","dgp_log_ev","any glare")

# window data
correlation_terms_plots(cor_data[cor_data$position %in% c("1","2"),],"dgp_adaptation","dgp_contrast","window","dgp","original")
correlation_terms_plots(cor_data[cor_data$position %in% c("1","2"),],"dgp_log_ev_adaptation","dgp_log_ev_contrast","window","dgp_log_ev","original")

correlation_terms_plots(cor_data[cor_data$position %in% c("1","2"),],"dgp_adaptation_optim_disturbing","dgp_contrast_optim_disturbing","window","dgp","disturbing")
correlation_terms_plots(cor_data[cor_data$position %in% c("1","2"),],"dgp_log_ev_adaptation_optim_disturbing","dgp_log_ev_contrast_optim_disturbing","window","dgp_log_ev","disturbing")

correlation_terms_plots(cor_data[cor_data$position %in% c("1","2"),],"dgp_adaptation_optim_any_glare","dgp_contrast_optim_any_glare","window","dgp","any glare")
correlation_terms_plots(cor_data[cor_data$position %in% c("1","2"),],"dgp_log_ev_adaptation_optim_any_glare","dgp_log_ev_contrast_optim_any_glare","window","dgp_log_ev","any glare")


# tidy up
rm(c1,c2,c3,c4,bins,position,column,metric,df_bins,df_join,df_glare,lm_df_agg,val_adjr2,val_interc_std_err,val_r2,val_p,
   val_intercept,val_slope,val_slope_std_err,val_sd_max,val_sd_mean,res,glare)
