# Raquel Viula, 2022
# "Discomfort Glare in Classrooms: an Investigation in Space"
# PhD Thesis, TU Delft

# glare data modelling preparation

#### load required libraries ####
library("plyr")
library("dplyr")


# identify working directory (of R project)
setwd("~/glare-spatial-model")
wd <- getwd()

#
#### read input data ####
#
glare_data <- read.csv(paste0(getwd(),"/input/glare_data.csv"),header = TRUE)
vote_data <- read.csv(paste0(getwd(),"/input/vote_data.csv"),header = TRUE)


#
#### calculate new glare parameters ####
#
is.nan.df <- function(x)
  do.call(cbind, lapply(x, is.nan))

# calculate dgp, ugp and dgp log(ev) per case
dgp <- function(E_v1,E_v2,l_s,omega_s,posindx){
  dgp <- first(((5.87*10^-5*E_v1)+coalesce((9.18*10^-2*log10(1+sum((l_s^2*omega_s)/(E_v2^1.87*posindx^2)))),0)+0.16))
}
lowlight_correction <- function(E_v){
  if(first(E_v) < 320){
    correction <- first((exp(1)^((0.024*E_v)-4))/(1+(exp(1)^((0.024*E_v)-4))))    
  } else {
    correction <- 1
  }
}
ugp <- function(lum_backg,l_s,omega_s,posindx){
  ugp <- first(coalesce(0.26*log10((0.25/lum_backg)*sum((l_s^2*omega_s)/coalesce(posindx^2,0))),0))
}
# calculate summaries and output to CSV
glare_parameters <- glare_data %>%
  group_by(year, subject, position, sequence) %>%
  dplyr::summarise(min_posindx = min(posindx),
                   mean_omega_s = mean(omega_s, na.rm = TRUE),
                   mean_l_s = mean(l_s, na.rm = TRUE),
                   E_v_ind = min(E_v)-min(E_v_dir),
                   lsmean_lavg = mean(l_s, na.rm = TRUE)/min(av_lum),
                   lsmean_ev = mean(l_s, na.rm = TRUE)/min(E_v),
                   lsmean_lt = mean(l_s, na.rm = TRUE)/min(Lt),
                   ev_lt = min(E_v)/min(Lt),
                   ev_lb = min(E_v)/min(lum_backg),
                   dgp_calc = dgp(E_v,E_v,l_s,omega_s,posindx),
                   dgp_lowlight = dgp(E_v,E_v,l_s,omega_s,posindx)*lowlight_correction(E_v),
                   dgp_log_ev = dgp(log10(E_v),E_v,l_s,omega_s,posindx),
                   dgp_log_ev_lowlight = dgp(log10(E_v),E_v,l_s,omega_s,posindx)*lowlight_correction(E_v),
                   ugp_calc = ugp(lum_backg,l_s,omega_s,posindx)
  )
# remove NULLs
glare_parameters[is.nan.df(glare_parameters)] <- 0


# merge vote and glare input data with new glare parameters
glare_data = merge(glare_data,glare_parameters,by=c('year','subject','position','sequence'),all = FALSE)
write.csv(glare_data, paste0(getwd(),"/outputs/ch7/glare_data_out.csv"),col.names=T,row.names=F)
vote_data = merge(vote_data,glare_parameters,by=c('year','subject','position','sequence'),all = FALSE)
write.csv(vote_data, paste0(getwd(),"/outputs/ch7/vote_data_out.csv"),col.names=T,row.names=F)

#
#### recode vote to discomfort variable and export vote data as CSV ####
#
vote_data$disturbing = recode(vote_data$vote, '1' = 0, '2' = 0, '3' = 1, '4' = 1)
vote_data$any_glare = recode(vote_data$vote, '1' = 0, '2' = 1, '3' = 1, '4' = 1)
write.csv(vote_data, paste0(getwd(),"/outputs/ch7/vote_data_out.csv"),col.names=T,row.names=F)


#
#### create global objects from output data ####
#
#### create glare data subsets for different positions ####
#
wall_glare_data = glare_data[which(glare_data$position %in% c("3","4")),]
window_glare_data = glare_data[which(glare_data$position %in% c("1","2")),]
#
wall_glare_data_1 = wall_glare_data[which(wall_glare_data$year %in% c("2016","2017","2018")),]
window_glare_data_1 = window_glare_data[which(window_glare_data$year %in% c("2016","2017","2018")),]

#
#### create vote data subsets for different positions ####
#
p1_vote_data = vote_data[which(vote_data$position == "1"),]
p2_vote_data = vote_data[which(vote_data$position == "2"),]
p3_vote_data = vote_data[which(vote_data$position == "3"),]
p4_vote_data = vote_data[which(vote_data$position == "4"),]
#
front_vote_data = vote_data[which(vote_data$position %in% c("2","3")),]
back_vote_data = vote_data[which(vote_data$position %in% c("1","4")),]
#
wall_vote_data = vote_data[which(vote_data$position %in% c("3","4")),]
window_vote_data = vote_data[which(vote_data$position %in% c("1","2")),]


#
#### recode indicator column names for plotting and tables
#
inds = c('av_lum','E_v','lum_backg','Lt','lum_sources','mean_l_s','omega_sources','mean_omega_s','min_posindx','E_v_dir','E_v_ind','dgp',
         'lsmean_lavg','lsmean_ev','lsmean_lt','ev_lt','ev_lb')
names(inds) = c('Lavg','Ev','Lb','Lt','Ls_omega','Ls_mean','omegas_total','omegas_mean','P_min','Ev_dir','Ev_ind','DGP',
                'ls_mean/Lavg','ls_mean/Ev','ls_mean/Lt','Ev/Lt','Ev/Lb')
new_inds = c("dgp_calc","dgp_lowlight","dgp_log_ev","dgp_log_ev_lowlight","ugp_calc")
names(new_inds) = c("DGP mean","DGP mean lowlight","DGP log Ev mean","DGP log Ev mean lowlight","ugp mean")
glare_inds = c("disturbing","any_glare")
names(glare_inds) = c("% disturbing glare","% any glare")
