% This is a script supporting the publication "On optimizing the sensor
% spacing for pressure measurements on wind turbine airfoils" by Fritz,
% Kelley and Brown.
%
% In this code, the pressure sensor layout is optimized for the FFA-W3-241 airfoil
% and the optimized layouts are compared against a cosine layout.
clear all; close all; clc;

%% Input
stin.iNsensors = 9; % Define number of sensors you want to optimize

%% Importing the airfoil data
stairfoil.sairfoil = 'FFA-W3-241';
stairfoil.d2geom = importdata('FFA-W3-241_geomNew.dat');
stairfoil.d2cp = importdata('FFA-W3-241_cp.dat','\t');
sttmp = importdata('FFA-W3-241_polars.dat','\t',1);
stairfoil.d2polars = sttmp.data;

stairfoil.d1prob = ones(length(stairfoil.d2polars),1)/length(stairfoil.d2polars); % Unlike in the publication equal weighting for all angles of attack chosen for simplicity of this script. By adjusting this array, it is possible to put more emphasis on certain angles of attack during the optimization routines

clear sttmp

%% Cosine spacing
d1xSensors = 1 - (cos(linspace(-pi,pi,stin.iNsensors))/2 + 0.5);
[~,iLEsensors] = min(d1xSensors);
d1cSensors = [1 - d1xSensors(1:iLEsensors),1 + d1xSensors(iLEsensors+1:end)];

% XFOIL
[~,iLEairfoil] = min(stairfoil.d2geom(:,1));
d1cCp = [1 - stairfoil.d2geom(1:iLEairfoil,1);1 + stairfoil.d2geom(iLEairfoil+1:end,1)];

stcosine.d1xSensors = d1xSensors;
stcosine.d1ySensors = interp1(d1cCp,stairfoil.d2geom(:,2),d1cSensors);
stcosine.d1cSensors = d1cSensors;
stcosine.d2cpSensors = interp1(d1cCp,stairfoil.d2cp(2:end,2:end)',d1cSensors');
stcosine.d2cpInterp = interp1(d1cSensors',stcosine.d2cpSensors,d1cCp,"linear");

stcosine.d2polarsDer = polarsFromCp(stairfoil.d2geom,stcosine.d2cpInterp,stairfoil.d2cp(2:end,1));

clear d1* i*

%% Run genetic algorithm
% Input variables
d1c = ones(1,stin.iNsensors);

ObjectiveFunction = @(d1c) errorCp(d1c,stairfoil);

iNvars = length(d1c);

dn = 1; %movement range of variables, choose between 0 and 2 (0*c ... 2*c)

d1LB = d1c-dn;
d1UB = d1c+dn;

d1LB(d1LB<0) = 0;
d1UB(d1UB>2) = 2;

mkdir History
mkdir hyHistory

% Optimizer settings
hybridopts = psoptimset('Display','iter','InitialMeshSize',1,'MaxFunEvals',Inf,'MaxIter',Inf);
hybridopts = psoptimset(hybridopts,'UseParallel','always','CompletePoll','on','Vectorized','off'); % Change 'UseParallel' flag if prallel processing toolbox is not available
hybridopts = psoptimset(hybridopts,'OutputFcns',@hyoutfun);

options = gaoptimset('Display','iter','Generations',Inf,'PopulationSize',5000,'StallGenLimit',15,'TolFun',0);
options = gaoptimset(options,'UseParallel','always','Vectorized','off');
options = gaoptimset(options,'OutputFcns',@outfun);
options = gaoptimset(options,'HybridFcn',{@patternsearch,hybridopts});

% Run genetic algorithm
[x,fval,exitflag,output] = ga(ObjectiveFunction,iNvars,[],[],[],[],d1LB,d1UB,[],options);

delete('History\*');
rmdir History
delete('hyHistory\*');
rmdir hyHistory

% Process optimization result
stga.d1c = sort(x);
[~,iLEsensors] = min(abs(stga.d1c - 1));
if stga.d1c(iLEsensors) > 1
    iLEsensors = iLEsensors - 1;
end
stga.d1x = [1 - stga.d1c(1:iLEsensors), stga.d1c(iLEsensors + 1:end) - 1];

[~,iLEaf] = min(stairfoil.d2geom(:,1));
stairfoil.d1cAF = [1 - stairfoil.d2geom(1:iLEaf,1);1 + stairfoil.d2geom(iLEaf + 1:end,1)];

stga.d1y = interp1(stairfoil.d1cAF,stairfoil.d2geom(:,2),stga.d1c);

stga.d2cpSensors = interp1(stairfoil.d1cAF,stairfoil.d2cp(2:end,2:end)',stga.d1c);
stga.d2cpInterp = interp1(stga.d1c',stga.d2cpSensors,stairfoil.d1cAF,'linear','extrap')';

stga.d2polarsDer = polarsFromCp(stairfoil.d2geom,stga.d2cpInterp',stairfoil.d2cp(2:end,1));

stga.derrLift = errorLift(stga.d1c,stairfoil);
stga.derrCp = errorCp(stga.d1c,stairfoil);

clear d* exitflag fval hybridopts i* O* o* x

%% Run SQP optimizer
% Input variables
d1c = linspace(0,2,stin.iNsensors); % The SQP optimizer is heavily dependant on the starting points of the design variables. It can be benefitial to run the optimizer multiple times with random starting points and then pick the best result afterwards.

ObjectiveFunction = @(d1c) errorCp(d1c,stairfoil);

d1UB = 2*ones(size(d1c));
d1LB = zeros(size(d1c));

% Optimizer settings
options = optimoptions('fmincon','Display','iter','MaxFunEvals',Inf,'MaxIter',Inf,'TolCon',1e-15,'UseParallel',false,'Algorithm','sqp');

% Run optimizer
[x,fval] = fmincon(ObjectiveFunction,d1c,[],[],[],[],d1LB,d1UB,[],options);

% Process optimization result
stsqp.d1c = sort(x);
[~,iLEsensors] = min(abs(stsqp.d1c - 1));
if stsqp.d1c(iLEsensors) > 1
    iLEsensors = iLEsensors - 1;
end
stsqp.d1x = [1 - stsqp.d1c(1:iLEsensors), stsqp.d1c(iLEsensors + 1:end) - 1];

[~,iLEaf] = min(stairfoil.d2geom(:,1));
stairfoil.d1cAF = [1 - stairfoil.d2geom(1:iLEaf,1);1 + stairfoil.d2geom(iLEaf + 1:end,1)];

stsqp.d1y = interp1(stairfoil.d1cAF,stairfoil.d2geom(:,2),stsqp.d1c);

stsqp.d2cpSensors = interp1(stairfoil.d1cAF,stairfoil.d2cp(2:end,2:end)',stsqp.d1c);
stsqp.d2cpInterp = interp1(stsqp.d1c',stsqp.d2cpSensors,stairfoil.d1cAF,'linear','extrap')';

stsqp.d2polarsDer = polarsFromCp(stairfoil.d2geom,stsqp.d2cpInterp',stairfoil.d2cp(2:end,1));

stsqp.derrLift = errorLift(stsqp.d1c,stairfoil);
stsqp.derrCp = errorCp(stsqp.d1c,stairfoil);

clear d* fval i* O* o* x

%% Plots
% Plot the sensor distributions
figure('Color','w','WindowState','maximized');
hold on; box on; grid on; grid('minor'); daspect([1 1 1]);
xlabel('x/c [-]'); ylabel('y/c [-]');
xlim([-0.05,1.05]);
legend('Location','northeast');
plot(stairfoil.d2geom(:,1),stairfoil.d2geom(:,2),'-k','DisplayName',stairfoil.sairfoil)
scatter(stcosine.d1xSensors,stcosine.d1ySensors,'or','filled','DisplayName','Cosine')
scatter(stga.d1x,stga.d1y,'sb','filled','DisplayName','GA')
scatter(stsqp.d1x,stsqp.d1y,'dy','filled','DisplayName','SQP')

% Plot the pressure distribution as would be measured by the cosine and
% optimized alyouts
d1aoaoPlot = [-5, 0, 5, 10, 15];

figure('Color','w','WindowState','maximized');
tiledlayout(2,3,'TileSpacing','compact','Padding','compact');
for i = 1:length(d1aoaoPlot)
    iplot = find(stairfoil.d2polars(:,1) == d1aoaoPlot(i));
    nexttile(i); hold on; box on; grid on; grid('minor');
    set(gca,'YDir','reverse');
    xlim([0,1]); 
    xlabel('x/c [-]'); ylabel('c_p [-]'); title(['AOA = ',num2str(d1aoaoPlot(i),'%6.2f'),' deg']);
    legend('Location','northeast');
    plot(stairfoil.d2cp(1,2:end),stairfoil.d2cp(iplot+1,2:end),'-k','DisplayName','XFOIL')
    plot(stairfoil.d2cp(1,2:end),stcosine.d2cpInterp(:,iplot),'-r','DisplayName','Cosine')
    plot(stairfoil.d2cp(1,2:end),stga.d2cpInterp(iplot,:),'-b','DisplayName','GA')
    plot(stairfoil.d2cp(1,2:end),stsqp.d2cpInterp(iplot,:),'-y','DisplayName','SQP')
end

% Plot the polars derived from the various sensor layouts vs the polars
% calculated by XFOIL as well as the resulting error
figure('Color','w','WindowState','maximized');
tiledlayout(2,3,'TileSpacing','compact','Padding','compact');

nexttile(1); hold on; box on; grid on; grid('minor'); % cl
xlabel('\alpha [deg]'); ylabel('c_l [-]');
legend('Location','best');
plot(stairfoil.d2polars(:,1),stairfoil.d2polars(:,2),'-k','DisplayName','XFOIL')
plot(stcosine.d2polarsDer(:,1),stcosine.d2polarsDer(:,2),'-r','DisplayName','Cosine')
plot(stga.d2polarsDer(:,1),stga.d2polarsDer(:,2),'-b','DisplayName','GA')
plot(stsqp.d2polarsDer(:,1),stsqp.d2polarsDer(:,2),'-y','DisplayName','SQP')

nexttile(2); hold on; box on; grid on; grid('minor'); % cdp (pressure drag)
xlabel('\alpha [deg]'); ylabel('c_{d,p} [-]');
legend('Location','best');
plot(stairfoil.d2polars(:,1),stairfoil.d2polars(:,4),'-k','DisplayName','XFOIL')
plot(stcosine.d2polarsDer(:,1),stcosine.d2polarsDer(:,3),'-r','DisplayName','Cosine')
plot(stga.d2polarsDer(:,1),stga.d2polarsDer(:,3),'-b','DisplayName','GA')
plot(stsqp.d2polarsDer(:,1),stsqp.d2polarsDer(:,3),'-y','DisplayName','SQP')

nexttile(3); hold on; box on; grid on; grid('minor'); % cm
xlabel('\alpha [deg]'); ylabel('c_m [-]');
legend('Location','best');
plot(stairfoil.d2polars(:,1),stairfoil.d2polars(:,5),'-k','DisplayName','XFOIL')
plot(stcosine.d2polarsDer(:,1),stcosine.d2polarsDer(:,4),'-r','DisplayName','Cosine')
plot(stga.d2polarsDer(:,1),stga.d2polarsDer(:,4),'-b','DisplayName','GA')
plot(stsqp.d2polarsDer(:,1),stsqp.d2polarsDer(:,4),'-y','DisplayName','SQP')

nexttile(4); hold on; box on; grid on; grid('minor'); % Error cl
xlabel('\alpha [deg]'); ylabel('E(c_l) [-]');
legend('Location','best');
plot(stcosine.d2polarsDer(:,1),stcosine.d2polarsDer(:,2) - stairfoil.d2polars(:,2),'-r','DisplayName','Cosine')
plot(stga.d2polarsDer(:,1),stga.d2polarsDer(:,2) - stairfoil.d2polars(:,2),'-b','DisplayName','GA')
plot(stsqp.d2polarsDer(:,1),stsqp.d2polarsDer(:,2) - stairfoil.d2polars(:,2),'-y','DisplayName','SQP')

nexttile(5); hold on; box on; grid on; grid('minor'); % Error cdp (pressure drag)
xlabel('\alpha [deg]'); ylabel('E(c_{d,p}) [-]');
legend('Location','best');
plot(stcosine.d2polarsDer(:,1),stcosine.d2polarsDer(:,3) - stairfoil.d2polars(:,4),'-r','DisplayName','Cosine')
plot(stga.d2polarsDer(:,1),stga.d2polarsDer(:,3) - stairfoil.d2polars(:,4),'-b','DisplayName','GA')
plot(stsqp.d2polarsDer(:,1),stsqp.d2polarsDer(:,3) - stairfoil.d2polars(:,4),'-y','DisplayName','SQP')

nexttile(6); hold on; box on; grid on; grid('minor'); % Error cm
xlabel('\alpha [deg]'); ylabel('E(c_m) [-]');
legend('Location','best');
plot(stcosine.d2polarsDer(:,1),stcosine.d2polarsDer(:,4) - stairfoil.d2polars(:,5),'-r','DisplayName','Cosine')
plot(stga.d2polarsDer(:,1),stga.d2polarsDer(:,4) - stairfoil.d2polars(:,5),'-b','DisplayName','GA')
plot(stsqp.d2polarsDer(:,1),stsqp.d2polarsDer(:,4) - stairfoil.d2polars(:,5),'-y','DisplayName','SQP')
