# Code and data from the CPAIOR 2021 paper "Learning Variable Activity Initialisation for Lazy Clause Generation Solvers"
#
# Ronald van Driel et al.
# Delft University of Technology, NL
# April 2021
# copyright (c) 2020-2021 by R. van Driel et al.
# GNU General Public Licence 3
#
# contact: n.yorke-smith@tudelft.nl


import subprocess
import glob
import ntpath
import re
from multiprocessing import Pool
import os.path
import sys
import itertools
import time
import random


modelFiles = glob.glob("FlatZincPaths/*/*.fzn.gz")
modelFiles = list(map(lambda x: x.replace('\\', '/'), modelFiles))
commands = {'chuffed0_OG': 'chuffed0_OG/build/fzn-chuffed',
            'chuffed1': 'chuffed1/build/fzn-chuffed',
            'chuffed0_vsids': 'chuffed0_vsids/build/fzn-chuffed'}

predictFiles = glob.glob("PredictFiles/Predictions/*.txt")
predictFiles = list(map(lambda x: x.replace('\\', '/'), predictFiles))
predictNames = list(map(lambda x: re.search("(?<=Predictions/)[^/]+(?=\\.txt)", x).group(0), predictFiles))
outputfile = 'output/'

def runChuffed(tup):
    fznFile = tup[0]
    command = tup[1]
    iter = tup[2]
    fileName = fznFile[fznFile.rfind('/')+1:][:(fznFile[fznFile.rfind('/')+1:]).rfind('.')]

    fznFileWithout = fznFile.rsplit( ".", 1 )[ 0 ]
    print(fznFileWithout)

    timeout = '10800000'
    filepath = outputfile + "run" + str(iter) + "/" + fileName + "_" + command + ".txt"
    if not os.path.exists(filepath):
        os.makedirs(os.path.dirname(filepath), exist_ok=True)
        f = open(filepath, "w+")
        p0= ""
        try:
            time.sleep(random.uniform(0.01, 0.11))  # sleep for a short random moment

            # gunzip fznFile to fznFileWithout -- both are paths
            if not os.path.exists(fznFileWithout):
                print(fznFileWithout + " doesn't exist. Attempting to gunzip " + fznFile)
                r_gunzip = subprocess.call("gunzip -k -f " + fznFile, shell=True)

            # run the (chuffed) solver
            r0 = subprocess.run([commands.get(command), '-s', '-t ' + timeout, str(fznFileWithout)], universal_newlines=True, stdout=subprocess.PIPE)
            p0 = str(r0.stdout)
            tout = "Time limit exceeded" in p0

        except:
            print("Error in " + fileName)
            print(sys.exc_info()[0])
            print(sys.exc_info()[1])
            print(sys.exc_info()[2])
            f.write("Error")
        finally:
            print(p0)
            time.sleep(random.uniform(0.9, 1.1))  # sleep for about 1 second
            time2 = re.search("(?<=time=)[0-9]+.*[0-9]+", p0).group(0)

            nodes = re.search("(?<=nodes=)[0-9]+", p0).group(0)

            objective = re.search("(?<=objective=)-*[0-9]+", p0).group(0)

            f.write(str(tout) + "\n" + time2 + "\n" + nodes + "\n" + objective)

            f.close()

            # delete fznFileWithout -- a path
            if os.path.exists(fznFileWithout):
                print(fznFileWithout + " exists. Attempting to delete it (might fail if another thread does it first)")
                r_delete = subprocess.call("rm " + fznFileWithout, shell=True)
    else:
        print(fileName + " Already processed!")


if __name__ == '__main__':
    iterations = 3

    completed = 0
    total = len(predictFiles) * len(commands) * iterations

    uncompletedFiles = list(itertools.product(predictFiles, commands.keys(), range(1, iterations + 1)))

    for txtFile in predictFiles:
        for com in commands.keys():
            for i in range(1,11):
                fileName = re.search("(?<=Predictions/).*(?=_P)", txtFile).group(0)
                if os.path.exists(outputfile + "run" + str(i) + "/" + fileName + "_" + com + ".txt"):
                    uncompletedFiles.remove((txtFile, com, i))
                    completed += 1

    print("Processed " + str(completed) + " out of " + str(total) + " files. ")

    flatZincs = []
    for tup in uncompletedFiles:
        txtFile = tup[0]
        com = tup[1]
        iter = tup[2]
        fileName = re.search("(?<=Predictions/).*(?=_P)", txtFile).group(0)
        try:
            model = [value for value in modelFiles if fileName + ".fzn.gz" in value][0]
            flatZincs.append((model, com, iter))
        except:
            print(fileName + ".fzn.gz not found in modelFiles")

    pool = Pool()
    pool.map_async(runChuffed, flatZincs)
    pool.close()
    pool.join()
