%[text] ## Simulation of Sinusoidal Position test ANDI vs INDI
%[text] This is a test to show the benefits of using ANDI vs INDI in a position chirping test of RW3c
%[text] ### Run this first section so that the simulink has all the needed parameters
clear all 
pathstr = fullfile(pwd,'Data/Simulation_position_sinusoidal_tracking');
%[text] Effectiveness derivation. Assume steady flight level ($\\left.\\alpha \\approx \\theta \\right)${"editStyle":"visual"}
syms m g theta T p1 p2 p3 s
Fx        = -sin(theta) * T %[output:567313ba]
Fz        = cos(theta)  * T -m*g %[output:83f8204d]
G_x_theta = diff(Fx/m,theta);
G_z_theta = diff(Fz/m,theta);
G_x_T     = diff(Fx/m,T);
G_z_T     = diff(Fz/m,T);
G         = simplify(expand([G_x_theta, G_x_T; G_z_theta, G_z_T])) %[output:9329a8b1]
TF_3rd_order = simplify(expand((p1/(p1+s))*(p2/(p2+s))*(p3/(p3+s)))) %[output:6c4ba1fa]
TF_3rd_order = expand((p1/(p1+s))^3) %[output:00a70a48]
%[text] Specify other important physical variables
w_T     = 10.1;   %[rad/sec]
g       = 9.81;   %[m/s2]
m       = 7.5;    %[kg]
T_i     = m*g;    %[N]
theta_i = 0;      %[rad]
%[text] Define Functions to get the RM and EC gains based on desired poles
k_13_s = @(omega_n,zeta,p_1)(omega_n^2*p_1)/(omega_n^2+2*zeta*omega_n*p_1);
k_23_s = @(omega_n,zeta,p_1)(omega_n^2+2*zeta*omega_n*p_1)/(2*zeta*omega_n+p_1);
k_33_s = @(omega_n,zeta,p_1)2*zeta*omega_n+p_1;

k_12_s = @(omega_n,zeta) omega_n/(2*zeta);
k_22_s = @(omega_n,zeta) 2*zeta*omega_n;
%[text] Define the Poles of the Attitude Dynamics Reference Model (Assume RM dynamics are attitude dynamics)
p_1_att   = 4.71;%
p_2_att   = p_1_att;
p_3_att   = p_1_att;
p_att_rm  = p_1_att;
p_att_ec  = sqrt(p_att_rm^3/w_T);
%[text] Approximate the 3rd order attitude dynamics 
tao   = (p_1_att*p_2_att+p_1_att*p_3_att+p_2_att*p_3_att)/(p_1_att*p_2_att*p_3_att);
w_att = 1/tao;
%[text] Define the Poles of the Position Dynamics
p_1  = 1.0;
damp = 1.0;
p_3  = w_att;
%[text] Calculate the RM gains
k_1_rm_pos = k_13_s(p_1/damp,damp,p_3);
k_2_rm_pos = k_23_s(p_1/damp,damp,p_3);
k_3_rm_pos = k_33_s(p_1/damp,damp,p_3);
k_pos_e_s  = [k_1_rm_pos,k_2_rm_pos,k_3_rm_pos];
%[text] Assuming we want equal EC dynamics, calculate the EC gains
k_1_e_pos     = k_1_rm_pos * k_2_rm_pos * k_3_rm_pos;
k_2_e_pos     = k_2_rm_pos * k_3_rm_pos;
k_3_e_pos     = k_3_rm_pos;
k_pos_e       = [k_1_e_pos, k_2_e_pos, k_3_e_pos];
%[text] INDI gains
k_12_s_pos = k_12_s(p_1,damp);
k_22_s_pos = k_22_s(p_1,damp);
k_pos_e_indi  = [k_12_s_pos*k_22_s_pos, k_22_s_pos, 1.0];
%[text] Calculate sinusoidal excitation
f0 = 0.8/(2*pi); % start frequency [rad/sec]
f1 = 0.8/(2*pi); % end frequency [rad/sec]
t_chirp = 45;    % duration of chirp [sec]
A = 1;           % amplitude of chirp [m]
%[text] ANDI attitude dynamics
att_dyn_andi_num = p_att_rm^3;
att_dyn_andi_den = [1,3*p_att_rm,3*p_att_rm^2,p_att_rm^3];
%[text] INDI attitude dynamics
att_dyn_indi_num = [w_T*p_att_rm^3,2*w_T*p_att_ec*p_att_rm^3,w_T*p_att_ec^2*p_att_rm^3];
att_dyn_indi_den = [1, (w_T + 3*p_att_rm), (3*p_att_rm^2 + 3*w_T*p_att_rm + 2*w_T*p_att_ec), (w_T*p_att_ec^2 + 6*w_T*p_att_ec*p_att_rm + p_att_rm^3 + 3*w_T*p_att_rm^2), (3*w_T*p_att_ec^2*p_att_rm + 6*w_T*p_att_ec*p_att_rm^2 + w_T*p_att_rm^3), (3*w_T*p_att_ec^2*p_att_rm^2 + 2*w_T*p_att_ec*p_att_rm^3),w_T*p_att_ec^2*p_att_rm^3];
%%
%[text] ### Run the simulation
out = sim("Data/Simulation_position_sinusoidal_tracking/chirp_pos_andi_indi_sim_real_dyn");
%%
%[text] ## Run the Simulation first! Then you can plot the results
LineWidth = 2.0;
FontSize = 10;

sim_data = out.Simulink_data{1}.Values;
figure(1)
cla %[output:6bdcc98e]
pl = tiledlayout(1,1,'TileSpacing','Compact','Padding','Compact'); %[output:6bdcc98e]
nexttile %[output:6bdcc98e]
hold on %[output:6bdcc98e]
plot(sim_data.time,sim_data.Data(:,1),'LineWidth',LineWidth) %[output:6bdcc98e]
plot(sim_data.time,sim_data.Data(:,2),'-.','LineWidth',LineWidth) %[output:6bdcc98e]
plot(sim_data.time,sim_data.Data(:,3),':','LineWidth',LineWidth) %[output:6bdcc98e]
grid on %[output:6bdcc98e]
grid minor %[output:6bdcc98e]
box  on %[output:6bdcc98e]
hold off %[output:6bdcc98e]
ylabel('Position ($m$)', 'Interpreter', 'latex', 'FontSize', FontSize); %[output:6bdcc98e]
xlabel('Time ($s$)', 'Interpreter', 'latex', 'FontSize', FontSize); %[output:6bdcc98e]
xlim([0,t_chirp-0.12]) %[output:6bdcc98e]
ylim([-1.5,1.5]) %[output:6bdcc98e]
leg = legend('Reference','ANDI','INDI','FontSize',FontSize,'NumColumns',3); %[output:6bdcc98e]
leg.Layout.Tile = 'north'; %[output:6bdcc98e]
set(leg, 'ItemTokenSize', [10, 1]); % [length, gap] %[output:6bdcc98e]
saveFigureAsPDF(pathstr,'simulation_chirp.pdf',83,60); %[output:6bdcc98e] %[output:5add0f31]
ex_ANDI = abs(sim_data.Data(:,2)-sim_data.Data(:,1));
ex_INDI = abs(sim_data.Data(:,3)-sim_data.Data(:,1));


figure(3)
cla %[output:5609a9e2]
pl = tiledlayout(1,1,'TileSpacing','Compact','Padding','Compact'); %[output:5609a9e2]
ax1 = nexttile; %[output:5609a9e2]
hold on %[output:5609a9e2]
plot(sim_data.time,ex_ANDI,'Color',[0.85 0.325 0.098],'LineStyle','-.','LineWidth',LineWidth) %[output:5609a9e2]
plot(sim_data.time,ex_INDI,'Color',[0.929 0.694 0.125],'LineStyle',':','LineWidth',LineWidth) %[output:5609a9e2]
hold off %[output:5609a9e2]
%title("Simulation Position Error [m]")
grid on %[output:5609a9e2]
grid minor %[output:5609a9e2]
box on %[output:5609a9e2]

leg = legend('ANDI','INDI','FontSize',FontSize,'NumColumns',2); %[output:5609a9e2]
leg.Layout.Tile = 'north'; %[output:5609a9e2]
set(leg, 'ItemTokenSize', [10, 1]); % [length, gap] %[output:5609a9e2]

xlim([0,t_chirp-0.12]) %[output:5609a9e2]
ylim([0,0.65]) %[output:5609a9e2]
ylabel('Position Error ($m$)', 'Interpreter', 'latex', 'FontSize', FontSize); %[output:5609a9e2]
xlabel('Time ($s$)', 'Interpreter', 'latex', 'FontSize', FontSize); %[output:5609a9e2]

% Save the figure as a PDF with A4 dimensions
saveFigureAsPDF(pathstr,'Sim_P_E_chirp.pdf',83,60) %[output:5609a9e2] %[output:9cc6c830]
fprintf('ANDI: Mean error %0.2f m\n', mean(ex_ANDI(sim_data.time>5))); %[output:28526580]
fprintf('INDI: Mean error %0.2f m\n', mean(ex_INDI(sim_data.time>5))); %[output:2560c462]
%%
%[text] Some frequency analysis
time_idx = (sim_data.time>8);
Fs = 1/mean(diff(sim_data.time)) %[output:58500a80]
f = f0;
N = length(sim_data.Data(time_idx,1));
Y_input = fft(sim_data.Data(time_idx,1));
Y_input = Y_input(1:N/2+1);
Y_andi = fft(sim_data.Data(time_idx,2));
Y_andi = Y_andi(1:N/2+1);
Y_indi = fft(sim_data.Data(time_idx,3));
Y_indi = Y_indi(1:N/2+1);
frequencies = Fs*(0:(N/2))/N;
[minimum,index_of_f] = min(abs(frequencies - f));

figure(10) %[output:3554a79f]
cla %[output:3554a79f]
hold on %[output:3554a79f]
plot(frequencies,abs(Y_input)) %[output:3554a79f]
plot(frequencies,abs(Y_andi)) %[output:3554a79f]
plot(frequencies,abs(Y_indi)) %[output:3554a79f]
hold off %[output:3554a79f]
xlim([0,10]) %[output:3554a79f]
legend('Input',"ANDI","INDI") %[output:3554a79f]
% Extract magnitude and phase
magnitude_andi = 20*log10(abs(Y_andi(index_of_f))/abs(Y_input(index_of_f))) %[output:601c6636]
phase_andi = rad2deg(angle(Y_andi(index_of_f)) - angle(Y_input(index_of_f))) %[output:50c692ea]
magnitude_indi = 20*log10(abs(Y_indi(index_of_f))/abs(Y_input(index_of_f))) %[output:7003a0dd]
phase_indi = rad2deg(angle(Y_indi(index_of_f)) - angle(Y_input(index_of_f))) %[output:5caf984d]

%[appendix]
%---
%[metadata:view]
%   data: {"layout":"inline","rightPanelPercent":9.5}
%---
%[output:567313ba]
%   data: {"dataType":"symbolic","outputData":{"name":"Fx","value":"-T\\,\\sin \\left(\\theta \\right)"}}
%---
%[output:83f8204d]
%   data: {"dataType":"symbolic","outputData":{"name":"Fz","value":"T\\,\\cos \\left(\\theta \\right)-g\\,m"}}
%---
%[output:9329a8b1]
%   data: {"dataType":"symbolic","outputData":{"name":"G","value":"\\left(\\begin{array}{cc}\n-\\frac{T\\,\\cos \\left(\\theta \\right)}{m} & -\\frac{\\sin \\left(\\theta \\right)}{m}\\\\\n-\\frac{T\\,\\sin \\left(\\theta \\right)}{m} & \\frac{\\cos \\left(\\theta \\right)}{m}\n\\end{array}\\right)"}}
%---
%[output:6c4ba1fa]
%   data: {"dataType":"symbolic","outputData":{"name":"TF_3rd_order","value":"\\frac{p_1 \\,p_2 \\,p_3 }{{\\left(p_1 +s\\right)}\\,{\\left(p_2 +s\\right)}\\,{\\left(p_3 +s\\right)}}"}}
%---
%[output:00a70a48]
%   data: {"dataType":"symbolic","outputData":{"name":"TF_3rd_order","value":"\\frac{{p_1 }^3 }{{p_1 }^3 +3\\,{p_1 }^2 \\,s+3\\,p_1 \\,s^2 +s^3 }"}}
%---
%[output:6bdcc98e]
%   data: {"dataType":"image","outputData":{"dataUri":"data:image\/png;base64,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","height":337,"width":560}}
%---
%[output:5add0f31]
%   data: {"dataType":"text","outputData":{"text":"Figure saved as PDF: \/home\/tomaso\/UANDI_controller_VSQP\/Data\/Simulation_position_sinusoidal_tracking\/simulation_chirp.pdf\n","truncated":false}}
%---
%[output:5609a9e2]
%   data: {"dataType":"image","outputData":{"dataUri":"data:image\/png;base64,iVBORw0KGgoAAAANSUhEUgAAA0gAAAH6CAYAAAA9emyMAAAAAXNSR0IArs4c6QAAIABJREFUeF7snQuYXVWV51diYgiRYIBgIBWlxCYq3UAEDAaqQQptgQ9H2hhEMSg28jAaHioCzSCO2oAdTcYEBmLjBIRgJhEmNuoghWiQh42taREmOFaUlIAgYBJCGvKab5\/LrbpVqap77\/mve\/c55\/7O9\/Gh1Fn78VuPvdbd5+wzYseOHTuMCwIQgAAEIAABCEAAAhCAAARsBAUSVgABCEAAAhCAAAQgAAEIQKBEgAIJS4AABCAAAQhAAAIQgAAEIPAKAQokTAECEIAABCAAAQhAAAIQgAAFEjYAAQhAAAIQgAAEIAABCECgPwF2kLAICEAAAhEJnHrqqfbggw9GHAFdF5HA9OnTbenSpUWcGnOCAAQg0HACFEgNR0wHEIAABIYm8MY3vtG6u7tBBAFXAtiVK04agwAEWowABVKLKZzpQgAC2SJAIpstfRRlNNhVUTTJPCAAgRgEKJBiUKdPCEAAAq8QIJHFFBpBALtqBFXahAAEWoUABVKraJp5QgACmSRAIptJteR+UNhV7lXIBCAAgYgEKJAiwqdrCEAAAiSy2EAjCGBXjaBKmxCAQKsQoEBqFU0zTwhAIJMESGQzqZbcDwq7yr0KmQAEIBCRAAVSRPh0DQEIQIBEFhtoBAHsqhFUaRMCEGgVAhRIraJp5gkBCGSSQFYS2U2bNllPT4+9\/vWvt7Fjx\/ZjFY4hf81rXmN77713v\/\/+9NNP24gRI2zixIm2fv16e+qpp5K\/h\/82fvx4mzRp0k7Mn3nmGXv55Zdt8uTJmdRHrYNat26djRs3zvbYY49kPmvXrrV99tknmXflFf574BP4PfHEE7Zx48bkz6NGjUpkJ0yYsFOXlW3XOp6B92XFrtKOHzkIQAACMQlQIMWkT98QgEDLE8hKIjtv3jz7H\/\/jf9jnPvc5O\/PMM\/vp5e1vf3tS9PzoRz\/qVwBcfvnltuuuu9pFF11kN998s33xi1+0N7zhDbZt2zZ78sknk6Jg9uzZdvbZZycFQbhCP\/\/xH\/9hS5YsybXuP\/CBD9ixxx5r55xzjv3ud7+zd73rXcn\/\/+Y3v9lvXu9+97vt0ksvtaOPPto+9alP2U9\/+lN73eteZ1u2bLE\/\/vGPtt9++9l5551nJ5xwQq9cZdtpIWXFrtKOHzkIQAACMQlQIMWkT98QgEDLE8hCIrt161abMWOGnXLKKXbHHXfY3XffvVOBtNdee9nb3vY2+9KXvtT7t4EF0g033GBdXV29f7\/vvvvs85\/\/vIUC65\/\/+Z8LXSCdfPLJSfF48cUX24knntjLYGCBFO758pe\/nPw9cF+2bJn9t\/\/235J\/Zs6cmfx3CqSWDwsAgAAEIhOgQIqsALqHAARam0C1AmnL0+vqBjR67yl1yfzgBz9ICpgf\/vCHSaG0YMGC5N\/lKxQ4V111VbLTEYqgQw89NPlTtQIp3POrX\/3KQvHwf\/7P\/7EDDjigqTtI21\/s6Z3DyF3b+jGp5W8DZSobGLiD9Pd\/\/\/c2f\/78ZAcuFInlR+2GK5DK7YWdu8D1gQcesJEjR1Ig1WW93AwBCEDAnwAFkj9TWoQABCBQM4FqBdK6y99vm39zf83thRsPWP5EXfd\/5CMfSQqi8LjYV77yleTxuG984xv9CqTFixfbww8\/nDwaF3aZRo8eXVOBFBqZPn26ffrTn7YPf\/jDTS2QnlvZ3juHPd67th+TWv42as8jbPyRSwdlObBAOumkk+yRRx5JHqML7yZdeeWViVwtBVJ4T6mzszN5hHH\/\/fenQKrLerkZAhCAgD8BCiR\/prQIAQhAoGYCsQukkJz\/3d\/9nf3sZz9LDhMI79OER8TuvfdeC4\/VhSvsIIUC6aCDDrL3v\/\/9ybs2c+bMqblACsXDe9\/73uTdpma+g1RLERTmN1TxlKZACodQHHfccXb99dcnhWEtBdILL7yQsL3tttvs4IMPpkCq2Xu4EQIQgEBjCFAgNYYrrUIAAhCoiUC1AmnDj5fZlmfqe8xuz1kX1tR3uCnsGN16663JIQLlK+xkXHDBBfaJT3yiX4EUkvc1a9Yk78qsXLky2U0KJ96VD2kY+A5SEN6+fXtSYIXH8UKh1MwCafOa+b1zGjv1vH5MavlbeMRuzJTSe0EDr6F2kMJ94cCKwCI8uhgKw\/BeUvmQhsp3kMptBqbHH3+8Pfjgg0mRyjtINZsvN0IAAhBoCAEKpIZgpVEIQAACtRGoViDV1kq6u1566SU74ogjkp2dN73pTb2N\/PKXv0zeGQrv0oTT68o7SKFACtfVV19t4Z63vOUtNmbMmGELpFAknH\/++fbzn\/88eS+nmQVSOiq1SQ1XIO3YscNmzZplRx55pN15550Jn+EKpC984Qu2evXqZAcpXBRItemAuyAAAQg0igAFUqPI0i4EIACBGgjELJC++93vJsVOeLzuVa96Ve9owzeRQuF03XXXJe8mDSyQNm\/ebO95z3uS+8Px1IPtID3\/\/PNJkRV2qD7zmc8kx32HqxUKpDDPxx57zMLBDeGo83DAxcACKeyshXe9wi7ct7\/9bVu6dGnyeB0FUg1Owy0QgAAEGkyAAqnBgGkeAhCAwHAEYhZIYaciFEIXXrjzI3nheO5QKIXDGgYWSGE+4R2lUPScddZZvQXSZZdd1jvVcFDBW9\/61mR3KryTU75apUAqF4OLFi2yb33rW70FUjjgonyFj8SG95Tmzp1rU6dO7f3v7CARMyAAAQjEJUCBFJc\/vUMAAi1OIGaB1OLoCz197KrQ6mVyEIBAgwlQIDUYMM1DAAIQyOoOEpopLgEKpOLqlplBAAKNJ0CB1HjG9AABCEBgSAIkshhHIwhgV42gSpsQgECrEKBAahVNM08IQCCTBEhkM6mW3A8Ku8q9CpkABCAQkQAFUkT4dA0BCECARBYbaAQB7KoRVGkTAhBoFQIUSK2iaeYJAQhkkgCJbCbVkvtBYVe5VyETgAAEIhKgQIoIn64hAAEIkMhiA40ggF01giptQgACrUKAAqlVNM08IQCBTBIgkc2kWnI\/KOwq9ypkAhCAQEQCFEgR4dM1BCAAgVNPPdUefPBBQEDAlUD4AO3SpUtd26QxCEAAAq1CgAKpVTTNPCEAAQhAAAIQgAAEIACBqgQokKoi4gYIQAACEIAABCAAAQhAoFUIUCC1iqaZJwQgAAEIQAACEIAABCBQlQAFUlVE3AABCEAAAhCAAAQgAAEItAoBCqRW0TTzhAAEIAABCEAAAhCAAASqEqBAqoqIGyAAAQhAAAIQgAAEIACBViFAgdQqmmaeEIAABCAAAQhAAAIQgEBVAhRIVRFxAwQgAAEIQAACEIAABCDQKgQokFpF08wTAhCAAAQgAAEIQAACEKhKgAKpKiJugAAEIAABCEAAAhCAAARahQAFUqtomnlCAAIQgAAEIAABCEAAAlUJUCBVRcQNEIAABCAAAQhAAAIQgECrEKBAahVNM08IQAACEIAABCAAAQhAoCoBCqSqiLgBAhCAAAQgAAEIQAACEGgVAhRIraJp5gkBCEAAAhCAAAQgAAEIVCVAgVQVETdAAAIQgAAEIAABCEAAAq1CgAKpVTTNPCEAAQhAAAIQgAAEIACBqgQokKoi4gYIQAACEIAABCAAAQhAoFUIUCC1iqaZJwQgAAEIQAACEIAABCBQlQAFUlVE3AABCEAAAhCAAAQgAAEItAoBCqRW0TTzhAAEIAABCEAAAhCAAASqEqBAqoqIGyAAAQhAAAIQgAAEIACBViFAgdQqmmaeEIAABCAAAQhAAAIQgEBVAhRIVRFxAwQgAAEIQAACEIAABCDQKgQokFpF08wTAhCAAAQgAAEIQAACEKhKgAKpKiJugAAEIAABCEAAAhCAAARahUDLFkinnnqqPfjgg62iZ+YJAQhAAAIQgAAEIACBphGYPn26LV26tGn9eXbUsgXSG9\/4Ruvu7k7Ncu3atdbe3p5aPgiqbcSWVxkWgYHHHFSOqh14zEEdgyqvMiwCA485qBxVParyHgzUMagMszAHlYHHHFSO6hxUeQ8G6hhUhlmYg8pAlQ8MVI7qGFT5mHpU2UlJtihMgZQSYJ4NtjxldQ4ehq+OIba8R+BROaoMPOagjkGVVxkWgYHHHFSOqh5VeQ8G6hhUhlmYg8rAYw4qR3UOqrwHA3UMKsMszEFloMpTIJUyxrQcPWwwZZoui1EgyQhbt4E8G36WtAZHXRsw1Bl6JAI+o8h3K9iij\/7gqHOEoc6QuKgxzLMNUiBpum9p6TwbfpYUB0ddGzDUGZIIwNCHgE8r+LTOEYY6Q+KixjDPNkiBpOm+paXzbPhZUhwcdW3AUGdIIgBDHwI+reDTOkcY6gyJixrDPNtgSxdIXV1dieYHHrYQnrUsX\/ytzzkGcrniiivs8ssvT26AWYlTGnv5xje+YV\/72tf6RSF41scz2OLs2bPx25Q2WLZdfLpkQor\/BX\/+1Kc+lSoWpI0hRZTDp\/U85MYbb+xdo1W7buWcCJ9Ol9sEm+ns7JQORNPKM026pQskTrHTTuJL+9JepcmqbcSWLy86yomGzCH9C6BlW1IZose+wgBbJi7iT3pMIqYQU7zWpzz7IztIWpEWRVpVWp4NFqftvyumJIQsgiyC+JOfPxFX8aes+BO2WAxbRI9x9ajm2lEKhFc6ZQcpJX2cjl\/YWMizkxjjj3EXwaz4Aj9YZMMOiqAHYko2bEnVgypfBFuOOQcKpJRFRkyxPCstJrfKvj0CT1bmEnMccNTpw1Bn6LGI+owi361giz76g6POEYY6Q+KixjDPuTY7SJruW1qa4OujfjjqHGGoMyQRgKEPAZ9W8GmdIwx1hsRFjSEFksYvinSelRYF2CCdEnx9NAFHnSMMdYYkAjD0IeDTCj6tc4ShzpC4qDHMc67NDpKm+5aWJvj6qB+OOkcY6gxJBGDoQ8CnFXxa5whDnSFxUWNIgaTxiyKdZ6VFAcYOUsOws4jpaGGoMyQRgKEPAZ9W8GmdIwx1hsRFjWGec212kFLq3iPwqG3kXd4j8MRmwBxKDhRbD2r\/zAE9lpcC1ZZUeWwRW8QW+xIz1Z9UefxR80cKpJRFRkwxVWk4nZ4UE3i0wOO1iBZBD\/hjNmxJ1YMqXwRbZg7FsGX0iB691ug8x0U1145ZJ7CDFJN+zvv2cNqcI3AZPhx1jDDUGXokdD6jyHcr2KKP\/uCoc4ShzpC4qDGkQNL4RZHOs9KiABukU4KvjybgqHOEoc6QRACGPgR8WsGndY4w1BkSFzWGec61W3oHqaurK9F8e3t7PwsIQaV88bc+NAO5VAZfmJU4pbGXnp4e6+jowAZfIZDGlipl0uoBufZ+75Kl0UM5mWjl+Bn8ua2tLVUswAb71mJ8Ws9DBtoTPp1uncan0+U2gXZnZ6d1d3drVVYk6ZYukPKqtEi2slO3\/Drlowk46hxhqDPkl1IY+hDwaQWf1jnCUGdIXNQYsoOk8YsirSrNI\/CobeRd3iPwxGbAHEruG1sPav\/MAT2WFyLVllR5bBFbxBb70kLVn1R5\/FHzRzXXjlIgvNIpO0gp6eN0elJM4NECj9ciWgQ94I\/ZsCVVD6p8EWyZORTDltEjevRao\/McFymQUhYZMcVUpeXZYHFav1+nWARZBPEnP38iruJPWfEnbLEYtoge4+pRzbVj1gnsIMWkn\/O+PQJPzhG4DB+OOkYY6gw9in2fUeS7FWzRR39w1DnCUGdIXNQYUiBp\/KJI51lpUYAN0inB10cTcNQ5wlBnSCIAQx8CPq3g0zpHGOoMiYsawzzn2pnfQdq+fbutXr3aNmzYYAcddJBNmDBhWG2FgPD444\/b\/vvvnxy1OtSVZ6Vp5uonTfD1YQlHnSMMdYYkAjD0IeDTCj6tc4ShzpC4qDHMc66d6QJp48aNdtppp9mmTZts8uTJSaG0YMECO\/roowfV2Fe+8hW7\/fbb7ZBDDrFf\/OIXdtZZZ9knPvGJQe\/Ns9I0c\/WTJvj6sISjzhGGOkMSARj6EPBpBZ\/WOcJQZ0hc1BjmOdfOdIE0b948e+ihh+ymm26yUaNG2cqVK+3KK6+0e++910aOHNlPaw888IDNmTPHvv\/979vee+9tjzzyiH3zm9+00MaIESN20nCelaaZq580wdeHJRx1jjDUGZIIwNCHgE8r+LTOEYY6Q+KixjDPuXamC6QTTjjBzjzzTDv55JMTDW3bts0OPvhgu\/nmm5N\/V16XXnqpvfrVr7bLL7+8Jm2qSvMIPGobeZf3CDyxGTCHkrvF1oPaP3NAj+WFQ7UlVR5bxBaxxb40TvUnVR5\/1PxRzbVrSugbdFOmC6TwztHixYtt+vTpvdM\/7rjjbO7cuXbSSSf1QzJr1iw77LDD7Le\/\/a09+uijtt9++9kll1xib33rWwdFpyoNp9OTYgKPFni8FtEi6AF\/zIYtqXpQ5Ytgy8yhGLaMHtGj1xqd57io5toNqn1qajbTBdLUqVNt2bJl\/XaLwq7SGWecYTNnzuw3wfDfX3rpJbv22muT4uj666+3G2+80e6++257zWtesxOMoLTBrtNPP91mz55dE7xWv6mnp2fYgzBanU+t84djraSGvg+GOsPQAhx1jjDUGWKLMPQh4NMKPl2dY8i3lyxZMuiN3d3d1RvI4B2ZLpCmTZtmixYtshkzZvSiO+qooyw8Tnf88cf3wxkew\/vbv\/1bO\/\/885P\/vmPHjuSwhuuuu86OOOKIQQukvCotK3bk8atGVuYScxxw1OnDUGcYWoCjzhGGOkNsEYY+BHxawafTc2QHKT27YSVPOeUUO\/HEE3t3dNavX588RnfnnXdae3t7P9nPfOYzNn78ePuv\/\/W\/9v73Qw89NCmwKJAaoyCChg9XOOocYagzJCmFoQ8Bn1bwaZ0jDHWGxEWNIQWSxm9I6VtuucVuuOEGW7Fihe2+++521VVX2f33358c5R2uO+64I3nEKxzYsGrVqmT36H\/\/7\/+dHAke\/hZ2msJ\/32233XbqI89KaxDuupsl+NaNbFABOOocYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNYZ5z7Uw\/Yhc+EnvZZZfZbbfdZuPGjbOJEycmO0Ll3aPwWF1nZ2dyvHe4Fi5cmLx79NrXvta2bt1q\/\/RP\/zTkN5NUpXkEHrWNvMt7BJ7YDJhDKXjG1oPaP3NAj+U0QLUlVR5bxBaxxb6kXPUnVR5\/1PxRzbW18kyTznSBVJ7aCy+8YJs3b04KpGpXOKghfGB2r732GvZWVWk4nZ4UE3i0wOO1iBZBD\/hjNmxJ1YMqXwRbZg7FsGX0iB691ug8x0U1166W8zfy77kokBoBQFVang0Wp\/X7dYpFkEUQf\/LzJ+Iq\/pQVf8IWi2GL6DGuHtVcuxH5f61tUiDVSor7diLgEXjA6rMT1+ocsUUfC4CjzhGGOkOPH558RpHvVrBFH\/3BMT1HCqT07KJJ5llp0aAN6Jig4aMJOOocYagzJCmFoQ8Bn1bwaZ0jDHWGxEWNYZ5zbXaQNN23tDTB10f9cNQ5wlBnSCIAQx8CPq3g0zpHGOoMiYsaQwokjV8U6TwrLQqwQTol+PpoAo46RxjqDEkEYOhDwKcVfFrnCEOdIXFRY5jnXJsdJE33LS1N8PVRPxx1jjDUGZIIwNCHgE8r+LTOEYY6Q+KixpACSeMXRTooraurK+m7\/F2l8kBCUClfQ\/2tp6fHOjo6+o29FrnK\/srBq165cqfhI7jhQ7mx5lAZfJkDelD8QbWlSvvLqz8wh7UW4mqIaWliclnvsW2JOZTW1LzrYWAcSbPGqT6t2hJz6DsEKY3+ysVRbD2ouWKQjzWH8K3S7u7uKHm+2ik7SCkJevwyo7aRd3mPX2ZiM2AOJQeKrQe1f+aAHit\/IBtYoNWzTGCLxbAl9IgesxIT8myL7CDVs3pk5N48Ky0jCOWkOCvziD0Oj+AXew6x+4ehjwbgqHOEoc7Q4wcLn1HkuxVs0Ud\/cEzPMc+5NjtI6fXe8pIEDR8TgKPOEYY6Q5JSGPoQ8GkFn9Y5wlBnSFzUGFIgafyiSOdZaVGADdIpwddHE3DUOcJQZ0giAEMfAj6t4NM6RxjqDImLGsM859rsIGm6b2lpgq+P+uGoc4ShzpBEAIY+BHxawad1jjDUGRIXNYYUSBq\/KNKq0jwCj9pG3uU9Ak9sBsyh5L6x9aD2zxzQY3khUm1JlccWsUVssS8tVP1JlccfNX9Uc+0oBcIrnbKDlJI+TqcnxQQeLfB4LaJF0AP+mA1bUvWgyhfBlplDMWwZPaJHrzU6z3GRAillkRFTTFVang0Wp\/X7dYpFkEUQf\/LzJ+Iq\/pQVf8IWi2GL6DGuHtVcO2adwA5STPo579sj8OQcgcvw4ahjhKHO0KPY9xlFvlvBFn30B0edIwx1hsRFjSEFksYvinSelRYF2CCdEnx9NAFHnSMMdYYkAjD0IeDTCj6tc4ShzpC4qDHMc67NDpKm+5aWJvj6qB+OOkcY6gxJBGDoQ8CnFXxa5whDnSFxUWNIgaTxiyIdlNbV1ZX03d7e3m8MIaiUL\/7Wh2Ygl8rgC7MSpzT20tPTYx0dHdjgKwTS2FKlTFo9INfe7zTCNHooJxOtHD+DP7e1taWKBdhg31qMT+t5yEB7wqfTrdP4dLrcJtDu7Oy07u7uKHm+2ik7SCrBFpbn1ykf5cNR5whDnSG\/lMLQh4BPK\/i0zhGGOkPiosaQHSSNXxRpVWkegUdtI+\/yHoEnNgPmUHLf2HpQ+2cO6LG8EKm2pMpji9gittiXFqr+pMrjj5o\/qrl2lALhlU7ZQUpJH6fTk2ICjxZ4vBbRIugBf8yGLal6UOWLYMvMoRi2jB7Ro9canee4SIGUssiIKZZnpcXkVtm3h9NmZS4xxwFHnT4MdYYeCZ3PKPLdCrbooz846hxhqDMkLmoM85xrs4Ok6b6lpQm+PuqHo84RhjpDEgEY+hDwaQWf1jnCUGdIXNQYUiBp\/KJI51lpUYAN0inB10cTcNQ5wlBnSCIAQx8CPq3g0zpHGOoMiYsawzzn2uwgabpvaWmCr4\/64ahzhKHOkEQAhj4EfFrBp3WOMNQZEhc1hhRIGr8o0qrSPAKP2kbe5T0CT2wGzKHkvrH1oPbPHNBjeSFSbUmVxxaxRWyxLy1U\/UmVxx81f1Rz7SgFwiudsoOUkj5OpyfFBB4t8HgtokXQA\/6YDVtS9aDKF8GWmUMxbBk9okevNTrPcZECKWWREVNMVVqeDRan9ft1ikWQRRB\/8vMn4ir+lBV\/whaLYYvoMa4e1Vw7Zp3ADlJM+jnv2yPw5ByBy\/DhqGOEoc7Qo9j3GUW+W8EWffQHR50jDHWGxEWNIQWSxi+KdFBaV1dX0nd7e3u\/MYSgUr74W\/9fhyu5VAZfmJXIpLGXnp4e6+jowAZfIZDGlipl0uoBufZ+75Kl0UM5mWjl+Bn8ua2tLVUswAb71mJ8Ws9DBtoTPp1uncan0+U2gXZnZ6d1d3dHyfPVTtlBUgm2sDy\/TvkoH446RxjqDPmlFIY+BHxawad1jjDUGRIXNYbsIGn8okjnWWlRgA3SKcHXRxNw1DnCUGdIIgBDHwI+reDTOkcY6gyJixrDPOfa7CBpum9paYKvj\/rhqHOEoc6QRACGPgR8WsGndY4w1BkSFzWGFEgavyjSqtI8Ao\/aRt7lPQJPbAbMoeS+sfWg9s8c0GN5IVJtSZXHFrFFbLEvLVT9SZXHHzV\/VHPtKAXCK52yg5SSPk6nJ8UEHi3weC2iRdAD\/pgNW1L1oMoXwZaZQzFsGT2iR681Os9xkQIpZZERUyzPSovJrbJvD6fNylxijgOOOn0Y6gw9EjqfUeS7FWzRR39w1DnCUGdIXNQY5jnXZgdJ031LSxN8fdQPR50jDHWGJAIw9CHg0wo+rXOEoc6QuKgxpEDS+EWRzrPSogAbpFOCr48m4KhzhKHOkEQAhj4EfFrBp3WOMNQZEhc1hnnOtdlB0nTf0tIEXx\/1w1HnCEOdIYkADH0I+LSCT+scYagzJC5qDCmQNH5RpFWleQQetY28y3sEntgMmEPJfWPrQe2fOaDH8kKk2pIqjy1ii9hiX1qo+pMqjz9q\/qjm2lEKhFc6ZQcpJX2cTk+KCTxa4PFaRIugB\/wxG7ak6kGVL4ItM4di2DJ6RI9ea3Se4yIFUsoiI6ZYUFpXV1cyhPb29n5DCcZYvob6W09Pj3V0dNQtV9lf2ehr6W+wca5atcra2tqizaHSaZkDelD8QbWlSvuL5dPMofSjSeVVb2wNcTXEtHrlKuN1bD0wh9Kamnc9DIwjada42P7AHPp+yE2jv3KRq\/q0qgc1VwzysebQ2dlp3d3dMdP91H2zg5QaHYIev2pA0WcnrtU5Yos+FgBHnSMMdYYeuy8+o8h3K9iij\/7gmJ4jO0jp2UWTzLPSokEb0DFBw0cTcNQ5wlBnSFIKQx8CPq3g0zpHGOoMiYsawzzn2uwgabpvaWmCr4\/64ahzhKHOkEQAhj4EfFrBp3WOMNQZEhc1hhRIGr8o0nlWWhRgg3RK8PXRBBx1jjDUGZIIwNCHgE8r+LTOEYY6Q+KixjDPuTY7SJruW1qa4OujfjjqHGGoMyQRgKEPAZ9W8GmdIwx1hsRFjSEFksYvirSqtDSBZ8vT62zzb+638e+clcw5TRuVsPIuXwQGzKFkkdhifAZF0INqR0VgwByIKeV1XvUHVb4ItgiDuP6k5tpRCoRXOmUHKSX9ep3u2WXzLPwTrvZrHrTRe0\/JfVJZL4PBUKttxJYvwgJShDmodlAEBkWYA3qMm8x4JebYInr0siU1JqjyRbDlmHOgQEpZZNQitn37dlu9erVt2LDBDjroIJswYcKgYuvXr7ennnqq39\/GjBlj++2336D3N1tpG368zJ5adF4ylkmfnN+7i1QLg6ze4xF4sjq3Zo4LjjptGOoMPRZRn1HkuxVs0Ud\/cNQ5wlBnSFzUGDY719ZG21860ztIGzdutNNOO802IOE4AAAgAElEQVQ2bdpkkydPTgqlBQsW2NFHH70TgxtuuMHmz59vkyZN6v3bm970JrvmmmsyUSCFQTw2c99kLOOPmWWT5sz31GOUtgi+PtjhqHOEoc6QRACGPgR8WsGndY4w1BkSFzWGFEgavyGl582bZw899JDddNNNNmrUKFu5cqVdeeWVdu+999rIkSP7yYXiaN26dRZkarliKG3tOdNtyzPrbOyB77ApV6yoZZiZvofg66MeOOocYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNYYxcWxtxn3Smd5BOOOEEO\/PMM+3kk09ORrxt2zY7+OCD7eabb07+XXl98YtfTP5+xRVX1MQmhtKeWniebbhnWTK+A5Y\/UdM4s3wTwddHO3DUOcJQZ0giAEMfAj6t4NM6RxjqDImLGsMYubY24pwUSOGdo8WLF9v06dN7R3zcccfZ3Llz7aSTTurH4DOf+Yz9+c9\/ttGjR9uTTz5pb37zm+38889PHs0b7ApKG+w6\/fTTbfbs2VX59vT0WFtbW9X7Km\/Ydue\/2PYf3ZD8p1GXLLc\/btpWdxuV7aUZQ5bkw1iYQ3wGRdCDakdFYFCEOaDHUoRWOcSWL8IcVIZFYFCEOaDH5sSUG2+80ZYsWTJoTtzd3V1XrpyVmzO9gzR16lRbtmxZv92isKt0xhln2MyZM\/sx\/Od\/\/mcLSggFTjjI4brrrrOf\/\/znduedd9rYsWN34q1WtWl+mRl4UMOz+x1u7e3tqW0hzRgqO4stH8YSewxq\/8yhZFEqx9jyzAE9lmMjtqj7cxH8SbWDIjAowhzQY9zYrubaqRNkB8FMF0jTpk2zRYsW2YwZM3qnetRRR9mll15qxx9\/\/LDT37Jli73tbW+za6+91oLMwEtVGk7HIpqVhApbxBaLYovYctxkxsuOSKzRo5ctqTFBlS+CLcecg5prO9Q5qZvIdIF0yimn2Iknntj7yFs4yvuwww5LdoUG7rw8\/fTTNm7cuOSf8hXeU1q4cKF1dHS4F0ipiRdI0CPwFAhH6qnAMTW6XkEY6gw9FlGfUeS7FWzRR39w1DnCUGdIXNQYUiBp\/IaUvuWWWywc371ixQrbfffd7aqrrrL777\/fbr\/99kTmjjvuSN7hCYXQrFmzLCjiy1\/+sr3qVa+yb3\/728mJdvfcc08iO\/DKs9IahLvuZgm+dSMbVACOOkcY6gxJBGDoQ8CnFXxa5whDnSFxUWOY51w70ztI4SOxl112md12223JztDEiROTR+7Ku0fhdLvOzk6bM2dO8g7Eeeedl7zcGg5qCPf\/0z\/9k7397W8fVLt5Vppmrn7SBF8flnDUOcJQZ0giAEMfAj6t4NM6RxjqDImLGsM859qZLpDKannhhRds8+bNSYFU7dqwYUNyy\/jx44e9NabSyh+Mbb\/mQRu995RqU8rs3wm+PqqBo84RhjpDEgEY+hDwaQWf1jnCUGdIXNQYxsy1tZGb5aJAUic5mHxMpZULpPHHzLJJc+Y3YnpNaZPg64MZjjpHGOoMSQRg6EPApxV8WucIQ50hcVFjGDPX1kZOgZSanxJ41p4z3bY8sy7pW\/lgrDIGD6dX+8\/CGJhDyQVUDnmXLwKDIsxBtaMiMGAOxYhJ6BE9lhNMNa6p8jFtkQIpdZkRT1BVmmKwTy08zzbcs4wCqQCJeczA4xV8izAHxR+9OKpjUOXRIwlZVmwZW8QWs2KLxNW4tqjm2vGqBHaQYrLPfd8egSf3EBwmAEcdIgx1hh5Jrc8o8t0KtuijPzjqHGGoMyQuagwpkDR+UaTzrLQowAbplODrowk46hxhqDMkEYChDwGfVvBpnSMMdYbERY1hnnPtlj6koaurK9H8wI\/OhqBSvvhbn3MM5FIZfGFW4pTGXsLR9AM\/ZgzP+nhW8kqrB+Ta+72Lhg3WZ4PlRCr4c\/g+X5pYgA229y44+LSehwy0J3wan252XAqf4unu7taqrEjSLV0g5VVpkWxlp275dcpHE3DUOcJQZ8gvpTD0IeDTCj6tc4ShzpC4qDFkB0njF0VaVZpH4FHbyLu8R+CJzYA5lNw3th7U\/pkDeiwvRKotqfLYIraILfalhao\/qfL4o+aPaq4dpUB4pVN2kFLSx+n0pJjAowUer0W0CHrAH7NhS6oeVPki2DJzKIYto0f06LVG5zkuUiClLDJiiqlKy7PB4rR+v06xCLII4k9+\/kRcxZ+y4k\/YYjFsET3G1aOaa8esE9hBikk\/5317BJ6cI3AZPhx1jDDUGXoU+z6jyHcr2KKP\/uCoc4ShzpC4qDGkQNL4RZHOs9KiABukU4KvjybgqHOEoc6QRACGPgR8WsGndY4w1BkSFzWGec612UHSdN\/S0gRfH\/XDUecIQ50hiQAMfQj4tIJP6xxhqDMkLmoMKZA0flGk86y0KMDYQWoYdhYxHS0MdYYkAjD0IeDTCj6tc4ShzpC4qDHMc67NDpKm+5aWJvj6qB+OOkcY6gxJBGDoQ8CnFXxa5whDnSFxUWNIgaTxiyKtKs0j8Kht5F3eI\/DEZsAcSu4bWw9q\/8wBPZYXItWWVHlsEVvEFvvSQtWfVHn8UfNHNdeOUiC80ik7SCnp43R6Ukzg0QKP1yJaBD3gj9mwJVUPqnwRbJk5FMOW0SN69Fqj8xwXKZBSFhkxxYLSurq6kiG0t7f3G0owxvLF3\/r\/klPJpdJpYVYik8Zeenp6rKOjAxt8hUAaW6qUSasH5Nr77QSm0UM5KWzl+Bn8ua2tLVUswAb71mJ8Ws9DBtoTPp1uncan0+U2gXZnZ6d1d3fHTPdT980OUmp0CHr8qgFFn524VueILfpYABx1jjDUGZYL7YE\/OPm03DqtYIs+uoZjeo7sIKVnF00yz0qLBm1AxwQNH03AUecIQ50hSSkMfQj4tIJP6xxhqDMkLmoM85xrs4Ok6b6lpQm+PuqHo84RhjpDEgEY+hDwaQWf1jnCUGdIXNQYUiBp\/KJIq0rzCDxqG3mX9wg8sRkwh5L7xtaD2j9zQI\/lhUi1JVUeW8QWscW+tFD1J1Uef9T8Uc21oxQIr3TKDlJK+jidnhQTeLTA47WIFkEP+GM2bEnVgypfBFtmDsWwZfSIHr3W6DzHRQqklEVGTDFVaXk2WJzW79cpFkEWQfzJz5+Iq\/hTVvwJWyyGLaLHuHpUc+2YdQI7SDHp57xvj8CTcwQuw4ejjhGGOkOPYt9nFPluBVv00R8cdY4w1BkSFzWGFEgavyjSeVZaFGCDdErw9dEEHHWOMNQZkgjA0IeATyv4tM4RhjpD4qLGMM+5NjtImu5bWprg66N+OOocYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNIQWSxi+KdJ6VFgUYO0gNw84ipqOFoc6QRACGPgR8WsGndY4w1BkSFzWGec612UHSdN\/S0gRfH\/XDUecIQ50hiQAMfQj4tIJP6xxhqDMkLmoMKZA0flGkg9K6urqSvtvb2\/uNIQSV8jXU33p6eqyjo6Nuucr+ysGrlv4GG+eqVausra0t2hwqgy9zQA+KP6i2VGl\/sXyaOZSO\/q+86o2tIa6GmFavXGW8jq0H5lBaU\/Ouh4FxJM0aF9sfmEPf50jS6K9cHKk+repBzRWDfKw5dHZ2Wnd3d5Q8X+2UHaSUBD1+mVHbyLu8xy8zsRkwh5IDxdaD2j9zQI\/lpUC1JVUeW8QWscW+xEz1J1Uef9T8kR2klEVGTLE8Ky0mt8q+PQJPVuYScxxw1OnDUGfokQj4jCLfrWCLPvqDo84RhjpD4qLGMM+5NjtImu5bWprg66N+OOocYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNIQWSxi+KdJ6VFgXYIJ0SfH00AUedIwx1hiQCMPQh4NMKPq1zhKHOkLioMcxzrs0Okqb7lpYm+PqoH446RxjqDEkEYOhDwKcVfFrnCEOdIXFRY0iBpPGLIq0qzSPwqG3kXd4j8MRmwBxK7htbD2r\/zAE9lhci1ZZUeWwRW8QW+9JC1Z9UefxR80c1145SILzSKTtIKenjdHpSTODRAo\/XIloEPeCP2bAlVQ+qfBFsmTkUw5bRI3r0WqPzHBcpkFIWGTHFVKXl2WBxWr9fp1gEWQTxJz9\/Iq7iT1nxJ2yxGLaIHuPqUc21Y9YJ7CDFpJ\/zvj0CT84RuAwfjjpGGOoMPYp9n1HkuxVs0Ud\/cNQ5wlBnSFzUGFIgafyiSOdZaVGADdIpwddHE3DUOcJQZ0giAEMfAj6t4NM6RxjqDImLGsM859rsIGm6b2lpgq+P+uGoc4ShzpBEAIY+BHxawad1jjDUGRIXNYYUSBq\/KNJ5VloUYOwgNQw7i5iOFoY6QxIBGPoQ8GkFn9Y5wlBnSFzUGOY5127pHaSurq5E8+3t7f0sIASV8sXf+tAM5FIZfGFW4pTGXnp6eqyjowMbfIVAGluqlEmrB+Ta+x3XnkYP5WSileNn8Oe2trZUsQAb7FuL8Wk9DxloT\/h0unUan06X2wTanZ2d1t3drVVZkaRbukBSlObxy4zaRt7lPX6Zic2AOZQiV2w9qP0zB\/RYXoNVW1LlsUVsEVvs\/8PswB8e68mX8ce4\/sQOUj3WmpF7VaXhdHpSTCIQN3BVuqJqz3mXxxazYYuqHaHHbOixCHrAFrNhS6oeVPki2HLMOai5dsySgR2kmPRz3rdH4Mk5Apfhw1HHCEOdocci6jOKfLeCLfroD446RxjqDImLGkMKJI1fFOk8Ky0KsEE6Jfj6aAKOOkcY6gxJBGDoQ8CnFXxa5whDnSFxUWOY51ybHSRN9y0tTfD1UT8cdY4w1BmSCMDQh4BPK\/i0zhGGOkPiosaQAknjF0U6D0pbd\/n7bdIn59vovadEYVStU4JvNUK1\/R2OtXEa7i4Y6gxJBGDoQ8CnFXxa5whDnSFxUWOYh1x7qBlmfgdp+\/bttnr1atuwYYMddNBBNmHChKraCjK\/\/e1v7fWvf72NHTt20PtVpXkEnmptrD1nuo098B02ac78QedQTb4aqNjyHoGHOXBYhocdebSBLeq2qDJEj6Wor3JU5bMwBnUOqnwRGBRhDugxbkxQc+1qeWwj\/57pAmnjxo122mmn2aZNm2zy5MlJobRgwQI7+uijh2Vy\/fXX25VXXmnf+c537PDDD89lgbTl6XW29tzpydjbr3lw0F0k1fFjyxch+DKHuMG37NyqLaNH9OhlS9hiMWwJPaLHrMSEPNsiBVKDSrh58+bZQw89ZDfddJONGjXKVq5cmRQ+9957r40cOXLQXsPO0T\/8wz\/Y5s2bbdGiRbktkMLkHpu5bzLHPT9wge15ymd2mq\/qNLHlSUqzsQAVQQ+qLReBQRHmgB6zERPQg74LVwR\/LMIcsOW4MYUCqUEF0gknnGBnnnmmnXzyyUkP27Zts4MPPthuvvnm5N8Dr61bt9rMmTNtzpw59oUvfMG+\/vWvN6xAatCUc9WsR+DJ1YQbNFg46mBhqDP0SIZ8RpHvVrBFH\/3BUecIQ50hcVFjSIGk8RtSOrxztHjxYps+vfSoWbiOO+44mzt3rp100kk7yS1cuND+8Ic\/2Fe\/+lU76qijKJAapJdyswRfH8Bw1DnCUGdIIgBDHwI+reDTOkcY6gyJixpDCiSN35DSU6dOtWXLlvXbLQq7SmeccUayU1R5Pfroo3b22Wfbv\/7rv9puu+1WU4E0WMenn366zZ49u0EzKlazPT091tbWVqxJRZgNHHXoMNQZhhbgqHOEoc4QW4ShDwGfVvDp6hxvvPFGW7JkyaA3dnd3V28gg3dk+pCGadOmJe8RzZgxoxdd2Bm69NJL7fjjj+\/9b1u2bLH3ve99dvHFFyeFUbjYQWq8tfHrlA9jOOocYagz5JdSGPoQ8GkFn9Y5wlBnSFzUGLKDpPEbUvqUU06xE088sXdHZ\/369XbYYYfZnXfeae3t7b1ya9assQ9+8IP92gn3hp2ks846y84999yd+siL0p5aeJ5teWad7X7MKTb+nbMaRDpdswTfdNwGSsFR5whDnSGJAAx9CPi0gk\/rHGGoMyQuagzzkmsPNstM7yDdcsstdsMNN9iKFSts9913t6uuusruv\/9+u\/3225O53HHHHckjXoMd2NDoHSSPwFNLG+WT7MYfM2un7yHVIj+caceW9wg8zIHTljzsyKMNbFG3RZUheixFfJWjKp+FMahzUOWLwKAIc0CPcWMCBZJWYA4pHT74etlll9ltt91m48aNs4kTJyaP3JV3j8Lpdp2dncmpdQOvohRI4WOxYQcpfDB2yhUr+k1TdfzY8kUIvswhbvAtO4Rqy+gRPXrZErZYDFtCj+gxKzEhz7ZIgdSgAqnc7AsvvJB81ygUSF5XXpQWHrHbcM+yZNoHLH\/Ca\/ou7Xg4rctAct4IHHUFwlBn6FEk+owi361giz76g6POEYY6Q+KixjAvufZgs8z0I3aaWoaXzovSnr31q\/bs8q8nk2m\/5kEbvfeURmKpq22Cb124hrwZjjpHGOoMSQRg6EPApxV8WucIQ50hcVFjmJdcmwKpgkBelLbhx8vsqUXnJSOf9Mn5mTqogeCrBY6yNBx1jjDUGZIIwNCHgE8r+LTOEYY6Q+KixjAvuTYFUg4LpBd\/c5\/1XF765tOesy5M\/snKRfD10QQcdY4w1BmSCMDQh4BPK\/i0zhGGOkPiosaQAknjF0VaVZpH4Km1jaFOsqtVfijAseU9Ag9z0E+sKoIeVDsoAoMizAE9lqK1yiG2fBHmoDIsAoMizAE9xo0paq4dpUB4pVPeQUpJv5lOVz7JLgy18qAGdQyx5YsQfJlD3OBbdl\/VltEjevSyJWyxGLaEHtFjVmJCnm2RAillkRFTTFVaMw12qJPs1DHElicpzcYCVAQ9qLZcBAZFmAN6zEZMQA\/6Ll4R\/LEIc8CW48YUNdeOWSe09A5SV1dXwr78XaXKXwvK\/zsLf9t257\/Y9h\/dUBprxUl2wfFjjrMy8MQeS6UT5W0sPT091tHR0S8O5G0Osf2oklfssZSTipi+mdYf8Om+ZCKt\/oI\/hw+YZ2HtSDuHLMjh0\/r6PjAWsq6ULLte38Sn62dWjiHhW6Xd3d0x65zUfbd0gZQXpWX1JDuPX2ZSW26BBOGoKxOGOsNyYTcwefBpuXVawRZ9dA1HnSMMdYbERY0hO0gavyjSeVNa+aCGLH0sluDrY7pw1DnCUGdIIgBDHwI+reDTOkcY6gyJixrDvOXalbNlB0nTfUtLE3x91A9HnSMMdYYkAjD0IeDTCj6tc4ShzpC4qDGkQNL4RZHOs9KiABukU4KvjybgqHOEoc6QRACGPgR8WsGndY4w1BkSFzWGec612UFKqXuPwKO2kXd5j8ATmwFzKDlQbD2o\/TMH9FheClRbUuWxRWwRW+xLzFR\/UuXxR80fKZBSFhkxxVSlxXS6cGjD+HfOIinNQGJO8NSCJ4mAXyJQBFuMGVexRWyxMifBFosR29FjXD2quXbMOoEdpJj06+g7iyfZeQSeOhAU9lY46qqFoc7Qo8DyGUW+W8EWffQHR50jDHWGxEWNIQWSxi+KdB6VVj7JLuweTfrk\/CjcvH9hiz6JDAyARUxXAgx1hiQCMPQh4NMKPq1zhKHOkLioMcxjrl2eMTtImu6bKr32nOm25Zl1NvbAd9iUK1Y0te\/BOiP4+qgAjjpHGOoMSQRg6EPApxV8WucIQ50hcVFjSIGk8YsinUelPbXwPNtwzzLLyreQCL4+pgtHnSMMdYYkAjD0IeDTCj6tc4ShzpC4qDHMY67NDtIb32jd3d2pNe8ReNQ28i7vEXhiM2AOJReKrQe1f+aAHsuLgWpLqjy2iC1ii32pmepPqjz+qPkjBVLqMiOeYFBaV1dXMoD29vZ+AwkOVb6G+ltPT491dHTULVfZX9lxa+lvsHGuWrXK2traos2hMvAwB\/Sg+INqS5X2F8unmUOpUK686o2tIa6GmFavXGW8jq0H5lBaU\/Ouh4FxJM0aF9sfmEPfj3dp9FcujlSfVvWg5opBPtYcOjs7pc2IeFWCGe8gpaTPrxL6rgG\/zGi\/zHj9ylgEPeCP2bAlVQ+qfBFsmTkUw5bRI3r0WqPzHBfZQUpZZMQUy7PSYnKr7NvDabMyl5jjgKNOH4Y6Q4+EzmcU+W4FW\/TRHxx1jjDUGRIXNYZ5zrXZQdJ033Tp8kl2oePYhzUQfH3UD0edIwx1hiQCMPQh4NMKPq1zhKHOkLioMaRA0vhFkc6r0son2QVo7dc8aKP3nhKFH0HDDzuLmM4ShjpDfBqGPgR8WsGndY4w1BkSFzWGec21w6zZQdJ033TpDT9eZk8tOi\/pt+2K5bbrgTOaPoZyhwRfH\/Rw1DnCUGdIIgBDHwI+reDTOkcY6gyJixpDCiSNXxTpvCrtxd\/cZz2Xz0yYTfrkfBv\/zllR+BE0\/LCziOksYagzxKdh6EPApxV8WucIQ50hcVFjmNdcO8yaHaSUuvcIPGnbeGzmvsmoRx52gr3p899MOQP9FLq0468csNpGbHmP4MkcsEUPO\/JoI7Ytqv0XgQFzKK0Qqi3ElmcO6LGc67SyLVIgpU7R4wmqSlMNXgme5YMaRuw\/zf7qqjtSQ1TnoMorDAg8fWpHD3oyhS2SzBBTiCmeP94RU4gpxBQzNddOneA6CLKD5ACx2U1UHtQQ8yQ7j8S82eyy2B8cda3AUGfokdD5jCLfrWCLPvqDo84RhjpD4qLGkAJJ4xdFOs9Ke3bZPAv\/hCvmSXYEXx\/ThaPOEYY6QxIBGPoQ8GkFn9Y5wlBnSFzUGOY512YHSdN9FOnKgxrYQYqiAtdOWcR0nDDUGZIIwNCHgE8r+LTOEYY6Q+KixpACSeMXRTooraurq7QL097ebwwhqJSvrP5t3eXvt\/984vc26pLl0eZQGXzzwKwc6LKm256eHuvo6MidDWaJZ6X95dWns8ATny65oRLPgj+3tbXlcl1R5+4ZW\/FpPQ8ZGAsVu\/bUbZbsrJax4NPp8+TOzk7r7u6OkuernbKDlJKgxy8zShthF+mP8z9lf7X4FylnoL\/Yroy\/PGi1jdjyHr8uMQds0cOOPNqIbYtq\/0VgwBz6itSBP1DWs9iptqTKo0f0mJU8J6YtsoNUT9TKyL2q0rIQPP\/fdf9obzrrS6mJqnNQ5WM6rVfgYg4sgl62hD\/phTL+mA1\/LIIe8Mds2JKqB1W+CLYccw5qrp06wXUQZAcpJUScjmQmK4kxtogtFsUWseViJKUxE7Ks+EIRGBRhDsSUuDGFAillkRFTLM9Ki8mtsm+PwJOVucQcBxx1+jDUGXokQz6jyHcr2KKP\/uCoc4ShzpC4qDHMc67NDpKm+5aWJvj6qB+OOkcY6gxJBGDoQ8CnFXxa5whDnSFxUWNIgaTxiyKdZ6VFATZIpwRfH03AUecIQ50hiQAMfQj4tIJP6xxhqDMkLmoM85xrs4Ok6b6lpQm+PuqHo84RhjpDEgEY+hDwaQWf1jnCUGdIXNQYUiBp\/KJI51lpUYCxg9Qw7CxiOloY6gxJBGDoQ8CnFXxa5whDnSFxUWOY51ybHaSUuvcIPGobeZf3CDyxGTCHkgPF1oPaP3NAj+WlQLUlVR5bxBaxxb7ETPUnVR5\/1PyRAillkRFTTFUaTqcnxQQeLfB4LaJF0AP+mA1bUvWgyhfBlplDMWwZPaJHrzU6z3FRzbVj1gnsIMWkn\/O+PZw25whchg9HHSMMdYYeCZ3PKPLdCrbooz846hxhqDMkLmoMKZA0flGkg9K6urqSvtvb2\/uNIQSV8sXf+tAM5FIZfGFW4pTGXnp6eqyjowMbfIVAGluqlEmrB+Ta+z0qmUYP5WSileNn8Oe2trZUsQAb7FuL8Wk9DxloT\/h0unUan06X2wTanZ2d1t3dHSXPVzutawfpl7\/8pd15550WjGXcuHEWioyTTz7ZJk6cqI6j6fJ5rmqbDmuIDvl1ykcTcNQ5wlBnyC+lMPQh4NMKPq1zhKHOkLioMcxzrl1TgfTkk0\/al7\/8ZRs9erT91V\/9VVIcbd261Z5\/\/nn7j\/\/4DzvqqKPsE5\/4hEaxydJ5VlqTUQ3ZHcHXRxNw1DnCUGdIIgBDHwI+reDTOkcY6gyJixrDPOfaVQukp556ypYvX24f\/ehH7TWvec2gpMLO0m9+8xs77bTTNJJNlB5Oadtf7LGRu7YNOxqPwKO2kXd5j8ATmwFzKLlJbD2o\/TMH9FgO+KotqfLYIraILfalX6o\/qfL4o+aPhS6QQvEzbdq0qqXL448\/brvttptNmDCh6r1ZuGEopYXiaNuLPTZ6ryMokKooisCjBQ4WQRbBShfDn\/RCm2QmGzGpCHrAH7NhS6oeVPki2HLMORS6QKpcwDdu3GjhcbvwmN2IESOSX4332muvpDDK20WBpCcjBJ5iLCAxgydFol+RiB7xR\/zJz59Y34rhT+gxrh5bpkA6\/\/zz7d\/+7d\/spZdeSt47OvLII+0Pf\/iDXXjhhXmrj5IDJvJ6skZWYHsEnqzMJeY44KjTh6HO0KPA8hlFvlvBFn30B0edIwx1hsRFjWGec+2q7yBVorn77rtt7733tte97nX2wAMP2P33358UGUuWLLExY8ZoFJssnWelNRnVkN0RfH00AUedIwx1hiQCMPQh4NMKPq1zhKHOkLioMcxzrl1XgbRt2zb78Y9\/bHvuuWdN7yVpWEvS27dvt9WrV9uGDRvsoIMOGvYdp3Cy3q9\/\/WvbtGmTTZ06ddjjx\/OsNA+uHm0QfD0o6o86+owi361giz76g6POEYY6Q5JSGPoQ8GkFn07PMc+5dl0F0mc+85nkO0ivfe1r7a\/\/+q\/tQx\/6kE2fPj05\/rsRV3jnKZyMFwqeyZMnJ4XSggUL7Oijj96pu3BIxEc+8pGkgArj+8UvfmGf\/\/zn7cMf\/vCgQ8uz0hrBOk2bBI001HaWgaPOEYY6Q5JSGPoQ8GkFn9Y5wlBnSFzUGOY5166rQLr11luTd482b96cPF63atUq+7BpqckAACAASURBVNWvfpW8l9SIa968efbQQw\/ZTTfdZKNGjbKVK1falVdeaffee6+NHDmyX5dz5sxJjiEPfw\/X9773Pbv44ouT7zQNvDf8fSilbfjZqTbm9e+3MVNmNmJKhWqT4OujTjjqHGGoMyQRgKEPAZ9W8GmdIwx1hsRFjWHLFEjPPPOMhWO\/wyl27e3tCbVwYEOj3j864YQT7Mwzz7STTz456Ss84nfwwQfbzTffnPy78gpj22WXXXpP1HvkkUfspJNOsvDvwcanFkgegUdtI+\/yHoEnNgPmUPLC2HpQ+2cO6LG8nqi2pMpji9gittiX3an+pMrjj5o\/tkyBFB5ZC7tG++67rx1xxBHJI3b77LOPVl4OIx3eOVq8eHHyGF\/5Ou6442zu3LlJ8TPY9fvf\/956enqSR\/EOO+wwu+iiiwa9LyhtsOv000+32bNnV51T6KOtbfiPyVZrRG0j7\/KBD3OIz6AIelDtqAgMijAH9FhaNVQOseWLMAeVYREYFGEO6LE5MeXGG29MDmwb7MrridF1PWIX3j9685vfnLwLdN9999lPf\/rT5ES77373u9VqgVR\/DwctLFu2rN9uUdhVOuOMM2zmzMEfgTv33HPtwQcftDe96U32hS98wd7ylrcMWSDlVWmpYDZAyOOXmQYMK3dNwlFXGQx1hqEFOOocYagzxBZh6EPApxV8Oj3HQu8gvfDCC8m7PUNdDz\/8cHJgw44dO+w\/\/\/M\/bezYselJDpCcNm2aLVq0yGbMmNH7l\/AO1KWXXmrHH3\/8sP3ccccdySEN4WjyiRMn7nRvnpXmBlhsiKAhAnxFHI46RxjqDElKYehDwKcVfFrnCEOdIXFRY5jnXLvqDtJPfvIT22OPPexv\/uZvhqQUjuL+l3\/5l+SRu3Hjxmk0K6RPOeUUO\/HEE3sfeVu\/fn3y2FzYySq\/A1W+\/Tvf+U5SSE2ZMqW3hfCe0n\/\/7\/990FPv8qw0N8BiQwRfESAFkg9Adj7g6EZAb4i4qDMkKYWhDwGfVvDp9BzznGtXLZACluuvv96ef\/55e8973mP7779\/sqP08ssvWzgYITzOds8999gFF1xg++23X3qKg0jecsstdsMNN9iKFSts9913t6uuuio5Pe\/2229P7g67ROE9oFAIhXeHxo8fb+Hku1e\/+tV21113WXjcLny3KRwRPvDKs9JcIQuNETQEeBWicNQ5wlBnSFIKQx8CPq3g0zpHGOoMiYsawzzn2jUVSAHPAw88kByY8LOf\/Sw5uS5cYeLhXaCPfvSjro\/WldURdqYuu+wyu+2225KdqfCoXHjkrrx7FE636+zstHDE9x\/\/+Ee78MILk1PrQgG3ZcsWu+SSS3pPwKulQHpp3XLb9MvP2tipc23s1POGtQqPwKO2kXd5j8ATmwFzKLlJbD2o\/TMH9FgO+KotqfLYIraILfalX6o\/qfL4o+aPLVEglc01vGv05z\/\/OdmtadTx3gMrk\/AeVPj20mDvEg28N9wXPiy71157DVvgDKa0zWvm2+Y1C5JvII2b9lUKpCo\/HBB4tMDBIsgiWOli+JNeaJPMZCMmFUEP+GM2bEnVgypfBFuOOYeWKpCq5My5+TMFkp6MEHiKsYDEDJ4UiX5FInrEH\/EnP39ifSuGP6HHuHqkQMpNWdQ30DwrLSu4PQJPVuYScxxw1OnDUGfoUWD5jCLfrWCLPvqDo84RhjpD4qLGMM+5ds3vIAVE4d2jZj1Wp6mkunSelVZ9ds25g+DrwxmOOkcY6gxJBGDoQ8CnFXxa5whDnSFxUWOY51y7rgLpyCOPtG9961t2wAEHaMQyIJ1npWUAXzIEgq+PJuCoc4ShzhCfhqEPAZ9W8GmdIwx1hsRFjWGec+26CqRw3Hc4Oa7ysITwIdZjjz1WIxhBOs9Ki4Br0C4Jvj6agKPOEYY6QxIBGPoQ8GkFn9Y5wlBnSFzUGOY5166rQPrYxz5mv\/zlL23SpEkJsW3bttnjjz9ua9as0QhGkA5K6+rqSnouHxv+3Mr25P+\/POHvbHPbRf3+Vh5iCDjla+DHalvtb5XBt9XmXmmy6tx7enqso6OjnxeobVbadSvYbiWvVpt7eQH3iEv4dImi4n\/Bn8P3+Vp9fVBjJD6t5xoDY6Fi1x7xRbUJ1TfTzgGf7suT680nwqd4uru7I2T5epd1FUgLFy60N7\/5zbbLLrv09vzDH\/7QvvSlL+kjaXILg1W15QKJY75rU0ZlMhW+HxW+IxWu1x63ykbu2lZTI+ovXLHlywF7YDJU0+RfuYk56I9rqgzRY19hgC2v3am4aaY\/Y4vYYmUSGtMfi2CLrA1x\/alldpDCN4bCrtGvfvUrC99DOuSQQ2zUqFEN+UhsPQtSmnsHU9pjM\/dNmhp\/zCybNGf+sM3idP2T2vI3pAK03WYstdF7HVGTWlSOseWLsIAUYQ6qHRSBQRHmgB7jJjNeiTm2iB69bEmNCap8EWw55hxapkBavXq1\/cM\/\/INt3brV9thjj+RUu6uvvtpmzJhRUzKcpZsGKm3L0+ts7bnTay6QsjSXWGOpDDxh9yjsIoUrfGQ37MJx1UbAI4DX1lNx74Khj27hqHOEoc7QI6HzGUW+W8EWffQHx\/QcW6ZAmjNnjp1++ul2+OGHJ7TCjtI111xjF154YXp6kSTzrLRIyHbqlqDhowk46hxhqDMkKYWhDwGfVvBpnSMMdYbERY1hnnPtut5BuvXWW+2DH\/xgP1p33XWXHXrooTZhwgSNYpOl86y0JqMasjuCr48m4KhzhKHOkEQAhj4EfFrBp3WOMNQZEhc1hnnOtesqkL7xjW\/Y7Nmzbffdd0+IhRPt5s+fb0uWLNEIRpDOs9Ii4Bq0S4KvjybgqHOEoc6QRACGPgR8WsGndY4w1BkSFzWGec616yqQwjtIZ555pu222262YcOG5B2kUDQdffTRGsEI0ju9g\/TnB2zjfacmI6nlHRqPwKO2kXd5j8ATmwFzKDlvbD2o\/TMH9FhehlRbUuWxRWwRW+xLClV\/UuXxR80fW6ZA+sUvfmGTJ0+2Rx99NDnFbtq0abl7tK7sdhRI8ZNaAo8WeLwW0SLogUUwG7ak6kGVL4ItM4di2DJ6RI9ea3Se42LLFEjhY5ZLly5NPoJXvsJBDWPHjo2wB6R1SYHkXyD95a4O2\/5iT\/INpPAtpFou1fFjy7MIsgiyCPJrb2WsU2MSMYWYQkwhphQlprRMgXT77bfbXnvtZQceeGCiu\/BNpHBwQzjdLm\/XQKW9+Jv7rOfy0tHUkz4538a\/c1beptT08Q5MBDb87FTb+uwDyTj2eG\/fV8CbPrCcdeiRUOVsyu7DhaEPUjjqHGGoM\/QoEn1Gke9WsEUf\/cExPceWKZD+y3\/5L\/brX\/96J1Ld3d3p6UWSHKi0DT9eZk8tOi8ZTdsVy23XA\/P3badmoyRo+BCHo84RhjpDklIY+hDwaQWf1jnCUGdIXNQYtkyBtHLlSjv22GN7H6kL7yHddttt9oEPfEAjGEE6KK2rqyvpub293SoLpFFnL7QR+0\/r\/Vvl8ELAKV9BrpX\/Vhl84VKyhDQ20dPTY+Hx1Va2JXXulfaXVg\/Itfc7bAOfTufTwZ\/DY+hpYgE22Lem4tN6rjHQnvBpfLrZcamzs9PyuIkSLKWuU+wWL15sJ554ou27774RShrfLgdWtc8um2fhn3Cxg1Qba36dqo1TtbvgWI1Q9b\/DsDqjWu6AYy2Uhr8HhjrD0AIcdY4w1BliixrDltlBet\/73mc33XRTcsx3+XryySdtn3320QhGkB6otM1r5tvmNQuSkYw7+BYb84Z3uIyqfGiBS2MZa4Tg66MQOOocYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNYcsUSDfccIM9\/\/zz9ta3vjUhFh6x+8EPfpB8Cylv13AFUjiBLZzENtzlEXjUNrIm\/9K65bbpl58tFZnTvmpjppQOvWgkx9gMPIInc9B\/LVYZoseSl6ocY8szh2zosQh6UG25CAyKMAf0GDcmtEyB9PGPf9z+9Kc\/2a677tpbIIVnC8P3kfJ2DfeIXfs1D9rovac0NLEvYuDZUufHdovAgDnEDb5lJ2URLIYe0GMx9FiEuIgtFsMW0WNcPbZMgbR69Wo7+OCD+xUOg\/23PBRLA5X21MLzbMM9y5Kh11Ig5WGOjR7jwMATHicM30IKV607SI0eYx7a9wjgeZhnI8cIQx+6cNQ5wlBn6FFg+Ywi361giz76g2N6joUukB599FH7\/e9\/b4cffnjyDaTy9dxzzyWP1o0ePdouueSS9PQiSQ5XIB2w\/IlIo8pXtwQNH33BUecIQ50hSSkMfQj4tIJP6xxhqDMkLmoMC10gXXzxxXbCCSfYjBkzbMmSJfa73\/3ODjjgAJs9e3byodj3vve99v3vf18jGEG6mQVS+IDq+COXRphlY7sk+PrwhaPOEYY6QxIBGPoQ8GkFn9Y5wlBnSFzUGBa6QLr66qvtc5\/7XEIoHMrwsY99zP7n\/\/yfvcS+8pWvFGIHSTOBwaXD42blU+zCwQ9Fuwi+PhqFo84RhjpDEgEY+hDwaQWf1jnCUGdIXNQYFrpAmj9\/vp133nm9hK666iq76KKLev9\/eMzuU5\/6lEYwgvRApYVdnq3PPpCcXldLMVNL4KnWZi1tDIcm7\/IegSc2A+ZQstDYelD7Zw7osRxrVVtS5bFFbBFb7Mt8VH9S5fFHzR8LXSDNmzfPLrzwwiELpMsvv9yuuOKKCCWO1mUzCqRqI1QdN+\/yBB4t8HgtokXQg+oLRWBQhDmgx2zEBPSg\/+hTBH8swhyw5bgxpdAFUngHadWqVclhDOF69tlnbc8990z+9\/bt2+3xxx9Pfj3O2zVQafU+DofTDb6AhO8ghe8hhasZ35NS9aDKF2EBKcIc0GPcRdCrWEePxdAjMQU9ZiUmEFPi2mKhC6TwjlG4JkyYsFMNFN5J+slPfmLf+c538lYfWVBaV1dXMu729nZ78Tf3Jf979MQp1rNpa+98wt8qr8pisNX\/Vhl4ylzG\/GmJ7fL0kn4FEsxKFjSUvfT09FhHR+l49MpFpfy\/W93OauEy8EcamKWzpcF8ejjb5W87rw\/Bn9va2ob0d5jVtqbi030\/PKeNZwNtjbV4+LV4KN\/Ep4fOX6rFs87OTgvfS83jNWJHqHKGuR577LHk1LqhrjVr1tjUqVNzN\/eBVe3ac6bblmfW2dgD32FTrliRu\/nEGLDHLzMxxp21PuGoawSGOsPQAhx1jjDUGWKLMPQh4NMKPp2eY6F3kNJjybZkMwqkysfN9nhv\/h5DrKZBgkY1QrX9HY61cRruLhjqDElKYehDwKcVfFrnCEOdIXFRY0iBpPGLIj1QaY\/N3DcZh+cOUr3v40QBIXRK8BXgVYjCUecIQ50hiQAMfQj4tIJP6xxhqDMkLmoMKZA0flGkByrtuZWl56JH7jLdXvvuW13GRIHkgrHwjbCI6SqGoc6QRACGPgR8WsGndY4w1BkSFzWGLVcgbdu2LSEWXl\/60Y9+ZMcff7xGMIL0UAXSmCkzbdy0r1YdkUfgUdvIu7xH4InNgDmUXCW2HtT+mQN6LAd91ZZUeWwRW8QW+1Iw1Z9UefxR88eWKZCuvfZau\/nmm+2JJ57oV0Dk8YQKVWk4nZ4UE3i0wOO1iBZBD\/hjNmxJ1YMqXwRbZg7FsGX0iB691ug8x0U11666W9HAG6qeYlfZ97HHHmtf\/\/rXbcqUKTZixAjbunWr\/eu\/\/qt97GMfa+AQG9N0npXWGCL1t+rhtPX3WjwJOOo6haHO0COh8xlFvlvBFn30B0edIwx1hsRFjWGec+26CqT58+fbnDlzbNSoUb3EyufDawibL51npTWf1uA9Enx9NAFHnSMMdYYkAjD0IeDTCj6tc4ShzpC4qDHMc65dV4F0+eWX229\/+1vbd9\/SiW\/heuSRR+z73\/++RjCCdKXStjy9ztaeOz0ZxZ6zLkz+8b62\/PkBG73XEd7NRm2P4OuDH446RxjqDEkEYOhDwKcVfFrnCEOdIXFRY9gyBdKll15qkyZNsjFjxvQSu\/fee+3GG2\/UCEaQrlTai7+5z3oun+leIL20brmFk+zCFQ5+CAdAFOki+PpoE446RxjqDEkEYOhDwKcVfFrnCEOdIXFRY9gyBdLzzz9vEyZMsJdeesm2b99uY8eOtQ0bNtj48eM1ghGk++0g\/fkB23jfqckoXj15rr3m0POqjqiWwFOtQKqljeEGknd5j8ATmwFzKFlobD2o\/TMH9FiOtaotqfLYIraILfZlPqo\/qfL4o+aPLVMgPfPMM\/b5z3\/e7rnnnuSI7yOPPNKuvPJKmzx5ctWCIms3DFUg1brTg9PpSTGBRws8XotoEfSAP2bDllQ9qPJFsGXmUAxbRo\/o0WuNznNcbJkC6fzzz7f999\/f3vWud9kee+xh4XjvO++80y677LKs1T9VxxOU1tXVldzXttufeneQQoH0xNZDe+Xb20sfkK009PC\/w+EUHR0dg\/4t\/Meh5Cr\/Vjb68O\/yVYtc+d5Vq1ZZW1tb8n\/rkSv3p86h0mmZA3pQ\/EG1pUr7y6s\/MIe1SVwNMS1NPCvrPbYtMYfSepR3PQyMI2nWONWnVVtiDn0\/5KbRX7nIja0HNVcM8rHm0NnZmdQKebzqOqRh8eLFduaZZ\/ab51133WVvf\/vbc\/eYXWVVu+HHy+ypRaXH6iZ9cr6Nf+esPOqy6WP2+FWj6YPOYIdw1JUCQ51hORkYWJz4tNw6rWCLPrqGo84RhjpD4qLGsGV2kMJhDLNnz+6ltXnz5mT3KDxmV3n0t4azOdJDFUhtVyy3XQ+c0ZxB5LwXgq+PAuGoc4ShzpBEAIY+BHxawad1jjDUGRIXNYYtUyDdfvvt9s1vftMOPvhgW79+vYUT7M466yw755xzNIIRpJtVID23svSI3rhDrrYxr\/9AhJk2rkuCrw9bOOocYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNYcsUSAHTL37xC\/vRj36UnGJ3zDHH2IwZ+dxtqVTas8vmWfgnXJ47SNtf7LG\/3FV6T2ns1Lk2dmr10\/E0U2yuNMHXhzccdY4w1BmSCMDQh4BPK\/i0zhGGOkPiosawpQqkgageeOABO+KI\/H0AtVJpm9fMt81rFiRTG3fwLTbmDe\/QLOIV6coCKXwDKRwAUaSL4OujTTjqHGGoMyQRgKEPAZ9W8GmdIwx1hsRFjWGhC6TwcdiTTjopKYI+\/vGP2x\/\/+MdeWmEX6U9\/+pOtXr1aIxhBeqgC6bXHrbKRu5ZOJBvu8gg8aht5l\/cIPLEZMIeSl8TWg9o\/c0CP5Xiv2pIqjy1ii9hiX\/al+pMqjz9q\/ljoAunuu++2N7\/5zbbvvvvaN77xDXvLW95iu+yyS0IsfAspHDV9ySWXVKsnUv89FGGhAAsfpD3ooIOSD9UOdYV7f\/3rX9vGjRvtgAMOsL333nvIe4d6xK79mgdt9N5Tqo4Xp9OTYgKPFni8FtEi6AF\/zIYtqXpQ5Ytgy8yhGLaMHtGj1xqd57hY6AKpslJ47rnnku8fVV4eihuqGgmFzmmnnWabNm1KPkYbCqUFCxbY0UcfvZNI+IhtOGFvxIgRts8++9jPf\/5zu+iiixL5wa48K61q9dakGxqp+yZNIRPdwFFXAwx1hh4Jnc8o8t0KtuijPzjqHGGoMyQuagzznGvX9R2km2++2T784Q\/3o3XVVVfZZz\/7WRs5cqRGcRDpefPm2UMPPWQ33XRTcoz4ypUrkyPFw+l5A\/sLjwI+\/fTTFr7VFK4f\/vCHdsEFF9jDDz886NjyrDR30CkbJPimBDdADI46RxjqDEkEYOhDwKcVfFrnCEOdIXFRY5jnXLumAun3v\/+9felLX0reM5g4cWIvrfAdpCeffNJ+8pOf2NixYzWKg0ifcMIJyYdpTz755OSv27ZtS44YD4Va+HflFYqjUDTttddeyX\/+3e9+Z+95z3uSAmnMmDE7td5spW342ak2\/sil7oxiNkjw9aEPR50jDHWGJAIw9CHg0wo+rXOEoc6QuKgxbHaurY22v3RNBVIQWbNmjV133XX2zne+s7eFUHgccsghw77roww2vHMUdoSmT5\/e28xxxx1nc+fOTQ6OGO76x3\/8R\/vLX\/5iCxcuHPS2ZiktHPMdTrMbtecRFEiKMRRYlkVMVy4MdYYkAjD0IeDTCj6tc4ShzpC4qDFsVq6tjXJw6ZoLpCAeDkEY+GhbIx1w6tSptmzZsn67RWFX6YwzzrCZM2cOyePrX\/+63XHHHXbrrbf27igNvDkorXxdN2eHHfomsyeeM1vxh48m7zJ5Xbut+ZCNfPkp2\/7qSbZx6i1ezWainZ6eHmtrq37iXyYGm+FBwFFXDgx1hqEFOOocYagzxBZh6EPApxV8ujrHG2+80ZYsWTLojd3d3dUbyOAdVQuk\/\/W\/\/pcddthh1t7ebuGdoHBwQvkKp9iFgxNuv\/32hkxt2rRptmjRon4foz3qqKMsvG90\/PHH79Tnyy+\/nLwPFY4ev\/baa4c98a6yqg2Pv2199oHkeO9wzHctl0dhqLaRd\/nAmTnEZ1AEPah2VAQGRZgDeiytPiqH2PJFmIPKsAgMijAH9Bg3phR6BynsxhxzzDEWipVQmOy66679jvn+93\/\/d7vllsbsjJxyyil24okn9u7orF+\/PinW7rzzzqRgq7zC+0lnn3128i5UKORGjx49bJ1DgRR\/ES5C8GUOcYNv2clZBIuhB\/RYDD0WIS5ii8WwRfQYV4+FLpAqq4zBthnDh2PDEdyNuELhdcMNN9iKFSts9913t3Bi3v3339+7YxUeowuPeIUDG771rW8lp9wtX77cXvWqV1UdTqXS1p4z3bY8s85GT5xi7dc+WFXWI\/h7tKE6fmz5IjBgDnGDLwVS\/3AV26dj948\/ZsMfi6AH1ZaLwKAIc0CPcWNCyxRI9913n23ZsiX5DlEoTsLjdx\/60Ifs3e9+d01FRb03hXeeLrvsMrvtttts3LhxyQl64ZG78u5RON2us7PT5syZY+HdpP\/7f\/\/vTl185zvfscMPP3yn\/z5YgTT2wHfYlCtW1DvMlr3fI\/C0LLyKicNRtwIY6gw9kiGfUeS7FWzRR39w1DnCUGdIXNQYtkyBFHZwPv3pT1s49js8\/nb11Vfb448\/bh\/\/+Mdr2rVJi\/mFF16wcKR45RHjadsqyzVLaZt++Vl7ad3ypNs93rtWHXam5Am+PuqAo84RhjpDEgEY+hDwaQWf1jnCUGdIXNQYNivX1kY5uHTVQxoqxX784x9bR0dHUhzNmDHDLrzwQrv77ruT95MmTJjQiPE1rM1mKY0CqWEqLEzDLGK6KmGoMyQRgKEPAZ9W8GmdIwx1hsRFjWGzcm1tlA4FUvhAa3gvKLz3c80119j3v\/\/95NtI3\/ve92zEiBGNGF\/D2qxU2nMrSwc+NOJbRRRIDVNhYRpmEdNVCUOdIYkADH0I+LSCT+scYagzJC5qDFumQAqYwjtI5RPinn\/++YRc3naPwpgHK5DGTJlp46Z9tSZr8Ag8aht5l\/cIPLEZMIeSu8TWg9o\/c0CP5cCv2pIqjy1ii9hiXxqm+pMqjz9q\/tgyBdLWrVuT0+LCMdvhAIVw\/PcnPvEJGzNmTE1FRZZuqlTaYzP3TYY2\/phZNmnO\/JqGidPpSTGBRws8XotoEfSAP2bDllQ9qPJFsGXmUAxbRo\/o0WuNznNcbJkC6corr7SHH37Y3vWud9kee+yRHNYQPhx7ySWX1FRUZOmmstK2PL3O1p47nQIphXLy7LRegYtFkEXQy5bwJ350yYotYYvYYlFsEVuOu0a3TIEUTrG76KKL+qXSP\/jBD+zYY4\/N3S7SYAXSnrMutPAPV20EPAJPbT0V+y446vqFoc7Qo9j3GUW+W8EWffQHR50jDHWGxEWNYcsUSOF7ROHbQ5XXwoUL7WMf+1jynaI8XUFpXV1dtuP5J23rV2YmQy8XSCGolK\/yN5cqf01J87dwEMSmN37Nto47pPc7TmqbjRhnPW1WBt965CrtBDmz8AHmcDokXPp+6arXxyrtKMh6+W2r2Sc+nd4Gy4lU+YPq2GBfREvjR\/i0nocMjIVp9FC263pjcpHk8On0a2r4Vml3d3eeyoPesdZ1zHc4ue7ZZ5+1Qw45xNavX2933XWX7b777rZgwYLcTb5ZO0jhG0jhJLtw7TZjqY3e64jcsRpqwPw65aNKOOocYagzLCc0AxN7n5ZbpxVs0UfXcNQ5wlBnSFzUGLbMDlI4we6b3\/xm7yENf\/u3f2vnnnuujR07ViMYQbq3QPrzA7bxvlOTEYydOtfGTj3PdTQUSK44C9kYi5iuVhjqDEkEYOhDwKcVfFrnCEOdIXFRY1j4AumOO+6wpUuX2qRJk5KPw+6zzz4asQxID1YghSO+w1HftVwegUdtI+\/yHoEnNgPmUPKW2HpQ+2cO6LEc91VbUuWxRWwRW+zLwlR\/UuXxR80fC10ghZPq3ve+99nb3\/52e+aZZ2zkyJG2YsWKWmqITN9DgRQ\/qSXwaIHHaxEtgh5YBLNhS6oeVPki2DJzKIYto0f06LVG5zkuFrpAuv766+3QQw9N\/glX2EH61Kc+Zfvtt1+mC6Bqgysr7cXf3Gc9l5d2jSZ9cr6Nf+esaqIuv5gXIXjm2Wm9AlcR9FiEOWCLxUhG0GMx9EhMQY9ea6waE1T5IthyzDkUukCaN29eUhSVrx\/+8Ic2ceLE3oLpscceswMOOKCmoiJLN5WVtuHHy+ypRaX3jtquWG67HjgjS8PM9Fg8Ak+mJ9ikwcFRBw1DnaHHIuoziny3gi366A+OOkcY6gyJixrDQhdIX\/ziF23q1Kn2qle9KqG0Zs0a22WXXewNb3iD7dixIzmwYfHixRrBCNLNKpC2NPgQiAjoersk+PrQh6POEYY6QxIBGPoQ8GkFn9Y5wlBnSFzUGBa6QLr44ovtu9\/9ro0ePXonzpG4jAAAIABJREFUSlu3brWXX345l2ecUyBpRk\/Q0PmVW2AR01nCUGeIT8PQh4BPK\/i0zhGGOkPiosaw0AXS8uXL7e\/\/\/u+TwxkGu2699Vb74Ac\/qBGMIF1WWqOP4d7+Yo\/95a7SR0AbcYx4BHS9XRJ8fejDUecIQ50hiQAMfQj4tIJP6xxhqDMkLmoMC10gbdu2rffxusEwVfu7hrZx0oMVSK89bpWN3LWtpk49Ao\/aRt7lPQJPbAbMoeQusfWg9s8c0GM58Ku2pMpji9gittiXhqn+pMrjj5o\/FrpAqqlayOFNqtJwOj0pJvBogcdrES2CHvDHbNiSqgdVvgi2zByKYcvoET16rdF5jotqrh2zvBixI5y00IJXUFpXV1cy8\/b29n4EgjGWr6H+1tPTYx0dpUfnKp2gmlxlf2Wjr6W\/wca5atUqa2sr7XjFmEOl0zIH9KD4g2pLlfaXV39gDmstxNUQ09LEs7LeY9sScyitR3nXw8A4kmaNU31atSXm0PdDbhr9lYvc2HpQc8UgH2sOnZ2duTynIOi+pQuk7u7uligNG\/UelMevGi2hgCqThKNuBTDUGZaTgYHFiU\/LrdMKtuijazjqHGGoMyQuagzZQdL4RZEuK+2phefZhnuWJWM4YPkTDRvLcyvbbcyUmTZu2lcb1sdQDVMgNR15XR2yiNWFa9CbYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNIQWSxi+KdDMLpFAchWvUnkfY+COXRplv6DSMw\/MkPYKvjyrhqHOEoc6QRACGPgR8WsGndY4w1BkSFzWGFEgavyjSvd9B+tmptvXZB5Ix7PHevnePPAcVjvkOuzixCyTPORE0\/GhWLmKVu32xdhz9Zta8lkgEfFjDUecIQ50h6wsMfQj4tIJPp+dIgZSeXTTJgQVSON47HPNd6+XhMGobeZf3WARjM2jEHMo7jrUWSCqDRsyhVj8q36fOQZUvAoMizAE9ljxC5RBbvghzUBkWgUER5oAe48YUCqR6s6EM3E+BFH8RLkLwbcQc6t1xZAHQbbkReqw3zKFH9JiVHwywRWyxKLaILVMg1bsWl+\/nFLuU5HC6Yi4g4RG3zWsW2LYXe2zM69+fHKxR7VJtIbZ8EYoDlWERGBRhDugxbjLjlRRji+jRy5bUmKDKF8GWY86BHaRqGWQG\/55npdWLc\/Oa+UnSH67dZiy10XsdUW8Tg97vEXhcBuLYSKNO\/BtuiEXk6KiSmpqCYU2Yqt4Ex6qIqt4Aw6qIaroBjjVhGvYmGOoMPYoLn1Hks5U859rsIDXB5ja8chBEve85eQ2tskAK71mFcXhcBF8Pij47cT4jyW8r2KKP7uCoc4ShzpCkFIY+BHxawafTc6RASs8ummRZaWvPmW5bnllnYw98h025YkVDxlMukELjjTopb7iBUyA1RK1ujRJ8dZQw1BmSlMLQh4BPK\/i0zhGGOkPiosaQAknjF0W6rLRmfKNo0y8\/ay+tW57MM0aB1CjABF8fsoNxLNtl+LBwLe9B+Ywkv61giz66g6POEYY6Q5JSGPoQ8GkFn07PkQIpPbtokkFpXV1dtvuvj03GUPmNouAM5au9vfSR1\/JV\/ltPT491dHQM+rfwH4eSq\/xb2elq6W+wNletWmVtbaXH5Wrpz3sOlUGDOfjpofI9qP\/c+3Tb94gvDGtn6KH0mGLllUd\/YA5rLcTVENPS6K8cB1V\/UPXAHErrUd71MHBdTbPGxbYl5tD3CHsa\/ZULddWnVT2ouWKQjzWHzs5O6+7ujpbrKx23\/DtIj83cN+E3\/phZNmnO\/JpZevyioLaRd\/ly8BmYDNWshAx8L6RRc6jnW0iqHTRqDuixHgI+76KptpB3+SLYMnMo+Q22GJ9BEfSg2lERGMScAztI9eUBmbi7rDQKpP47ZPUop4iBJ82BGiqH2PIxg2flrmbMQrkIDIowB9UXisCAOVAgERf7MhE1Jqjy+KPmjxRI9WTVGbk3KG3NAz+xtedOT0ZU7w5SRqZR0zDSJP21NOwReGrpp5n31PuRVo+xFZGjB5d62oBhPbSGvheOOkcY6gw9klKfUeS7FWzRR39wTM+RAik9u2iSAwukPWddaOGfRl7hsSnPY7ZrHSsFUq2kzJpxaMfA0RB8a9fPUHfCUGdIUgpDHwI+reDTOkcY6gyJixpDCiSNXxTpZhZIjTpmu1ZwW\/78QHLrpl99NinQvC6Crw9JOOocYagzJBGAoQ8Bn1bwaZ0jDHWGxEWNIQWSxi+KdFDa\/3v4pxYeqQrX2KlzbezU8xoylsoCabcZS230Xkc0pJ9mN0rw9SE+GMdwLHw4Hj5cRbIZH2I7t4It+pCFo84RhjpDklIY+hDwaQWfTs+RAik9u2iSyQ7Sz2+xjfedmqpA8nAYtY28y3ssgrEZNGoOlQVStW8hqQwaNYd6nFudgypfBAZFmAN6LHmNyiG2fBHmoDIsAoMizAE9xo0pFEj1ZEIZuXdggVQtCR04bJxOX8SzFnxDUZLmo6yqLQwmHx6LrLV4V\/vPmh7ShAgYxF0EyzpT9aDKF8GWmUMxbBk9okfiohkFUpqMJrJMUNrD3\/u29Vw+MxnJpE\/Ot\/HvnFXzqFjIi1cgpX0UUrWF2PJFWMhVhkVgUIQ5oEeSyqwkldhiMWwRPcbVIwVSzWVFdm7Ms9Lqobj9xZ7e96zC7kjYKfO6PAKP11g82tn068vtpbU3Jk3t8d61Hk3W1EbRONY0aeebYOgDFI46RxjqDD2KfZ9R5LsVbNFHf3BMzzHPufaIHTt27Eg\/9fxKNlNp9bxPohANj2W9atc2G7lrW28zlQWS90EURQ0agVklQ0UntcgWlWMtc\/e6B4Y+JOGoc4ShzpACCYY+BHxawafTc2xmrp1+lINLtnSB1NXVlVBpb2\/vRyc4Q\/ny+Fvl+ySb2y6yyW87uyH9lY\/zDqfklecw8uWnbLc1H0r6KxdIXvOrDBpebVaCaZU2e3p6rKOjdJpi+WqVuXv5XyUvrzbLSZpnLMh6m\/h0SduK\/wV\/bmtra\/i6oo4z63aNT+t5yMBYqNh11u2lkf6AT6fPkzs7O627u9u7dmlKey1dID36k6ubcpRyZYFU72EQTbGClJ3wq0pKcAPEhuIYjvkOu4\/hivGBYZ\/ZNacVbNGHMxx1jjDUGZaT3YE\/UPq03DqtYIs+uoZjeo7sIKVnF00yKO03P\/i0bV6zIBlDvd+a8XAYtY28y3ssgrEZNHIOtX5gWGXQyDnU6uDqHFT5IjAowhzQY8ljVA6x5YswB5VhERgUYQ7oMW5MoUCqNQvK0H1Baf925Sft2WXzklG1XbHcdj1wRs0jxOn0RTxLwbf8rlaawxlUWxhKvtZ319T+s6SHmh1wwI0wiLsIltWh6kGVL4ItM4di2DJ6RI\/ERY75TpvTRJUbWCC1X\/Ogjd57Ss1jYiEvVoGkPAap2kJs+SIs5CrDIjAowhzQI0llVpJKbLEYtoge4+qRHaSay4rs3KgWSNmZSd9Iwo5DeGQwvK9Svip3Iep9jLDaHD0CT7U+mvX3RnKqNocicaw210b9HYY+ZOGoc4ShztCj2PcZRb5bwRZ99AfH9BwpkNKziyYZlHbfBe+1DfcsS8ZQ7w5SmoE\/t7LdvL9FVDmOcpJf+ZhYIxP\/ogWN8M7P9hf\/aK+e8n4LJwE26yoax2Zxq+wHhj7U4ahzhKHOkAIJhj4EfFrBp9NzpEBKzy6aZLOVFoqjcDWyQCq\/1D\/YiWehf3aQopnbsB0TfHW9wFBnSFIKQx8CPq3g0zpHGOoMiYsaw2bn2tpo+0tn\/pjv7du32+rVq23Dhg120EEH2YQJE4ad\/0svvWS\/\/\/3v7YADDrARI0YMeW9Q2q9XvL\/3GOU0L+fXo4hmFEj1jMfjXoKvB8Xh3+X6y10dFg6QCFejbdRnNnFawRZ9uMNR5whDnSFJKQx9CPi0gk+n50iBlJ7dsJIbN2600047zTZt2mSTJ09OCqUFCxbY0UcfPajcI488YhdccIE99thj9uijj9qYMWOGLZB+ddN02\/rsA6mSTw+HUdvIu7zHIhibQaPnUPktpKEKJJVBo+dQS3hQ56DKF4FBEeaAHkveonKILV+EOagMi8CgCHNAj3FjCgVSLRlQinvmzZtnDz30kN100002atQoW7lypV155ZV277332siRI\/u1+Nvf\/tbOOeccO\/vss+1zn\/tcXQXSyF3b+h1sUMtQcTp9Ec9S8A07NZWHW9RiA+V7VFsYTr6Wj8Wq\/WdJD\/Vwr7wXBnEXwWb4Qq22odpCbPki+GMR5qDaQREYFGEO6DHu2kCBVOvKVed9J5xwgp155pl28sknJ5Lbtm2zgw8+2G6++ebk35XX888\/b6NHj7Y\/\/elP9q53vaumAuknH5lmm39zv42eOMXar32wrtHhdMUpkN7wutEWCqRwjdrzCBt\/5NKm2oJqS6o8i2DcBcSruECP6NHLlogpxVnf2ttL7z+nvVRbyLt8EeJqzDlQIKX1vCpy4Z2jxYsX2\/Tp03vvPO6442zu3Ll20kknDSr9u9\/9ruYC6eq\/etkO2m27\/enlEXb6w6XH8U4\/\/XSbPXt2g2bU\/GbHdZ9vozatTjpe\/zd3uw6gp6fH2traXNuM0djIl5+y3dZ8KOn6P\/c+3V563elNHUZRODYV2oDOYOhDH446RxjqDEMLcNQ5wlBniC3WxvDGG2+0JUuWDHpzd3d3bY1k7K5MH9IwdepUW7ZsWb\/dorCrdMYZZ9jMmTPlAknZQapXj+WX7dM8zldrX+XHsSp3QTb87NTkPatG9Kv+MlTrvIp+Hxx1DcNQZxhagKPOEYY6Q2wRhj4EfFrBp9NzZAcpPbthJadNm2aLFi2yGTNm9N531FFH2aWXXmrHH3+8XCB1\/d1E2\/LMulSP2NU75WYWSGFs5Rf6KZDq1VTz7yf46sxhqDMkKYWhDwGfVvBpnSMMdYbERY0hBZLGb0jpU045xU488cTeR97Wr19vhx12mN1555021HO19Txi9+\/XT06OUE7z3km9U25koVLvWLzuJ\/j6kKzGsVxcN8NOfWbU\/FaqMWz+iPLZIxx1vcFQZ0hSCkMfAj6t4NPpOVIgpWc3rOQtt9xiN9xwg61YscJ23313u+qqq+z++++322+\/PZG74447kndgKg9saFaB5OEwaht5l\/dYBGMzaMYcqu0+qgyaMYdqIUKdgypfBAZFmAN6LHmKyiG2fBHmoDIsAoMizAE9xo0pFEjVsp+Ufw8fib3sssvstttus3HjxtnEiROTR+7Ku0fhdLvOzk6bM2eOLVy40L72ta\/t1FN4cSw8ljfwCkr74dv+M\/nPYw98h025YkVdo8Tp9EU8K8G3fIpd2g+xqrZQTb7at5Cqyddi2GobeZfPii1y4tTaIZ8OqMWO0WPcZKhSR3mPCer4scVs2CJ6jKsHCqRaV66U973wwgu2efPmpEDyuiiQ9AKnSIHnuZXtttuMpTZ6ryPqNjGVQ2z5IizkKsMiMCjCHNBj3GSmHPzQg74+FsEfizAHbDluTKFAqjuljC9QWSCNP2aWTZozP\/6gxBFsXjPfNq9ZkLQSPnoaTq4LiX+4GvH+ikfgEafsJq58KFYdRJE4qizSysMwLbn+cnDUOcJQZ+iRmPuMIt+tYIs++oNjeo4USOnZRZN8x5vbbclfv5T036wCqVyspH2UqxqsygKpvBtS7nPMlJk2btpXqzVR198JGnXhGvJmOOocYagzJCmFoQ8Bn1bwaZ0jDHWGxEWNIQWSxi+KdLMLpMr3SMq7O94Tf2ndcgv9hGvg42Lhv1MgeRP3aY9FTOcIQ50hiQAMfQj4tIJP6xxhqDMkLmoMKZA0flGkg9Iemr8j6Xvs1Lk2dup5DR3HYI+\/NbTDJjRO8PWBDEedIwx1hiQCMPQh4NMKPq1zhKHOkLioMaRA0vhFkVYLJI\/Ao7aRd3mPwBObAXMouW9sPaj9t8IcwjffwnuJw10qx9jyraDHWhZM9BA\/JmGLrA1lX21lf6RAqiViZ+yeD7z7jXbdnMF3kIIxl6+Bx+6W\/9bT02MdHR39ZlWLXBAot1l2mnrlyp2uWrUq+Q5UZZuVDtnoOVQ6PXNAD4o\/qLZUaX959QfmsNZCXA0xbai4W4tuY9sScyitcXnXw0BbS7PGqT6t2hJz6CuU0+ivXOTG1oOaKwb5WHMIn+Lp7u7OWAVQ23BG7Nixo1QltNilVrXqLwJF+HUJBvxClpVfyLDFYthio\/S44Wen2vgjl9a0yqljUOWLsDYUYQ7okZjC+tYXMtP6g5pr1xS0G3QTBVKDwGah2S1\/fsA23ndqMpRwQEM4yc7zSuswnmMoQltw1LUIQ52hR1LrMwr\/VsIx\/uERw7TfOqtnRNhiPbSGvheOOkcY6gyLEBdD\/EvywEO+mupbjwpFCiSFXiTZZittuBPmvBCEBKDsCOHgifDtIwokL7qNa4dFTGcLQ51hERKBoSiU42I4QbTRF7boQxiOOkcY6gyLEBfDDnq4at1F96FWaqXZubbn2Ft2B+msw99gF75hS8Jy0ifn2\/h3zvLkulNblQVSI3ZzQoeVBVLYLRq11\/Qhj\/32mCzB14Oi\/jKxzyjy3Qq26KO\/onIMnzkI8dB7F30w6kVl6GNhtbcCx9pZDXUnDHWGWSuQQp637cWeunaCQvx79ZT31yXjQ44CyYtjU9v59AffaF\/4UOn1q2Y8dtGMAqmpAB1OLmv2eLPaXy2LWKMfl8wqm1rHVQvDWttq5fvgqGsfhjrDrCWlPjNqfitFtMXnVrYnIMOPHd7fdsxDoRkKpM1rFjRt7qrVsoP0\/9s7E2AriquPH5aHPFAIq8J7SD1Q0BIUFyIoBATcIKgIolGiZlFBMRCNmhhlcfugKBGMW4lLJK6IwqcmmhgxGotFMRKFGLQEhQefooiAQFjfV6df+jpvuPfOTHfP9PTc\/1RZCHe6+5zfOd3Tp1ddghbS6wZIJhoe3TxcT2\/iI2ibQVI6+GcHvR8GXQZJ6VCsmuvqoJs+CwyyoAPsWFtLdDnYTp8FHXQZxskgzJH9Jso3kYcuR396uZ+QtxCEWTKmW34aGah0mXU5qKZHgKRiLctpJpx8KI1qt0dIUTl5LjU56qRIEqk6i7cQ3TxcT4+GJ1pnSI6ctTzru2PoTTA0kQd80X6nFnaMVp\/iGi3WrQtZsGMWdEirHXk\/yZ6Ni4X7+r8Ffp9Oqw5ROlv5AiSZPsyewrXL51FFp56Bd8CleQDPZTsiQIri7Sl5V9doOg67Y+UMMUXKj87yPh0Z0vABS4MMugyhAzqlsknT9SXd9FnwRTBAfQpTn3gWgQ8iKranTNeXdNPHVR95Pwkv2eeHA4Rilz+nVYco3UAdHfwHV5V3HR+l6Ny7OjLE5QdRFbGlg25fO6qeJt8v2UMabBrNGyAFNXBRje2dfi5rfWIuEDNdjolKH1W3rL6v23BllUsUvcAwCq3C74KjPsckGIZdYqWvjb0cinG0uencHpHoJSfhi9GlSi6Fd+8uB9S2AqTkNN6\/JK4rfKhDmOWIcchps6+tqw8CJF2CEdLLJVLeJKYDFxkg8ahSWateuZGmoKn4CGoYG1VRKTOLaUr9I2bCpmBogqL+MkEzUgTnwh2fBk0qtZbNBJei9kbcvhj1wBZ+n5+y1r3UFLKUKogj6+WaTkmjDGKYtDw2y9PxF5c58oA879eyVVcQINn0esWykzaaqaneIHW9AVKY9blB+RX73eVGQ0dv02nBUZ8oGOZnKNudsAMkrnDkzg7vw1AdEdb3uMI5xM2w2IEtfql4Kdaer5ZQ\/SYVqWSF70ucnujOgEe8FPRzj7tO60uY3hyS7mubJFGyM0iP3VRFQ79fi9I\/i8OVQT5VVbVHSspH9bfPVi6ig1ZeKLLxH0+pmqftdN5Gw7YsJmxkS4fq6mrq27f2pmtdP7OlA8ttqq6o6OBNY1sWLl9FhzjSeWcbdlTeQBXHjS7qZ67UaXltAgd+aWEt7cf1ubKy0mp98PpS8w8GiJk274BZ2pjl+96iTuv3Q\/xtoQt2j6Md1M0zbXXapW\/cwIEDadWqVXW+O678BQFSiJNg8hnTxIiCbh6up5eNlr9jHaXy2GYAHWqtZdsOuuVnVYeoB8LocrSdPqt2jNImpoFBGmTIui\/yIEHQxce6DGBHfN+8g7YqfTXMIEVtvVPwvncGKezyE6\/YaHj0O8VZaHyhg90PCAcAvMwK9dGuHXQ\/oqbSoz6mww+yYIe0tyl83HfQxvt1\/3iA2h7aQ2v\/iS6HuNKHPaiEGfhnzqN2QePSIawcuuXbrI8IkMJaOUXvzT29ko4+aJ+QqMvc9ZEl03HYKPcYFBNMRwabFcZkoGmbQRo46jJwWQc++EQus1IZ3YIv1m1hdH3Jdnq\/L\/MoO1+pEGU\/Ztp0iPxxSsGMbtxtSpijrrNux7BHnfMSS5snuNm0Ay8x3vjh7+mQvg+oVKNcGps6mKhLJvJQZYAAScv17CSWAVJZmw5Udf+SRIWQAZJ\/XbhpIaLeOB21fNUKE7WcrL8PjmoWlpvVmx47jdbvOX6\/PR9quZZ2qqz5ojykgH0kqScphmk+yc8E62Ic5R40LkfnLkETcqY5j6R8Ma0MTB1x7SLHqKddxmVDBEhxkY0xXxsBkj9g4b9HGdkMg8M7sibf5yMeg6biw+Ttf8fFRkNFz7jTgKM+YTDUZ2hilNGMFOZysXHEdVK+yLrt21FdcB+KDCJ4IK7ZSU+l8kj0YpYuxjEtnT9znhpPTkn5YjzSpyfXtHLkFRQ8+JNvL1paBhEQIKXHj0NLkoYAKbSwEV70bspGgBQBnMVX09r4WkQSuWgw3B9Z1CO+XQuQTI0OR3a2gARJ+aL3JL98IkU9oMM0B938THDkDmRcA4S6+iWR3gTDJOTUKUPnfqOw5aaNY5gLcL0Bkso++7Bsgt5DgBREKIW\/L\/99FbVvSfsdfxqnqHEveYtT9nx5p63RSFp\/U+WBoz7JfAxtzB7oa2IuB5VRdpd8kfXbtfY5MYKapsclhmni5pfFBEf+5jbtMU3rkII0MyomGy\/l39j+jswuPebglx\/\/tSlx2MuEL5qWK0n9dWRHgKRDz1JaGSCpji6ZqDC6ebienk0PHewzyIId8vlR1A9I1nxRZQYhawxUPi+2GWShPqZFh7bf3KccQOv6gW0GPIOw+aPHqXHjcq0l9roc4kofdsBZt3zbdjRRvok8VDkiQFL5CllO8\/ezKujgRjVUflRv6jD5ucjSqDqLtyDdPFxPb7PSSjvoMoQOtSR1Oeqk50BoV4vTS\/KkItbdO4JaKFD0X4ZdqMHTsYNtPzBVp20zSANHXQZZ0CHtDLyH1BS6D2nt8nnUoduwyP0bF\/op8rCrQsvHmM+WhT+iTZ1na8+i6fqC7fQ26yMCJK3qZycxG+2V4\/5jJUCSI9t7mh5DbQfOVwZgu9Lplm+z0prqTEEHuwGSrEulGCDJDhJbQAZAunXS9fSyPnY8uIx4hJmfqOvvbTNAmxKuTfHOkOYbAMi6HWX9L7bETJeBy74o\/WNz9wUlFyDluyNK1xdU0yNAUu7i20uoGyDpSC47darL+4qV7d2YJ9\/TuQOhWFmqFUaHXRbTgqO6VeUyi32NDqHWZyxSz8jRlLItCTtDFKRmVnzRGzzG1f4VYpk0w3wnWfH+rG3LrjN+SmqQ\/5j8PYijd48djvrOTz6IoUl7pSkvOXvEf0YdIMmnh2scve1foVPukrIXAqSkSBssh4322muviRz9F0xyZZCPyd\/Kq6dSo01\/zuUtR35Mluf9aBQKkEyV5200TOXpNXGp5FldXU19+9aOdsunVHQ3Vf+8vEzlyfmUmh1cqtPewaAdlTeIZZbe9prbwvUba4iD5yR9gutzZWVl7N8Vv\/4Vx43OtR\/e37Z1mr7fMisX\/DqoTq\/7xwPE31R+\/J1AF\/RLon3x+z241FaRqP26pOp0FJ9o3\/Bd4pM8pf\/zXYDyOWTX\/4oLsvnhwYPqrQfH0qcN008eOHAgrVq1qk7\/xpW\/1KupqalxRViTctqIar13FJkY1cjHI9\/xj\/yxLLRGWYepa6MqOrrGmRYc9en6Gdbs20X16jfSz9jRHLwjiFHaGpd8UeWUviTMmRRDr439y6xUDuhIgk2UMnQ4ZkH\/KKwKvavD0ET5WckjjRzDHPWdBv42+tqm9C7ZAEmeYpfEEZHSWEkESKYcI0w+aWw0wsidtnfAUd8ifoZyAy9fkmn6MmZ9aePPQTV4cM0X5RLDxp0upSbdJsYPNkQJaWLIfKIEyCHUS+wVHY6y\/rOwruqvA1reEba1vB95ZxZ18kxr2rgGgL366vhinNy4fvM3jvuxvJQ4jQ8CpDRaJUAmGSCprk83UWF083A9PZsIOthnkAU7mAiQsuSLqvszXGPAejZoUik6CfJxTYd8nyroYL9d1LVBEu1qoaOuZYBoYm+mLoe40wcNAuiWn4Qdg7rQLuuAACnIuin8\/dXpVXT8YSSi7vKu4yNL6LLDpqUjgYan1hK6vqSbPg0yqOpQaJmBykEoqjKkpT7xkqvPvtitdWKT6wxc9mWTI9a27ZgFO+gyTIJBoZkyUyflJqFDUOermB3kMlP\/LKHcg8erB3TbxLQzCOJn+\/uEACmshVL03r2ndKBTW+2lViOvFf9FfVQbT++68XybZ6PIoSqD7QqDjkBdK9u2o+sfgKgXwhaqY7btoFK+f6aIN+P6NyCXYpvS\/IMBQm2VE5xU7GCyTXO9Pib5fSl21Hcp2NG7bD\/fSZaf\/310Sd4PJwPHUg2QOECsX15JZa175ZomW\/UBAVKUL3BK3tUNkFTVSOr4xSQ2qepWOFWGWUsHjnoWNRUg6UlhJ7XpjbpZ8EXV\/VemLJgkQ+9pdfKoa6m\/63tvwnDMp78pO2YhnzAMs6CnXwf5TTB1\/LtLHE1\/E3T9AwGSLkEL6SecfCiNareHDrlqBjU7ZWQiEniDFi7QVOXgEew9AAAaJElEQVT1Cy\/XJct\/j6sclxqNRAysWAg4KoLzJANDfYYmZi7MSKGWi9ysbbvTnKQvenWVs2Wq+8\/UqMeXKgxHr\/6uB4RxkAzDMI5ys5ZnmjnKYFDW\/yQGx6PYFwFSFFopefejEe2FJEkGSEl9uP0BkqlLJP2mS3OjkRI3CyUGOIbCVPQlL0OepeW6xhcxe5cY6JfiRg6qR3y7GCAFBQP8e9I+kGR9znfUd9o6SKq1RpVjVvRX5eZNp8rQRNlJ5sFBgup+8jByppVjoRVJpi8QD8Oo0DsIkHToWUrrj7qTECOpACkJXVzsTCXFJWo5aW18o+ph830vw6B1+TblTKps1fbNRV9M2xJLFxkm5ZdRylHluG3ZDbRzzRxRVCnOKsn9N6y7KsModrL9bhL1P80ck9Bfx8YIkHToWUqr2oGQ4pqoMLp5uJ7eRJBlmwF0qK0Rtu3gLV\/1DqQ06aDaLLIO\/lvUo8ygZIWBywdVpK0+6fiiDTuYOuJaty4kaUfuz\/CMebOTnyJv+7ep82ytQ1uS1KGQnwXZodBR5+ir1SUaxDGonqumR4AURDaFv+saTdVZvCh083A9vQuNbxjXhR3sB0i6Ax4u+6J3Znpr1yepY9feObflJRjeO4KC\/Bm+bN+XXfZFm983\/304tn05KTv6j\/r2XnGwsf0dJREgeS8D91\/xoOsHSdmxWNvssg66fe2gb1acv9erqampibOAtOatazQdh5WjPbqNl44Maaj0aZBBlyF0sD+DZOrUHl1fsJXeGyDtqLyBKo4brdzs2tLB1Givd02+6p4E2wzQpkRrUwotqS0VO25ffgv9Z9WjApp\/v7EuAxd9UQZIsv6XEoNC90LZtKNuX1v5Y2YgYckGSPKQhoaj76FOg86tg5IrlHz8SwRM\/Nbik4uprFUvcUeHdFyT5Xk7jPlu0jahA3PxNjym8vSPQJrkEgdrE3lWV1dT3759E\/XBOPzaZp7yo7irxenEQYJNWUz4BOcRVgdvff9P20uofa9Jyr7kep2uPOgL2rrwR0J\/7iB93ujsWNvyfDbi+lxZWRnafib8Rd77xDrvWDlT6Ly5+4LEdTfZXnu\/KcXqQ6NNf6by6qmiaHliK75H3zUBXh\/NOpe1y+dR01XXCOX9d6Dp6G6jTkf5Bsg2pOG2ZTn9vQNEOrrr1umBAwfSqlWrDIQryWdR8gFS5eS51OSokxIj7x\/diKNg7ygqL6\/xTj+bLM\/EyIxJeVzNCxz1LQeG+gzlR1Zn34gZKaLl4p9BlAFCXNcbBEmXtC96ZxAP6DBCnODoDRaC5E3r72E54qjvwhYMyzCtPpAWudLMMW3tn99mmEFKixdHkEPOIFXdt4TK2naIkFL9VVNLgcJIwIGYyi3yYfKW76S50Yiih+13wVHfApIh17Fty66LbVBAX9J4c\/AOjhzUezaVtak7MxlUuou+aPtiWD9TGwy\/23fxfdqz8e2SCpC8\/IOOfQ\/y\/6z9bsMXs8Yw7QNHaWv\/ECBloAbYOCfexPr4NKFH42vGGuCoz1EyLPU7UHQ\/lq76YpqOunWVoX4tNJuDCsdSP+Lff2CDCkOzVkwmt7jrf9o5xq2\/jhUxg6RDz1Ja3QDJRIXRzcP19CZGZmwzgA61Fdi2HWT53g5S1DtQ0qKDTpO4fvEkarzhMeUZhCwwgA7262Oa2gTV+qTrR0ky8F8OL5fWu6RDITuF0aHYUd9h0gf5iG4ettMn6YuYQQryJgd+lwFS1E6UVE3X4W06rCkdwCBdwYFOtdO1pe306\/7xQG6jNnNQ2XtnWwfd8llvbtf8J1lF8QtdGWynz0K7Ch3Qrkb9RvsDJHknkm59dMUXWX9u9\/xHfJuQ30QeunbQTW9TB8wgRfkCp+RduQepy9z1ShLpOCxPg\/OFbjp52HR4CWzy5Mk0ceJEJX5RPwA6o0vFBNS1gQk76HJMgw66Muimf+R\/htE5Ry0Tplbde6crg8303o3qqvozO9u+qMPQ1BJmHRlMMDTRpujqoJvehA5RfDHf0jpdHXTTm2AQVoZCM+dRGKb1GxuWgRwg4j\/Lj7yeyg8fI1SKkr4QA12OujKESV\/siG8THMLIkI8fAiStLrK5xPv27aN\/\/vOftGXLFjr66KOpRYsWBTPXDZB0pOZ9EuVdx+tkEZg2iTWpLjt+IMAEXwBHfdgnn9CJXpxQe6Wb6v03+lLYy8HUATAu+6IpBrpWtMHQu\/eO5VddGaGru8n0UTh69deZQTUpfxryisIwDfKakIH7PqZPsHSBY1raP78NXWBXyO8yc8z31q1badSoUbRt2zaqqKgQgdLMmTOpX79+eXW3FSAlEbh4R1L52Fd535KJxsebh8uOb5qFTn7MceXiNxI7TVFH1rSmhS+asYzLHL2zaKY7SFHo2mIovy0sq84sYhRd43w3Ckdpe7Z7gyaVYoltWp\/dG9bSxjl30iFjZ8QuYhSGsQvjKYAZ7P5ybaJXrOjol1aOXp3S0v4hQNLxtJjS3nnnnbR06VL6wx\/+QA0bNqQXXniBpkyZQm+99RbVr19\/v1I5QCpr04Gq7l8Sk0T5s5WnzMQ5ym1qqUkQGBcajSAd0vB77yOq6MlT2tLBY+9y5oNhiht\/KPmY\/e0rFmrpLn2RO4kujZ5L\/U3xlHsRVAOEJOv0ltfnUMO2lVp293LjtnXfjmrrR7yHZWi6kyhtz8FB0x7TqKx1L1NuFUs+QYFCWI5e4bj+q+w\/jEXBAplKv+f+Bz9xXjOiwjAJFqvHnEh8B6VJ3XmZ4d7t1WL7guknrRzz6RnHDJoOT5fY+fXMzAzS4MGD6bLLLqNhw4YJHffu3UvHHHMMPfHEE+JP\/2OrIZX3tDQ76alUj3KFqRAuO34Y\/ZJ654qeHenajrtFcUney5WUfoXK4Y7C5r89IzrIPKKquh+Q85e+GOdH0jQv1tnbSW418lrtInT1T7JO8yBVs\/4jxUhyh8nPaevOI6g71zwXSwcpinBhGa6dOFz4fvlRvY0FiVHktPku+\/3qK08UurPf57usPSxHrx4cJPLgI6+cSPPD+n9+b+0y+0L6q8jP9b9Rh+G54FiFoUq5UdPIdo+\/Aay\/N1BSHTTigWG+JDqOFTNxcJQ+0Lz\/+dTslJFRETrzfhzsklI+MwES7zmaNWsWnXjiiTl2gwYNonHjxtHQoUONB0iqG9a8gujmYTu9Cce3rYNu+WxP3Tz+flYFHX7aeaKToNJQ6pZvQgcVGfjjyDNHO1YsEp1kflSDJOmLPADBj8rouYoOOvWZ9ecgiWdRmEHD0fdQp0HnarX9a5fPo0MOaaekvzfQVBUiCkO2ffXEEWImnxlwkBQlfSEZdfPQTR+2XZQdJDmT4F1ypSuD7fRBbQrrvuVvc4jrANf9fPU+LEevH3j396adAcstg+RCgyNRdeBBAm9wqMLQX6+iyhA2Pdd\/fvIFx6rtKrf\/\/rZfV\/642kXWn\/2f2\/6glUwmdNDNQzW9CR9U\/R7ppstMgNS1a1eaM2dOndkinlX66U9\/SiNG7D+adO1VP6LFS5bQ\/32tixDpQUCPwMGNauiLXfX0MkFq5wjA7s6ZDAKDgHECaAeMI0WGKSLAkxZPPWV+2WMSKmYmQDr22GPp3nvvpZNOOinHrU+fPvTb3\/6WzjzzzCRYogwQAAEQAAEQAAEQAAEQAAHHCWQmQDr\/\/PNpyJAhdPHFFwuTbN68mU444QT6y1\/+QlVVVY6bCeKDAAiAAAiAAAiAAAiAAAgkQSAzAdKTTz5JjzzyCD333HPUvHlzmjp1Ki1atIjmz5+fBEeUAQIgAAIgAAIgAAIgAAIgkAECmQmQ+JLYm2++mebNm0dNmzalNm3aiCV3mD3KgJdCBRAAARAAARAAARAAARBIiEBmAiTJ69tvv6UdO3aIAAkPCIAACIAACIAACIAACIAACEQhkLkAKYryeBcEQAAEQAAEQAAEQAAEQAAEvARKMkDiGab33nuP6tevT3z63QEHHACviEBg\/fr1tHXr1jopWrZsiVm7EAx37txJn376KXXp0oXq1at7tPeXX35J\/\/rXv4hZduvWbb\/fQ2RfMq8U4rhq1Sravbv20l35VFRU0IEHHlgybMIoykuSP\/jgA1GP2Rfbtm1bJxl8MQxFomIc4YvhGO7Zs0f44rZt24iv6\/Cv\/oAv6nOEL4ZjKN\/iev3xxx\/ToYceSuXl5bnE8MVoHF1\/u+QCJO6AXnrppWJvEnekNmzYQM888wxxJwpPOAKXX345LVu2jL73ve\/lEgwbNozGjBkTLoMSfYt975prrqGPPvqIPvzwwzqB+dy5c+mWW24RAfuaNWtEw\/zQQw9RWVlZidIqrHYxjnxhdKtWrepwmzRpUp3j\/0sdKH\/k+bRPDtDbtWtHb7\/9Nt1www00atQogQa+GM5DgjjCF4M5clv34x\/\/mFq0aCG+J++++y79+te\/posuugi+GIwv90YQR\/hiBJhE9OCDD9KUKVNE37Bnz57wxWj4MvN2yQVII0eOJL646tprrxVGvPHGG4n3Ld19992ZMWrcilxwwQU0fPhwOu+88+IuKjP582gUB5CjR4+m66+\/vk6AxKP47JMcEPE9Xjw7wgEnd1gvvPDCzDAwoUgxjjU1NdS5c2d64403qEOHDiaKy2QefDccDwzNmjVL6PfKK6+IwH358uViFB++GM7sxThy8AlfDOY4duxYMbvLnVF+XnzxRfrNb35D77\/\/PnwxGF\/ujWIc4YsRQBKJmaOf\/\/znYi87H\/TFARK+0dEYZuXtkgqQtmzZQscccwy99dZbuRkjXmrHI1g8xe9f8pQVI5vWY\/DgwXT11VfjAt4IYDdt2iRmNb744gs69dRT6wRIfFfXbbfdRm+++WYux\/vvv5+WLl1KDz\/8cIRSsv9qMY5cv3v06CFGoXlEGk9+Ahwc8fLi1q1bixc++eQTOuOMM0SAxMElfDGc5xTjyIMc8MVgjjwL17hxYzrooIPEyzw7PHToUPEnfDGYn3yjGEf4YniOvNxzxIgRxAEnrzy46667RICEb3R4hll6s6QCJO4A8Mg8L3GSwRA3LDxiumTJEuyhCenZffr0ob59+9Lq1avFbEfv3r1FwORdqxsyq5J7jTuj\/gCJg6DXXnuN+C4v+fzxj3+k6dOni3\/Hsz+BfByrq6vpBz\/4AV1xxRW0ePFi0fE6++yziWc88RQmcNNNN9E333xD99xzjwjI4Ytq3uLlCF+MxpD3ZTKzmTNnigveecknfDEaQ347H0f4YniO3AZ+9tlnNG3aNOJ+jgyQ4IvhGWbpzZIKkHhkmZct8f4P+chRZx69r6yszJJtY9OF93B17NhRdDy3b98u9s7wnpnf\/e53sZWZlYzzdezvu+8+euedd+jRRx\/NqckjVsyVZzvxhAuQuHPAyxdPP\/106t+\/v5gZ4eU6vHxM7mkAy7oEuAPAwfjTTz8tZpTgi2oe4ucIX4zG8corrxSDlIcddpgYuT\/yyCPhi9EQirfzcYQvhgPJ\/UJeAv\/SSy+JGU1vgIR2MRzDrL1VUgHSypUrxbIwXmPaoEEDYcu1a9dSv379sCxHw7N5mSJPS\/OyCJwIWBxkvgDpscceoz\/96U9iQ6h8nn32WXrkkUfo5Zdf1rBMdpPm45hPW16quGDBAmKeeL4jsGvXLrruuuvEkk9mJJckwhejeUkhjvDFaBzl2xys8yENXGe5TUS7qM8x352QaBfrcuUDu8455xwxoMaBET\/eAAntopofup6qpAIkXg7Ge5BeeOEFcbQtP3\/961\/FQQ18khOeYAJ79+4lPubbuwmeR16GDBlCK1aswDK7AIT5OvYLFy6kcePGCR+USz9vv\/120XnF4SH5gebjyPV748aN1L59+1winpXjjd\/PP\/98sHOXyBtch3mklJfE3nnnnXVO\/IMvhneCYhzhi+E48qAQH0zj\/Z7wN5rbPd6ziXZRn2OvXr3QLgZg5MFz\/1LszZs3i5kkXrLN+wnhi+F8MUtvlVSAxIb7xS9+Iew3Y8YMccz3JZdcQnwEJgdJeIIJ8JJE3rN16623ilkjZsjrxT\/\/\/PM6e2iCcyrNN\/J17JnhKaecIk6546Vg69atE3vl+GSnAQMGlCYohUCTlyPyEfTz588XAyBff\/21+Ojxpm\/eI4enlgAHjTxIxMd5y5l0yQa+GN5LinGEL4bjyN\/fZs2aiUC9UaNGYsCSl4m9\/vrr4m4utIv6HHmvMNrFcBy9b3lnkNAuRueXhRQlFyB99dVXdNVVV9G\/\/\/1vcckfj15xsIQDBsK7M3\/EJkyYIE7C4iPSu3fvLjrzuEuqMEPe\/MmHLvif2bNni6l8XqbInXhessP7uviY0fHjx4c3Som8GcSRf+fjq\/lOFT544Ic\/\/CFNnDhRdL7w1BLgUyi5\/fM\/8s4P+GI4TwniCF8M5siDQXzlBi\/P5uO+uSPKg5U8QMQPfDGYIb8RxBG+GI5joQAJvhidXxZSlFyAJI3GxwU3bNgwd7xoFoyZtA48Qt+0aVPsOzIEnu\/x4QCeR1Sxl0sdKi99Yt9s2bLlfjMk6rmWVkr4ohl7wxfDceQ7Z\/gOLnn0vDcVfDEcQ36rGEf4YniOhd6EL+ozdCmHkg2QXDISZAUBEAABEAABEAABEAABEEiGAAKkZDijFBAAARAAARAAARAAARAAAQcIIEBywEgQEQRAAARAAARAAARAAARAIBkCCJCS4YxSQAAEQAAEQAAEQAAEQAAEHCCAAMkBI0FEEAABEAABEAABEAABEACBZAggQEqGM0oBARAAARAAARAAARAAARBwgAACJAeMBBFBAARAAARAAARAAARAAASSIYAAKRnOKAUEQAAEQAAEQAAEQAAEQMABAgiQHDASRAQBEAABEAABEAABEAABEEiGAAKkZDijFBAAARAAARAAARAAARAAAQcIIEBywEgQEQRAAARAAARAAARAAARAIBkCCJCS4YxSQAAEQAAEQAAEQAAEQAAEHCCAAMkBI0FEEAABEAABEAABEAABEACBZAggQEqGM0oBARAAARAAARAAARAAARBwgAACJAeMBBFBAARAAARAAARAAARAAASSIYAAKRnOKAUEQAAEQAAEQAAEQAAEQMABAgiQHDASRAQBEAABEAABEAABEAABEEiGAAKkZDijFBAAARAAARAAARAAARAAAQcIIEBywEgQEQRAID4CEydOpD179lDTpk1p69at1LhxY9q3bx\/t2rWLzj33XKqsrKTx48fT448\/TmVlZbEI8uGHH4pyq6qq9st\/wYIF1KdPH2rUqFEsZSNTEAABEAABEACBugQQIMEjQAAESprAFVdcQTNmzKDy8nLq1q0bTZo0iUaMGEEvv\/wybdy4kUaOHEmvvvoqDRkyJBZOCxcupDVr1tAFF1yQN\/8dO3bQ3XffTb\/61a+oQYMGsciATEEABEAABEAABL4jgAAJ3gACIFCyBPbu3Uvvv\/8+HXvssYKBN0Div7\/zzjvUs2fP2Phs2rRJBD4PPfQQ1atXr2A57733Hr355ps0bty42GRBxiAAAiAAAiAAArUEECDBE0AABEDgvwT8ARL\/8\/PPP0+zZ8+mRx99lDZv3ky33367CKS2bdtGvPxt9OjRVL9+fTHjtGHDBnriiSfowAMPpHXr1tGcOXPo22+\/pbVr19LUqVOpRYsWdVjPmjVL\/P2yyy7L\/fvy5ctpxYoVtGXLFurYsSOddtpp4rczzzxTyMIzXXhAAARAAARAAATiI4AAKT62yBkEQMAxAvkCJA5UevToQe+++64IcK6++mpq06YNTZgwgd544w0aM2aM+JP\/7eyzzxYBEwczP\/nJT2jmzJlibxP\/f+fOnYn3O3mfiy++mPi\/QYMG5f6Z\/3799ddTkyZNiPcmyaV9\/O8\/+9nPqF+\/fo5RhbggAAIgAAIg4BYBBEhu2QvSggAIxEggX4DEy\/AOP\/zwXIDES+K6dOlCl19+Oa1evVoERbxMjx8OnnhJ3tChQ8V\/HDzJp127djRgwIA60nNgdNttt1GvXr1y\/37HHXfQxx9\/LGas+PCIhg0bit+uvPJKOvnkk+miiy6KkQCyBgEQAAEQAAEQQIAEHwABEACB\/xKIGiB9+umndNZZZ+0XIA0cOFAETm+\/\/bZYfscPn4rnP4lu2LBhdM0111Dfvn1zNuAleWeccYYIts4\/\/\/zcv\/MyPJ6Z4pP18IAACIAACIAACMRHAAFSfGyRMwiAgGMEjjjiCLr11lvpvPPOy0m+e\/du6tq1Ky1dupRatmwpDlUImkHiWR6eFRo7dixdcskltHPnTnr44YfFLJD3ufHGG+mEE07IBT28j4mDoKeffpoWLVoklujJZ\/jw4eKEve7duztGFeKCAAiAAAiAgFsEECC5ZS9ICwIgEAMBPnBh\/vz5dPPNNxPP\/nAQxEER34f00ksviXuQ+Lf+\/fuLpXWdOnUShzU8++yzNG3aNHrwwQfFHiPeI8SzULxsjgOqX\/7yl9S6dWvx3\/Tp06mioqKO9Hw63TPPPENTpkwR\/z548GDioGnx4sViqZ7cb8QBFh8DPm\/evBi0R5YgAAIgAAIgAAJeAgiQ4A8gAAIgEBMBDrC++eYbMfNU6LnpppvEXqX27duLk\/F4xqp58+a5pXmcjmef+KCI448\/PiZJkS0IgAAIgAAIgIAkgAAJvgACIAACFgnwRbB8DxIvxWvWrNl+kvBs0vbt2\/c74MGiyCgaBEAABEAABDJNAAFSps0L5UAABFwh8PXXX+edaeLLZP33J7miE+QEARAAARAAARcJIEBy0WqQGQRAAARAAARAAARAAARAIBYCCJBiwYpMQQAEQAAEQAAEQAAEQAAEXCSAAMlFq0FmEAABEAABEAABEAABEACBWAggQIoFKzIFARAAARAAARAAARAAARBwkQACJBetBplBAARAAARAAARAAARAAARiIYAAKRasyBQEQAAEQAAEQAAEQAAEQMBFAv8PkyBsYwBvYWIAAAAASUVORK5CYII=","height":337,"width":560}}
%---
%[output:9cc6c830]
%   data: {"dataType":"text","outputData":{"text":"Figure saved as PDF: \/home\/tomaso\/UANDI_controller_VSQP\/Data\/Simulation_position_sinusoidal_tracking\/Sim_P_E_chirp.pdf\n","truncated":false}}
%---
%[output:28526580]
%   data: {"dataType":"text","outputData":{"text":"ANDI: Mean error 0.07 m\n","truncated":false}}
%---
%[output:2560c462]
%   data: {"dataType":"text","outputData":{"text":"INDI: Mean error 0.18 m\n","truncated":false}}
%---
%[output:58500a80]
%   data: {"dataType":"textualVariable","outputData":{"name":"Fs","value":"500"}}
%---
%[output:3554a79f]
%   data: {"dataType":"image","outputData":{"dataUri":"data:image\/png;base64,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","height":337,"width":560}}
%---
%[output:601c6636]
%   data: {"dataType":"textualVariable","outputData":{"name":"magnitude_andi","value":"0.8744"}}
%---
%[output:50c692ea]
%   data: {"dataType":"textualVariable","outputData":{"name":"phase_andi","value":"1.6947"}}
%---
%[output:7003a0dd]
%   data: {"dataType":"textualVariable","outputData":{"name":"magnitude_indi","value":"2.2980"}}
%---
%[output:5caf984d]
%   data: {"dataType":"textualVariable","outputData":{"name":"phase_indi","value":"6.0243"}}
%---
