# R=script for the article: 
## Towards Effective Human Intervention in Algorithmic Decision-Making

# Packages
require(tidyverse)
require(readxl)
require(scales)
require(gplots)
require(rcompanion)
require(here)
require(psych)
require(quantreg)
require(mediation)
require(rstatix)
require(flexplot)

# Setting the working directory.
setwd(paste0(here(), "/"))

# Loading data.
dt = read_excel("cleaned_data.xlsx")
# Print column names.
names(dt)

# Remove unnecessary columsn.
dt = dt[, c(2:5, 8:41)]
dt = dt[, c(1, 5:35)] 

# Renaming column names.
names(dt) = c("ID",  
              "AI.Literacy.Lik", "Affinity.Technology.Lik", "Exp.Human.Lik",
              "Exp.AI.Lik", "Good.Public.Admin.Lik", "Agreement.Policy.Lik",
              "Complexity.Lik", "Profile.Cat", "Model.Type.Cat", 
              "Data.Type.Cat", "Ability.Agg.Dbl", "Ability.Q1.Lik",
              "Ability.Q2.Lik", "Ability.Q3.Lik", "Ability.Q4.Lik",
              "Ability.Q5.Lik", "Ability.Q6.Lik", "Benevolence.Agg.Dbl",
              "Benevolence.Q1.Lik", "Benevolence.Q2.Lik", "Benevolence.Q3.Lik",
              "Benevolence.Q4.Lik", "Benevolence.Q5.Lik", "Integrity.Agg.Dbl",
              "Integrity.Q1.Lik", "Integrity.Q2.Lik", "Integrity.Q3.Lik",
              "Integrity.Q4.Lik", "Integrity.Q5.Lik", "Integrity.Q6.Lik",
              "Fairness.Perception.Lik")

# Hypothesis testing.
# -----------------------------------------------------
# H1A: Profile -> Ability
# -----------------------------------------------------

# Normality assumption.
shapiro.test(dt$Ability.Agg.Dbl)
# Homoscedasticity assumption.
bartlett.test(dt$Ability.Agg.Dbl ~ dt$Profile.Cat)
# Kruskal-Wallis
kruskal.test(dt$Ability.Agg.Dbl ~ dt$Profile.Cat)
# Effect size.
ordinalEtaSquared(x = dt$Ability.Agg.Dbl,
                  g = dt$Profile.Cat)
# Visualiation.
plotmeans(dt$Ability.Agg.Dbl ~ dt$Profile.Cat)

# -----------------------------------------------------
# H1B: Profile -> Benevolence
# -----------------------------------------------------

# Normality assumption.
shapiro.test(dt$Benevolence.Agg.Dbl)
# Homoscedasticity assumption.
bartlett.test(dt$Benevolence.Agg.Dbl ~ dt$Profile.Cat)
# Kruskal-Wallis
kruskal.test(dt$Benevolence.Agg.Dbl ~ dt$Profile.Cat)
# Effect size.
ordinalEtaSquared(x = dt$Benevolence.Agg.Dbl,
                  g = dt$Profile.Cat)
# Visualiation.
plotmeans(dt$Benevolence.Agg.Dbl ~ dt$Profile.Cat)

# -----------------------------------------------------
# H1C: Model Type -> Integrity
# -----------------------------------------------------

# Normality assumption.
shapiro.test(dt$Integrity.Agg.Dbl)
# Homoscedasticity assumption.
bartlett.test(dt$Integrity.Agg.Dbl ~ dt$Model.Type.Cat)
# Kruskal-Wallis
kruskal.test(dt$Integrity.Agg.Dbl ~ dt$Model.Type.Cat)
# Effect size.
ordinalEtaSquared(x = dt$Integrity.Agg.Dbl,
                  g = dt$Model.Type.Cat)
# Visualiation.
plotmeans(dt$Integrity.Agg.Dbl ~ dt$Model.Type.Cat)

# -----------------------------------------------------
# H1D: Data Provenance -> Integrity
# -----------------------------------------------------

# Normality assumption.
shapiro.test(dt$Integrity.Agg.Dbl)
# Homoscedasticity assumption.
bartlett.test(dt$Integrity.Agg.Dbl ~ dt$Data.Type.Cat)
# Kruskal-Wallis
kruskal.test(dt$Integrity.Agg.Dbl ~ dt$Data.Type.Cat)
# Effect size.
ordinalEtaSquared(x = dt$Integrity.Agg.Dbl,
                  g = dt$Data.Type.Cat)
# Visualiation.
plotmeans(dt$Integrity.Agg.Dbl ~ dt$Data.Type.Cat)

# -----------------------------------------------------
# Exploratory: Profile -> Integrity
# -----------------------------------------------------

# Normality assumption.
shapiro.test(dt$Integrity.Agg.Dbl)
# Homoscedasticity assumption.
bartlett.test(dt$Integrity.Agg.Dbl ~ dt$Profile.Cat)
# Kruskal-Wallis
kruskal.test(dt$Integrity.Agg.Dbl ~ dt$Profile.Cat)
# Effect size.
ordinalEtaSquared(x = dt$Integrity.Agg.Dbl,
                  g = dt$Profile.Cat)
# Visualiation.
plotmeans(dt$Integrity.Agg.Dbl ~ dt$Profile.Cat)

# Linear Models.
# -----------------------------------------------------
# H2: Ability + Benevolence + Integrity -> Fairness
# -----------------------------------------------------

# Quantile regression.
# Null model.
m0 = rq(Fairness.Perception.Lik ~ 1,
        data = dt,
        tau = 0.5)
# Main model.
m = rq(Fairness.Perception.Lik ~ Ability.Agg.Dbl + Benevolence.Agg.Dbl + Integrity.Agg.Dbl,
       data = dt,
       tau = 0.5)
# Model summary.
summary(m, se = "ker")
# p-value for the overall model.
anova(m, m0)
# R2 values for the regression.
nagelkerke(m)

# -----------------------------------------------------
# Interaction Effect: Ability + Benevolence + Integrity -> Fairness
# -----------------------------------------------------

# Linear regression.
m = lm(Integrity.Agg.Dbl ~ Profile.Cat * Model.Type.Cat * Data.Type.Cat,
       data = dt)
# Model summary.
summary(m)

# -----------------------------------------------------
# Exploratory: Policy Agreement -> Integrity
# -----------------------------------------------------

# Quantile regression.
# Null model.
m0 = rq(Integrity.Agg.Dbl ~ 1,
        data = dt,
        tau = 0.5)
# Main model.
m = rq(Integrity.Agg.Dbl ~ Agreement.Policy.Lik,
       data = dt,
       tau = 0.5)
# Model summary.
summary(m, se = "ker")
# p-value for the overall model.
anova(m, m0)
# R2 values for the regression.
nagelkerke(m)

# -----------------------------------------------------
# Exploratory: Policy Agreement -> Fairness
# -----------------------------------------------------

# Linear regression.
m = lm(Fairness.Perception.Lik ~ Agreement.Policy.Lik,
       data = dt)
# Model summary.
summary(m)

# Mediation Analysis.
# -----------------------------------------------------
# Exploratory: Ability (Mediator) -> Profile (IV) on Fairness (DV)
# -----------------------------------------------------

# Step 1: Predicting the dependent variable with the independent variable.
base_model = lm(Fairness.Perception.Lik ~ Profile.Cat,
                data = dt)
summary(base_model)
visualize(base_model)

# Step 2: Predicting the mediator with the independent variable.
mediate_model = lm(Ability.Agg.Dbl ~ Profile.Cat,
                   data = dt)
summary(mediate_model)
visualize(mediate_model)

# Step 3: The full model, including the mediator.
full_model = lm(Fairness.Perception.Lik ~ Ability.Agg.Dbl + Profile.Cat,
                data = dt)
summary(full_model)

# Step 4. Mediation analysis.
results = mediate(mediate_model, full_model, 
                  treat = "Profile.Cat",
                  mediator = "Ability.Agg.Dbl",
                  boot = TRUE,
                  sims = 500)
summary(results)
mediate_plot(Fairness.Perception.Lik ~ Ability.Agg.Dbl + Profile.Cat, data = dt)

# -----------------------------------------------------
# Exploratory: Integrity (Mediator) -> Profile (IV) on Fairness (DV)
# -----------------------------------------------------

# Step 1: Predicting the dependent variable with the independent variable.
base_model = lm(Fairness.Perception.Lik ~ Profile.Cat,
                data = dt)
summary(base_model)
visualize(base_model)

# Step 2: Predicting the mediator with the independent variable.
mediate_model = lm(Integrity.Agg.Dbl ~ Profile.Cat,
                   data = dt)
summary(mediate_model)
visualize(mediate_model)

# Step 3: The full model, including the mediator.
full_model = lm(Fairness.Perception.Lik ~ Integrity.Agg.Dbl + Profile.Cat,
                data = dt)
summary(full_model)

# Step 4. Mediation analysis.
results = mediate(mediate_model, full_model, 
                  treat = "Profile.Cat",
                  mediator = "Integrity.Agg.Dbl",
                  boot = TRUE,
                  sims = 500)
summary(results)
mediate_plot(Fairness.Perception.Lik ~ Integrity.Agg.Dbl + Profile.Cat, data = dt)

# -----------------------------------------------------
# Exploratory: Integrity (Mediator) -> Policy Agreement (CV) on Fairness (DV)
# -----------------------------------------------------

# Step 1: Predicting the dependent variable with the independent variable.
base_model = lm(Fairness.Perception.Lik ~ Agreement.Policy.Lik,
                data = dt)
summary(base_model)
visualize(base_model)

# Step 2: Predicting the mediator with the independent variable.
mediate_model = lm(Integrity.Agg.Dbl ~ Agreement.Policy.Lik,
                   data = dt)
summary(mediate_model)
visualize(mediate_model)

# Step 3: The full model, including the mediator.
full_model = lm(Fairness.Perception.Lik ~ Integrity.Agg.Dbl + Agreement.Policy.Lik,
                data = dt)
summary(full_model)

# Step 4. Mediation analysis.
results = mediate(mediate_model, full_model, 
                  treat = "Agreement.Policy.Lik",
                  mediator = "Integrity.Agg.Dbl",
                  boot = TRUE,
                  sims = 500)
summary(results)
mediate_plot(Fairness.Perception.Lik ~ Integrity.Agg.Dbl + Agreement.Policy.Lik, data = dt)
