# LAP-DTR: Layer-Adaptive Partitioning with Dynamic Task Redistribution

Energy-aware model partitioning framework for Vision Transformers across heterogeneous edge devices.

## Overview

This project implements the LAP-DTR framework for distributing Vision Transformer (ViT) models across multiple edge devices while optimizing for energy consumption, accuracy, and latency. The framework supports three partitioning strategies: Meta, Ensembled, and Hybrid.

## Features

- Energy-aware model partitioning for ViTs
- Dynamic task redistribution based on device capabilities
- Attention-based feature summarization for reduced communication
- Support for heterogeneous edge device networks
- Real-time monitoring and adaptive layer complexity
- Fault tolerance with intermediate result caching

## Architecture

### Partitioning Strategies

1. **Meta Partitioning**: Layer-wise independent partitioning with parallel execution
2. **Ensembled Partitioning**: Combines similar dimensional layers to reduce communication
3. **Hybrid Partitioning**: Dynamic allocation based on device capabilities and task requirements

### Key Components

- **Device Management**: Models edge devices with energy, computation, and memory constraints
- **LAP-DTR Framework**: Core partitioning and redistribution algorithms
- **Distributed Training**: Multi-device training with load balancing
- **Distributed Inference**: Real-time inference with dynamic adaptation

## Installation

```bash
pip install -r requirements.txt
```

## Quick Start

### Basic Usage

```python
from config import Config
from device import DeviceManager
from lap_dtr import LAPDTRFramework
from training import DistributedTrainer
from inference import DistributedInference

# Initialize configuration
config = Config()

# Setup devices
device_manager = DeviceManager()
devices = device_manager.create_device_network(num_devices=4)

# Create LAP-DTR framework
framework = LAPDTRFramework(config, devices)

# Load and partition model
model = framework.load_vit_model('vit_base_patch16_224')
partitioned_models = framework.partition_model(model, strategy='hybrid')

# Training
trainer = DistributedTrainer(framework, config)
trainer.train(partitioned_models, train_loader)

# Inference
inference_engine = DistributedInference(framework, config)
results = inference_engine.run_inference(test_data)
```

### Configuration

Modify `config.py` to adjust:
- Model specifications (ViT variant, input size)
- Device parameters (energy, compute capacity, memory)
- Training hyperparameters
- Partitioning strategy weights

### Running Experiments

```bash
# Train with hybrid partitioning
python main.py --mode train --strategy hybrid --devices 4

# Run inference
python main.py --mode inference --model_path checkpoints/model.pth

# Evaluate partitioning strategies
python main.py --mode evaluate --compare_strategies
```

## Evaluation Metrics

The framework tracks:
- **Latency**: Total inference time across devices
- **Energy Consumption**: Per-device energy usage
- **Accuracy**: Model performance metrics
- **Resource Utilization**: Compute and memory usage
- **Communication Overhead**: Data transfer between devices

## Model Support

Supported Vision Transformer variants:
- ViT (Vision Transformer)
- DeiT (Data-efficient Image Transformers)
- Swin Transformer
- Custom ViT architectures

## Device Requirements

- CUDA-capable GPUs (recommended)
- Minimum 4GB RAM per device
- Network connectivity for distributed operation
- Energy monitoring support (optional)
