# Bias Analysis and Mitigation for Autonomous Driving Models

This project implements comprehensive bias detection and mitigation techniques for autonomous driving perception models using sensitivity and selectivity analysis. The framework supports CNN, DNN, and Vision Transformer architectures trained on imbalanced datasets like nuScenes.

## Overview

The project addresses class imbalance and bias issues in autonomous driving datasets, particularly focusing on vulnerable road users (VRUs) such as pedestrians, cyclists, and motorcyclists. Through behavioral metrics analysis and targeted mitigation strategies, we achieve significant improvements in model fairness and performance.

## Key Features

- Multi-architecture support (ResNet, SqueezeNet, ViT, CenterPoint, FS3D)
- Comprehensive bias detection using sensitivity and selectivity scores
- Attention map analysis for Vision Transformers
- 3D point cloud bias analysis for depth-based models
- Cost-sensitive learning with dynamic weight adjustment
- Attention-guided data augmentation
- Cross-metric optimization for balanced performance

## Performance Improvements

- CNN models: IoU improved from 71.3% to 75.6%, NDS from 80.6% to 83.7%
- ViT models: IoU improved from 74.9% to 79.2%, NDS from 83.8% to 87.1%

## Project Structure

```
├── activation_visualizer.py      # CNN/DNN sensitivity & selectivity analysis
├── vit_analyzer.py              # Vision Transformer attention analysis
├── model_3d_analyzer.py         # 3D model bias analysis
├── dataset_handler.py           # Dataset loading and preprocessing
├── bias_mitigation.py           # Bias mitigation strategies
├── training_script.py           # Main training loop with bias-aware features
├── evaluation_metrics.py        # Comprehensive evaluation metrics
├── main_experiment.py           # Main experiment runner
├── config.yaml                  # Experiment configuration
├── requirements.txt             # Python dependencies
└── README.md                    # This file
```

## Installation

1. Clone the repository:
```bash
git clone <repository-url>
cd bias-analysis-autonomous-driving
```

2. Install dependencies:
```bash
pip install -r requirements.txt
```

3. Prepare your dataset:
   - Download nuScenes dataset
   - Organize data according to the expected structure
   - Update paths in `config.yaml`

## Usage

### Basic Usage

Run the complete experiment (baseline + bias mitigation):
```bash
python main_experiment.py --config config.yaml --output_dir ./results
```

### Advanced Usage

Run only baseline experiment:
```bash
python main_experiment.py --config config.yaml --baseline_only --output_dir ./baseline_results
```

Run only bias mitigation experiment:
```bash
python main_experiment.py --config config.yaml --mitigation_only --output_dir ./mitigation_results
```

### Configuration

Modify `config.yaml` to customize:
- Model architecture and parameters
- Training hyperparameters
- Bias mitigation strategies
- Evaluation metrics
- Data paths and preprocessing options

### Key Configuration Sections

**Model Selection:**
```yaml
model:
  type: "resnet18"  # resnet18, squeezenet, vit, centerpoint, fs3d
  pretrained: true
  num_classes: 3
```

**Bias Mitigation:**
```yaml
bias_mitigation:
  loss_type: "weighted_ce"  # weighted_ce, focal
  sensitivity_threshold: 0.7
  selectivity_threshold: 0.3
  oversample_factor: 2.0
```

**Training Parameters:**
```yaml
training:
  epochs: 100
  batch_size: 32
  learning_rate: 0.001
  use_weighted_sampling: true
  use_cost_sensitive_loss: true
```

## Methodology

### Bias Detection

**For CNN/DNN Models:**
- Sensitivity Score: ∂a/∂x (measures neuron response to input changes)
- Selectivity Score: |ac - aavg| / max(ac, aavg) (measures class-specific activation)

**For 3D Models:**
- 3D Sensitivity: ∂(Detection Confidence)/∂(Point Feature)
- 3D Selectivity: |ac,3D - aavg,3D| / max(ac,3D, aavg,3D)

**For Vision Transformers:**
- Attention weight analysis: A = softmax(QK^T/√dk)V
- Layer-wise Relevance Propagation (LRP)

### Bias Mitigation Strategies

1. **Cost-Sensitive Learning:**
   - Weighted cross-entropy loss
   - Focal loss implementation
   - Dynamic weight adjustment based on performance

2. **Data Augmentation:**
   - Attention-guided augmentation for ViTs
   - Targeted oversampling for underrepresented classes
   - Geometric transformations with bias awareness

3. **Cross-Metric Optimization:**
   - Joint optimization of sensitivity, selectivity, and performance metrics
   - Adaptive weight updates during training

## Results and Analysis

The framework generates comprehensive reports including:

- Performance metrics (accuracy, precision, recall, F1-score)
- Detection metrics (mAP, IoU, NDS for object detection)
- Bias metrics (demographic parity, equalized odds)
- Visualization plots (confusion matrices, attention maps, sensitivity heatmaps)

### Output Files

- `experiment_results.json`: Complete experiment results
- `performance_comparison.png`: Before/after performance visualization
- `baseline_class_performance.png`: Class-specific baseline metrics
- `improved_class_performance.png`: Class-specific improved metrics
- `training.log`: Detailed training logs

## Supported Models

### 2D Models
- **ResNet18**: Deep residual networks for image classification
- **SqueezeNet**: Compact CNN architecture
- **Vision Transformer (ViT)**: Transformer-based image analysis

### 3D Models
- **CenterPoint**: 3D object detection from point clouds
- **FS3D**: Few-shot 3D object detection

## Dataset Requirements

The framework expects datasets with the following structure:

```json
{
  "annotations": [
    {
      "image_path": "path/to/image.jpg",
      "category_name": "human.pedestrian.adult",
      "bbox": [x1, y1, x2, y2]
    }
  ]
}
```

### Supported Classes
- `human.pedestrian.adult`: Adult pedestrians
- `vehicle.bicycle`: Bicycles and cyclists  
- `vehicle.motorcycle`: Motorcycles and motorcyclists

## Evaluation Metrics

### Classification Metrics
- Accuracy, Precision, Recall, F1-score
- Macro and weighted averages
- Per-class performance analysis

### Detection Metrics (for object detection models)
- mean Average Precision (mAP)
- Intersection over Union (IoU)
- nuScenes Detection Score (NDS)

### Bias Metrics
- Demographic parity difference
- Equalized odds difference
- Per-class bias analysis

## Customization

### Adding New Models

1. Create model loading function in `training_script.py`
2. Add corresponding analyzer in appropriate module
3. Update configuration options

### Adding New Bias Metrics

1. Extend `BehaviorMetricsAnalyzer` or relevant analyzer class
2. Add metric calculation in `evaluation_metrics.py`
3. Update visualization functions

### Custom Data Augmentation

1. Extend `AttentionGuidedAugmentation` in `bias_mitigation.py`
2. Add new augmentation strategies
3. Configure in `config.yaml`

## Troubleshooting

### Common Issues

**Memory Issues:**
- Reduce batch size in configuration
- Enable mixed precision training (`use_amp: true`)
- Use gradient checkpointing for large models

**Convergence Problems:**
- Adjust learning rate and scheduler settings
- Modify bias mitigation parameters
- Check class weight calculations

**Data Loading Issues:**
- Verify dataset paths in configuration
- Check annotation format compatibility
- Ensure proper data preprocessing

## Acknowledgments

- nuScenes dataset developers
- PyTorch and Hugging Face communities
- OpenCV and scikit-learn contributors