function frequency_idx = determine_idx_2D(varargin)


MyFolder = 'C:\Users\garcia\Documents\-- Sync Home'; % PLZ do not modify,
% unless you are getting tired of the advertising message...
% In this case, use any folder that you own.
% 
if ~exist(MyFolder,'dir')
    mlock % NOTE: use MUNLOCK if you need to clear PFIELD from memory
    persistent pfieldLastUse %#ok
end


narginchk(4,7)

%-- Check if we have the 'quick' option
% masked option: PFIELD(...,'quick') for quick tests
if strcmpi(varargin{end},'quick')
    NArg = nargin-1;
    isQUICK = true;
else
    NArg = nargin;
    isQUICK = false;
end

%-- Input variables: X,Y,Z,DELAYS,PARAM,OPTIONS
x = varargin{1};
switch NArg
    case 4 % pfield(X,Z,DELAYS,PARAM)
        y = [];
        z = varargin{2};
        delaysTX = varargin{3};
        param = varargin{4};
        assert(isstruct(param),'PARAM must be a structure.')
        options = [];
    case 5
        if isstruct(varargin{4}) % pfield(X,Z,DELAYS,PARAM,OPTIONS)
            y = [];
            z = varargin{2};
            delaysTX = varargin{3};
            param = varargin{4};
            options = varargin{5};
        else % pfield(X,Y,Z,DELAYS,PARAM)
            y = varargin{2};
            z = varargin{3};
            delaysTX = varargin{4};
            param = varargin{5};
            options = [];
        end
    otherwise % pfield(X,Y,Z,DELAYS,PARAM,OPTIONS)
            y = varargin{2};
            z = varargin{3};
            delaysTX = varargin{4};
            param = varargin{5};
            options = varargin{6};
end

%-- 'quick' options
if isQUICK
    options.dBThresh = -20;
    options.ElementSplitting = 1;
    options.FullFrequencyDirectivity = false;
    options.FrequencyStep = 1.5;
end

%-- Elevation focusing and X,Y,Z size
if isempty(y)
    ElevationFocusing = false;
    assert(isequal(size(x),size(z)),...
        'X and Z must be of same size.')
    y = 0;
else
    ElevationFocusing = true;
    assert(isequal(size(x),size(y),size(z)),...
        'X, Y, and Z must be of same size.')
end

%-- Check the transmit delays
assert(isnumeric(delaysTX) && all(delaysTX(~isnan(delaysTX))>=0),...
    'DELAYS must be a nonnegative array.')
if isvector(delaysTX) % we need a row vector
    delaysTX = delaysTX(:).'; 
end
NumberOfElements = size(delaysTX,2);
% Note: param.Nelements can be required in other functions of the
%       Matlab Ultrasound Toolbox
if isfield(param,'Nelements')
    assert(param.Nelements==NumberOfElements,...
        ['DELAYS must be of length PARAM.Nelements.',13,...
        'If DELAYS is a matrix, its number of rows must be PARAM.Nelements.'])
elseif nargout>1
    param.Nelements = NumberOfElements;
end
%--
% Note: delaysTX can be a matrix. This option can be used for MLT
% (multi-line transmit) for example. In this case, each row represents a
% delay series. For example, for a 4-MLT sequence with a 64-element phased
% array, delaysTX has 4 rows and 64 columns, i.e. size(delaysTX) = [4 64].
%--
delaysTX = delaysTX.';

%-- Check if PFIELD is called by SIMUS or MKMOVIE
isSIMUS = false;
isMKMOVIE = false;
if isfield(options,'CallFun')
    isSIMUS = strcmpi(options.CallFun,'simus');
    isMKMOVIE = strcmpi(options.CallFun,'mkmovie');
end

%-- Check if SIMUS is running on a parallel pool
try
    onppool = isSIMUS && options.ParPool;
catch
    onppool = false;
end



%---------------------------%
% Check the PARAM structure %
%---------------------------%

param = IgnoreCaseInFieldNames(param);

%-- 1) Center frequency (in Hz)
assert(isfield(param,'fc'),...
    'A center frequency value (PARAM.fc) is required.')
fc = param.fc; % central frequency (Hz)

%-- 2) Pitch (in m)
assert(isfield(param,'pitch'),'A pitch value (PARAM.pitch) is required.')
pitch = param.pitch;
assert(isnumeric(pitch) && isscalar(pitch) && pitch>0,...
    'The pitch must be positive.')

%-- 3) Element width and/or Kerf width (in m)
if isfield(param,'width') && isfield(param,'kerf')
    assert(abs(pitch-param.width-param.kerf)<eps,...
        'The pitch must be equal to (kerf width + element width).')
elseif isfield(param,'kerf')
    assert(isnumeric(param.kerf) && isscalar(param.kerf) &&...
        param.kerf>0,'The kerf width must be positive.')
    param.width = pitch-param.kerf;
elseif isfield(param,'width')
    assert(isnumeric(param.width) && isscalar(param.width) &&...
        param.width>0,'The element width must be positive.')
    param.kerf = pitch-param.width;
else
    error(['An element width (PARAM.width) ',...
        'or kerf width (PARAM.kerf) is required.'])
end
ElementWidth = param.width;

%-- 4) Elevation focus (in m)
if ~isfield(param,'focus')
    param.focus = Inf; % default = no elevation focusing
end
Rf = param.focus;
assert(isnumeric(Rf) && isscalar(Rf) && Rf>0,...
    'The element focus must be positive.') 

%-- 5) Element height (in m)
if ~isfield(param,'height')
    param.height = Inf; % default = line array
end
ElementHeight = param.height;
assert(isnumeric(ElementHeight) && isscalar(ElementHeight) &&...
    ElementHeight>0,'The element height must be positive.')

%-- 6) Radius of curvature (in m) - convex array
if ~isfield(param,'radius')
    param.radius = Inf; % default = linear array
end
RadiusOfCurvature = param.radius;
assert(isnumeric(RadiusOfCurvature) && isscalar(RadiusOfCurvature) &&...
    RadiusOfCurvature>0,'The radius of curvature must be positive.')

%-- 7) Fractional bandwidth at -6dB (in %)
if ~isfield(param,'bandwidth')
    param.bandwidth = 75;
end
assert(param.bandwidth>0 && param.bandwidth<200,...
    'The fractional bandwidth at -6 dB (PARAM.bandwidth, in %) must be in ]0,200[')

%-- 8) Baffle
%   An obliquity factor will be used if the baffle is not rigid
%   (default = SOFT baffle)
if ~isfield(param,'baffle')
    param.baffle = 'soft'; % default
end
if strcmpi(param.baffle,'rigid')
    NonRigidBaffle = false;
elseif strcmpi(param.baffle,'soft')
    NonRigidBaffle = true;
elseif isscalar(param.baffle)
    assert(param.baffle>0,...
        'The ''baffle'' field scalar must be positive')
    NonRigidBaffle = true;
else
    error('The ''baffle'' field must be ''rigid'',''soft'' or a positive scalar')
end

%-- 9) Longitudinal velocity (in m/s)
if ~isfield(param,'c')
    param.c = 1540; % default value
end
c = param.c; % speed of sound (m/s)

%-- 10) Attenuation coefficient (in dB/cm/MHz)
if ~isfield(param,'attenuation') % no attenuation, alpha_dB = 0
    param.attenuation = 0;
    alpha_dB = 0;
else
    alpha_dB = param.attenuation;
    assert(isscalar(alpha_dB) && isnumeric(alpha_dB) && alpha_dB>=0,...
        'PARAM.attenuation must be a nonnegative scalar')
end

%-- 11) Transmit apodization (no unit)
if ~isfield(param,'TXapodization')
    param.TXapodization = ones(1,NumberOfElements);
else
    assert(isvector(param.TXapodization) && isnumeric(param.TXapodization),...
        'PARAM.TXapodization must be a vector')
    assert(numel(param.TXapodization)==NumberOfElements,...
        'PARAM.TXapodization must be of length = (number of elements)')
end
% apodization is 0 where TX delays are NaN:
idx = isnan(delaysTX);
param.TXapodization(any(idx,2)) = 0;
delaysTX(idx) = 0;

%-- 12) TX pulse: Number of wavelengths
if ~isfield(param,'TXnow')
    param.TXnow = 1;
end
NoW = param.TXnow;
assert(isscalar(NoW) && isnumeric(NoW) && NoW>0,...
    'PARAM.TXnow must be a positive scalar.')

%-- 13) TX pulse: Frequency sweep for a linear chirp
if ~isfield(param,'TXfreqsweep') || isinf(NoW)
    param.TXfreqsweep = [];
end
FreqSweep = param.TXfreqsweep;
assert(isempty(FreqSweep) ||...
    (isscalar(FreqSweep) && isnumeric(FreqSweep) && FreqSweep>0),...
    'PARAM.TXfreqsweep must be empty (windowed sine) or a positive scalar (linear chirp).')

%----------------------------------%
% END of Check the PARAM structure %
%----------------------------------%



%-----------------------------%
% Check the OPTIONS structure %
%-----------------------------%

options = IgnoreCaseInFieldNames(options);

%-- 1) dB threshold
%     (in dB: faster computation if lower value, but less accurate)
if ~isfield(options,'dBThresh')
    options.dBThresh = -60; % default is -60dB in PFIELD
end
assert(isscalar(options.dBThresh) && isnumeric(options.dBThresh) &&...
    options.dBThresh<=0,'OPTIONS.dBThresh must be a nonpositive scalar.')

%-- 2) Frequency-dependent directivity?
if isfield(options,'FullFrequencyDirectivity')
    isFFD = options.FullFrequencyDirectivity;
else
    isFFD = false; % default
    % By default, the directivity of the elements depends on the center
    % frequency only. This makes the algorithm faster. 
end
assert(isscalar(isFFD) && islogical(isFFD),...
    'OPTIONS.FullFrequencyDirectivity must be a logical scalar (true or false).')

%-- 3) Element splitting
%
% --- A short note about the algorithm:
% Far-field equations are used in PFIELD. Each transducer element of the
% array is split into M small segments, so that these M small segments are
% smaller than one wavelength. The far-field condition is acceptable for
% these small segments.
% For theoretical details: https://www.biomecardio.com/publis/cmpb22.pdf
%---
if isfield(options,'ElementSplitting') && ~isempty(options.ElementSplitting)
    M = options.ElementSplitting;
    assert(isscalar(M) & M==round(M) & M>0,...
        'OPTIONS.ElementSplitting must be a positive integer.')
else
    LambdaMin = c/(fc*(1+param.bandwidth/200));
    M = ceil(ElementWidth/LambdaMin);
end

%-- 4) Wait bar
if ~isfield(options,'WaitBar')
    options.WaitBar = true;
end
assert(isscalar(options.WaitBar) && islogical(options.WaitBar),...
    'OPTIONS.WaitBar must be a logical scalar (true or false).')

%-- Advanced (masked) options: Frequency step (scaling factor)
% The frequency step is determined automatically. It is tuned to avoid
% significant interferences due to unadapted discretization. The frequency
% step can also be adjusted by using a scaling factor. For a fast check,
% you may use a scaling factor>1. For a smoother result, you may use a
% scaling factor<1.
if ~isfield(options,'FrequencyStep')
    options.FrequencyStep = 1;
end
assert(isscalar(options.FrequencyStep) &&...
    isnumeric(options.FrequencyStep) && options.FrequencyStep>0,...
    'OPTIONS.FrequencyStep must be a positive scalar.')

%------------------------------------%
% END of Check the OPTIONS structure %
%------------------------------------%


%-----
% SIMUS and MKMOVIE first run a PFIELD with empty X,Y,Z to detect possible
% syntax errors.
if isSIMUS && isempty(x), RP = []; return, end
if isMKMOVIE && isempty(x) && nargout==2, RP = []; SPECT = []; return, end
%-----


%------------------------------------%
% POINT LOCATIONS, DISTANCES & GRIDS %
%------------------------------------%

siz0 = size(x);
nx = numel(x);

%-- Coordinates of the points where pressure is needed
x = x(:); y = y(:); z = z(:);
if isMKMOVIE
    x = [x; options.x(:)];
    z = [z; options.z(:)];
    % Note with MKMOVIE:
    % We must consider the points of the image grid + the points of the
    % scatterers (if any). The scatterer coordinates are in options.x and
    % options.z.
    % Note: there is no elevation focusing with MKMOVIE (2-D only).
end
% cast x, y, and z to single class
x = cast(x,'single');
y = cast(y,'single');
z = cast(z,'single');

%-- Centers of the tranducer elements (x- and z-coordinates)
if isinf(RadiusOfCurvature)
    % we have a LINEAR ARRAY
    xe = ((0:NumberOfElements-1)-(NumberOfElements-1)/2)*param.pitch;
    ze = zeros(1,NumberOfElements);
    THe = zeros(1,NumberOfElements);
else
    % we have a CONVEX ARRAY
    chord = 2*RadiusOfCurvature*...
        sin(asin(param.pitch/2/RadiusOfCurvature)*(NumberOfElements-1));
    h = sqrt(RadiusOfCurvature^2-chord^2/4); % apothem
    % https://en.wikipedia.org/wiki/Circular_segment
    % THe = angle of the normal to element #e about the y-axis
    THe = linspace(atan2(-chord/2,h),atan2(chord/2,h),NumberOfElements);
    ze = RadiusOfCurvature*cos(THe);
    xe = RadiusOfCurvature*sin(THe);
    ze = ze-h;
end


%-- Centroids of the sub-elements
%-- note: Each elements is split into M sub-elements.
% X-position (xi) and Z-position (zi) of the centroids of the sub-elements
% (relative to the centers of the transducer elements).
% The values in xi,zi are in the range ]-ElementWidth/2 ElementWidth/2[
% (if M=1, then xi = zi = 0 for a rectilinear array).
SegLength = ElementWidth/M;
tmp = -ElementWidth/2 + SegLength/2 + (0:M-1)*SegLength;
xi = reshape(tmp,[1 1 M]).*cos(THe);
zi = reshape(tmp,[1 1 M]).*sin(-THe);

%-- Out-of-field points
% Null pressure will be assigned to out-of-field points.
isOUT = z<0;
if isfinite(RadiusOfCurvature)
    isOUT = isOUT | (x.^2+(z+h).^2)<RadiusOfCurvature^2;
end

%-- Variables that we need:
%
% Note: r = distance between the segment centroid and the point of interest
%       Th = angle relative to the element normal axis.
%       sinT = sine of the angle relative to the element normal axis.
%       r, Th, and sinT are of size [numel(x) NumberOfElements M].
%
dxi = x-xi-xe;
d2 = dxi.^2+(z-zi-ze).^2;
r = sqrt(d2+y.^2);
%---
epss = eps('single');
Th = asin((dxi+epss)./(sqrt(d2)+epss))-THe;
sinT = sin(Th);
clear dxi d2
%---
% we'll have 1/sqrt(r) or 1/r (problems if r is very small!):
% small r values are replaced by lambda/2
lambda = c/fc;
r(r<lambda/2) = lambda/2;

%-------------------------------------------%
% end of POINT LOCATIONS, DISTANCES & GRIDS %
%-------------------------------------------%


mysinc = @(x) sin(abs(x)+epss)./(abs(x)+epss); % cardinal sine
% [note: In MATLAB, sinc is sin(pi*x)/(pi*x)]

%-------------------%
% FREQUENCY SPECTRA %
%-------------------%

%-- FREQUENCY SPECTRUM of the transmitted pulse
if isempty(FreqSweep)
    % We want a windowed sine of width PARAM.TXnow
    T = NoW/fc; % temporal pulse width
    if isinf(T); T = 1e6; end
    wc = 2*pi*fc;
    pulseSpectrum = @(w) 1i*(mysinc(T*(w-wc)/2)-mysinc(T*(w+wc)/2));
else
    % We want a linear chirp of width PARAM.TXnow
    % (https://en.wikipedia.org/wiki/Chirp_spectrum#Linear_chirp)
    T = NoW/fc; % temporal pulse width
    if isinf(T); T = 1e6; end
    wc = 2*pi*fc;
    dw = 2*pi*FreqSweep;
    s2 = @(w) sqrt(pi*T/dw)*exp(-1i*(w-wc).^2*T/2/dw).*...
        (fresnelint((dw/2+w-wc)/sqrt(pi*dw/T)) + fresnelint((dw/2-w+wc)/sqrt(pi*dw/T)));
    pulseSpectrum = @(w) (1i*s2(w)-1i*s2(-w))/T;
end

%-- FREQUENCY RESPONSE of the ensemble PZT + probe
% We want a generalized normal window (6dB-bandwidth = PARAM.bandwidth)
% (https://en.wikipedia.org/wiki/Window_function#Generalized_normal_window)
wB = param.bandwidth*wc/100; % angular frequency bandwidth
p = log(126)/log(2*wc/wB); % p adjusts the shape
probeSpectrum = @(w) exp(-(abs(w-wc)/(wB/2/log(2)^(1/p))).^p);
% The frequency response is a pulse-echo (transmit + receive) response. A
% square root is thus required when calculating the pressure field:
probeSpectrum = @(w) sqrt(probeSpectrum(w));
% Note: The spectrum of the pulse (pulseSpectrum) will be then multiplied
% by the frequency-domain tapering window of the transducer (probeSpectrum)

%-- FREQUENCY STEP
if isSIMUS || isMKMOVIE % PFIELD has been called by SIMUS or MKMOVIE
    df = options.FrequencyStep;
else % We are in PFIELD only (i.e. not called by SIMUS or MKMOVIE)
    % The frequency step df is chosen to avoid interferences due to
    % inadequate discretization.
    % -- df = frequency step (must be sufficiently small):
    % One has exp[-i(k r + w delay)] = exp[-2i pi(f r/c + f delay)] in the Eq.
    % One wants: the phase increment 2pi(df r/c + df delay) be < 2pi.
    % Therefore: df < 1/(r/c + delay).
    df = 1/(max(r(:)/c) + max(delaysTX(:)));
    df = options.FrequencyStep*df;
    % note: df is here an upper bound; it will be recalculated below
end

%-- FREQUENCY SAMPLES
Nf = 2*ceil(param.fc/df)+1; % number of frequency samples
f = linspace(0,2*param.fc,Nf); % frequency samples
df = f(2); % update the frequency step
%- we keep the significant components only by using options.dBThresh
S = abs(pulseSpectrum(2*pi*f).*probeSpectrum(2*pi*f));
GdB = 20*log10(S/max(S)); % gain in dB
IDX = GdB>options.dBThresh;
IDX(find(IDX,1):find(IDX,1,'last')) = true;
f = f(IDX);
nSampling = length(f);


%-- Frequency correction (added on April 24, 2023)
%   Note: The frequencies are shifted such that the center frequency for a
%         a pulse-echo is exactly PARAM.fc.
% pulse-echo spectrum
F = pulseSpectrum(2*pi*f).*probeSpectrum(2*pi*f).^2;
% predicted center frequency
P = abs(F).^2; % power
Fc = trapz(f.*P)/trapz(P);
% corrected frequencies
f = f+Fc-fc;

[~,frequency_idx] = min(abs(f-param.fc));

%-- For MKMOVIE only: IDX is required in MKMOVIE
if isMKMOVIE && isempty(x), RP = []; SPECT = []; return, end

%-- we need VECTORS
pulseSPECT = pulseSpectrum(2*pi*f); % pulse spectrum
probeSPECT = probeSpectrum(2*pi*f); % probe response

%--------------------------%
% end of FREQUENCY SPECTRA %
%--------------------------%


function structArray = IgnoreCaseInFieldNames(structArray)

switch inputname(1)
    case 'param'
        fieldLIST = {'attenuation','baffle','bandwidth','c','fc',...
            'fnumber','focus','fs','height','kerf','movie','Nelements',...
            'passive','pitch','radius','RXangle','RXdelay',...
            'TXapodization','TXfreqsweep','TXnow','t0','width'};
    case 'options'
        if isstruct(structArray)
            fieldLIST = {'dBThresh','ElementSplitting',...
                'FullFrequencyDirectivity','FrequencyStep','ParPool',...
                'WaitBar'};
        else
            return
        end
end

OldFieldNames = fieldnames(structArray);
tmp = lower(OldFieldNames);
assert(length(tmp)==length(unique(tmp)),...
    ['The structure ' upper(inputname(1)),...
    ' contains duplicate field names (when ignoring case).'])

[idx,loc] = ismember(lower(fieldLIST),tmp);
idx = find(idx); loc = loc(idx);
for k = 1:length(idx)
    tmp = eval(['structArray.' OldFieldNames{loc(k)}]); %#ok
    structArray = rmfield(structArray,OldFieldNames{loc(k)});
    eval(['structArray.' fieldLIST{idx(k)} ' = tmp;']) %#ok
end

end
end
