function frequency_idx = determine_idx(varargin)

% this function determines the index of the correct frequency to extract
% the complex data 


MyFolder = 'C:\Users\garcia\Documents\-- Sync Home'; % PLZ do not modify,
% unless you are getting tired of the advertising message...
% In this case, use any folder that you own.
% 
if ~exist(MyFolder,'dir')
    mlock % NOTE: use MUNLOCK if you need to clear PFIELD from memory
    persistent pfieldLastUse %#ok
end


%-- Input variables: X,Y,Z,DELAYS,PARAM,OPTIONS
x = varargin{1};
y = varargin{2};
z = varargin{3};
assert(isequal(size(x),size(y),size(z)),...
    'X, Y, and Z must be of same size.')
delaysTX = varargin{4};
param = varargin{5};
if nargin==6
    options = varargin{6};
else
    options = [];
end

%-- Check the transmit delays
assert(isnumeric(delaysTX) && all(delaysTX(~isnan(delaysTX))>=0),...
    'DELAYS must be a nonnegative array.')
if isvector(delaysTX) % we need a row vector
    delaysTX = delaysTX(:).'; 
end
NumberOfElements = size(delaysTX,2);

%--
% Note: delaysTX can be a matrix. This option can be used for MPT
% (multi-plane transmit) for example. In this case, each row represents a
% delay series. For example, for a 4-MPT sequence with a 1024-element
% matrix array, delaysTX has 4 rows and 1024 columns, i.e. size(delaysTX) =
% [4 1024].
%--
delaysTX = delaysTX.';

%-- Check if PFIELD3 is called by SIMUS3
isSIMUS3 = false;
if isfield(options,'CallFun')
    isSIMUS3 = strcmpi(options.CallFun,'simus3');
end




%---------------------------%
% Check the PARAM structure %
%---------------------------%

param = IgnoreCaseInFieldNames(param);

%-- 1) Center frequency (in Hz)
assert(isfield(param,'fc'),...
    'A center frequency value (PARAM.fc) is required.')
fc = param.fc; % central frequency (Hz)

%-- 2) Coordinates of the transducer elements (xe,ye)
% note: ze = 0 in this PFIELD3 version.
assert(isfield(param,'elements'),...
    ['PARAM.elements must contain the x- and y-locations ',...
    'of the transducer elements.'])
assert(size(param.elements,1)==2,...
    ['PARAM.elements must have two rows that contain the ',...
    'x (1st row) and y (2nd row) coordinates of the transducer elements.'])
xe = param.elements(1,:);
ye = param.elements(2,:);
assert(numel(xe)==NumberOfElements,...
    'The number of elements must match the number of transmit delays.')

%-- 3) Element width (in m)
assert(isfield(param,'width'),...
    'An element width (PARAM.width) is required.')
ElementWidth = param.width;
assert(isnumeric(ElementWidth) && isscalar(ElementWidth) &&...
    ElementWidth>0,'The element width must be positive.')

%-- 4) Element height (in m)
assert(isfield(param,'height'),...
    'An element height (PARAM.height) is required with PFIELD3 and SIMUS3.')
ElementHeight = param.height;
assert(isnumeric(ElementHeight) && isscalar(ElementHeight) &&...
    ElementHeight>0,'The element height must be positive.')

%-- 5) Fractional bandwidth at -6dB (in %)
if ~isfield(param,'bandwidth')
    param.bandwidth = 75;
end
assert(param.bandwidth>0 && param.bandwidth<200,...
    'The fractional bandwidth at -6 dB (PARAM.bandwidth, in %) must be in ]0,200[')

%-- 6) Baffle
%   An obliquity factor will be used if the baffle is not rigid
%   (default = SOFT baffle)
if ~isfield(param,'baffle')
    param.baffle = 'soft'; % default
end
if strcmpi(param.baffle,'rigid')
    NonRigidBaffle = false;
elseif strcmpi(param.baffle,'soft')
    NonRigidBaffle = true;
elseif isscalar(param.baffle)
    assert(param.baffle>0,...
        'The ''baffle'' field scalar must be positive')
    NonRigidBaffle = true;
else
    error('The ''baffle'' field must be ''rigid'',''soft'' or a positive scalar')
end

%-- 7) Longitudinal velocity (in m/s)
if ~isfield(param,'c')
    param.c = 1540; % default value
end
c = param.c; % speed of sound (m/s)

%-- 8) Attenuation coefficient (in dB/cm/MHz)
if ~isfield(param,'attenuation') % no attenuation, alpha_dB = 0
    param.attenuation = 0;
    alpha_dB = 0;
else
    alpha_dB = param.attenuation;
    assert(isscalar(alpha_dB) && isnumeric(alpha_dB) && alpha_dB>=0,...
        'PARAM.attenuation must be a nonnegative scalar')
end

%-- 9) Transmit apodization (no unit)
if ~isfield(param,'TXapodization')
    param.TXapodization = ones(1,NumberOfElements);
else
    assert(isvector(param.TXapodization) && isnumeric(param.TXapodization),...
        'PARAM.TXapodization must be a vector')
    assert(numel(param.TXapodization)==NumberOfElements,...
        'PARAM.TXapodization must be of length = (number of elements)')
end
% apodization is 0 where TX delays are NaN:
idx = isnan(delaysTX);
param.TXapodization(any(idx,2)) = 0;
delaysTX(idx) = 0;

%-- 10) TX pulse: Number of wavelengths
if ~isfield(param,'TXnow')
    param.TXnow = 1;
end
NoW = param.TXnow;
assert(isscalar(NoW) && isnumeric(NoW) && NoW>0,...
    'PARAM.TXnow must be a positive scalar.')

%-- 11) TX pulse: Frequency sweep for a linear chirp
if ~isfield(param,'TXfreqsweep') || isinf(NoW)
    param.TXfreqsweep = [];
end
FreqSweep = param.TXfreqsweep;
assert(isempty(FreqSweep) ||...
    (isscalar(FreqSweep) && isnumeric(FreqSweep) && FreqSweep>0),...
    'PARAM.TXfreqsweep must be empty (windowed sine) or a positive scalar (linear chirp).')

%----------------------------------%
% END of Check the PARAM structure %
%----------------------------------%



%-----------------------------%
% Check the OPTIONS structure %
%-----------------------------%

options = IgnoreCaseInFieldNames(options);

%-- 1) dB threshold
%     (in dB: faster computation if lower value, but less accurate)
if ~isfield(options,'dBThresh')
    options.dBThresh = -60; % default is -60dB in PFIELD3
end
assert(isscalar(options.dBThresh) && isnumeric(options.dBThresh) &&...
    options.dBThresh<=0,'OPTIONS.dBThresh must be a nonpositive scalar.')

%-- 2) Frequency-dependent directivity?
if isfield(options,'FullFrequencyDirectivity')
    isFFD = options.FullFrequencyDirectivity;
else
    isFFD = false; % default
    % By default, the directivity of the elements depends on the center
    % frequency only. This makes the algorithm faster. 
end
assert(isscalar(isFFD) && islogical(isFFD),...
    'OPTIONS.FullFrequencyDirectivity must be a logical scalar (true or false).')

%-- 3) Element splitting
%
% --- A short note about the algorithm:
% Far-field equations are used in PFIELD3. Each transducer element of the
% array is split into M-by-N small rectangles, so that these MN rectangles
% have size smaller than one wavelength by one wavelength. The far-field
% condition is acceptable for these small rectangles.
%---
if isfield(options,'ElementSplitting') && ~isempty(options.ElementSplitting)
    assert(numel(options.ElementSplitting)==2,...
        'OPTIONS.ElementSplitting must be a two-element vector.')
    M = options.ElementSplitting(1);
    N = options.ElementSplitting(2);
    assert(isscalar(M) & M==round(M) & M>0 &...
        isscalar(N) & N==round(N) & N>0,...
        'OPTIONS.ElementSplitting must contain two positive integers.')
else
    LambdaMin = c/(fc*(1+param.bandwidth/200));
    M = ceil(ElementWidth/LambdaMin);
    N = ceil(ElementHeight/LambdaMin);
end

%-- 4) Wait bar
if ~isfield(options,'WaitBar')
    options.WaitBar = true;
end
assert(isscalar(options.WaitBar) && islogical(options.WaitBar),...
    'OPTIONS.WaitBar must be a logical scalar (true or false).')

%-- Advanced (masked) options: Frequency step (scaling factor)
% The frequency step is determined automatically. It is tuned to avoid
% significant interferences due to unadapted discretization. The frequency
% step can also be adjusted by using a scaling factor. For a fast check,
% you may use a scaling factor>1. For a smoother result, you may use a
% scaling factor<1.
if ~isfield(options,'FrequencyStep')
    options.FrequencyStep = 1;
end
assert(isscalar(options.FrequencyStep) &&...
    isnumeric(options.FrequencyStep) && options.FrequencyStep>0,...
    'OPTIONS.FrequencyStep must be a positive scalar.')

%------------------------------------%
% END of Check the OPTIONS structure %
%------------------------------------%



%------------------------------------%
% POINT LOCATIONS, DISTANCES & GRIDS %
%------------------------------------%


%-- Coordinates of the points where pressure is needed
x = x(:); y = y(:); z = z(:);

%-- Cast x, y, and z to single class
x = cast(x,'single');
y = cast(y,'single');
z = cast(z,'single');

%-- Centroids of the sub-elements
%-- note: Each elements is split into M-by-N sub-elements.
% X-position (xi) and Y-position (yi) of the centroids of the sub-elements
% (relative to the centers of the transducer elements).
% The values in xi are in the range ]-ElementWidth/2 ElementWidth/2[.
% The values in yi are in the range ]-ElementHeight/2 ElementHeight/2[.
% (if M = 1 and N = 1, then xi = yi = 0).
SegWidth = ElementWidth/M;
xi = -ElementWidth/2 + SegWidth/2 + (0:M-1)*SegWidth;
SegHeight = ElementHeight/N;
yi = -ElementHeight/2 + SegHeight/2 + (0:N-1)*SegHeight;
[xi,yi] = meshgrid(xi,yi);
xi = reshape(xi,[1 1 M*N]);
yi = reshape(yi,[1 1 M*N]);


%-- Variables that we need:
%
% Note: We work in an ISO spherical system for each sub-element
%       r = distance between the segment centroid and the point of interest
%       sinT = sine theta: theta is the polar angle.
%       cosT = cosine theta.
%       sinP = sine phi: phi is the azimuthal angle.
%       cosP = cosine phi.
%       They are of size [numel(x) NumberOfElements M*N].
%
dxi = x-xi-xe;
dyi = y-yi-ye;
d2 = dxi.^2+dyi.^2;
r = sqrt(d2+z.^2);
%---
epss = eps('single');
clear dxi dyi d2
%---
% The term 1/r is present in the equations (problems if r is very small!):
% small r values are replaced by lambda/2
lambda = c/fc;
r(r<lambda/2) = lambda/2;

%-------------------------------------------%
% end of POINT LOCATIONS, DISTANCES & GRIDS %
%-------------------------------------------%

mysinc = @(x) sin(abs(x)+epss)./(abs(x)+epss); % cardinal sine
% [note: In MATLAB, sinc is sin(pi*x)/(pi*x)]

%-------------------%
% FREQUENCY SPECTRA %
%-------------------%

%-- FREQUENCY SPECTRUM of the transmitted pulse
if isempty(FreqSweep)
    % We want a windowed sine of width PARAM.TXnow
    T = NoW/fc; % temporal pulse width
    if isinf(T); T = 1e6; end
    wc = 2*pi*fc;
    pulseSpectrum = @(w) 1i*(mysinc(T*(w-wc)/2)-mysinc(T*(w+wc)/2));
else
    % We want a linear chirp of width PARAM.TXnow
    % (https://en.wikipedia.org/wiki/Chirp_spectrum#Linear_chirp)
    T = NoW/fc; % temporal pulse width
    if isinf(T); T = 1e6; end
    wc = 2*pi*fc;
    dw = 2*pi*FreqSweep;
    s2 = @(w) sqrt(pi*T/dw)*exp(-1i*(w-wc).^2*T/2/dw).*...
        (fresnelint((dw/2+w-wc)/sqrt(pi*dw/T)) + fresnelint((dw/2-w+wc)/sqrt(pi*dw/T)));
    pulseSpectrum = @(w) (1i*s2(w)-1i*s2(-w))/T;
end

%-- FREQUENCY RESPONSE of the ensemble PZT + probe
% We want a generalized normal window (6dB-bandwidth = PARAM.bandwidth)
% (https://en.wikipedia.org/wiki/Window_function#Generalized_normal_window)
wB = param.bandwidth*wc/100; % angular frequency bandwidth
p = log(126)/log(2*wc/wB); % p adjusts the shape
probeSpectrum = @(w) exp(-(abs(w-wc)/(wB/2/log(2)^(1/p))).^p);
% The frequency response is a pulse-echo (transmit + receive) response. A
% square root is thus required when calculating the pressure field:
probeSpectrum = @(w) sqrt(probeSpectrum(w));
% Note: The spectrum of the pulse (pulseSpectrum) will be then multiplied
% by the frequency-domain tapering window of the transducer (probeSpectrum)



%-- FREQUENCY STEP
if isSIMUS3 % PFIELD3 has been called by SIMUS3
    df = options.FrequencyStep;
else % We are in PFIELD3 only (i.e. not called by SIMUS3)
    % The frequency step df is chosen to avoid interferences due to
    % inadequate discretization.
    % -- df = frequency step (must be sufficiently small):
    % One has exp[-i(k r + w delay)] = exp[-2i pi(f r/c + f delay)] in the Eq.
    % One wants: the phase increment 2pi(df r/c + df delay) be < 2pi.
    % Therefore: df < 1/(r/c + delay).
    df = 1/(max(r(:)/c) + max(delaysTX(:)));
    df = options.FrequencyStep*df;
    % note: df is here an upper bound; it will be recalculated below
end

%-- FREQUENCY SAMPLES
Nf = 2*ceil(param.fc/df)+1; % number of frequency samples
f = linspace(0,2*param.fc,Nf); % frequency samples
df = f(2); % update the frequency step
%- we keep the significant components only by using options.dBThresh
S = abs(pulseSpectrum(2*pi*f).*probeSpectrum(2*pi*f));
GdB = @(f) 20*log10(S/max(S)); % gain in dB
IDX = GdB(f)>options.dBThresh;
IDX = IDX(find(IDX,1):find(IDX,1,'last'));
f = f(IDX);


[~,frequency_idx] = min(abs(f-param.fc));

end





function structArray = IgnoreCaseInFieldNames(structArray)

switch inputname(1)
    case 'param'
        fieldLIST = {'attenuation','baffle','bandwidth','c','elements','fc',...
            'fnumber','focus','fs','height','kerf','movie','Nelements',...
            'passive','pitch','radius','RXangle','RXdelay',...
            'TXapodization','TXfreqsweep','TXnow','t0','width'};
    case 'options'
        if isstruct(structArray)
            fieldLIST = {'dBThresh','ElementSplitting',...
                'FullFrequencyDirectivity','FrequencyStep','ParPool',...
                'WaitBar'};
        else
            return
        end
end

OldFieldNames = fieldnames(structArray);
tmp = lower(OldFieldNames);
assert(length(tmp)==length(unique(tmp)),...
    ['The structure ' upper(inputname(1)),...
    ' contains duplicate field names (when ignoring case).'])

[idx,loc] = ismember(lower(fieldLIST),tmp);
idx = find(idx); loc = loc(idx);
for k = 1:length(idx)
    tmp = eval(['structArray.' OldFieldNames{loc(k)}]); %#ok
    structArray = rmfield(structArray,OldFieldNames{loc(k)});
    eval(['structArray.' fieldLIST{idx(k)} ' = tmp;']) %#ok
end

end
