% with this script the data from the pressure profile measurements are analysed
% and compared to the simulations 
clearvars;

% define the varibales of the TX transducer
frequency = 2.5e6; % frequency of the simulation
radius = in2m(1.5/2); % transform to meters the radius of the TX

rotation_phi_deg = 0; % rotation of the TX transducer according to Euler angles in degrees
rotation_theta_deg = 0;
rotation_psi_deg = 0;

lateral_shift_x = 0; % shift of centre of TX with respect to centre of RX in meters 
lateral_shift_y = 0; 

% define the variables of the RX transducer
RX_x = 20*1e-3; % dimension of RX in x coordinate
RX_y = RX_x; % dimension of RX in y coordinate
test_distance = 50*1e-3; % distance at which to perform the simulation in meters

medium = set_medium('water');
lateral_step = 1e-3; % this is the step from the measurement data 

%% load the data from the hydrophone measurements 

% measured data to the pressure at the desired driving voltage 
pressure_scaling_coeff = 10; % multiplication coefficient to convert measured data to the pressure at the desired driving voltage for the measurements done at 2.5 MHz is 10 times 
[filename,pathname] = uigetfile('*.mat','Select the mat file containing the data');

load(strcat(pathname,filename));
acoustic_data = acoustic{1,2};
acoustic_info_exp = acoustic{1,1};
% clearvars acoustic

%% calculate the average power at the receiver

% find the indices of the data corresponding to the area 
ROI_indices = find(abs([acoustic_data.Z])'<= RX_x*1e3/2 & abs([acoustic_data.Y])'<= RX_y*1e3/2);
rows_ROI = length(find([acoustic_data(ROI_indices).Y]' == 0)); % determine the number of rows and columns in the region of interest
col_ROI = find(diff([acoustic_data(ROI_indices).Z]'),1);

voltage_matrix_ROI = transpose(reshape([{acoustic_data(ROI_indices).line}]',col_ROI,rows_ROI)); % extracts the signal recorded at each point of the selected region of interest. 
% Each cell contains the recording 

% find the sensitivity of the needle at that frequency
needle = acoustic_info_exp.needle.data;
total_sensitivity = needle{1, 2}.sens_needle;
[~, sensitivity_index] = min(abs(frequency/1e6 - needle{1, 2}.freq_needle)); % find the closest frequency of the calibrated ones to the actual measuring frequency
sensitivity = total_sensitivity(sensitivity_index); % sensitivity = mV/MPa

% calculate the power at the receiver 
[mean_power_measurement_phase, mean_power_measurement_pkpk] = ...
    calculate_power_measurements(rows_ROI,col_ROI,voltage_matrix_ROI,sensitivity,...
                                        frequency,medium,pressure_scaling_coeff,RX_x,RX_y,'y');


%% simulate average power at receiver to estimate tilt 

data_matrix = zeros(length(RX_x)*length(lateral_shift_x)*length(lateral_shift_y)...
    *length(rotation_theta_deg)*length(rotation_psi_deg),8);
% disp(['Number of iterations: ',num2str(size(data_matrix,1))])
k=1;

for a = 1:length(RX_x)        
    for b = 1: length(lateral_shift_x)
        for c = 1:length(lateral_shift_y)
            for d = 1:length(rotation_theta_deg)
                for e = 1:length(rotation_psi_deg)

                   disp(['Iteration ',num2str(k),'/',num2str(size(data_matrix,1))])    
                        % shift the position of the TX transducer such that the area on
                        % whcih to calculate the power is still at the centre
                        coord_shift = rotate_vector_forward([lateral_shift_x(b) lateral_shift_y(c) test_distance], ...
                                        [deg2rad(rotation_phi_deg) deg2rad(rotation_theta_deg(d)) deg2rad(rotation_psi_deg(e))]);  
                
                        xdc = get_circ(radius, -coord_shift + [0 0 test_distance],...
                                        [deg2rad(rotation_phi_deg) deg2rad(rotation_theta_deg(d)) deg2rad(rotation_psi_deg(e))]);
                
                        % this measures are in meters
                        xmin = -0.04;
                        xmax = -xmin;
                        ymin = -0.04;
                        ymax = -ymin;
                        zmin = test_distance;
                        zmax = test_distance;
                        
                        [delta, xpoints, ypoints, zpoints] = create_input_coordinate_grid(xmin,xmax,ymin,ymax,zmin,zmax,lateral_step);
                        coord_grid=set_coordinate_grid(delta, xmin, xmax, ymin, ymax, zmin, zmax);
                        
                        ndiv= 20;
                        
                        p_cw=cw_pressure(xdc, coord_grid, medium, ndiv, frequency); % simulate the pressure filed
%                         power_cw=cw_power(xdc, coord_grid, medium, ndiv, frequency); % simulate the pressure filed

                
                        x = xmin:delta(1):xmax;
                        y = ymin:delta(2):ymax;
                        z = zmin:delta(3):zmax;
                        
                        clearvars xmin xmax ymin ymax zmin zmax xpoints ypoints zpoints 
                        
                        % extracts the pressure values that are in the area corresponding to the
                        % size of the receiver 
                        ROI_x_simulation = find(round(abs(x*1e3)) <= RX_x(a)*1e3/2);
                        ROI_y_simulation = find(round(abs(y*1e3)) <= RX_y(a)*1e3/2);
                        ROI_pressure_simulation = p_cw(ROI_x_simulation, ROI_y_simulation);
                        
                        [mean_power_simulation_phase, mean_power_simulation] = ...
                            calculate_power_simulation_CW(frequency,medium,ROI_pressure_simulation,RX_x(a),RX_y(a),'n');
        
                        % create matrix to store the data
                        data_matrix(k,:) = [RX_x(a)*1e3, lateral_shift_x(b), lateral_shift_y(c), test_distance, rotation_theta_deg(d), rotation_psi_deg(e),...
                                            mean_power_simulation, mean_power_simulation_phase];
                        k=k+1;

            
                end
            end
        end
    end
end

data_table = array2table(data_matrix,'VariableNames',{'RX_lat_size','shift_x','shift_y','distance','theta_deg','psi_deg','avgP','avgP_phase'});
clearvars data_matrix a b c d e ROI* x y z 
% calculate ratio of measurement power
measurement_power_ratio = mean_power_measurement_phase/mean_power_measurement_pkpk;

% calculate the power ratio for the simulations
sub_data = zeros(length(rotation_theta_deg)*length(rotation_phi_deg),3);
j=1;
for a = 1:length(rotation_theta_deg)
    for b = 1:length(rotation_psi_deg)
        simulation_power_ratio_table = data_table(data_table.theta_deg==rotation_theta_deg(a) & data_table.psi_deg==rotation_psi_deg(b),:);
        sub_data(j,:) = [rotation_theta_deg(a), rotation_psi_deg(b), simulation_power_ratio_table.avgP_phase/simulation_power_ratio_table.avgP];
        j=j+1;
    end
end

simulation_power_ratio_table = array2table(sub_data,'VariableNames',{'theta_deg','psi_deg','power_ratio'});
clearvars sub_data

% filter out only the ones that have power ratio similar to the ones from
% the measurements 

selection = simulation_power_ratio_table(simulation_power_ratio_table.power_ratio < 1.05*measurement_power_ratio & ...
    simulation_power_ratio_table.power_ratio > 0.95*measurement_power_ratio,:);






