"""ui_window.py: Contains all functions to perform data analysis and create extension user interface in the form of a widget window. """

__author__      = "Niklas Biermann"
__copyright__   = "Copyright 2023, Niklas Biermann"
__all__ = ["WidgetWindow"]

import csv

# Python API
import logging
import os.path
from functools import partial
from typing import List, Optional

import carb

# External packages
import numpy as np
import omni.client
import omni.kit.app
import omni.kit.ui
import omni.kit.viewport.utility
import omni.ui as ui
import omni.usd
from omni.ui import color as cl
from pxr import Usd

from .draw_util import ViewportScene
from .quality_checks import DQCheck

# Paths to read CSV files to
outputpath_psc = "c:/omniverse/position_check.csv"
outputpath_dmc = "c:/omniverse/2D_check.csv"
outputpath_pfc = "c:/omniverse/performance_check.csv"
outputpath_efc = "c:/omniverse/empty_file_check.csv"
outputpath_ncc = "c:/omniverse/naming_check.csv"
outputpath_scc = "c:/omniverse/scaling_check.csv"

# Design constants for the widgets
LABEL_PADDING = 120
BUTTON_HEIGHT = 50
GENERAL_SPACING = 5

WINDOW_WIDTH = 300
WINDOW_HEIGHT = 400

BUTTON_SELECTED_STYLE = {
    "Button": {
        "background_color": 0xFF5555AA,
        "border_color": 0xFF5555AA,
        "border_width": 2,
        "border_radius": 5,
        "padding": 5,
    }
}

BUTTON_BASE_STYLE = {
    "Button": {
        "background_color": cl("#292929"),
        "border_color": cl("#292929"),
        "border_width": 2,
        "border_radius": 5,
        "padding": 5,
    }
}

# For python it is recommended to use std python logging, which also redirected to Carbonite
# It also captures file path and log
logger = logging.getLogger(__name__)


class WidgetWindow(ui.Window):
    def __init__(self, title: str, delegate=None, **kwargs):
        """
        Constructor for the Window UI widget of the extension.
        """
        # Setup the base widget window
        super().__init__(title, width=WINDOW_WIDTH, height=WINDOW_HEIGHT, visible=True, **kwargs)
        self.__label_width = GENERAL_SPACING
        self._viewport_scene = None 
        for key, value in kwargs.items():
            print("{} is {}".format(key, value))
        self._parent_ext_id = kwargs.get("parent_ext_id", "None")

        # Models
        self._source_prim_model = ui.SimpleStringModel()

        self.deferred_dock_in("Property", ui.DockPolicy.CURRENT_WINDOW_IS_ACTIVE)

        # Build the actual window UI
        self._build_window()

    def _destroy(self):
        """
        Destructor for the Window UI widget of the extension.

        """
        # It will destroy all the children
        super()._destroy()
        logger.info("Window destroyed")

    @property
    def label_width(self):
        """The width of the attribute label"""
        return self.__label_width

    @label_width.setter
    def label_width(self, value):
        """The width of the attribute label"""
        self.__label_width = value
        self.frame.rebuild()

    def _get_selected_prim_names_from_model_or_viewport(self):
        """Reads the selection of prims from the model.
        The selection button needed to be clicked before.
        If the selection model is empty, the selection from the viewport is used.
        If the selection model is empty AND no selection is done inside the viewport returns a empty list."""
        prim_names = []
        if not self._source_prim_model.get_value_as_string():  # No selection in the model
            logger.info("Getting selection directly from current selection and not from the model...")
            prim_names = DQCheck.get_selection()
            logger.info(f"prim_names from selection and not from model: {prim_names}")
            if not prim_names:
                logger.info("Selection neither in model or in the viewport!")
        else:  # There are selections in the model
            prim_names = [i.strip() for i in self._source_prim_model.as_string.split(",")]

        return prim_names

    def _on_position_check(self):
        """Called when the user checks the Position-Checkbox. Initiates the Position-Check from .quality_checks"""
        logger.info("Clicked Action")
        logger.info("CSV exported")
        stage = Usd.Stage.Open(self._source_prim_model.get_value_as_string())
        DQCheck.position_check(stage)

    def _on_dimensioncheck(self):
        """Called when the user checks the 2D-Checkbox. Initiates the 2D-Check from .quality_checks"""
        logger.info("Clicked Action")
        logger.info("CSV exported")
        stage = Usd.Stage.Open(self._source_prim_model.get_value_as_string())
        DQCheck.dimension_check(stage)

    def _on_fps_check(self):
        """Called when the user checks the Performance-Checkbox. Initiates the Performance-Check from .quality_checks"""
        logger.info("Clicked Action")
        logger.info("CSV exported")
        stage = Usd.Stage.Open(self._source_prim_model.get_value_as_string())
        DQCheck.performance_check(stage)

    def _on_empty_file_check(self):
        """Called when the user checks the Empty-File-Checkbox. Initiates the Empty-File-Check from .quality_checks"""
        logger.info("Clicked Action")
        logger.info("CSV exported")
        stage = Usd.Stage.Open(self._source_prim_model.get_value_as_string())
        DQCheck.empty_file_check(stage)

    def _on_naming_check(self):
        """Called when the user checks the Naming-Checkbox. Initiates the Naming-Check from .quality_checks"""
        logger.info("Clicked Action")
        logger.info("CSV exported")
        stage = Usd.Stage.Open(self._source_prim_model.get_value_as_string(), load=Usd.Stage.LoadNone)
        DQCheck.naming_check(stage)

    def _on_scaling_check(self):
        """Called when the user checks the Scaling-Checkbox. Initiates the Scaling-Check from .quality_checks"""
        logger.info("Clicked Action")
        logger.info("CSV exported")
        stage = Usd.Stage.Open(self._source_prim_model.get_value_as_string())
        DQCheck.scaling_check(stage)

    def _on_position_check_light(self):
        """
        Function to initiate drawing of bounding box around selected prim.
        """

        logger.info("_on_position_check_light clicked")
        # Get the active (which at startup is the default Viewport)
        viewport_window = omni.kit.viewport.utility.get_active_viewport_window()

        # Issue an error if there is no Viewport
        if not viewport_window:
            carb.log_error("No Viewport Window to add XXX scene to")
            return

        # Build out the scene
        self._viewport_scene = ViewportScene(viewport_window, self._parent_ext_id)
        bbs = self._handle_bbs_compute()
        # Only the last draw call is drawn
        self._viewport_scene.draw_bounding_box(bbs)

    def _handle_bbs_compute(self):
        """
        Function to call bounding box calculation for selected prim.
        """
        bbs = self._calc_bb_from_selected()
        return bbs

    def _calc_bb_from_selected(self):  
        """
        Function to compute bounding box for selected prim in prim tree or 3D environment.
        """
        logger.info("Computing BB!")
        prim_names = self._get_selected_prim_names_from_model_or_viewport()

        if not prim_names:  # Nothing selected
            return None
        else:  # If there are prims selected (model or viewport)
            bbs = []
            for prim_name in prim_names:
                prim_bb = DQCheck.compute_path_bbox(
                    prim_name
                )  # Replace with cache Version for performance on bigger scenes
                bbs.append(prim_bb)
                logger.info(f"prim_bb: {prim_bb}")
            return bbs

    def _build_source(self):  
        """Build the widgets of the "USD Stage Path" group"""
        with ui.CollapsableFrame("Source", name="group"):
            with ui.VStack(height=0, spacing=GENERAL_SPACING):
                with ui.HStack():
                    ui.Label("USD Stage Path", name="attribute_name", width=self.label_width)
                    ui.StringField(model=self._source_prim_model)

    def _run(self):
        """Executed upon push of Validation-button depending on what checkboxes are checked."""
        if self.checkbox_position.model.get_value_as_bool():
            self._on_position_check()
        if self.checkbox_dimension.model.get_value_as_bool():
            self._on_dimensioncheck()
        if self.checkbox_performance.model.get_value_as_bool():
            self._on_fps_check()
        if self.checkbox_emptyfile.model.get_value_as_bool():
            self._on_empty_file_check()
        if self.checkbox_naming_convention.model.get_value_as_bool():
            self._on_naming_check()
        if self.checkbox_scaling.model.get_value_as_bool():
            self._on_scaling_check()
    
    def _build_position_check(self):
        """Function to build all user interface components of the position check."""
        # Build Position-Check(-Light)
        with ui.HStack():
            # Checkbox creation
            self.checkbox_position = ui.CheckBox(width=ui.Percent(5))
            with ui.CollapsableFrame("Position-Check", height=40, width=ui.Percent(95), collapsed=True):
                with ui.VStack():
                    # Button for Position-Check-Light
                    ui.Button(
                        "Position-Check Light",
                        clicked_fn=partial(self._on_position_check_light),
                        style={"background_color": cl.grey, "color": cl.white},
                    )
                    # Actual Position-Check, analysis of CSV file
                    if os.path.isfile(outputpath_psc):
                        with open(outputpath_psc, "r") as csv_file:
                            position_reader = csv.reader(csv_file, delimiter=";")
                            self.z_rejection_count = 0
                            self.z_acceptance_count = 0
                            self.y_rejection_count = 0
                            self.y_acceptance_count = 0
                            self.x_rejection_count = 0
                            self.x_acceptance_count = 0
                            next(position_reader)
                            for row_x in position_reader:
                                if "False" in row_x[4]:
                                    self.x_rejection_count += 1
                                if "True" in row_x[4]:
                                    self.x_acceptance_count += 1
                                if "False" in row_x[5]:
                                    self.y_rejection_count += 1
                                if "True" in row_x[5]:
                                    self.y_acceptance_count += 1
                                if "False" in row_x[6]:
                                    self.z_rejection_count += 1
                                if "True" in row_x[6]:
                                    self.z_acceptance_count += 1
                        # Creation of progress bar for each dimension of positional deviations
                        x_position_bar = ui.ProgressBar()
                        x_position_bar.model.set_value(
                            self.x_rejection_count / (self.x_acceptance_count + self.x_rejection_count)
                        )
                        ui.Label("Percentage of prims outside 99.99% x confidence interval")
                        y_position_bar = ui.ProgressBar()
                        y_position_bar.model.set_value(
                            self.y_rejection_count / (self.y_acceptance_count + self.y_rejection_count)
                        )
                        ui.Label("Percentage of prims outside 99.99% y confidence interval")
                        z_position_bar = ui.ProgressBar()
                        z_position_bar.model.set_value(
                            self.z_rejection_count / (self.z_acceptance_count + self.z_rejection_count)
                        )
                        ui.Label("Percentage of prims outside 99.99% z confidence interval")
                    # Create z-distribution histogram
                    ui.Label(
                        "Histogram - Z-Distribution",
                        style={"background_color": cl.black, "color": cl.white},
                    )
                    if os.path.isfile(outputpath_psc):
                        with open(outputpath_psc, "r") as csv_file:
                            leser = csv.reader(csv_file, delimiter=";")
                            z_values = []
                            next(leser)
                            for row in leser:
                                if row[3].isdigit or "-" in row[3]:
                                    z_values.append(float(row[3]))
                            histogram, edges = np.histogram(z_values, bins=50)
                            counter_float = [float(i) for i in histogram]
                            ui.Plot(
                                ui.Type.HISTOGRAM,
                                0.0,
                                float(max(counter_float)),
                                *counter_float,
                                height=250,
                                style={"color": cl.red},
                            )
                        ui.Button(
                            f"Min: {min(edges)}, Max: {max(edges)}, Bin-Range: {edges[1]-edges[0]}",
                            style={"background_color": cl.black, "color": cl.white},
                            tooltip="Histogram Legend",
                            tooltip_offset_y=22,
                        )
            omni.ui.Spacer(height=10)

    def _build_scaling_check(self):
        """Function to build all user interface components of the scaling check."""
        # Scaling-Check 
        with ui.HStack():
            self.checkbox_scaling = ui.CheckBox(width=ui.Percent(5))
            with ui.CollapsableFrame("Scaling-Check", height=40, width=ui.Percent(95), collapsed=True):
                with ui.VStack():
                    scaling_bar = ui.ProgressBar()
                    if os.path.isfile(outputpath_scc):  
                        with open(outputpath_scc, "r") as csv_file:
                            scaling_reader = csv.reader(csv_file, delimiter=";")
                            self.scaling_error_count = 0
                            self.no_error_count = 0
                            next(scaling_reader)
                            for row in scaling_reader:
                                if row[3] == "False":
                                    self.scaling_error_count += 1
                                if row[3] == "True":
                                    self.no_error_count += 1
                                scaling_bar.model.set_value(
                                    self.scaling_error_count / (self.no_error_count + self.scaling_error_count)
                                )
                    ui.Label("Percentage of falsely scaled prims")
                    ui.Label(
                        "Histogram - Volume Distribution",
                        style={"background_color": cl.black, "color": cl.white},
                    )
                    if os.path.isfile(outputpath_scc):  
                        with open(outputpath_scc, "r") as csv_file:
                            scaling_histogram_reader = csv.reader(csv_file, delimiter=";")
                            volumes = []
                            next(scaling_histogram_reader)
                            for row in scaling_histogram_reader:
                                if row[1].isdigit or "-" in row[1]:
                                    volumes.append(float(row[1]))
                            volume_histogram, edges = np.histogram(volumes, bins=50)
                            volumes_float = [float(i) for i in volume_histogram]
                            ui.Plot(
                                ui.Type.HISTOGRAM,
                                0.0,
                                float(max(volumes_float)),
                                *volumes_float,
                                height=250,
                                style={"color": cl.red},
                            )
                        ui.Button(
                            f"Min: {min(edges)}, Max: {max(edges)}, Bin-Range: {edges[1]-edges[0]}",
                            style={"background_color": cl.black, "color": cl.white},
                            tooltip="Histogram Legend",
                            tooltip_offset_y=22,
                        )
            omni.ui.Spacer(height=10)

    def _build_dimension_check(self):
        """Function to build all user interface components of the dimension check."""
        with ui.HStack():
            self.checkbox_dimension = ui.CheckBox(width=ui.Percent(5))
            with ui.CollapsableFrame("2D-Check", height=40, width=ui.Percent(95), collapsed=True):
                with ui.VStack():
                    dimension_bar = ui.ProgressBar()
                    if os.path.isfile(outputpath_dmc): 
                        with open(outputpath_dmc, "r") as csv_file:
                            dimension_reader = csv.reader(csv_file, delimiter=";")
                            self.two_d_count = 0
                            self.three_d_count = 0
                            next(dimension_reader)
                            for row in dimension_reader:
                                if row[1] == "0.0":
                                    self.two_d_count += 1
                                if row[1] != "0.0":
                                    self.three_d_count += 1
                                dimension_bar.model.set_value(
                                    self.two_d_count / (self.three_d_count + self.two_d_count)
                                )
                    ui.Label("Percentage of 2D prims")
            omni.ui.Spacer(height=10)

    def _build_performance_check(self):
        """Function to build all user interface components of the performance check."""
        with ui.HStack(): 
            self.checkbox_performance = ui.CheckBox(width=ui.Percent(5))
            with ui.CollapsableFrame(
                "Performance-Check", height=40, width=ui.Percent(95), collapsed=True
            ):
                with ui.VStack():
                    if os.path.isfile(outputpath_pfc):  
                        with open(outputpath_pfc, "r") as csv_file:
                            performance_reader = csv.reader(csv_file, delimiter=";")
                            next(performance_reader)
                            mesh_count = 0
                            for row in performance_reader:
                                if row[2] != "":
                                    mesh_count += 1
                            ui.Label(f"Total prim count: {mesh_count}")
                            self.performance_severity = 0
                            if mesh_count >= 50000:
                                self.performance_severity = mesh_count
            omni.ui.Spacer(height=10)

    def _build_empty_file_check(self):
        """Function to build all user interface components of the empty-file-check."""
        with ui.HStack():  
            self.checkbox_emptyfile = ui.CheckBox(width=ui.Percent(5))
            with ui.CollapsableFrame(
                "Empty-File-Check", height=40, width=ui.Percent(95), collapsed=True
            ):
                with ui.VStack():
                    if os.path.isfile(outputpath_efc):  
                        with open(outputpath_efc, "r") as csv_file:
                            empty_file_reader = csv.reader(csv_file, delimiter=";")
                            next(empty_file_reader)
                            self.empty_file_count = 0
                            for row in empty_file_reader:
                                if row[0] != "":
                                    self.empty_file_count += 1
                            ui.Label(f"Empty file count: {self.empty_file_count}")
            omni.ui.Spacer(height=10)

    def _build_naming_check(self):
        """Function to build all user interface components of the naming convention check."""
        with ui.HStack():
            self.checkbox_naming_convention = ui.CheckBox(width=ui.Percent(5))
            with ui.CollapsableFrame(
                "Naming-Convention-Check", height=40, width=ui.Percent(95), collapsed=True):
                with ui.VStack():
                    if os.path.isfile(outputpath_ncc):  
                        with open(outputpath_ncc, "r") as csv_file:
                            naming_reader = csv.reader(csv_file, delimiter=";")
                            self.violation_count = 0
                            next(naming_reader)
                            for row in naming_reader:
                                if row[1] == "False":
                                    self.violation_count += 1
                            ui.Label(f"Naming convention violation count: {self.violation_count}")
            omni.ui.Spacer(height=10)

    def _build_window(self):  
        """
        Function to build the main widget window and its contents.
        """
        # Define the UI of the key widget window
        with self.frame:
            # Vertical Stack of menus: Everything underneath is stapled vertically
            with ui.VStack():
                # Build text field to insert virtual factory URL
                self._build_source()
                # Validation button to run checks whose checkboxes are ticked
                ui.Button("Validation", clicked_fn=partial(self._run), width=ui.Percent(100))
                # Build collapsible frame for all rules
                with ui.CollapsableFrame("Rule Selection", height=80, width=ui.Percent(100), collapsed=False):
                    # Vertical Stack in each collapsable frame
                    with ui.VStack():
                            # Build collapsable frame for each rule and related KPIs
                            self._build_position_check()
                            self._build_scaling_check()
                            self._build_dimension_check()
                            self._build_performance_check()
                            self._build_empty_file_check()
                            self._build_naming_check()
                # Calculate overall severity score
                # Severity score is added as KPI to assess the overall level of data quality within the checked virtual factory model (see thesis chapter 6.4)
                self.severity = int( 
                            (1 / 3) * (self.x_rejection_count + self.y_rejection_count + self.z_rejection_count) * 7
                            + self.scaling_error_count * 7
                            + self.two_d_count * 10
                            + self.performance_severity * 8
                            + self.empty_file_count * 9
                            + self.violation_count * 5
                        )
                ui.Label(f"Severity Score: {self.severity}") 

        logger.info("Window built")
