clear all;close all;clc %#ok<CLALL>

readraw=0; % set to 1 if wanting to read the raw data; set to 0 if wanting to load the MAT-file (for higher speed)
fullsample=0; % set to 1 if wanting to evaluate the full sample (N = 60); set to 0 if wanting to exclude invalid participants

%   1. t        - Time stamp
%   2. CrossW   - Crossing Window, Availability for pedestrian to cross  (Open(O)/Close(C))
%   3. Crossing - Whether the pedestrian is crossing or not, (Y/N)
%   4. P.x      - Pedestrian x position in the scene relative to HIKER origin reference (m)
%   5. P.y      - Pedestrian y position in the scene relative to HIKER origin reference (m)
%   6. P.z      - Pedestrian z position in the scene relative to HIKER origin reference (m)
%   7. P.speed  - Pedestrian speed (m/s)
%   8. P.hr     - Pedestrian head rotation angle (clockwise in degrees), 0 deg when participant enters HIKER and looks straight
%   9. P.r.w    - Scalar value In quaternion
%  10. P.r.x    - Vector in quaternion-direction of axis around which we rotate
%  11. P.r.y    - Vector in quaternion-direction of axis around which we rotate
%  12. P.r.z    - Vector in quaternion-direction of axis around which we rotate
%  13. C1.x     - Vehicle x position (m)
%  14. C1.z     - Vehicle z position (m), car drives at z = 0.748 m
%  15. C1.speed - Speed of vehicle in mph
%  16. eHMI_On  - eHMI switched on (Y/N)
%  17. eHMI_Int - eHMI intensity (0 to 1)

% Block 1: Trial 1-12
% Block 2: Trial 13-24
% Block 3: Trial 25-36
% Block 4: Trial 38-49
% Trials 37 & 50 are failure trials. In the failure trials, the eHMI is on but the car does not stop

% Per block:
% 4 trials No Braking & eHMI off
% 4 trials Braking at 43 m (1 of which with eHMI off)
% 4 trials Braking at 33 m (1 of which with eHMI off)

% Participant 1, 3, 5, ... belong to the +1 Group
% Participant 2, 4, 6, ... beling to the -1 Group

% Lane boundaries from 2.826 m till -1.363 m
% This corresponds to 2.050 m 6.239 m after correcting the z position so that z = 0 m corresponds to the starting position of the pedestrian
%%
if readraw==1
    [BrakingDistanceAll,eHMIOn,Speed340]=deal(NaN(60,50)); % create NaN matrix
    Measures=NaN(4,60,50); % create NaN matrix
    [PedestrianZall,VehicleXall,eHMIintensityall,VehicleSpeedall]=deal(NaN(60,50,2172)); % create NaN matrix
    for pp=1:60 % loop over 60 participants
        disp(pp) % display participant number
        filename=['PedsimEHMI_exp2_' num2str(pp)];
        Xtext=readmatrix(filename,'outputtype','char');
        Q=NaN(size(Xtext,1),1);
        for i=1:size(Xtext,1);if strfind(Xtext{i,18},'Q');Q(i)=1;end;end % Store a 1 in the Q vector if the row contains the questions
        TrialIndexesStart=[2;find(Q(1:end-1)==1)+2]; % Start Indexes of each trial
        TrialIndexesEnd=find(Q==1)-1; % End indexes of each trial
        Questions=Xtext(Q==1,:); % All question results
        for i=1:50 % loop over the 50 trials
            TrialDataText=Xtext(TrialIndexesStart(i):TrialIndexesEnd(i),:);
            
            Time=str2double(TrialDataText(:,1));Time=Time-Time(1); % Time vector starting at 0
            PedestrianZ=-(str2double(TrialDataText(:,6))-4.876);   % ensure that pedestrian starts at 0 m, and forward walking is positive
            VehicleX=   -(str2double(TrialDataText(:,13))-1.8);    % vehicle X is the position along the road
            eHMIintensity=str2double(TrialDataText(:,17));         % eHMI intensity
            VehicleSpeed=str2double(TrialDataText(:,15));          % Vehicle speed
            
            Timenew=0:0.01:max(Time); % New time vector (100 Hz)
            PedestrianZi    =interp1(Time,PedestrianZ,Timenew);   % Interpolate the data to 100 Hz
            VehicleXi       =interp1(Time,VehicleX,Timenew);      % Interpolate the data to 100 Hz
            eHMIintensityi  =interp1(Time,eHMIintensity,Timenew); % Interpolate the data to 100 Hz
            VehicleSpeedi   =interp1(Time,VehicleSpeed,Timenew);  % Interpolate the data to 100 Hz
            
            eHMIOn(pp,i)=sum(str2double(TrialDataText(:,17))); % Calculate the sum of the eHMI intensity values (used for defining whether the eHMI was on in this trial)
            Speed340(pp,i)=str2double(TrialDataText(340,15));  % Extract speed some time into the trial (will be used later on to extract whether the trial was a yielding 33 m trial, yielding 43 m trial, or a no-yielding trial)
            
            Measures(1,pp,i)=str2double(Questions{i,end-4}(1:strfind(Questions{i,end-4},';')-1)); % Response to Q1 "I experienced the situation as risky"
            Measures(2,pp,i)=str2double(Questions{i,end-2}(1:strfind(Questions{i,end-2},';')-1)); % Response to Q2 "I could comprehend the behavior and appearance of the approaching vehicle"
            Measures(3,pp,i)=str2double(Questions{i,end}(1:strfind(Questions{i,end},';')-1));     % Response to Q3 "I trust the behavior and appearance of the automated vehicle"
            
            PedestrianZall(pp,i,1:length(PedestrianZi))=PedestrianZi;       % Store pedestrian walking distance in a matrix
            VehicleXall(pp,i,1:length(VehicleXi))=VehicleXi;                % Store vehicle position in a matrix
            eHMIintensityall(pp,i,1:length(eHMIintensityi))=eHMIintensityi; % Store eHMI intensity in a matrix
            VehicleSpeedall(pp,i,1:length(VehicleSpeedi))=VehicleSpeedi;    % Store vehicle speed in a matrix
        end
    end
    save('all_data','eHMIOn','Speed340','Measures','PedestrianZall','VehicleXall','eHMIintensityall','VehicleSpeedall')
else
    load all_data
end
%% Figure 3. Vehicle speed as a function of the distance between the automated vehicle and the pedestrian.
opengl hardware % set OpenGL rendering method.
figure(3);hold on;box on;grid on

plot(squeeze(VehicleXall(1,2,1:2:end)),sm2km(squeeze(VehicleSpeedall(1,2,1:2:end))),'k:','Linewidth',5) % Speed for non-yielding vehicle 
plot(squeeze(VehicleXall(1,1,1:2:end)),sm2km(squeeze(VehicleSpeedall(1,1,1:2:end))),'-','Linewidth',5,'color',[0 128 0]/255) % Speed for yielding vehicle (33 m)
plot(squeeze(VehicleXall(1,5,1:2:end)),sm2km(squeeze(VehicleSpeedall(1,5,1:2:end))),':','Linewidth',5,'color',[255 165 0]/255) % Speed for yielding vehicle (43 m)

onindex=find(squeeze(eHMIintensityall(2,1,:))>0,1,'first'); % eHMI onset -1 s (33 m)
plot(squeeze(VehicleXall(2,1,onindex)),sm2km(squeeze(VehicleSpeedall(2,1,onindex))),'o','markeredgecolor',[0 128 0]/255,'markerfacecolor','w','markersize',14,'linewidth',5) % plot marker
onindex=find(squeeze(eHMIintensityall(1,1,:))>0,1,'first'); % eHMI onset +1 s (33 m)
plot(squeeze(VehicleXall(1,1,onindex)),sm2km(squeeze(VehicleSpeedall(1,1,onindex))),'o','markeredgecolor',[0 128 0]/255,'markerfacecolor',[0 128 0]/255,'markersize',14,'linewidth',5) % plot marker
onindex=find(squeeze(eHMIintensityall(2,3,:))>0,1,'first'); % eHMI onset -1 s (43 m)
plot(squeeze(VehicleXall(2,3,onindex)),sm2km(squeeze(VehicleSpeedall(2,3,onindex))),'o','markeredgecolor',[255 165 0]/255,'markerfacecolor','w','markersize',14,'linewidth',5) % plot marker
onindex=find(squeeze(eHMIintensityall(1,5,:))>0,1,'first'); % eHMI onset +1 s (43 m)
plot(squeeze(VehicleXall(1,5,onindex)),sm2km(squeeze(VehicleSpeedall(1,5,onindex))),'o','markeredgecolor',[255 165 0]/255,'markerfacecolor',[255 165 0]/255,'markersize',14,'linewidth',5) % plot marker

legend('Non-yielding vehicle','Yielding vehicle (33 m)','Yielding vehicle (43 m)','eHMI onset -1 s (33 m)','eHMI onset + 1 s (33 m)','eHMI onset -1 s (43 m)','eHMI onset +1 s (43 m)','location','southeast')
set(gca,'xlim',[-3 60],'ylim',[0 50],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
xlabel('Distance between vehicle and pedestrian (m)')
ylabel('Vehicle speed (kph)')
h=findobj('FontName','Helvetica'); set(h,'FontSize',28,'Fontname','Arial')
set(gcf,'WindowState','maximized');print(gcf,'Figure3.png','-dpng','-r300');
%% Compute the 4th measure: Road entering time
for i=1:60 % loop over 50 pp
    for i2=1:50 % loop over 50 trials
        road_entering_time=find(PedestrianZall(i,i2,:)>2.050,1,'first')/100; % first moment that the pedestrian position exceeds 2.05 m
        if ~isempty(road_entering_time) % store in matrix when road_entering_time exists
            Measures(4,i,i2)=road_entering_time;
        end
    end
end
%% Extract trial type
for pp=1:60 % loop over participants
    BrakingDistanceAll(pp,Speed340(pp,:)> 27 &Speed340(pp,:) < 29.5)=33; % store in 60 x 50 matrix BrakingDistanceAll that this is a yielding 33 m trial
    BrakingDistanceAll(pp,Speed340(pp,:)> 24 &Speed340(pp,:) < 26)=43;   % store in 60 x 50 matrix BrakingDistanceAll that this is a yielding 43 m trial
    BrakingDistanceAll(pp,Speed340(pp,:)> 29.6)=999;                     % store in 60 x 50 matrix BrakingDistanceAll that this is a non-yielding trial (no braking)
end
eHMIOn=eHMIOn>0; % make the eHMI intensity sum binary. It the sum exceeds 0, this must have been an eHMI On trial. There are 26 eHMI On trials per participant
%% Sort the measures (scores) in matrices, making a distinction between vehicle behavior
BlockStart=[1 13 25 38]; % trial numbers where each block starts
BlockEnd= [12 24 36 49]; % trial numbers where each block ends
MeanMeasuresVehBehaviorPerBlock =NaN(4,60,4,6);  % 4 measures x 60 participants x 4 blocks x 4 vehicle behaviors
MeasuresVehBehavior             =NaN(4,60,7,26); % 4 measures x 60 participants x 7 vehicle behaviors x 26 trials
[PedestrianZVehBehavior,VehicleXVehBehavior]=deal(NaN(60,7,26,2172)); % (60 participants x 7 vehicle behaviors x 26 trials x 2172 data points)
for Blocks=1:4 % loop over 4 blcoks
    MeasuresBlock=Measures(:,:,BlockStart(Blocks):BlockStart(Blocks)+11); % 4 measures x 60 participants x 12 trials per block matrix
    for pp=1:60 % loop over 60 participants
        Index=NaN(6,6); % Create for this participant a 7 x 6 matrix containing the indexes of trials (1 to 12) within that block per trial type
        Index(1,1:3)=  find(BrakingDistanceAll(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==33  & eHMIOn(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==1); % 33 m, eHMI on (3 trials)
        Index(2,1)  =  find(BrakingDistanceAll(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==33  & eHMIOn(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==0); % 33 m, eHMI off (1 trial)
        Index(3,1:3)=  find(BrakingDistanceAll(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==43  & eHMIOn(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==1); % 43 m, eHMI on (3 trials)
        Index(4,1)  =  find(BrakingDistanceAll(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==43  & eHMIOn(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==0); % 43 m, eHMI off (1 trial)
        Index(5,1:4)=  find(BrakingDistanceAll(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==999 & eHMIOn(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==0); % non-yielding, eHMI off (4 trials)
        Index(6,1:6)=  find(eHMIOn(pp,BlockStart(Blocks):BlockStart(Blocks)+11)==1);                                                                        % eHMI on (6 trials)
        
        if Blocks==1 % only do this once
            IndexAll=NaN(7,26); % Create for this participant a 7 x 26 matrix containing the indexes of trials (1 to 50) per trial type
            IndexAll(1,1:12)=find(BrakingDistanceAll(pp,:)==33 & eHMIOn(pp,:)==1);  % 33 m, eHMI on (12 trials)
            IndexAll(2,1:4) =find(BrakingDistanceAll(pp,:)==33 & eHMIOn(pp,:)==0);  % 33 m, eHMI off (4 trials)
            IndexAll(3,1:12)=find(BrakingDistanceAll(pp,:)==43 & eHMIOn(pp,:)==1);  % 43 m, eHMI on (12 trials)
            IndexAll(4,1:4) =find(BrakingDistanceAll(pp,:)==43 & eHMIOn(pp,:)==0);  % 43 m, eHMI off (4 trials)
            IndexAll(5,1:16)=find(BrakingDistanceAll(pp,:)==999 & eHMIOn(pp,:)==0); % non-yielding, eHMI off (16 trials)
            IndexAll(6,1:26)=find(eHMIOn(pp,:)==1);                                 % eHMI on (26 trials; i.e., 6 per block x 4 blocks = 24 + 2 failure trials = 26)
            IndexAll(7,1:18)=find(BrakingDistanceAll(pp,:)==999);                   % non-yielding (18 trials)
            
            for Measure_nr=1:4 % loop over the 4 measures
                for VehicleBehavior=1:7 % loop over the 7 types of vehicle behavior as defined above
                    Measures_per_pp_per_behavior=squeeze(Measures(Measure_nr,pp,IndexAll(VehicleBehavior,IndexAll(VehicleBehavior,:)>0)))'; % vector with scores for that participant and that vehicle behavior type
                    MeasuresVehBehavior(Measure_nr,pp,VehicleBehavior,1:length(Measures_per_pp_per_behavior))=Measures_per_pp_per_behavior;
                    % store the measures for that participant & vehicle type in a 4 measures x 60 participants x 6 vehicle behavior x 26 trials matrix
                end
            end
            for VehicleBehavior=1:7 % loop over the 7 types of vehicle behavior as defined above
                temp=squeeze(PedestrianZall(pp,IndexAll(VehicleBehavior,IndexAll(VehicleBehavior,:)>0),:)); % Pedestrian position of trials of a particular vehicle behavior x 2172 time stamps
                PedestrianZVehBehavior(pp,VehicleBehavior,1:size(temp,1),1:size(temp,2))=temp; % Store in matrix (60 participants x 7 vehicle behaviors x 26 trials x 2172 data points)
                temp=squeeze(VehicleXall(pp,IndexAll(VehicleBehavior,IndexAll(VehicleBehavior,:)>0),:)); % Vehicle position of trials of a particular vehicle behavior x 2172 time stamps
                VehicleXVehBehavior(pp,VehicleBehavior,1:size(temp,1),1:size(temp,2))=temp; % Store in matrix (60 participants x 7 vehicle behaviors x 26 trials x 2172 data points)
            end
        end
        
        for Measure_nr=1:4  % loop over the 4 measures
            for VehicleBehavior=1:6 % loop over the 6 types of vehicle behavior as defined above
                MeanMeasuresVehBehaviorPerBlock(Measure_nr,pp,Blocks,VehicleBehavior)=mean(MeasuresBlock(Measure_nr,pp,Index(VehicleBehavior,Index(VehicleBehavior,:)>0)));
                % 4 measures x 60 participants x 4 block numbers x 6 vehicle behaviors
            end
        end
    end
end
%% Figure 8. Pedestrian location at the moment the front of the car passed
PedPositionWhenCarPasses=deal(NaN(60,18));
for pp=1:60 % loop over 60 participants
    for trial=1:18 % loop over the 18 trials in which the vehicle was not yielding
        [~,carpasses]=min(abs(squeeze(VehicleXVehBehavior(pp,7,trial,:)))); % find the index of the minimum vehicle position in that trial, i.e.,the moment the car passed
        if ~isempty(carpasses)
            pedestrian_position_in_trial=squeeze(PedestrianZVehBehavior(pp,7,trial,:)); % pedestrian position in that trial
            PedPositionWhenCarPasses(pp,trial)=pedestrian_position_in_trial(carpasses); % store the pedestrian position when the non-yielding vehicle passes
        end
    end
end

temp=sum(PedPositionWhenCarPasses'>2.050); % number of trials per participant where they were on the road or had already crossed the road at the moment the non-yielding AV passed
disp('Number of times participants had walked onto the road, per participant in the +1 Group:')
disp(temp(1:2:end)) % number of trials walking onto the road +1
disp('Number of times participants had walked onto the road, per participant in the -1 Group:')
disp(temp(2:2:end)) % number of trials walking onto the road -1

invalidpp=find(sum(PedPositionWhenCarPasses'>2.050)>6);   % Participants who were on the road or had already crossed the road at the moment the non-yielding AV passed in more than 6 out of 18 non-yielding trials (i.e., 4 trials per block plus 2 failure trials) were excluded from all analyses.
validpp=find(~ismember(1:60,invalidpp));                  % Valid participants are participants who are not invalid particiapnts
odd=1:2:60;
validpp_odd=odd(~ismember(odd,invalidpp));                % Valid participants in the +1 Group
even=2:2:60;
validpp_even=even(~ismember(even,invalidpp));             % Valid participants in the -1 Group

if fullsample==1
validpp_odd=odd;validpp_even=even;validpp=1:60; % uncomment for generating the Figures and Tables of the appendices
end

figure(8)
hold on;box on;grid on
hl1=plot(-9999,-9999,'ko','markerfacecolor','k','markersize',13); % place dummy marker (used for creating the legend)
hl2=plot(-9999,-9999,'mo','markerfacecolor','m','markersize',13); % place dummy marker (used for creating the legend)

h1=patch([-1 20 20 -1],[2.050 2.050 6.239 6.239],[0 0 0],'facealpha',.1,'edgecolor','none');
for i=1:18 % loop over 18 non-yielding trials
    for pp=1:length(validpp_odd) % loop over odd participants (+1 Group)
        h2=scatter(-.15+i,PedPositionWhenCarPasses(validpp_odd(pp),i),240,'k','filled');
        alpha(h2,.2) % make the marker transparent
    end
end

for i=1:18 % loop over 18 non-yielding trials
    for pp=1:length(validpp_even) % loop over odd participants (-1 Group)
        h3=scatter(.15+i,PedPositionWhenCarPasses(validpp_even(pp),i),240,'m','filled');
        alpha(h3,.2)  % make the marker transparent
    end
end

for i=1:18  % loop over 18 non-yielding trials
    n1=sum(PedPositionWhenCarPasses(validpp_odd,i)>2.050 & PedPositionWhenCarPasses(validpp_odd,i)<6.239); % Number of participants who were on the road when the car passed, +1 Group
    N1=length(validpp_odd); % Number of participants in the +1 Group
    n2=sum(PedPositionWhenCarPasses(validpp_even,i)>2.050 & PedPositionWhenCarPasses(validpp_even,i)<6.239); % Number of participants who were on the road when the car passed, -1 Group
    N2=length(validpp_even); % Number of participants in the +1 Group
    text(i,7.8,[num2str(round(100*n1/N1)) '%'],'horizontalalignment','center') % Percentage of participants on the road, +1 Group
    text(i,7.5,[num2str(round(100*n2/N2)) '%'],'horizontalalignment','center','color','m') % Percentage of participants on the road, -1 Group
    if ismember(i,[13 18]) % in case the non-yielding trial is a failure trial
        x1= [n1 N1-n1;n2 N2-n2];
        [~,p] = fishertest(x1); % perform Fisher's exact test
        disp('Sample sizes: [+1 Group: on road, not on road; -1 Group: on road, not on road]')
        disp(x1)
        disp('p-value of Fisher''s exact test')
        disp(p)
    end
end

plot([-10 100],[7.3 7.3],'k:','Linewidth',3) % plot horizontal line
text(2/18,1.03,'Block 1','Units','Normalized','horizontalalignment','center')
text(6/18,1.03,'Block 2','Units','Normalized','horizontalalignment','center')
text(10/18,1.03,'Block 3','Units','Normalized','horizontalalignment','center')
text(12.5/18,1.03,'FT 1','Units','Normalized','horizontalalignment','center','color','r')
text(15/18,1.03,'Block 4','Units','Normalized','horizontalalignment','center')
text(17.3/18,1.03,'FT 2','Units','Normalized','horizontalalignment','center','color','r')
lg=legend([h1 hl1(1) hl2(1)],'Road','+1 Group','-1 Group','location','northeast','position',[0.05 0.49 0.0817708316879968 0.101765313840482]);
xlabel('Trial number (non-yielding trials only)')
ylabel('Pedestrian position perpendicular to the road (m)')
h=findobj('FontName','Helvetica'); set(h,'FontSize',20,'Fontname','Arial')
set(gca,'xlim',[0.5 18.5],'xtick',[1 4 5 8 9 12 14 17],'ylim',[0 8],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.05])
set(gcf,'WindowState','maximized');print(gcf,'Figure8.png','-dpng','-r300');
%% Figure 6. Mean pedestrian position perpendicular to the road as a function of elapsed time per Group, yielding condition, and eHMI condition (Blocks 1–3 only). Also shown are the moments the eHMI turned on, the moments the vehicle came to a stop, and the moment the vehicle passed in the eHMI Off trials.

d1=squeeze(nanmean(PedestrianZVehBehavior(:,1,:,:),3)); % 33 m, eHMI on (12 trials)
d2=squeeze(nanmean(PedestrianZVehBehavior(:,2,:,:),3)); % 33 m, eHMI off (4 trials)
d3=squeeze(nanmean(PedestrianZVehBehavior(:,3,:,:),3)); % 43 m, eHMI on (12 trials)
d4=squeeze(nanmean(PedestrianZVehBehavior(:,4,:,:),3)); % 43 m, eHMI off (4 trials)
d5=squeeze(nanmean(PedestrianZVehBehavior(:,5,:,:),3)); % non-yielding, eHMI off (16 trials)

time=0.01:0.01:length(d1)/100; % time vector (100 Hz)

figure(6)
subplot(1,2,1);hold on;box on;grid on
title('+1 Group')
patch([-1 17 17 -1],[2.050 2.050 6.239 6.239],[0 0 0],'facealpha',.1,'edgecolor','none')
plot(time(1:2:end),nanmean(d1(validpp_odd,1:2:end)),'-','Linewidth',5,'color',[0 128 0]/255) % 33 m, eHMI on (plot all at 50 Hz to prevent visual jitter)
plot(time(1:2:end),nanmean(d2(validpp_odd,1:2:end)),':','Linewidth',5,'color',[0 128 0]/255) % 33 m, eHMI off
plot(time(1:2:end),nanmean(d3(validpp_odd,1:2:end)),'-','Linewidth',5,'color',[255 165 0]/255) % 43 m, eHMI on
plot(time(1:2:end),nanmean(d4(validpp_odd,1:2:end)),':','Linewidth',5,'color',[255 165 0]/255) % 43 m, eHMI off
plot(time(1:2:650),nanmean(d5(validpp_odd,1:2:650)),'k','Linewidth',5) % non-yielding, eHMI off
onindex=find(squeeze(eHMIintensityall(1,1,:))>0,1,'first');
plot(time(onindex),0,'o','markeredgecolor',[0 128 0]/255,'markerfacecolor',[0 128 0]/255,'markersize',18) % eHMI onset, 33 m
onindex=find(squeeze(eHMIintensityall(1,5,:))>0,1,'first');
plot(time(onindex),0,'o','markeredgecolor',[255 165 0]/255,'markerfacecolor',[255 165 0]/255,'markersize',18) % eHMI onset, 43 m
plot(time(788),0,'*','color',[0 128 0]/255,'linewidth',3,'markersize',18) % AV stops, 33 m
plot(time(856),0,'*','color',[255 165 0]/255,'linewidth',3,'markersize',18) % AV stops, 43 m
plot(time(598),0,'d','markeredgecolor',[0 0 0]/255,'markerfacecolor',[0 0 0]/255,'markersize',18) % AV passes, No yielding
set(gca,'xlim',[0 15],'ylim',[0 7],'pos',[0.04 0.09 0.42 0.87],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01],'xtick',0:1:15)
xlabel('Elapsed time (s)')
ylabel('Pedestrian position perpendicular to the road (m)')
legend('Road','33 m & On','33 m & Off','43 m & On','43 m & Off','No yielding & Off','eHMI onset, 33 m','eHMI onset, 43 m','AV stops, 33 m','AV stops, 43 m','AV passes, No yielding','location','northwest','autoupdate','off');

subplot(1,2,2);hold on;box on;grid on
title('-1 Group')
patch([-1 17 17 -1],[2.050 2.050 6.239 6.239],[0 0 0],'facealpha',.1,'edgecolor','none')
plot(time(1:2:end),nanmean(d1(validpp_even,1:2:end)),'-','Linewidth',5,'color',[0 128 0]/255) % 33 m, eHMI on
plot(time(1:2:end),nanmean(d2(validpp_even,1:2:end)),':','Linewidth',5,'color',[0 128 0]/255) % 33 m, eHMI off
plot(time(1:2:end),nanmean(d3(validpp_even,1:2:end)),'-','Linewidth',5,'color',[255 165 0]/255) % 43 m, eHMI on
plot(time(1:2:end),nanmean(d4(validpp_even,1:2:end)),':','Linewidth',5,'color',[255 165 0]/255) % 43 m, eHMI off
plot(time(1:2:650),nanmean(d5(validpp_even,1:2:650)),'k','Linewidth',5) % non-yielding, eHMI off
onindex=find(squeeze(eHMIintensityall(2,1,:))>0,1,'first');
plot(time(onindex),0,'o','markeredgecolor',[0 128 0]/255,'markerfacecolor',[0 128 0]/255,'markersize',18) % eHMI onset, 33 m
onindex=find(squeeze(eHMIintensityall(2,3,:))>0,1,'first');
plot(time(onindex),0,'o','markeredgecolor',[255 165 0]/255,'markerfacecolor',[255 165 0]/255,'markersize',18) % eHMI onset, 43 m
plot(time(788),0,'*','color',[0 128 0]/255,'linewidth',3,'markersize',18) % AV stops, 33 m
plot(time(856),0,'*','color',[255 165 0]/255,'linewidth',3,'markersize',18)  % AV stops, 43 m
plot(time(598),0,'d','markeredgecolor',[0 0 0]/255,'markerfacecolor',[0 0 0]/255,'markersize',18) % AV passes, No yielding
set(gca,'xlim',[0 15],'ylim',[0 7],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01],'pos',[0.56 0.09 0.42 0.87],'xtick',0:1:15)
xlabel('Elapsed time (s)')
ylabel('Pedestrian position perpendicular to the road (m)')
legend('Road','33 m & On','33 m & Off','43 m & On','43 m & Off','No yielding & Off','eHMI onset, 33 m','eHMI onset, 43 m','AV stops, 33 m','AV stops, 43 m','AV passes, No yielding','location','northwest','autoupdate','off');
h=findobj('FontName','Helvetica'); set(h,'FontSize',18,'Fontname','Arial')
set(gcf,'WindowState','maximized');print(gcf,'Figure6.png','-dpng','-r300');
%% Figure 7. Mean values of perceived risk of the situation (left top), comprehension of the behavior and appearance of the vehicle (right top), trust in the behavior and appearance of the vehicle (left bottom), and the moment the pedestrian entered the road (right bottom)
figure(7)
subplot(2,2,1)
hold on;box on;grid on
h1 = shadedErrorBar(1:26, nanmean(squeeze(MeasuresVehBehavior(1,validpp_odd,6,:))),  nanstd(squeeze(MeasuresVehBehavior(3,validpp_odd,6,:))),  {'Color','k','LineWidth',1}); set(h1.patch, 'FaceAlpha',0.1); % +1 Group
h2 = shadedErrorBar(1:26, nanmean(squeeze(MeasuresVehBehavior(1,validpp_even,6,:))), nanstd(squeeze(MeasuresVehBehavior(3,validpp_even,6,:))), {'Color','m','LineWidth',1}); set(h2.patch, 'FaceAlpha',0.1); % -1 Group
h3 = plot(nanmean(squeeze(MeasuresVehBehavior(1,validpp_odd,6,:))),'k-o','Linewidth',2,'Markersize',13); % +1 Group
h4 = plot(nanmean(squeeze(MeasuresVehBehavior(1,validpp_even,6,:))),'m-o','Linewidth',2,'Markersize',13,'markerfacecolor','m'); % -1 Group
text(3/26.5,1.05,'Block 1','Units','Normalized','horizontalalignment','center')
text(9/26.5,1.05,'Block 2','Units','Normalized','horizontalalignment','center')
text(15/26.5,1.05,'Block 3','Units','Normalized','horizontalalignment','center')
text(18.5/26.5,1.05,'FT 1','Units','Normalized','horizontalalignment','center','color','r')
text(22/26.5,1.05,'Block 4','Units','Normalized','horizontalalignment','center')
text(25.5/26.5,1.05,'FT 2','Units','Normalized','horizontalalignment','center','color','r')
legend([h3 h4],'+1 Group','-1 Group','location','northwest')
ylabel('Risk')
set(gca,'xlim',[0.5 27],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01],'ylim',[1 10],'ytick',1:10,'pos',[0.05 0.58 0.42 0.38],'xtick',[1 6 7 12 13 18 20 25])

subplot(2,2,2)
hold on;box on;grid
h1 = shadedErrorBar(1:26, nanmean(squeeze(MeasuresVehBehavior(2,validpp_odd,6,:))),  nanstd(squeeze(MeasuresVehBehavior(3,validpp_odd,6,:))),  {'Color','k','LineWidth',1}); set(h1.patch, 'FaceAlpha',0.1); % +1 Group
h2 = shadedErrorBar(1:26, nanmean(squeeze(MeasuresVehBehavior(2,validpp_even,6,:))), nanstd(squeeze(MeasuresVehBehavior(3,validpp_even,6,:))), {'Color','m','LineWidth',1}); set(h2.patch, 'FaceAlpha',0.1); % -1 Group
h3 = plot(nanmean(squeeze(MeasuresVehBehavior(2,validpp_odd,6,:))),'k-o','Linewidth',2,'Markersize',13); % +1 Group
h4 = plot(nanmean(squeeze(MeasuresVehBehavior(2,validpp_even,6,:))),'m-o','Linewidth',2,'Markersize',13,'markerfacecolor','m'); % -1 Group
text(3/26.5,1.05,'Block 1','Units','Normalized','horizontalalignment','center')
text(9/26.5,1.05,'Block 2','Units','Normalized','horizontalalignment','center')
text(15/26.5,1.05,'Block 3','Units','Normalized','horizontalalignment','center')
text(18.5/26.5,1.05,'FT 1','Units','Normalized','horizontalalignment','center','color','r')
text(22/26.5,1.05,'Block 4','Units','Normalized','horizontalalignment','center')
text(25.5/26.5,1.05,'FT 2','Units','Normalized','horizontalalignment','center','color','r')
legend([h3 h4],'+1 Group','-1 Group','location','southwest')
ylabel('Comprehension')
set(gca,'xlim',[0.5 27],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01],'ylim',[1 10],'ytick',1:10,'pos',[0.57 0.58 0.42 0.38],'xtick',[1 6 7 12 13 18 20 25])

subplot(2,2,3)
hold on;box on;grid
h1 = shadedErrorBar(1:26, nanmean(squeeze(MeasuresVehBehavior(3,validpp_odd,6,:))),  nanstd(squeeze(MeasuresVehBehavior(3,validpp_odd,6,:))),  {'Color','k','LineWidth',1}); set(h1.patch, 'FaceAlpha',0.1); % +1 Group
h2 = shadedErrorBar(1:26, nanmean(squeeze(MeasuresVehBehavior(3,validpp_even,6,:))), nanstd(squeeze(MeasuresVehBehavior(3,validpp_even,6,:))), {'Color','m','LineWidth',1}); set(h2.patch, 'FaceAlpha',0.1); % -1 Group
h3 = plot(nanmean(squeeze(MeasuresVehBehavior(3,validpp_odd,6,:))),'k-o','Linewidth',2,'Markersize',13); % +1 Group
h4 = plot(nanmean(squeeze(MeasuresVehBehavior(3,validpp_even,6,:))),'m-o','Linewidth',2,'Markersize',13,'markerfacecolor','m'); % -1 Group
text(3/26.5,1.05,'Block 1','Units','Normalized','horizontalalignment','center')
text(9/26.5,1.05,'Block 2','Units','Normalized','horizontalalignment','center')
text(15/26.5,1.05,'Block 3','Units','Normalized','horizontalalignment','center')
text(18.5/26.5,1.05,'FT 1','Units','Normalized','horizontalalignment','center','color','r')
text(22/26.5,1.05,'Block 4','Units','Normalized','horizontalalignment','center')
text(25.5/26.5,1.05,'FT 2','Units','Normalized','horizontalalignment','center','color','r')
legend([h3 h4],'+1 Group','-1 Group','location','southwest')
xlabel('Trial number (eHMI On trials only)')
ylabel('Trust')
set(gca,'xlim',[0.5 27],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01],'ylim',[1 10],'ytick',1:10,'pos',[0.05 0.09 0.42 0.38],'xtick',[1 6 7 12 13 18 20 25])

subplot(2,2,4)
% pre-Failure trial 1
hold on;box on;grid
h1=shadedErrorBar(1:18, nanmean(squeeze(MeasuresVehBehavior(4,validpp_odd,6,1:18))),  nanstd(squeeze(MeasuresVehBehavior(3,validpp_odd,6,1:18))),  {'Color','k','LineWidth',1}); set(h1.patch, 'FaceAlpha',0.1); % +1 Group
h2=shadedErrorBar(1:18, nanmean(squeeze(MeasuresVehBehavior(4,validpp_even,6,1:18))), nanstd(squeeze(MeasuresVehBehavior(3,validpp_even,6,1:18))), {'Color','m','LineWidth',1}); set(h2.patch, 'FaceAlpha',0.1); % -1 Group
plot(1:18,nanmean(squeeze(MeasuresVehBehavior(4,validpp_odd,6,1:18))),'k-o','Linewidth',2,'Markersize',13); % +1 Group
plot(1:18,nanmean(squeeze(MeasuresVehBehavior(4,validpp_even,6,1:18))),'m-o','Linewidth',2,'Markersize',13,'markerfacecolor','m'); % -1 Group

% post-failure trial 1
h1 = shadedErrorBar(20:25, nanmean(squeeze(MeasuresVehBehavior(4,validpp_odd,6,20:25))),  nanstd(squeeze(MeasuresVehBehavior(3,validpp_odd,6,20:25))),  {'Color','k','LineWidth',1}); set(h1.patch, 'FaceAlpha',0.1); % +1 Group
h2 = shadedErrorBar(20:25, nanmean(squeeze(MeasuresVehBehavior(4,validpp_even,6,20:25))), nanstd(squeeze(MeasuresVehBehavior(3,validpp_even,6,20:25))), {'Color','m','LineWidth',1}); set(h2.patch, 'FaceAlpha',0.1); % -1 Group
h3 = plot(20:25,nanmean(squeeze(MeasuresVehBehavior(4,validpp_odd,6,20:25))),'k-o','Linewidth',2,'Markersize',13); % +1 Group
h4 = plot(20:25,nanmean(squeeze(MeasuresVehBehavior(4,validpp_even,6,20:25))),'m-o','Linewidth',2,'Markersize',13,'markerfacecolor','m'); % -1 Group

text(3/26.5,1.05,'Block 1','Units','Normalized','horizontalalignment','center')
text(9/26.5,1.05,'Block 2','Units','Normalized','horizontalalignment','center')
text(15/26.5,1.05,'Block 3','Units','Normalized','horizontalalignment','center')
text(18.5/26.5,1.05,'FT 1','Units','Normalized','horizontalalignment','center','color','r')
text(22/26.5,1.05,'Block 4','Units','Normalized','horizontalalignment','center')
text(25.5/26.5,1.05,'FT 2','Units','Normalized','horizontalalignment','center','color','r')
legend([h3 h4],'+1 Group','-1 Group','location','southwest')
xlabel('Trial number (eHMI On trials only)')
ylabel('Entering road (s)')
set(gca,'xlim',[0.5 27],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01],'pos',[0.57 0.09 0.42 0.38],'xtick',[1 6 7 12 13 18 20 25])
h=findobj('FontName','Helvetica'); set(h,'FontSize',18,'Fontname','Arial')
set(gcf,'WindowState','maximized');print(gcf,'Figure7.png','-dpng','-r300');
%% Figure 9. Mean pedestrian position perpendicular to the road as a function of elapsed time in failure trial 1. A distinction is made between pedestrians who were on the road when the car passed, and pedestrians who were not on the road when the car passed.
PedPositionFT1=squeeze(nanmean(PedestrianZVehBehavior(:,7,13,:),3));

figure(9)
hold on;box on;grid on
patch([-1 17 17 -1],[2.050 2.050 6.239 6.239],[0 0 0],'facealpha',.1,'edgecolor','none')
plot([time(1) time(end)],[-9999 -9999],'k-','Linewidth',2) % plot a dummy line (used for creating the legend)
plot([time(1) time(end)],[-9999 -9999],'r-','Linewidth',5) % plot a dummy line (used for creating the legend)
onindex=find(squeeze(eHMIintensityall(1,37,:))>0,1,'first');
plot(time(onindex),0,'mo','markerfacecolor','m','markersize',17) % plot magenta marker for the eHMI onset moment
plot(time(598),0,'kd','markerfacecolor','k','markersize',17) % plot black marker for the moment the AV passes
plot(time(1:2:end),PedPositionFT1(intersect(validpp,find(PedPositionWhenCarPasses(:,13)> 2.050 & PedPositionWhenCarPasses(:,13)< 6.239)),1:2:end),'r-','Linewidth',5) % on the road
plot(time(1:2:end),PedPositionFT1(intersect(validpp,find(PedPositionWhenCarPasses(:,13)<=2.050 | PedPositionWhenCarPasses(:,13)>=6.239)),1:2:end),'k-','Linewidth',2) % not on the road
set(gca,'xlim',[0 6.5],'ylim',[0 6.5],'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01],'xtick',0:1:15)
xlabel('Elapsed time (s)')
ylabel('Pedestrian position perpendicular to the road (m)')
legend('Road','Not on the road when the car passed','On the road when the car passed','eHMI onset','AV passes','location','northwest','autoupdate','off');
h=findobj('FontName','Helvetica'); set(h,'FontSize',18,'Fontname','Arial')
set(gcf,'WindowState','maximized');print(gcf,'Figure9.png','-dpng','-r300');
%% Table 1. Means of the participants’ results per dependent measure and experimental condition
disp('Mean risk for the 4 blocks, +1 Group')
disp(round(squeeze(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(1,validpp_odd,:,1:5)))),2))
disp('Mean comprehension for the 4 blocks, +1 Group')
disp(round(squeeze(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(2,validpp_odd,:,1:5)))),2))
disp('Mean trust for the 4 blocks, +1 Group')
disp(round(squeeze(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(3,validpp_odd,:,1:5)))),2))
disp('Mean road entering time (s) for the 4 blocks, +1 Group')
disp(round(squeeze(nanmean(squeeze(MeanMeasuresVehBehaviorPerBlock(4,validpp_odd,:,1:5)))),2))

disp('Mean risk for the 4 blocks, -1 Group')
disp(round(squeeze(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(1,validpp_even,:,1:5)))),2))
disp('Mean comprehension for the 4 blocks, -1 Group')
disp(round(squeeze(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(2,validpp_even,:,1:5)))),2))
disp('Mean trust for the 4 blocks, -1 Group')
disp(round(squeeze(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(3,validpp_even,:,1:5)))),2))
disp('Mean road entering time (s) for the 4 blocks, -1 Group')
disp(round(squeeze(nanmean(squeeze(MeanMeasuresVehBehaviorPerBlock(4,validpp_even,:,1:5)))),2))

disp('Mean risk, Comprehension, Trust, Road entering time for Failure trial 1 and 2, +1 Group')
disp(round(squeeze(mean(MeasuresVehBehavior(:,validpp_odd,6,[19 26]),2)),2)')
disp('Mean risk, Comprehension, Trust, Road entering time for Failure trial 1 and 2, -1 Group')
disp(round(squeeze(mean(MeasuresVehBehavior(:,validpp_even,6,[19 26]),2)),2)')
%% -1 vs +1 Group comparison using independent-samples t-tests
[p,t,df]=deal(NaN(1,4));
for measure=1:4
    v1=mean(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_odd,1:3,[1 3])),2),3);  % Extract participant means for Blocks 1-3, eHMI On trials only, +1 Group
    v2=mean(mean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_even,1:3,[1 3])),2),3); % Extract participant means for Blocks 1-3, eHMI On trials only, -1 Group
    [~,p(measure),~,stats]=ttest2(v1,v2);
    t(measure)=stats.tstat;
    df(measure)=stats.df;
end

disp('Group 1 vs +1 comparison, df, t, p for Risk, Comprehension, Trust, and Time for entering road')
disp(df)
disp(round(t,2))
disp(round(p,3))
%% -1 vs +1 Group comparison using independent-samples t-tests, for Failure trial 1 and Failure trial 2
disp('Group 1 vs +1 comparisons for Failure Trial 1: df, t, p')
[~,p,~,stats]=ttest2(MeasuresVehBehavior(1:3,validpp_odd,6,19)',MeasuresVehBehavior(1:3,validpp_even,6,19)'); % Measures 1-3 (only questionnaires), Failure Trial 1 (i.e., 19th eHMI On trial)
disp(stats.df)
disp(round(stats.tstat,2))
disp(round(p,3))
disp('Group 1 vs +1 comparisons for Failure Trial 2: df, t, p')
[~,p,~,stats]=ttest2(MeasuresVehBehavior(1:3,validpp_odd,6,26)',MeasuresVehBehavior(1:3,validpp_even,6,26)'); % Measures 1-3 (only questionnaires), Failure Trial 2 (i.e., 26th eHMI On trial)
disp(stats.df)
disp(round(stats.tstat,2))
disp(round(p,3))
%% Comparison of Block 3 with Block 4 using paired-samples t-tests, for the +1 and -1 Groups separately
[p,t,df]=deal(NaN(4,2));
for measure=1:4
    v1=mean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_odd,3,[1 3])),2); % Extract participant means for Block 3, eHMI On trials only, +1 Group
    v2=mean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_odd,4,[1 3])),2); % Extract participant means for Block 4, eHMI On trials only, +1 Group
    v3=mean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_even,3,[1 3])),2); % Extract participant means for Block 3, eHMI On trials only, -1 Group
    v4=mean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_even,4,[1 3])),2); % Extract participant means for Block 4, eHMI On trials only, -1 Group
    
    [~,p(measure,1),~,stats]=ttest(v1,v2); % +1 Group
    df(measure,1)=stats.df;
    t(measure,1)=stats.tstat;
    
    [~,p(measure,2),~,stats]=ttest(v3,v4); % -1 Group
    df(measure,2)=stats.df;
    t(measure,2)=stats.tstat;
end
disp('Block 3 vs Block 4 comparison of +1 and -1 Group (4 measures')
disp('df')
disp(df)
disp('t')
disp(round(t,2))
disp('p')
disp(round(p,3))
%% Comparison ehMI on with eHMI off (+1 Group)
[p,t,df]=deal(NaN(4,1));

for measure=1:4
    v1=nanmean(squeeze(nanmean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_odd,1:3,[1 3])),2)),2); % Extract participant means for Blocks 1-3, eHMI On yielding trials only, +1 Group
    v2=nanmean(squeeze(nanmean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_odd,1:3,[2 4])),2)),2); % Extract participant means for Blocks 1-3, eHMI Off yielding trials only, +1 Group
    [~,p(measure,1),~,stats1]=ttest(v1,v2); % +1 Group
    t(measure,1)=stats1.tstat; % eHMI on
    df(measure,1)=stats1.df; % eHMI off
 end
disp('eHMI on vs. eHMI off: df, t, and p for all four measures, +1 Group')
disp([[df';round(t',2)];round(p',3)])
%% Comparison ehMI on with eHMI off (-1 Group)
[p,t,df]=deal(NaN(4,1));
for measure=1:4
    v1=nanmean(squeeze(nanmean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_even,1:3,[1 3])),2)),2); % Extract participant means for Blocks 1-3, eHMI On yielding trials only, -1 Group
    v2=nanmean(squeeze(nanmean(squeeze(MeanMeasuresVehBehaviorPerBlock(measure,validpp_even,1:3,[2 4])),2)),2); % Extract participant means for Blocks 1-3, eHMI Off yielding trials only, -1 Group
    [~,p(measure,1),~,stats1]=ttest(v1,v2); % +1 Group
    t(measure,1)=stats1.tstat; % eHMI on
    df(measure,1)=stats1.df; % eHMI off
 end
disp('eHMI on vs. eHMI off: df, t, and p for all four measures, -1 Group')
disp([[df';round(t',2)];round(p',3)])
%% Table S3. Standard deviations of the participants’ results per dependent measure and experimental condition (after exclusion of participants, N = 53)

disp('SD risk for the 4 blocks, +1 Group')
disp(round(squeeze(std(squeeze(MeanMeasuresVehBehaviorPerBlock(1,validpp_odd,:,1:5)))),2))
disp('SD comprehension for the 4 blocks, +1 Group')
disp(round(squeeze(std(squeeze(MeanMeasuresVehBehaviorPerBlock(2,validpp_odd,:,1:5)))),2))
disp('SD trust for the 4 blocks, +1 Group')
disp(round(squeeze(std(squeeze(MeanMeasuresVehBehaviorPerBlock(3,validpp_odd,:,1:5)))),2))
disp('SD road entering time (s) for the 4 blocks, +1 Group')
disp(round(squeeze(nanstd(squeeze(MeanMeasuresVehBehaviorPerBlock(4,validpp_odd,:,1:5)))),2))

disp('SD risk for the 4 blocks, -1 Group')
disp(round(squeeze(std(squeeze(MeanMeasuresVehBehaviorPerBlock(1,validpp_even,:,1:5)))),2))
disp('SD comprehension for the 4 blocks, -1 Group')
disp(round(squeeze(std(squeeze(MeanMeasuresVehBehaviorPerBlock(2,validpp_even,:,1:5)))),2))
disp('SD trust for the 4 blocks, -1 Group')
disp(round(squeeze(std(squeeze(MeanMeasuresVehBehaviorPerBlock(3,validpp_even,:,1:5)))),2))
disp('SD road entering time (s) for the 4 blocks, -1 Group')
disp(round(squeeze(nanstd(squeeze(MeanMeasuresVehBehaviorPerBlock(4,validpp_even,:,1:5)))),2))

disp('SD Risk, Comprehension, Trust, road entering time for Failure trial 1 and 2, +1 Group')
disp(round(squeeze(std(MeasuresVehBehavior(:,validpp_odd,6,[19 26]),[],2)),2)')
disp('SD Risk, Comprehension, Trust, road entering time for Failure trial 1 and 2, -1 Group')
disp(round(squeeze(std(MeasuresVehBehavior(:,validpp_even,6,[19 26]),[],2)),2)')
%% The raw questionnaire data are not provided because of reasons of privacy. Hence, the code is commented out

% [Qn,~,Q]=xlsread('Pre-Experiment Questionnaire.csv','A2:R61');
% QQ(:,1)=Qn(:,1); % Participant number
% QQ(:,2)=Qn(:,2); % What's your age?
% QQ(:,3)=strcmp(Q(:,4),'Female')+2*strcmp(Q(:,4),'Male'); % What's your gender?
% QQ(:,4)=strcmp(Q(:,9),'Less than a Year')+2*strcmp(Q(:,9),'1 - 5 years')+3*strcmp(Q(:,9),'5 - 10 years')+ 4*strcmp(Q(:,9),'More than 10 years'); % How long have you been living in the UK?
% QQ(:,5)=strcmp(Q(:,13),'Left-hand traffic (LHT) (Driving on left lane)')+2*strcmp(Q(:,13),'Right-hand traffic (RHT) (Driving on right lane)')+3*strcmp(Q(:,13),'Both LHT and RHT'); % I'm used to ...
% 
% QQ=sortrows(QQ,1);
% 
% disp(' ')
% disp('All 60 participants')
% disp(['60 participants (' num2str(sum(QQ(:,3)==2)) ' males and '  num2str(sum(QQ(:,3)==1)) ' females) aged between ' num2str(min(QQ(:,2))) ' and ' num2str(max(QQ(:,2))) ' years (M = ' num2str(round(mean(QQ(:,2)),1)) '; SD = ' num2str(round(std(QQ(:,2)),1)) ') took part in the study.'])
% disp([num2str(sum(QQ(:,5)==1)) ' participants were used to left-hand traffic, ' num2str(sum(QQ(:,5)==2)) ' participants were used to right-hand traffic, and the remaining ' num2str(sum(QQ(:,5)==3)) ' participants were used to both left- and right-hand traffic.'])
% 
% disp(' ')
% disp('+1 Group; retained participants')
% disp([num2str(length(validpp_odd)) ' participants (' num2str(sum(QQ(validpp_odd,3)==2)) ' males and '  num2str(sum(QQ(validpp_odd,3)==1)) ' females) aged between ' num2str(min(QQ(validpp_odd,2))) ' and ' num2str(max(QQ(validpp_odd,2))) ' years (M = ' num2str(round(mean(QQ(validpp_odd,2)),1)) '; SD = ' num2str(round(std(QQ(validpp_odd,2)),1)) ') took part in the study.'])
% disp([num2str(sum(QQ(validpp_odd,5)==1)) ' participants were used to left-hand traffic, ' num2str(sum(QQ(validpp_odd,5)==2)) ' participants were used to right-hand traffic, and the remaining ' num2str(sum(QQ(validpp_odd,5)==3)) ' participants were used to both left- and right-hand traffic.'])
% 
% disp(' ')
% disp('-1 Group, retained participants')
% disp([num2str(length(validpp_even)) ' participants (' num2str(sum(QQ(validpp_even,3)==2)) ' males and '  num2str(sum(QQ(validpp_even,3)==1)) ' females) aged between ' num2str(min(QQ(validpp_even,2))) ' and ' num2str(max(QQ(validpp_even,2))) ' years (M = ' num2str(round(mean(QQ(validpp_even,2)),1)) '; SD = ' num2str(round(std(QQ(validpp_even,2)),1)) ') took part in the study.'])
% disp([num2str(sum(QQ(validpp_even,5)==1)) ' participants were used to left-hand traffic, ' num2str(sum(QQ(validpp_even,5)==2)) ' participants were used to right-hand traffic, and the remaining ' num2str(sum(QQ(validpp_even,5)==3)) ' participants were used to both left- and right-hand traffic.'])
