import math
from constraint_functions import *

def get_tow(tow_width, tow_height, element_size):
    """This function takes the tow width, the tow height and the suggested element size as an input.
    First the proposed element size is tested, if the given dimensions do not work with the tow width and tow height
    the element size is updated.
    With the (updated) element size the nodes and elements can be generated. The output of the function is the code
    required by abaqus to generate the tow part and the tow instances"""

    # check if the element size in direction z is compatible with the width of the tow
    # tow length is 2 * tow width. Element size in x and z direction chosen to be equal
    # element size needs to be chosen such that there is a round number of elements/nodes
    dir_z = round(tow_width/element_size + 1)  # no of nodes in z direction
    element_size_z = tow_width / (dir_z - 1)
    if element_size_z != element_size:
        message = f"Element size in direction x and z updated from {element_size} to {element_size_z}"
        print(message)

    element_size_x = element_size_z
    dir_x = 4 * dir_z - 3

    # check if the element size in direction y is compatible with the height of the tow
    # element size needs to be chosen such that there is a round number of elements/nodes
    dir_y = round(tow_height/element_size + 1)  # no of nodes in y direction
    element_size_y = tow_height / (dir_y - 1)
    if element_size_y != element_size:
        message = f"Element size in direction y updated from {element_size} to {element_size_y}"
        print(message)

    no_nodes = 0
    nodes_tow = ""
    towdict = {}

    # the coordinates for each node given the (updated) element dimensions are generated
    # The inner for loop controls the z coordinate (tow width), after one full loop the x coordinate (tow length)
    # is updated. After a full loop in the x-direction the y coordinate (tow height) is updated
    for l in range(dir_y):
        for k in range(dir_x):
            for j in range(dir_z):
                no_nodes += 1  # node number is updated each round

                #   node number,   x-coordinate,   y-coordinate,   z-coordinate
                nodes_tow += f"""     {no_nodes},   {k*element_size_x},   {l*element_size_y},   {j*element_size_z}
"""
                towdict[f"{k*element_size_x},   {l*element_size_y},   {j*element_size_z}"] = f"{no_nodes}"

    q = 1
    no_elements = 0
    elements = ""

    # elements are defined by giving the nodes of each element in the appropriate order.
    # Similarly to the nodes generation the z-direction is taken as the basis
    for p in range(dir_y-1):  # - 1  because there is one less element than nodes
        for m in range(dir_x-1):  # - 1  because there is one less element than nodes
            for n in range(dir_z-1):  # - 1  because there is one less element than nodes
                no_elements += 1

                # element number,   node1_side1,  node2_side1,  node3_side1,  node4_side1
                #                   node1_side2,  node2_side2,  node3_side2,  node4_side2
                elements += (f"{no_elements},   {q},   {q+1},   {q+1+dir_z},   {q+dir_z},   {q+dir_x*dir_z},   {q+dir_x*dir_z+1},"
                             f"   {q+dir_x*dir_z+dir_z+1},   {q+dir_x*dir_z+dir_z} \n")

                # nodes for each side are given in the same continuous pattern, i.e.
                #  el1, 1, 2, 4, 3, 7, 8, 10, 9
                #  el2, 3, 4, 6, 5, 9, 10, 12, 11
                #  1----2           7----8
                #  |    |           |    |
                #  3----4           9----10
                #  |    |           |    |
                #  5----6           11----12
                #  side 1            side 2

                q += 1  # q is used to track the first node for each element, for each additional one in z direction +1
            q += 1  # when going to the next row in x-direction we need to go + 1 nodes
        q += dir_z  # when going to the next row in z-direction we need to jump the whole line of nodes

# the code required to define the part for a tow
    tow_part = f"""*Part, name=Tow1
*Node
{nodes_tow}*Element, type=C3D8R
{elements}*Nset, nset=Set-Tow, generate
   1,  {no_nodes},    1  
*Elset, elset=Set-Tow, generate
   1,  {no_elements},    1
*Orientation, name=Ori-1
          1.,           0.,           0.,           0.,           0.,          1.
1, 0.
** Section: Section-Tow
*Solid Section, elset=Set-Tow, orientation=Ori-1, material=Material-Tow
,
*End Part"""

# the code required to define the four instances in the current RVE
    tow_instance = f"""*Instance, name=Tow1-b, part=Tow1
*End Instance
**  
*Instance, name=Tow2-b, part=Tow1
     0.,     0.,     {tow_width}
*End Instance
**
*Instance, name=Tow3-b, part=Tow1
     0.,     0.,     {2*tow_width}
*End Instance
** 
*Instance, name=Tow4-b, part=Tow1
     0.,     0.,     {3*tow_width}
*End Instance
** 
*Instance, name=Tow1-t, part=Tow1
    0,  {2*tow_height},  0
    0,  {2*tow_height},  0,  {0.5*math.sqrt(2)},  {2*tow_height},  {0.5*math.sqrt(2)},    180.
*End Instance
** 
*Instance, name=Tow2-t, part=Tow1
    {tow_width},  {2*tow_height},  0
    {tow_width},  {2*tow_height},  0,  {tow_width + 0.5*math.sqrt(2)},  {2*tow_height},  {0.5*math.sqrt(2)},    180.
*End Instance
** 
*Instance, name=Tow3-t, part=Tow1
    {2*tow_width},  {2*tow_height},  0
    {2*tow_width},  {2*tow_height},  0,  {2*tow_width + 0.5*math.sqrt(2)},  {2*tow_height},  {0.5*math.sqrt(2)},    180.
*End Instance
** 
*Instance, name=Tow4-t, part=Tow1
    {3*tow_width},  {2*tow_height},  0
    {3*tow_width},  {2*tow_height},  0,  {3*tow_width + 0.5*math.sqrt(2)},  {2*tow_height},  {0.5*math.sqrt(2)},    180.
*End Instance"""
    return tow_part, tow_instance, dir_x, dir_y, dir_z
    # the code for the definition of a tow part and for tow instances is returned


def get_stitches(stitch_order, tow_width, tow_height, stitch_section_full, stitch_area, stitch_section_kd, knockdown,
                 sys):
    """This function takes the placement of the stitches (location and top/bottom) from stitch_order, the tow width,
    the tow height and the area of a stitch as the input. These inputs are used to create to code required to define
    the part(s) for the stitch(es) and the instance(s) for the stitch(es)"""

    stitch_part = ""
    stitch_instance = ""

    for j in range(len(stitch_order)):  # for loop depending on the amount of stitches

        # define stitch instance for each stitch
        stitch_instance += f"""**
*Instance, name=Stitch{j}1, part =Stitch{j}
*End Instance
**
*Instance, name=Stitch{j}2, part =Stitch{j}
*End Instance
"""

        # define stitch part for each stitch
        stitch_part += f"""**
*Part, name=Stitch{j}
*Node
"""
        for i in range(len(stitch_order[j][1])):  # gives for loop depending on the amount of points
            # for this particular stitch j

            if stitch_order[j][1][i][0] == 1:  # if bottom get coordinates from bottom dictionary
                x_coord = 0
            elif stitch_order[j][1][i][0] == 2:  # if top get coordinates from top dictionary
                x_coord = tow_width
            elif stitch_order[j][1][i][0] == 3:  # if top get coordinates from top dictionary
                x_coord = 2*tow_width
            elif stitch_order[j][1][i][0] == 4:  # if top get coordinates from top dictionary
                x_coord = 3*tow_width
            elif stitch_order[j][1][i][0] == 5:  # if top get coordinates from top dictionary
                x_coord = 4*tow_width
            else:
                sys.exit(f"{stitch_order[j][1][i][0]} is not a valid input. Input should be 1, 2, 3, 4 or 5")

            if stitch_order[j][1][i][1] == 1:  # if bottom get coordinates from bottom dictionary
                z_coord = 0
            elif stitch_order[j][1][i][1] == 2:  # if top get coordinates from top dictionary
                z_coord = tow_width
            elif stitch_order[j][1][i][1] == 3:  # if top get coordinates from top dictionary
                z_coord = 2*tow_width
            elif stitch_order[j][1][i][1] == 4:  # if top get coordinates from top dictionary
                z_coord = 3*tow_width
            elif stitch_order[j][1][i][1] == 5:  # if top get coordinates from top dictionary
                z_coord = 4*tow_width
            else:
                sys.exit(f"{stitch_order[j][1][i][1]} is not a valid input. Input should be 1, 2, 3, 4 or 5")

            if stitch_order[j][1][i][2] == 'b':  # if bottom get coordinates from bottom dictionary
                y_coord = 0
            elif stitch_order[j][1][i][2] == 't':  # if top get coordinates from top dictionary
                y_coord = 2*tow_height
            else:
                sys.exit(f"{stitch_order[j][1][i][1]} is not a valid input. Input should be 'b' or 't'")

            stitch_part += f"""     {i + 1},     {x_coord},     {y_coord},     {z_coord} \n"""

        stitch_part += f"""*Element, type = T3D2
"""
        for i in range(len(stitch_order[j][1])-1):  # no of connections/elements is one less than the amount of points

            stitch_part += f"""    {i+1},  {i+1},  {i+2}\n"""

        stitch_part += f"""*Elset, elset = Set-Stitch-full{j} 
{stitch_section_full[j][1]}
** Section: SectionStitch-full{j}
*Solid Section, elset = Set-Stitch-full{j}, material = Material-Stitch
{stitch_area},
*Elset, elset = Set-Stitch-kd{j}
{stitch_section_kd[j][1]}
** Section: SectionStitch-kd{j}
*Solid Section, elset = Set-Stitch-kd{j}, material = Material-Stitch
{stitch_area*knockdown},
*End Part
"""

    return stitch_part, stitch_instance  # return the code required for the stitch parts and instances

def get_constraints(stitch_order, dir_x, dir_z, sys):
    slave_sets = ""
    master_sets = ""
    slave_surface = ""
    master_surface = ""
    constraints = ""

    for j in range(len(stitch_order)):  # for loop depending on the amount of stitches

        node_dict1 = {"1": 1, "2": 1 + (dir_z - 1) * dir_z, "3": 1 + (dir_z - 1) * 2*dir_z,
                      "4": 1 + (dir_z - 1) * 3 * dir_z, "5": 1 + (dir_z - 1) * 4 * dir_z}
        node_dict2 = {"1": dir_z, "2": dir_z * dir_z, "3": (dir_z + dir_z - 1) * dir_z,
                      "4": (2*dir_z + dir_z - 2) * dir_z,  "5": (3*dir_z + dir_z - 3) * dir_z}
        counter = 1

        for k in stitch_order[j][1]:
            print(k)
            if k[2] == 't':
                if k[0] == 1:
                    toptow = [1, 1]
                    node = [node_dict1[str(k[1])], node_dict1[str(k[1])]]
                elif k[0] == 2:
                    toptow = [1, 2]
                    node = [node_dict2[str(k[1])], node_dict1[str(k[1])]]
                elif k[0] == 3:
                    toptow = [2, 3]
                    node = [node_dict2[str(k[1])], node_dict1[str(k[1])]]
                elif k[0] == 4:
                    toptow = [3, 4]
                    node = [node_dict2[str(k[1])], node_dict1[str(k[1])]]
                elif k[0] == 5:
                    toptow = [4, 4]
                    node = [node_dict2[str(k[1])], node_dict2[str(k[1])]]
                else:
                    print("This is not an accepted value")

                master_sets += f"""*Nset, nset=m_Set-{counter}-{j}1, instance=TOW{toptow[0]}-{k[2]}
{node[0]},
*Nset, nset=m_Set-{counter}-{j}2, instance=TOW{toptow[1]}-{k[2]}
{node[1]},
"""
            if k[2] == 'b':
                if k[1] == 1:
                    bottow = [1, 1]
                    node = [node_dict1[str(k[0])], node_dict1[str(k[0])]]
                elif k[1] == 2:
                    bottow = [1, 2]
                    node = [node_dict2[str(k[0])], node_dict1[str(k[0])]]
                elif k[1] == 3:
                    bottow = [2, 3]
                    node = [node_dict2[str(k[0])], node_dict1[str(k[0])]]
                elif k[1] == 4:
                    bottow = [3, 4]
                    node = [node_dict2[str(k[0])], node_dict1[str(k[0])]]
                elif k[1] == 5:
                    bottow = [4, 4]
                    node = [node_dict2[str(k[0])], node_dict2[str(k[0])]]
                else:
                    print("This is not an accepted value")

                master_sets += f"""*Nset, nset=m_Set-{counter}-{j}1, instance=TOW{bottow[0]}-{k[2]}
{node[0]},
*Nset, nset=m_Set-{counter}-{j}2, instance=TOW{bottow[1]}-{k[2]}
{node[1]},
"""

            master_surface += f"""*Surface, type=NODE, name=m_Set-{counter}-{j}1_CNS_, internal
m_Set-{counter}-{j}1, 1.
*Surface, type=NODE, name=m_Set-{counter}-{j}2_CNS_, internal
m_Set-{counter}-{j}2, 1.
"""

            slave_sets += f"""*Nset, nset=s_Set-{counter}-{j}1, instance = Stitch{j}1
{counter},
*Nset, nset=s_Set-{counter}-{j}2, instance = Stitch{j}2
{counter},
"""

            slave_surface += f"""*Surface, type=NODE, name=s_Set-{counter}-{j}1_CNS_, internal
s_Set-{counter}-{j}1, 1.
*Surface, type=NODE, name=s_Set-{counter}-{j}2_CNS_, internal
s_Set-{counter}-{j}2, 1.
"""

            constraints += f"""** Constraint: Constraint-{j}-{counter}1
*Tie, name=Constraint-{j}-{counter}1, adjust=yes, position tolerance=0.0001
s_Set-{counter}-{j}1_CNS_, m_Set-{counter}-{j}1_CNS_
** Constraint: Constraint-{j}-{counter}2
*Tie, name=Constraint-{j}-{counter}2, adjust=yes, position tolerance=0.0001
s_Set-{counter}-{j}2_CNS_, m_Set-{counter}-{j}2_CNS_
"""

            counter += 1

    return slave_sets, master_sets, slave_surface, master_surface, constraints

def get_interactions(dir_x, dir_z, dir_y):
    int_surface = ""
    int_def = ""

    int_surface = f"""*Elset, elset = _CP1_1-b-S4, internal, instance = tow1-b, generate
    {dir_z-1}, {(dir_x-1)*(dir_z-1)*(dir_y-1)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP1_1-b
_CP1_1-b-S4, S4
*Elset, elset = _CP1_2-b-S6, internal, instance = tow2-b, generate
    1, {(dir_x-1)*(dir_z-1)*(dir_y-1)-(dir_z-2)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP1_2-b
_CP1_2-b-S6, S6
*Elset, elset = _CP1_2-b-S4, internal, instance = tow2-b, generate
    {dir_z-1}, {(dir_x-1)*(dir_z-1)*(dir_y-1)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP1_2-b2
_CP1_2-b-S4, S4
*Elset, elset = _CP1_3-b-S6, internal, instance = tow3-b, generate
    1, {(dir_x-1)*(dir_z-1)*(dir_y-1)-(dir_z-2)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP1_3-b
_CP1_3-b-S6, S6
*Elset, elset = _CP1_3-b-S4, internal, instance = tow3-b, generate
    {dir_z-1}, {(dir_x-1)*(dir_z-1)*(dir_y-1)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP1_3-b2
_CP1_3-b-S4, S4
*Elset, elset = _CP1_4-b-S6, internal, instance = tow4-b, generate
    1, {(dir_x-1)*(dir_z-1)*(dir_y-1)-(dir_z-2)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP1_4-b
_CP1_4-b-S6, S6
*Elset, elset=_CP2_1-b_S2, internal, instance=tow1-b, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_1-b
_CP2_1-b_S2, S2
*Elset, elset=_CP2_2-b_S2, internal, instance=tow2-b, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_2-b
_CP2_2-b_S2, S2
*Elset, elset=_CP2_3-b_S2, internal, instance=tow3-b, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_3-b
_CP2_3-b_S2, S2
*Elset, elset=_CP2_4-b_S2, internal, instance=tow4-b, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_4-b
_CP2_4-b_S2, S2
*Elset, elset=_CP2_1-t_S2, internal, instance=tow1-t, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_1-t
_CP2_1-t_S2, S2
*Elset, elset=_CP2_2-t_S2, internal, instance=tow2-t, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_2-t
_CP2_2-t_S2, S2
*Elset, elset=_CP2_3-t_S2, internal, instance=tow3-t, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_3-t
_CP2_3-t_S2, S2
*Elset, elset=_CP2_4-t_S2, internal, instance=tow4-t, generate
    {(dir_x-1)*(dir_z-1)*(dir_y-2)+1},  {(dir_x-1)*(dir_z-1)*(dir_y-1)},   1
*Surface, type=ELEMENT, name=S2_4-t
_CP2_4-t_S2, S2
*Elset, elset = _CP6_1-t-S4, internal, instance = tow1-t, generate
    {dir_z-1}, {(dir_x-1)*(dir_z-1)*(dir_y-1)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP6_1-t
_CP6_1-t-S4, S4
*Elset, elset = _CP6_2-t-S6, internal, instance = tow2-t, generate
    1, {(dir_x-1)*(dir_z-1)*(dir_y-1)-(dir_z-2)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP6_2-t
_CP6_2-t-S6, S6
*Elset, elset = _CP6_2-t-S4, internal, instance = tow2-t, generate
    {dir_z-1}, {(dir_x-1)*(dir_z-1)*(dir_y-1)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP6_2-t2
_CP6_2-t-S4, S4
*Elset, elset = _CP6_3-t-S6, internal, instance = tow3-t, generate
    1, {(dir_x-1)*(dir_z-1)*(dir_y-1)-(dir_z-2)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP6_3-t
_CP6_3-t-S6, S6
*Elset, elset = _CP6_3-t-S4, internal, instance = tow3-t, generate
    {dir_z-1}, {(dir_x-1)*(dir_z-1)*(dir_y-1)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP6_3-t2
_CP6_3-t-S4, S4
*Elset, elset = _CP6_4-t-S6, internal, instance = tow4-t, generate
    1, {(dir_x-1)*(dir_z-1)*(dir_y-1)-(dir_z-2)}, {dir_z-1}
*Surface, type = ELEMENT, name=CP6_4-t
_CP6_4-t-S6, S6
"""

    int_def = """** Interaction: CP1-1-B-2-B
*Contact Pair, interaction=INTPROP-00, type=SURFACE TO SURFACE, adjust=0.00001
CP1_2-b, CP1_1-b
** Interaction: CP1-1-B2-2-B
*Contact Pair, interaction=INTPROP-00, type=SURFACE TO SURFACE, adjust=0.00001
CP1_3-b, CP1_2-b2
** Interaction: CP1-1-B2-2-B
*Contact Pair, interaction=INTPROP-00, type=SURFACE TO SURFACE, adjust=0.00001
CP1_4-b, CP1_3-b2
** Interaction: CP2-1-T-1-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_1-t, S2_1-b
** Interaction: CP2-1-T-2-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_1-t, S2_2-b
** Interaction: CP2-1-T-3-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_1-t, S2_3-b
** Interaction: CP2-1-T-4-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_1-t, S2_4-b
** Interaction: CP2-2-T-1-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_2-t, S2_1-b
** Interaction: CP2-2-T-2-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_2-t, S2_2-b
** Interaction: CP2-2-T-3-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_2-t, S2_3-b
** Interaction: CP2-2-T-4-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_2-t, S2_4-b
** Interaction: CP2-3-T-1-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_3-t, S2_1-b
** Interaction: CP2-3-T-2-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_3-t, S2_2-b
** Interaction: CP2-3-T-3-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_3-t, S2_3-b
** Interaction: CP2-3-T-4-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_3-t, S2_4-b
** Interaction: CP2-4-T-1-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_4-t, S2_1-b
** Interaction: CP2-4-T-2-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_4-t, S2_2-b
** Interaction: CP2-4-T-3-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_4-t, S2_3-b
** Interaction: CP2-4-T-4-B
*Contact Pair, interaction=INTPROP-090, type=SURFACE TO SURFACE, adjust=0.00001
S2_4-t, S2_4-b
** Interaction: CP6-1-T-2-T
*Contact Pair, interaction=INTPROP-00, type=SURFACE TO SURFACE, adjust=0.00001
CP6_2-t, CP6_1-t
** Interaction: CP6-2-T2-3-T
*Contact Pair, interaction=INTPROP-00, type=SURFACE TO SURFACE, adjust=0.00001
CP6_3-t, CP6_2-t2
** Interaction: CP6-3-T2-4-T
*Contact Pair, interaction=INTPROP-00, type=SURFACE TO SURFACE, adjust=0.00001
CP6_4-t, CP6_3-t2"""
    return int_surface, int_def

def get_rve(tow_width, tow_height, dir_x, dir_y, dir_z, module):
## Creating Ref. Points ##

    # Node 1 = RP1: Rigid body movement Y-axis
    # Node 2 = RP2: Rigid body movement Z-axis
    # Node 3 = RP3: Rigid body movement X-axis
    # Node 4 = RP4: G12
    # Node 5 = RP5: G13
    # Node 6 = RP6: G23

    refpoints = f"""*Node
      1,       {2*tow_width},       {2.4*tow_height},         {2*tow_width}
*Node
      2,       {2*tow_width},       {tow_height},         {4.4*tow_width}
*Node
      3,       {4.8*tow_width},       {tow_height},         {2*tow_width}
*Node
      4,       {5.6*tow_width},       {tow_height},         {2*tow_width}
*Node
      5,       {6.4*tow_width},       {tow_height},         {2*tow_width}
*Node
      6,       {7.2*tow_width},         {tow_height},         {2*tow_width}
*Nset, nset=RP1
 1,
*Nset, nset=RP2
 2,
*Nset, nset=RP3
 3,
*Nset, nset=RP4
 4,
*Nset, nset=RP5
 5,
*Nset, nset=RP6
 6,"""
    setone = ""
    setonedof = ""
    setoneno = ""
    altsetone = ""
    settwo = ""
    settwodof = ""
    settwono = ""
    altsettwo = ""
    setthree = ""
    setthreedof = ""
    setthreeno = ""
    altsetthree = ""

    if module == "E":
        setoneno = [3, 2, 2]
        setonedof = [1, 2, 3]
        setone = ["""
RP4, 1, -1.""", "", ""]
        altsetone = ["""
RP4, 1, 1.""", "", ""]

        settwono = [2, 3, 2]
        settwo = ["", """
RP5, 2, -1.""", ""]
        altsettwo = ["", """
RP5, 2, 1.""", ""]
        settwodof = [1, 2, 3]

        setthreeno = [2, 2, 3]
        setthree = ["", "", """
RP6, 3, -1."""]
        altsetthree = ["", "", """
RP6, 3, 1."""]
        setthreedof = [1, 2, 3]

    elif module == "G":
        setoneno = [3, 3, 3]
        setonedof = [1, 2, 3]
        setone = ["""
RP3, 1, -1.""", """
RP4, 2, -1.""", """
RP5, 3, -1."""]
        altsetone = ["""
RP3, 1, 1.""", """
RP4, 2, 1.""", """
RP5, 3, 1."""]

        settwono = [3, 3, 3]
        settwo = ["""
RP4, 1, -1.""", """
RP1, 2, -1.""", """
RP6, 3, -1."""]
        altsettwo= ["""
RP4, 1, 1.""", """
RP1, 2, 1.""", """
RP6, 3, 1."""]
        settwodof = [1, 2, 3]

        setthreeno = [3, 3, 3]
        setthree = ["""
RP5, 1, -1.""", """
RP6, 2, -1.""", """
RP2, 3, -1."""]
        altsetthree = ["""
RP5, 1, 1.""", """
RP6, 2, 1.""", """
RP2, 3, 1."""]
        setthreedof = [1, 2, 3]

    towset = ["1-t", "2-t", "3-t", "4-t", "1-b", "2-b", "3-b", "4-b"]

    bc_topbot, rve_topbot = topbot(towset, dir_x, dir_z, settwo, altsettwo, settwono, settwodof)
    bc_rightleft, rve_rightleft = rightleft(dir_x, dir_y, dir_z, setthree, altsetthree, setthreeno, setthreedof)
    bc_frontback, rve_frontback = frontback(dir_x, dir_y, dir_z, setone, altsetone, setoneno, setonedof)
    bc_edges1, rve_edges1 = edges1(dir_x, dir_y, dir_z, setone, altsetone, setoneno, setonedof)
    bc_edges2, rve_edges2 = edges2(dir_x, dir_z, settwo, altsettwo, settwono, settwodof)
    bc_edges3, rve_edges3 = edges3(dir_x, dir_y, dir_z, setthree, altsetthree, setthreeno, setthreedof)
    rve_C1_C7 = C1_C7(module)

    bc_constraints = f"""{bc_topbot}{bc_frontback}{bc_rightleft}{bc_edges1}{bc_edges2}{bc_edges3}"""
    rve_constraints = f"{rve_topbot}{rve_frontback}{rve_rightleft}{rve_edges1}{rve_edges2}{rve_edges3}{rve_C1_C7}"
    return refpoints, bc_constraints, rve_constraints


