% Check whether the model reproduces the Crabtree effect at the correct
% dilution rate by simulating a chemostat at different dilution rates.
clear all; close all; clc;

% Proteome allocation
phi_ba = [0.0057    0.0120    0.1166    0.0464    0.0068    0.0411    0.0018    0.7079    0.0618];

f_seed = phi_ba;

% Initial conditions
c0 = [1e-6 100 0 0 6.36 0 33.12 0 0 0 0];     % mM

% Chemostat conditions
tcycle = 1; tfeed = 1;

% Dilution rate range
D = [0.4:-0.01:0.26    0.25:-0.025:0.05];

% Monte-Carlo iterations
n = 1000; p = 0.5;

t_runs = []; upd = [];
for i = 1:length(D)

    % Reset the objective function
    c_opt = 5; 
    
    % Set timeframe
    ti = 0; tf = 20/D(i);        % h
    tspan = [ti tf];
    
    % Monte-Carlo optimizer
    for j = 1:n
        % Generate random proteome
        f = f_seed .* (1 + p.*(rand(1,9)-0.5));
        f = f./sum(f);
        
        % Solve system of ODE's
        options = odeset('NonNegative',1:numel(c0),'AbsTol',1e-4);
        [t, c] = ode15s(@(t,c) protmdl(t,c,f,D(i),tcycle,tfeed), tspan, c0, options);
        [~,v(i,:)] = protmdl(t(end),c(end,:)',f,D(i),tcycle,tfeed);
        
        % Check for imbalanced states
        if max(c(:,7)) < 1
            % If phosphate is never high, it must be imbalanced
            unb = 1;
        elseif max(c(:,8)) > 10
            % If intermediates reach high concentrations, there must be imbalance
            unb = 1;
        elseif max(c(:,9)) > 10
            unb = 1;
        elseif max(c(:,10)) > 10
            unb = 1;
        elseif isreal(c) == 0
            unb = 1;
        elseif max(v(:,10)) < 0.99*D(i)
            unb = 1;
        elseif min(c(:,2)) < 1e-3;
            unb = 1;
        else
            unb = 0;
        end
        
        % The seed proteome is updated if the objective function is lower than
        % the previous seed & if no imbalanced state is reached
         if c(end,1) < c_opt && unb == 0
            f_seed  = f;
            c_opt   = c(end,1);
            cstore = c(end,:);
            
            % Calculate q-rates
            q = [v(i,1)...                                  % Glucose
                 v(i,4)...                                  % Ethanol 
                 v(i,5)...                                  % Glycerol
                 0.5*v(i,7) + 0.5*v(i,6)...                 % O2
                 v(i,4) + 3.*v(i,6) + 0.2954.*v(i,10)...    % CO2
                 v(i,10)];                                  % Biomass
             
            ustore = v(end,10);
            vstore = v(end,:);

            % Keep a list of update iterations (diagnostic purposes)
            upd(end+1) = i;

            disp('Update') 
            
        end

    runsleft = (length(D)-i)*n + (n-j);
    disp(['Simulation ' num2str(j) ' of ' num2str(n) ' complete, opt ' num2str(i) ' of ' num2str(length(D))])
    
    % Return the optimal proteome, qrates and glucose concentration
    fvals(i,:) = f_seed;
    csvals(i,1) = c_opt;
  
    end
    
    if exist('q')
        qvals(i,:) = q;
        cvals(i,:) = cstore;
        uvals(i,:) = ustore;
        vvals(i,:) = vstore;
    else
        qvals(i,:) = NaN(1,6);
        cvals(i,:) = NaN(1,11);
        uvals(i,:) = NaN;
        vvals(i,:) = NaN(1,11);
    end
end

% Save workspace
save('ProteomeOptimization_Crabtree_X.mat');

% Plot biomass specific rates
figure(1)
names = {'Glucose consumption', 'Ethanol production', 'Glycerol production', 'O_2 consumption', 'CO_2 production', 'Biomass production'};
for i = 1:5
    subplot(2,3,i)
    plot(D, qvals(:,i), 'LineWidth', 1.2, 'Color', winter(1)); hold on
    xlabel('Dilution rate (h^{-1})')
    ylabel(['(mol/mol_x/h)'])
    title([names{i}, ' rate'])
end
subplot(2,3,6)
plot(D, csvals, 'LineWidth', 1.2, 'Color', winter(1))
ylabel('Residual Glucose concentration (mM)'); xlabel('Dilution rate (1/h)')
ylim([0 1.1*max(csvals)])
xlim([0 max(D)])
title('Residual Glucose')

% Plot proteomes
figure(2)
sectornames = {'Upt', 'Uglc', 'Lglc', 'Ferm', 'ESnk', 'Resp' 'Treh', 'Grwt', 'Struc'};
for i = 1:9
    subplot(3,3,i)
    plot(D,fvals(:,i), '.', 'LineWidth', 1.2, 'Color', winter(1)); hold on
    ylabel('Proteome sector size'); xlabel('Dilution rate (1/h)')
    ylim([0 1.1*max(fvals(:,i))])
    xlim([0 max(D)])
    title(sectornames{i})
    tix = get(gca, 'ytick')';
    set(gca,'yticklabel',num2str(tix,'%.3f'))
end

% qs as a function of qetoh and qo2
figure(3)
plot(qvals(:,1), qvals(:,2), 'LineWidth', 1.2, 'Color', winter(1)); hold on
plot(qvals(:,1), qvals(:,4), 'LineWidth', 1.2, 'Color', cool(1))
xlabel('Glucose uptake rate (mol/mol_x/h)')
ylabel('Rate (mol/mol_x/h)')
legend('q_{EtOH}', 'q_{O2}')

% Plot growth rate
figure(4)
plot(D, uvals, 'LineWidth', 1.2, 'Color', winter(1))
ylabel('Growth rate (h^{-1})'); xlabel('Dilution rate (1/h)')

