% Use a Monte-Carlo approach to optimize for the most competitive proteome
% for feast/famine conditions. The model of Siem is adapted to correct for
% the amount of energy & substrate invested to adapt the proteome.

clear all; close all; clc;

% Set operation parameters
D = 0.05;           % 1/h
tcycle = 1;         % h
tfeed = 1;          % h   
tend = 100;         % h

% Set amound of iterations
n = 1000;

% Set search range around seed proteome 
% (p = 2 equals 100% variation)
p = 0.5;

% Seed proteome (obtained from proteome data on glucose and De Godoy(2008))
phi_ba = [0.0057    0.0120    0.1166    0.0464    0.0068    0.0411    0.0018    0.7079    0.0618];

phi = phi_ba; %ones(1,9)./9;
f_seed = phi;

% Set initial objective function
c_opt_i = 5; %cs (mM)
c_opt = c_opt_i;

t_runs = []; upd = [];
for i = 1:n
    % Generate random proteome
    f = f_seed.*(1+p.*(rand(1,length(f_seed))-0.5));
    f = f./sum(f);
    
    % Initial conditions
    c0 = [1e-6 100 0 0 6.36 0 33.12 0 0 0 0];
    
    % Conditions chemostat
    chsttime = 100;
    
    % Run chemostat 
    options = odeset('NonNegative',1:numel(c0),'AbsTol',1e-4);
    [t,c] = ode15s(@(t,c) protmdl(t,c,f,D,1,1),[0 chsttime],c0,options);
    
    % Retrieve fluxes at all timepoints
    v = [];
    for k = 1:length(t)
        [dcdt, vs] = protmdl(t(k),c(k,:)',phi,D,tcycle,tfeed);
        v = [v; vs'];
    end

    % Calculate objective function based on last cycle
    cs_t_avg = trapz(t,c(:,1).*t)/trapz(t,t);
    
    % Check for imbalanced states
    if max(c(:,7)) < 1
        % If phosphate is never high, it must be imbalanced
        unb = 1;
    elseif max(c(:,8)) > 10
        % If intermediates reach high concentrations, there must be imbalance
        unb = 1;
    elseif max(c(:,9)) > 10
        unb = 1;
    elseif max(c(:,10)) > 10
        unb = 1;
    elseif isreal(c) == 0
        unb = 1;
    elseif max(v(:,10)) < 0.99*D
        % If growth rate is never near dilution rate; this is also often
        % where all fluxes are equal to zero
        unb = 1;
    else
        unb = 0;
    end
    
    % The seed proteome is updated if the objective function is lower than
    % the previous seed & if no imbalanced state is reached
    if cs_t_avg < c_opt && unb == 0
        f_seed  = f;
        c_opt   = cs_t_avg;
        f_opt   = f;
        
        % Keep a list of update iterations (diagnostic purposes)
        upd(end+1) = i; 
        disp('Update')         
    end

    % Give an estimate of the remaining simulation time
    disp(['Simulation ' num2str(i) ' of ' num2str(n) ' complete'])
end

% Final result
disp(['Objective function, time-weighted average susbtrate concentration = ', num2str(c_opt), 'mM'])

% Make a proteomap
figure(1)
names = {'Upt', 'Uglc', 'Lglc', 'Ferm', 'ESnk', 'Resp' 'Treh', 'Grwt', 'Struc'};
textscale = 0.3;
Proteomap(f_opt,names,textscale)

% Save workspace
save('ProteomeOptimization_CH_X');

%% Simulate with optimized proteome

% Solve system of ODE's
options = odeset('NonNegative',1:numel(c0),'AbsTol',1e-4);
[t, c] = ode15s(@(t,c) protmdl(t,c,f_opt,D,1,1), [0 chsttime], c0, options);

% Retrieve fluxes at all timepoints
v = [];
for i = 1:length(t)
    [dcdt, vs] = protmdl(t(i),c(i,:)',f_opt,D,1,1);
    v = [v; vs'];
end

% Concentrations
figure(2)
cnames = {'Glucose', 'Biomass', 'EtOH', 'Glycerol', 'ATP', 'NADH', 'Pi', 'G6P', 'FBP', 'Pyr', 'Trehalose'};
for i = 1:size(c,2)
    subplot(3,4,i)
    plot(t, c(:,i), 'LineWidth', 1.2); hold on
    xlabel('time (h)')
    ylabel('C (mol/m^3)')
    title(cnames{i})
end

% Fluxes
figure(3)
vnames = {'Upt', 'UGlc', 'LGlc', 'Ferm', 'ESnk', 'Resp, TCA', 'Resp, NDE', 'Trsn', 'Trdg', 'Grwt', 'Mtn'};
for i = 1:size(v,2)
    subplot(3,4,i)
    plot(t, v(:,i), 'LineWidth', 1.2); hold on
    xlabel('time (h)')
    ylabel('v (mol/mol_x/h)')
    title(vnames{i})
end