% Use a Monte-Carlo approach to optimize for the most competitive proteome
% for feast/famine conditions. The model of Siem is adapted to correct for
% the amount of energy & substrate invested to adapt the proteome.
% NO STRESS RESPONSE
clear all; close all; clc;

% Set operation parameters
D = 0.1;            % 1/h
tcycle = 400/3600;  % h
tfeed = 20/3600;    % h   
tend = 25;
numcycles = tend/tcycle;

% Set amound of iterations
n = 10;

% Set search range around seed proteome 
% (p = 2 equals 100% variation)
p = 0.5;

% Seed proteome (obtained from proteome data on glucose and De Godoy(2008))
% Choose the one that is most applicable.
phi_ba = [0.016    0.05    0.2    0.0464    0.0068    0.0411    0.0018    0.7079    0.0618];

phi = phi_ba;
f_seed = phi;

% Set initial objective function
c_opt_i = 10; %cs (mM)
c_opt = c_opt_i;

t_runs = []; upd = [];
for i = 1:n
    % Generate random proteome with minimal sector size for struc
    f = f_seed.*(1+p.*(rand(1,length(f_seed))-0.5));
    fstrucmin = 0.0618;
    f(9) = max(f(9), fstrucmin);
    f = f./sum(f);

    % Initial conditions
    c0 = [1e-6 100 0 0 6.36 0 33.12 0 0 0 0];
    
    % Conditions chemostat
    chsttime = 100;
    
    % Run chemostat to obtain steady-state concentrations for initial conditions
    options = odeset('NonNegative',1:numel(c0),'AbsTol',1e-4);
    [t,c] = ode15s(@(t,c) protmdl(t,c,f,D,1,1),[0 chsttime],c0,options);
    c0 = c(end,:); 

    % Set initial values for next cycle
    c0 = c(end,:);   
    
    % Stress response penalty based on chemostat
    f_mtn_req = 0.0203/(1+(c0(1)/0.02)^21)+0.0618;
    delta_f = f(9) - f_mtn_req;
    if delta_f < 0 
        penalty = abs(delta_f)*100*0.0103;
    else
        penalty = 0;
    end

    % Don't switch off fermentation in the first cycle
    fermsw = 1;
    
    % Simulate the feast/famine cycles piecewise
    % Only the data of the last cycle is saved.
    for j = 1:numcycles
        
        % Simulate i-th cycle for feast/famine proteome
        [t,c] = ode15s(@(t,c) protmdl_ff(t,c,f,D,tcycle,tfeed,penalty,fermsw),[0 tcycle],c0,options);
        
        % Retrieve fluxes at all timepoints
        v = [];
        for k = 1:length(t)
            [dcdt, vs] = protmdl_ff(t(k),c(k,:)',phi,D,tcycle,tfeed,penalty,fermsw);
            v = [v; vs'];
        end
        
        % Set initial values for next cycle
        c0 = c(end,:);
        
        % Stress response penalty based on the maximum and minimum residual
        % glucose concentration for the next cycle
        cav = (max(c(:,1)) + min(c(:,1)))/2;
        f_mtn_req = 0.0203/(1+(cav(1)/0.02)^21)+0.0618;
        delta_f = f(9) - f_mtn_req;
        if delta_f < 0 
            penalty = abs(delta_f)*100*0.0103;
        else
            penalty = 0;
        end

        % Switch off fermentation if the maximum glucose concentration in the
        % previous cycle is too low
        if max(c(:,1)) < 0.085           % mM
            fermsw = 0;
        else
            fermsw = 1;
        end
        
    end
    
    % Calculate objective function based on last cycle
    % (only if ode is solved)
    if length(t)>1
        cs_t_avg = trapz(t,c(:,1).*t)/trapz(t,t);
    else
        cs_t_avg = c_opt + 10;
    end
    
    % Check for imbalanced states
    if max(c(:,7)) < 1
        % If phosphate is never high, it must be imbalanced
        unb = 1;
    elseif max(c(:,8)) > 10
        % If intermediates reach high concentrations, there must be imbalance
        unb = 1;
    elseif max(c(:,9)) > 10
        unb = 1;
    elseif max(c(:,10)) > 10
        unb = 1;
    elseif isreal(c) == 0
        unb = 1;
    elseif max(v(:,10)) < 0.99*D
        % If growth rate is never near dilution rate; this is also often
        % where all fluxes are equal to zero
        unb = 1;
    else
        unb = 0;
    end
    
    % The seed proteome is updated if the objective function is lower than
    % the previous seed & if no imbalanced state is reached
    if cs_t_avg < c_opt && unb == 0
        f_seed  = f;
        c_opt   = cs_t_avg;
        f_opt   = f;
        
        % Keep a list of update iterations (diagnostic purposes)
        upd(end+1) = i; 
        disp('Update')         
    end

    % Give an estimate of the remaining simulation time
    disp(['Simulation ' num2str(i) ' of ' num2str(n) ' complete'])

end

% Final result
disp(['Objective function, time-weighted average susbtrate concentration = ', num2str(c_opt), 'mM'])


%% Make a proteomap
figure(1)
names = {'Upt', 'Uglc', 'Lglc', 'Ferm', 'ESnk', 'Resp' 'Treh', 'Grwt', 'Struc'};
textscale = 0.3;
Proteomap(f_opt,names,textscale)

% Save workspace
save('ProteomeOptimization_FF_X');