% MATLAB code for the simulations presented in the paper:
% De Winter, J. C., & Dodou, D. (2010). Five-point Likert items: t test versus Mann-Whitney-Wilcoxon. Practical Assessment, Research & Evaluation, 15, 1-12.
% This code was tested with Matlab versions R2008b and re-tested with Matlab R0218b

clc;clear all;close all %#ok<CLALL>
%%
D= [.00 .01 .03 .06 .90   % 1. Very strongly agree
    .01 .03 .06 .30 .60   % 2. Strongly agree
    .05 .10 .20 .45 .20   % 3. Agree peak
    .10 .15 .20 .30 .25   % 4. Agree flat
    .10 .20 .30 .25 .15   % 5. Neutral to agree
    .00 .20 .50 .20 .10   % 6. Neutral peak
    .15 .20 .25 .20 .20   % 7. Neutral flat
    .80 .12 .04 .03 .01   % 8. Very strongly disagree
    .70 .20 .06 .03 .01   % 9. Strongly disagree
    .25 .35 .20 .15 .05   % 10. Disagree flat
    .10 .25 .30 .20 .15   % 11. Neutral to disagree
    .01 .04 .50 .30 .15   % 12. Certainly not disagree
    .15 .05 .15 .25 .40   % 13. Multimodal
    .45 .05 .00 .05 .45]; % 14. Strong multimodal
Table1=NaN(size(D,1),4);
for i2=1:size(D,1) % loop over the 14 distributions
    temp=D(i2,:)*100;
    P=[ones(temp(1),1);2*ones(temp(2),1);3*ones(temp(3),1);4*ones(temp(4),1);5*ones(temp(5),1)];
    Table1(i2,:)=[mean(P) std(P) skewness(P) kurtosis(P)];
end
disp('Table 1: Mean, SD, skewness, kurtosis')
disp(round(Table1,2))

CD=NaN(size(D,1),size(D,1));
for i2=1:size(D,1) % loop over the 14 distributions
    for i3=1:size(D,1) % loop over the 14 distributions
    temp=D(i2,:)*100000;
    P1=[ones(temp(1),1);2*ones(temp(2),1);3*ones(temp(3),1);4*ones(temp(4),1);5*ones(temp(5),1)];
    temp=D(i3,:)*100000;
    P2=[ones(temp(1),1);2*ones(temp(2),1);3*ones(temp(3),1);4*ones(temp(4),1);5*ones(temp(5),1)];
    temp=tiedrank([P1;P2]);
    P1r=temp(1:length(P1));
    P2r=temp(length(P1)+1:end);
    CD(i2,i3)=abs(cohend(P1,P2))-abs(cohend(P1r,P2r));
    end
end
disp('Table 3. Absolute of Cohen''s d effect size of difference between untransformed data minus absolute of Cohen''s d effect size of difference between rank-transformed data')
disp(round(CD,2))
%%
nm=[10 10
    30 30
    200 200
    5 20
    100 10];
reps=10000;
Dcum=cumsum(D,2);
for i1=1:size(nm,1) % loop across the different sample sizes
    pT=NaN(size(Dcum,1),size(Dcum,1),reps);
    pW=NaN(size(Dcum,1),size(Dcum,1),reps);
    n = nm(i1,1); m = nm(i1,2);
    for i2=1:size(Dcum,1) % loop over the 14 distributions
        for i3=1:size(Dcum,1) % loop over the 14 distributions
            for i4=1:reps
                L1=NaN(n,1);L2=NaN(m,1);
                X1=rand(n,1);X2=rand(m,1); % draw random variables between 0 and 1
                for i5=1:n % generate Likert item data for sample 1
                    L1(i5)=find(X1(i5)<Dcum(i2,:),1);
                end
                for i5=1:m % generate Likert item data for sample 2
                    L2(i5)=find(X2(i5)<Dcum(i3,:),1);
                end
                [~,pT(i2,i3,i4)]=ttest2(L1,L2); % conduct t test
                V=tiedrank([L1;L2]);L1=V(1:n);L2=V(n+1:end); % transform to ranks
                [~,pW(i2,i3,i4)]=ttest2(L1,L2); % conduct t test on rank-transformed data (MWW)
            end
        end
    end
    Tsuccess=squeeze(sum(pT<.05,3)); % determine if null hypothesis is rejected for t test
    Wsuccess=squeeze(sum(pW<.05,3)); % determine if null hypothesis is rejected for MWW
    
    Type1=round(100*[diag(Tsuccess)/reps diag(Wsuccess)/reps],2); % Type I error rates of t test and MWW, expressed as a percentage
    if n==m % equal sample sizes
        AboveD=1-(triu(Tsuccess)+tril(Tsuccess)')./(2*reps); % above the diagonal: Type II error rate (between 0 and 1) for the t test (average of above and below diagonal if sample sizes are equal)
        BelowD=(tril(Tsuccess-Wsuccess)+triu(Tsuccess-Wsuccess)')./(2*reps); % below the diagonal: difference between the Type II error rate of MWW minus the Type II error rate for the t test, average of above and below diagonal
        AboveD=AboveD-tril(AboveD); % create zeros below the diagonal
        BelowD=BelowD+diag(NaN(size(Dcum,1),1)); % create NaNs on the diagonal
        Matrix=AboveD+BelowD; % combine the two matrices; above the diagonal: Type II error rate; below the diagonal: Type II error rate MWW - Type II error rate t test; on the diagonal: NaNs
        Matrix=round(100*Matrix); % transform 0-1 to a percentage, and round to nearest integer
        disp('Type II error percentage for the t test (above diagonal), Type II error percentage for MWW minus Type II error percentage for the t test (below diagonal;  A positive number means that the t test was more powerful than MWW), and Type I error percentages for the t test and MWW (two rightmost columns)')
        disp(['n = ' num2str(n) ', m = ' num2str(m)])
        disp([Matrix Type1]) % Display output
    else % unequal sample sizes
        TypeII=round(100*(1-Tsuccess./reps)); % Type II error percentage for the t test
        TypeII=TypeII+diag(NaN(size(Dcum,1),1)); % create NaNs on the diagonal
        disp('Type II error percentage for the t test, and Type I error percentages for the t test and MWW (two rightmost columns)')
        disp(['n = ' num2str(n) ', m = ' num2str(m)])
        disp([TypeII Type1]) % display output
        
        TypeIIdiff=round(100*((Tsuccess-Wsuccess)./reps)); % Type II error percentage for MWW minus Type II error percentage for the t test
        TypeIIdiff=TypeIIdiff+diag(NaN(size(Dcum,1),1)); % create NaNs on the diagonal
        disp('Type II error percentage for MWW minus Type II error percentage for the t test ( A positive number means that the t test was more powerful than MWW)')
        disp(['n = ' num2str(n) ', m = ' num2str(m)])
        disp(TypeIIdiff)
    end
end
%%
function cd = cohend(X1,X2)
M1=mean(X1);
M2=mean(X2);
n1=length(X1);
n2=length(X2);
SDpooled=sqrt((std(X1).^2*(n1-1)+std(X2).^2*(n2-1))/(n1+n2-2));
cd=(M1-M2)./SDpooled;
end