/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2020 ENERCON GmbH
    Copyright (C) 2018-2020 OpenCFD Ltd
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "flyingActuationDiskSource.H"
//#include "fvMesh.H"
//#include "fvMatrix.H"
//#include "volFields.H"

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

template<class AlphaFieldType, class RhoFieldType>
void Foam::fv::flyingActuationDiskSource::calc
(
    const AlphaFieldType& alpha,
    const RhoFieldType& rho,
    fvMatrix<vector>& eqn,
    interpolationCellPoint<vector> UInterp
)
{

    //Zero out the total force // 
    // Commented by Clem 2024_01_21 Actually totalForce_ = totalForceBlade_ + totalForceBlade_
    totalForce_ = vector(0.0, 0.0, 0.0);
    localForce_ = vector(0.0, 0.0, 0.0);

    //Zero out the rotor force //Added by Clem 2024_01_21
    totalForceRotor_ = vector(0.0, 0.0, 0.0);
    localForceRotor_ = vector(0.0, 0.0, 0.0);

    //Zero out the blade force //Added by Clem 2024_01_21
    totalForceBlade_ = vector(0.0, 0.0, 0.0);
    localForceBlade_ = vector(0.0, 0.0, 0.0);

    // Zero out force vector and field
    forceField_ *= dimensionedScalar("zero", forceField_.dimensions(), 0.0);

    // Check dimensions of force field and correct if necessary
    if (forceField_.dimensions() != eqn.dimensions()/dimVolume)
    {
        forceField_.dimensions().reset(eqn.dimensions()/dimVolume);
    }

    rotorPowerCalc_ = 0.0;           //Added by Clem 2024_01_23

    bladePowerCalc_ = 0.0;           //Added by Clem 2024_01_24

    switch (forceMethod_)
    {
        // case forceMethodType::FROUDE:
        // {
        //     calcFroudeMethod(alpha, rho, eqn);
        //     break;
        // }

        case forceMethodType::VARIABLE_SCALING:
        {
            calcVariableScalingMethod(alpha, rho, eqn, UInterp);
            break;
        }

        default:
            break;
    }
}



template<class AlphaFieldType, class RhoFieldType>
void Foam::fv::flyingActuationDiskSource::calcVariableScalingMethod
(
    const AlphaFieldType& alpha,
    const RhoFieldType& rho,
    fvMatrix<vector>& eqn,
    interpolationCellPoint<vector> UInterp
)
{
    const vectorField& U = eqn.psi();
    vectorField& Usource = eqn.source();
    const scalarField& cellsV = mesh_.V();
    const volVectorField& cellsC = mesh_.C();   //Added by Clem 2023_10_29


    //Added by Clem 2024_02_29 to adjust pitch
    if (bladeNumberofActuatorPoints_ > 0)
    {
        scalar currentTime = mesh_.time().value();
        if(timeStartAdjustPitch_ > 0 && timeStartAdjustPitch_ <= currentTime)
        {

            for(label bladeIndex = 0; bladeIndex < bladeNumber_; bladeIndex++)
            {
                scalar misTargetAOANow = bladeAOAMidofThatBlade_[bladeIndex] - bladeAOATarget_[bladeIndex];

                misTargetAOAFiltered_[bladeIndex] = 
                    (misTargetAOANow + misTargetAOAFiltered_[bladeIndex]) / bladePitchTimeScale_ * exp( - 1 / bladePitchTimeScale_ *  mesh_.time().deltaT().value() );

                scalar deltaPitch = bladePitchSpeed_ * mesh_.time().deltaT().value();

                if(misTargetAOAFiltered_[bladeIndex] > (0.75 * deltaPitch ) )
                {
                    for (label indexofAL = 0; indexofAL < bladeNumberofActuatorPoints_; indexofAL++ )
                    {
                        if(bladeNumIndicator_[indexofAL] == bladeIndex)
                        {
                            bladePitch_[indexofAL] = bladePitch_[indexofAL] - deltaPitch;
                        }
                    }
                }

                if(misTargetAOAFiltered_[bladeIndex] < (-0.75 * deltaPitch ) )
                {
                    for (label indexofAL = 0; indexofAL < bladeNumberofActuatorPoints_; indexofAL++ )
                    {
                        if(bladeNumIndicator_[indexofAL] == bladeIndex)
                        {
                            bladePitch_[indexofAL] = bladePitch_[indexofAL] + deltaPitch;
                        }
                    }
                }


            }
        } 
    }

    // Monitor and average monitor-region U and rho
    vector Uref(Zero);
    scalar rhoRef = 0.0;
    label szMonitorCells = monitorCells_.size();

    for (const auto& celli : monitorCells_)
    {
        Uref += U[celli];
        rhoRef = rhoRef + rho[celli];
    }
    reduce(Uref, sumOp<vector>());
    reduce(rhoRef, sumOp<scalar>());
    reduce(szMonitorCells, sumOp<label>());

    if (szMonitorCells == 0)
    {
        FatalErrorInFunction
            << "No cell is available for incoming velocity monitoring."
            << exit(FatalError);
    }


    Uref /= szMonitorCells;
    const scalar magUref = mag(Uref);
    rhoRef /= szMonitorCells;

    // Monitor and average U and rho on actuator disk
    vector Udisk(Zero);
    scalar rhoDisk = 0.0;
    scalar rotorTotalV = 0.0;

    scalar bladeTotalV = 0.0;  //Added by Clem 2023_10_30

    for (const auto& celli : cells_)
    {
        Udisk += U[celli]*cellsV[celli];
        rhoDisk += rho[celli]*cellsV[celli];
        rotorTotalV += cellsV[celli];
    }
    reduce(Udisk, sumOp<vector>());
    reduce(rhoDisk, sumOp<scalar>());
    reduce(rotorTotalV, sumOp<scalar>());

    if (rotorTotalV < SMALL)
    {
        FatalErrorInFunction
            << "No cell in the actuator disk."
            << exit(FatalError);
    }

    Udisk /= rotorTotalV;
    const scalar magUdisk = mag(Udisk);
    rhoDisk /= rotorTotalV;

    if (mag(Udisk) < SMALL)
    {
        FatalErrorInFunction
            << "Velocity spatial-averaged on actuator disk is zero." << nl
            << "Please check if the initial U field is zero."
            << exit(FatalError);
    }

    // Interpolated thrust/power coeffs from power/thrust curves
    const scalar rotorCt = sink_*rotorUvsCtPtr_->value(magUref);
    const scalar rotorCp = sink_*rotorUvsCpPtr_->value(magUref);

    // Calibrated thrust/power coeffs from power/thrust curves (LSRMTK:Eq. 6)
    const scalar rotorCtStar = rotorCt*sqr(magUref/magUdisk);
    const scalar rotorCpStar = rotorCp*pow3(magUref/magUdisk);

    // Start added by Clem 2023_11_13
    scalar rotorTCarrier = 0.0;
    scalar rotorPCarrier = 0.0;

    if(rotorForcePrescribed_ == 0)
    {
        rotorTCarrier = 0.5*rhoRef*diskArea_*magSqr(Udisk & freeStreamDir_)*rotorCtStar;
        rotorPCarrier = 0.5*rhoRef*diskArea_*pow3(mag(Udisk & freeStreamDir_))*rotorCpStar;
    }
    else
    {
        rotorTCarrier = rotorTPrescribed_;
        rotorPCarrier = rotorTCarrier * mag(Udisk & freeStreamDir_);
    }

    const scalar rotorT = rotorTCarrier;
    const scalar rotorP = rotorPCarrier;

    // End added by Clem 2023_11_13


    // Added by Clem 2024_10_22
    if (rotorNumberofActuatorPoints_ > 0)
    {
        *outputFileDisk_ << mesh_.time().value() << "," ;   
    }

    // Apply rotor force
    label conunterSmear = 0;
    for (label actuatorPointIndex = 0; actuatorPointIndex < rotorNumberofActuatorPoints_; actuatorPointIndex++)  // modified by Clem 2024_09_10
    {
        //vector cellCenterNow = cellsC[celli] + rotorStreamWiseShift_; //Adjusted by Clem 2024_02_12

        vector cellCenterNow = rotorActuatorPointsPositions_[actuatorPointIndex]; //Adjusted by Clem 2024_09_10

        vector rotorForceVectorToApply = Foam::Zero;

        label sampleCellI = findCell(cellCenterNow);
        vector inflowVelocityRotor = UInterp.interpolate( cellCenterNow, sampleCellI );

        if (nonUniformDiskLoad_ == 1)
        {
            rotorForceVectorToApply = 0.5 * rhoRef * 
                (inflowVelocityRotor &  diskDir_) * (inflowVelocityRotor & diskDir_)
                * rotorFrontalAreaPerCell_[actuatorPointIndex] * rotorCtStar * diskDir_;    //Added by Clem 2024_01_23, modified by Clem 2024_09_10
        }
        else
        {
        //    rotorForceVectorToApply = (cellsV[celli]/rotorTotalV*rotorT)*diskDir_;    //Added by Clem 2023_10_29, commented by Clem 2024_09_10
        }

        label rotorOrBlade = 0; //Added by Clem 2024_01_21  0 for rotor; 1 for blade

        applyForceField(eqn, U, cellCenterNow, rotorForceVectorToApply, 
            rotorLableListtoSmear_[conunterSmear], rotorSmearingLengthX_[actuatorPointIndex], rotorSmearingLengthY_[actuatorPointIndex], rotorSmearingLengthZ_[actuatorPointIndex], 
            rotorCompensationFactor_, rotorOrBlade);      //Added by Clem 2023_10_29  2023_11_12 2014_11_06
        conunterSmear += 1;

        // out put 
        // start Added by Clem 2024_1022
        *outputFileDisk_ << cellCenterNow.x() << "," 
                << cellCenterNow.y() << "," << cellCenterNow.z() << ",";

        *outputFileDisk_ << inflowVelocityRotor.x() << "," 
                << inflowVelocityRotor.y() << "," << inflowVelocityRotor.z() << ",";

        *outputFileDisk_ << rotorForceVectorToApply.x() * -1.0 << "," 
                << rotorForceVectorToApply.y() * -1.0 << "," << rotorForceVectorToApply.z() * -1.0 << ",";
        // end Added by Clem 2024_1022

    }

    // Added by Clem 2024_1022
    if (rotorNumberofActuatorPoints_ > 0)
    {
            *outputFileDisk_ << endl; // Added by Clem 2024_01_22
    }

    // Pout << "conunterSmear: " << conunterSmear << endl;  //Added by Clem 2023_11_12


    scalar bladeTtoReport = 0.0;
    scalar bladePtoReport = 0.0;

    //Added By Clem 2024_01_21 ??
    if (bladeNumberofActuatorPoints_ > 0)
    {
        bladeFlowAngle_ = scalarList(bladeNumberofActuatorPoints_, 0.0);
    }

    if(bladeUvsCtPtr_->value(magUref) == 0.0 && bladeForcePrescribed_ == 0)
    {
        const scalar bladeCt = 0.0;
        const scalar bladeCp = 0.0;

        const scalar bladeCtStar = 0.0;
        const scalar bladeCpStar = 0.0;

        const scalar bladeT = 0.0;
        const scalar bladeP = 0.0;


        bladeTtoReport = bladeT;
        bladePtoReport = bladeP;

    }
    else
    {
        // Commented by Clem 2024_02_08
        //for (const auto& celli : bladeCells_)
        //{
        //    bladeTotalV += cellsV[celli];
        //}
        //reduce(bladeTotalV, sumOp<scalar>());

        //if (bladeTotalV < SMALL)
        //{
        //    FatalErrorInFunction
        //        << "No cell in the blade disk."
        //        << exit(FatalError);
        //}


        // Interpolated thrust/power coeffs from power/thrust curves //Added by Clem 2023_10_30
        const scalar bladeCt = sink_*bladeUvsCtPtr_->value(magUref);
        const scalar bladeCp = sink_*bladeUvsCpPtr_->value(magUref);


        // Calibrated thrust/power coeffs from power/thrust curves (LSRMTK:Eq. 6)
        const scalar bladeCtStar = bladeCt*sqr(magUref/magUdisk);
        const scalar bladeCpStar = bladeCp*pow3(magUref/magUdisk);


        // Start added by Clem 2023_11_13
        scalar bladeTCarrier = 0.0;
        scalar bladePCarrier = 0.0;

        if(bladeForcePrescribed_ == 0)
        {
            bladeTCarrier = 0.5*rhoRef*diskArea_*magSqr(Udisk & freeStreamDir_)*bladeCtStar;
            bladePCarrier = 0.5*rhoRef*diskArea_*pow3(mag(Udisk & freeStreamDir_))*bladeCpStar;
        }
        else
        {
            bladeTCarrier = bladeTPrescribed_;
            bladePCarrier = bladeTCarrier * mag(Udisk & freeStreamDir_);
        }

        const scalar bladeT = bladeTCarrier;
        const scalar bladeP = bladePCarrier;


        bladeTtoReport = bladeT;
        bladePtoReport = bladeP;

        // End added by Clem 2023_11_13

        // Added by Clem 2024_01_22
        if (bladeNumberofActuatorPoints_ > 0)
        {
            *outputFile_ << mesh_.time().value() << "," ;   
        }

        label conunterSmear = 0;
        label counterForFlowAngle = 0;
        // Apply blade force
        // for (const label celli : bladeCells_)
        for (label actuatorPointIndex = 0; actuatorPointIndex < bladeNumberofActuatorPoints_; actuatorPointIndex++)
        {
            vector cellCenterNow = bladeActuatorPointsPositions_[actuatorPointIndex];
            //vector bladeForceVectorToApply = (cellsV[celli]/bladeTotalV*bladeT)*bladeDir_;    //Added by Clem 2023_10_29 //Commented by Clem 2024_01_23

            // Added by Clem 2024_02_03, update inflowVelocity_
            if(velSamplingRadius_ > 0.0)
            {   
                //Pout << "Going in samplingInflowVel at " << cellCenterNow << ": " << endl;
                //Adjusted by Clem 2024_03_06, 2024_11_06
                scalar characteristicSmear = sqrt(bladeSmearingLengthX_[actuatorPointIndex] * bladeSmearingLengthX_[actuatorPointIndex] / 3 
 + bladeSmearingLengthY_[actuatorPointIndex] * bladeSmearingLengthY_[actuatorPointIndex] / 3 
 + bladeSmearingLengthZ_[actuatorPointIndex] * bladeSmearingLengthZ_[actuatorPointIndex] / 3);
                scalar velSamplingRadius = characteristicSmear * 1.5;
                samplingInflowVel(cellCenterNow, velSamplingRadius, velSamplingNumber_, UInterp);
            }
            else
            {
                label sampleCellI = findCell(cellCenterNow);
                inflowVelocity_ = UInterp.interpolate( cellCenterNow, sampleCellI );
                //inflowVelocity_ = U[celli];
            }

            // Added by Clem 2024_02_08
            label sampleCellI = findCell(cellCenterNow);
            vector inflowVelocityOnPoint = UInterp.interpolate( cellCenterNow, sampleCellI );

            // Start Added by Clem 2024_01_22
            scalar samplingFlowAngle;
            samplingFlowAngle = Foam::atan(inflowVelocity_.z() / inflowVelocity_.x());
            bladeFlowAngle_[counterForFlowAngle] = samplingFlowAngle;

            // Added by Clem 2024_01_30  to make Cl = 2 * pi * sin(alpha)
            //scalar bladeLocalCl = 2 * Foam::constant::mathematical::pi
            //    * sin( samplingFlowAngle + bladeTwist_);

            // Adjusted by Clem 2024_02_29
            bladeAOA_[actuatorPointIndex] = 
                (samplingFlowAngle * 180 / Foam::constant::mathematical::pi + 
                bladeTwist_[actuatorPointIndex] + bladePitch_[actuatorPointIndex]);

            scalar bladeLocalCl = interpolate(bladeAOA_[actuatorPointIndex], alphaList_, clList_);   // Added by Clem 2024_02_12

            scalar bladeLocalCd = interpolate(bladeAOA_[actuatorPointIndex], alphaList_, cdList_);   // Added by Clem 2024_02_12

            vector bladeLiftVectorToApply = 0.5 * rhoRef * 
                ( inflowVelocity_.x() * inflowVelocity_.x() + inflowVelocity_.z() * inflowVelocity_.z() ) *   // Added by Clem 2024_02_12
                bladeLocalCl * bladeChord_ * bladeSpanPerAL_ * 
                ( (inflowVelocity_/ mag(inflowVelocity_) ) ^ lateralDirection_ );

            //CHECK THE DIRECTION WHEN COME BACK
            vector bladeDragVectorToApply = 0.5 * rhoRef * 
                ( inflowVelocity_.x() * inflowVelocity_.x() + inflowVelocity_.z() * inflowVelocity_.z() ) *   // Added by Clem 2024_02_12
                bladeLocalCd * bladeChord_ * bladeSpanPerAL_ * 
                ( inflowVelocity_ / mag(inflowVelocity_) );

            bladeDragVectorToApply.y() = 0.0;   // Added by Clem 2024_02_12



            vector bladeForceVectorToApply = bladeLiftVectorToApply + bladeDragVectorToApply;   // Adjusted by Clem 2024_02_12


            label rotorOrBlade = 1; //Added by Clem 2024_01_21  0 for rotor; 1 for blade

            // End Added by Clem 2024_01_22


            // Start Added by Clem 2024_01_25 for tip correction
            scalar f = 1.0;

            if(bladeEndEffectsActive_ == 1)
            {

                // commented by Clem 2024_01_27 Prandtl attempt
                //if (samplingFlowAngle >= 0)
                //{
                //    samplingFlowAngle = max(0.001, samplingFlowAngle);
                //}
                //else
                //{
                //    samplingFlowAngle = min(-0.001, samplingFlowAngle);
                //}
                //scalar acosArg = Foam::exp
                //        (
                //            -1/2.0*(1.0/bladeRelativeDistToRoot_[counterForFlowAngle] - 1)/sin(samplingFlowAngle)  //Note phi is alpha for our case
                //        );
                //f = 2.0/Foam::constant::mathematical::pi*acos(min(1.0, acosArg));


                // tablet values, Commented by Clem 2024_01_27
                f = tipCorrectionFactor_[counterForFlowAngle];
            }

            bladeForceVectorToApply = f * bladeForceVectorToApply;

            // out put file time, (x, y, z, u, v, w, fx, fy, fz)

            *outputFile_ << cellCenterNow.x() << "," 
                << cellCenterNow.y() << "," << cellCenterNow.z() << ",";

            *outputFile_ << inflowVelocity_.x() << "," 
                << inflowVelocity_.y() << "," << inflowVelocity_.z() << ",";

            *outputFile_ << inflowVelocityOnPoint.x() << "," 
                << inflowVelocityOnPoint.y() << "," << inflowVelocityOnPoint.z() << ",";

            *outputFile_ << bladeLiftVectorToApply.x() * -1.0 << "," 
                << bladeLiftVectorToApply.y() * -1.0 << "," << bladeLiftVectorToApply.z() * -1.0 << ",";

            *outputFile_ << bladeDragVectorToApply.x() * -1.0 << "," 
                << bladeDragVectorToApply.y() * -1.0 << "," << bladeDragVectorToApply.z() * -1.0 << ",";


            *outputFile_ << f << ",";

            *outputFile_ << bladeLocalCl << ",";

            *outputFile_ << bladePitch_[actuatorPointIndex] << ",";

            // End Added by Clem 2024_01_25

            applyForceField(eqn, U, cellCenterNow, bladeForceVectorToApply, 
                bladeLableListtoSmear_[conunterSmear], bladeSmearingLengthX_[actuatorPointIndex], bladeSmearingLengthY_[actuatorPointIndex], bladeSmearingLengthZ_[actuatorPointIndex],  
                bladeCompensationFactor_, rotorOrBlade);      //Added by Clem 2023_10_29
        
            conunterSmear += 1;
            counterForFlowAngle += 1;
        }

        if (bladeNumberofActuatorPoints_ > 0)
        {
            *outputFile_ << endl; // Added by Clem 2024_01_22
        }

        eqn += forceField_; //Added by Clem to try something 2024_02_13


        if (bladeNumberofActuatorPoints_ > 0)
        {
            //Added By Clem 2024_02_29, to get the AOA at mid span of each blade
            for(label bladeIndex = 0; bladeIndex < bladeNumber_; bladeIndex++)
            {
                bladeAOAMidofThatBlade_[bladeIndex] = bladeAOA_[indexForBladeMid_[bladeIndex]];
            }
        }


    }
    //Added by Clem 2023_10_30

    //Added by Clem 2023_10_30  Modified 2023_11_14 .value() is to get the value without dimension
    //vector localSum = (sum(forceField_)).value();
    totalForce_ = returnReduce(localForce_, sumOp<vector>());
    //totalForce_ *= -1.0;

    // Added by Clem 2024_01_24
    totalForceRotor_ = returnReduce(localForceRotor_, sumOp<vector>());
    //totalForceRotor_ *= -1.0;

    // Added by Clem 2024_01_24
    totalForceBlade_ = returnReduce(localForceBlade_, sumOp<vector>());
    //totalForceBlade_ *= -1.0;

    reduce(rotorPowerCalc_, sumOp<scalar>());  //Added by Clem 2024_01_23
    reduce(bladePowerCalc_, sumOp<scalar>());  //Added by Clem 2024_01_23


    if
    (
        mesh_.time().timeOutputValue() >= writeFileStart_
        && mesh_.time().timeOutputValue() <= writeFileEnd_
    )
    {
        Ostream& os = file();
        writeCurrentTime(os);

        os  << Uref << tab << rotorCpStar << tab << rotorCtStar << tab
            << Udisk << tab << rotorT << tab 
            << rotorP << tab << bladeTtoReport << tab 
            << bladePtoReport << tab << totalForce_ << tab
            << totalForceRotor_ << tab << totalForceBlade_ << tab
            << rotorPowerCalc_ << tab << bladePowerCalc_ << endl;
    }

}



// Added by Clem 2023_10_29
// Modified by Clem 2023_11_12

void Foam::fv::flyingActuationDiskSource::applyForceField
(
    fvMatrix<vector>& eqn,
    const vectorField& U,
    vector cellCenterNow,
    vector forceVectorToApply,
    labelList domainCells,  //Cells selected to smear
    scalar smearingLengthX,
    scalar smearingLengthY,
    scalar smearingLengthZ,
    scalar compensationFactor,
    label rotorOrBlade
)
{
    // Calculate projection width
    // scalar sphereRadius = (smearingLength*Foam::sqrt(Foam::log(1.0/0.001)));

    //scalar sphereRadius = 3 * smearingLength;

    // Apply force to the cells within the element's sphere of influence
    //forAll(mesh_.cells(), cellI)
    for (const label cellI : domainCells)
    {
        //Adjusted by Clem 2024_11_06
        // scalar dis = mag(mesh_.C()[cellI] - cellCenterNow);
        scalar disX = (mesh_.C()[cellI] - cellCenterNow).x();
        scalar disY = (mesh_.C()[cellI] - cellCenterNow).y();
        scalar disZ = (mesh_.C()[cellI] - cellCenterNow).z();

        //if (dis <= sphereRadius)
        //{
        scalar factor = Foam::exp( -Foam::sqr(disX/smearingLengthX) -Foam::sqr(disY/smearingLengthY) -Foam::sqr(disZ/smearingLengthZ) )
                      / ( (smearingLengthX*smearingLengthY*smearingLengthZ) //(Foam::pow(smearingLength, 3)           // it is divided by Volume at here -> force becomes force density Adjusted by Clem 2024_11_06
                      * Foam::pow(Foam::constant::mathematical::pi, 1.5))
                      /compensationFactor;    // To adjust the loss due to smearing, decided empirically
        // forceField is opposite forceVector  // Should consider the volume since the dimension should be force Clem 2023_10_30
        //Usource[cellI] += forceVectorToApply*factor*mesh_.V()[cellI]; // force per density

        vector forceSmearedApplyToCell = forceVectorToApply*factor*mesh_.V()[cellI]; // become force per density after smearing
        
        // Added by Clem 2023_01_21;  0 for rotor; 1 for blade
        if(rotorOrBlade == 0)
        {
            localForceRotor_ -= forceSmearedApplyToCell;
            rotorPowerCalc_ += forceSmearedApplyToCell & U[cellI];   // The work actuator posed on the the flow, by Clem 2024_02_12
        }

        // Added by Clem 2023_01_21;  0 for rotor; 1 for blade
        if(rotorOrBlade == 1)
        {
            localForceBlade_ -= forceSmearedApplyToCell;
            bladePowerCalc_ += forceSmearedApplyToCell & U[cellI];
        }

        //Usource[cellI] += forceSmearedApplyToCell;
        localForce_ -= forceSmearedApplyToCell;

        //totalForce_ -= Usource[cellI]; // force per density
        forceField_[cellI] -= forceVectorToApply*factor;  //force per volume per density

        //}
    }
    

}



// Added by Clem, sampling the inflow velocity with multiple points over a circular ring
// 2024_02_03
void Foam::fv::flyingActuationDiskSource::samplingInflowVel
(
    vector cellCenterNow,
    scalar velSamplingRadius,
    label velSamplingNumber,
    interpolationCellPoint<vector> UInterp
)
{

    vector velocitySum = vector(0.0, 0.0, 0.0);
        
    for (label point = 0; point < velSamplingNumber; point++)
    {

        vector sampleVelocity = vector(VGREAT, VGREAT, VGREAT);

        // distribute the points evenly in terms of angular distance
        scalar pointAngle = Foam::constant::mathematical::pi * 2.0 * point/
                                velSamplingNumber;
        scalar freeStreamDist = velSamplingRadius * Foam::cos(pointAngle);
        scalar verticalDist = velSamplingRadius * Foam::sin(pointAngle);
        vector samplePoint = cellCenterNow +
                                 freeStreamDist * freeStreamDir_ +
                                 verticalDist * (freeStreamDir_ ^ lateralDirection_);

        // Sample the velocity
        label sampleCellI = -1;
        if (bladeNumberofActuatorPoints_ > 0)
        {
            //Pout << "Going in findCell at : " << samplePoint << endl;
            sampleCellI = findCell(samplePoint);
        }

        if (sampleCellI >= 0)
        {
            //Pout << "Going in UInterp.interpolate at : " << samplePoint << endl;
            sampleVelocity = UInterp.interpolate
            (
                samplePoint,
                sampleCellI
            );
            //Pout << "sampleVelocity at " << samplePoint << " :" << sampleVelocity << endl;
        }

        // Reduce inflow velocity over all processors
        //reduce(sampleVelocity, minOp<vector>());

        // If inflow velocity is not detected, position is not in the mesh
        if (not (sampleVelocity[0] < VGREAT))
        {
            // Raise fatal error since inflow velocity cannot be detected
            FatalErrorIn("void actuatorLineElement::calculateForce()")
                << "Inflow velocity point for " << name_
                << " not found in mesh"
                << abort(FatalError);
        }

        velocitySum = velocitySum + sampleVelocity;
        //Pout << "velocitySum at " << samplePoint << " :" << velocitySum << endl;
    }

    //Pout << endl;
    //Pout << endl;

    inflowVelocity_ = velocitySum / velSamplingNumber;
}


Foam::label Foam::fv::flyingActuationDiskSource::findCell
(
    const point& location
)
{
    if (Pstream::parRun())
    {
        if (meshBoundBox_.containsInside(location))
        {
            if (debug)
            {
                Pout<< "Looking for cell containing " << location
                    << " inside bounding box:" << endl
                    << meshBoundBox_ << endl;
            }
            return mesh_.findCell(location);
        }
        else
        {
            if (debug)
            {
                Pout<< "Cell not inside " << meshBoundBox_ << endl;
            }
            return -1;
        }
    }
    else
    {
        return mesh_.findCell(location);;
    }
}




// ************************************************************************* //
