function [sigma, tau, M] = allandev(freqSignal, sampleFreq, varargin)
%ALLANDEV Returns the allan deviation for the given frequency signal and
%sample frequency.
%
%       Author: Ties Verschuren (c) 2019
%
%[sigma, tau, M] = allandev(freqSignal, sampleFreq, ___)
%   
%   INPUT
%       freqSignal      Frequency signal to calculate the allan deviation
%                       from.
%       sampleFreq      Sample frequency: used to calculate tau values.
%                       Expected unit: Hz
%   OPTIONAL INPUT (___)
%       minSamples      The minimal values to average over for sigma,
%                       default 100.
%       minGateTime     Minimal gate time to start from, default smallest
%                       possible (1/sampleFreq).
%       quiet           Disable progress output. Default true. When set to
%                       false, some progress data is shown.
%       
%   OUTPUT
%       sigma           The allan deviation. (2 sample method)
%       tau             The gate times for which the allan deviation is
%                       calculated.
%       M               The number of samples used per gate time in the
%                       calculation.

%% Default values for optional parameters
minSamples = 100; %minimal 100 
minGateTime = 1/sampleFreq;
pointsPerDecade = 5;
quiet = true;

%% parse optional parameters
if nargin > 2
    for i = 1:2:(nargin - 2)
        if size(varargin) < (i + 1)
            error('allandev: No value given for optional parameter %s', varargin{i});
        end
        value = varargin{i+1};
        switch varargin{i}
            case 'minSamples'
                minSamples = value;
            case 'minGateTime'
                minGateTime = value;
            case 'pointsPerDecade'
                pointsPerDecade = value;
            case 'quiet'
                quiet = value;
            otherwise
                error('allandev: Unknown option "%s" given.', varargin{i});
        end
    end
end

%% Parameter validation
if ~isnumeric(minSamples) || minSamples < 0
    error('allandev: Parameter minSamples should be a positive number, got %s', minSamples)
end

if ~isnumeric(minGateTime) || minGateTime < 0
    error('allandev: Parameter minGateTime should be a positive number, got %s', minSamples)
end

if ~isnumeric(pointsPerDecade) || pointsPerDecade < 0
    error('allandev: Parameter pointsPerDecade should be a positive number, got %s', pointsPerDecade)
end

if ~islogical(quiet)
    error('allandev: Parameter quiet should be a logical value (i.e. 1 or 0), got %s', quiet);
end


%% Calculate initial values
numSamples = numel(freqSignal);
freqAvg = mean(freqSignal); % average frequency

% Take the frequency fraction:  normalized frequency values to their mean.
y = freqSignal ./ freqAvg;

% Number of loops (number of different gate times used)
startTauNum = max(round(minGateTime * sampleFreq), 1); % starting index for gate time
endTauNum = max(floor(numSamples / (minSamples + 1 )), 1);

%% Calculate evaluation points
startTau = startTauNum * 1/sampleFreq;
endTau = endTauNum * 1/sampleFreq;
decades = log10(endTau) - log10(startTau);
evaluationTaus = logspace(log10(startTau), log10(endTau), round(decades * (pointsPerDecade + 1)));
evaluationPoints = unique(round(evaluationTaus * sampleFreq));
numPoints = numel(evaluationPoints);

%% Show status if needed
nbytes = 0;
if ~quiet
    fprintf('allandev: calculate for %d points\n', numPoints); % print status message
end

%% Initialize empty loop variables
sigma = zeros(numPoints, 1);
tau = sigma;
M = sigma;

%% Calculation
for i = 1:numPoints
    sampleIdx = evaluationPoints(i);
    % Tau is the time
    tau(i) = (1/sampleFreq) * sampleIdx;
    % num samples is always the number of average bins minus 1
    M(i) = floor(numSamples / sampleIdx) - 1;
    % sum for this tau value
    loopSum = 0;

    % Average for the first bin
    mm = 1;
    y_m = mean( y((1+(mm-1)*sampleIdx):(mm*sampleIdx)) );
    % loop over the other bins
    for m = 1:M(i)
        y_mplus1 = mean( y((1+m*sampleIdx):((m+1)*sampleIdx)) );
        loopSum = ((y_mplus1 - y_m)^2)/2 + loopSum;
        % store the average for the next loop
        y_m = y_mplus1;
    end
    % Calculate the allan deviation
    sigma(i) = sqrt( loopSum /(M(i)+1) );
    
    if ~quiet
        fprintf(repmat('\b',1,nbytes))
        nbytes = fprintf('allandev: pocessed %.2f %%\n', i/numPoints*100); % print status message
    end
end
% fprintf(repmat('\b',1,nbytes))
if ~quiet
    fprintf('allandev: Done\n');
end
end

