function paths = getSignalPathsFromVariable(structVar, varargin)
%GETSIGNALSFROMVARIABLE Returns the signal paths from the given structVar
%   
%   INPUT
%       structVar   Variable that is ordered as a struct
%
%   OUTPUT
%       paths       The paths and their type that exists in the struct. 
%                   Layout (per row): { {'path','parts'}, 'signalType'}
%                   So to access the struct path: paths{:, 1}
%                   To access the signalType: paths{:, 2}
%
%       PATHTYPES
%           valueTimesignal     Time signal containing a struct with
%               timestamp and value fields.
%           sharedTimesignal    Time signal conataining the data. The
%               timestamp data is available by replacing the last part of
%               the path with timestamp.
%
% Author: Ties Verschuren (c) 2020
    
    if nargin < 2
        % first run of this function loop. The parent will be dev*
        parent = fieldnames(structVar);
        structVar = structVar.(parent{1});
    elseif nargin ~= 2 || numel(varargin) ~= 1
        error('UHFLIStreamParser:getSignalPathsFromVariable:invalidArgumentCount', ...
            'Invalid number of arguments passed to this function.');
    else
        parent = varargin{1};
    end
    
    paths = []; % default empty output
    
    fields = fieldnames(structVar);
    
    for i = 1:numel(fields)
        if strcmp(fields{i}, 'header')
            % skip header fields
            % fprintf('Skipped header value...\n');
            continue;
        end
        % initialize loop values
        currentValue = structVar.(fields{i});
        currentParent = [parent {fields{i}}];
        currentPaths = []; % structure: { cellWithPathParts, signalType }
        
        if isa(currentValue, 'struct')
            if isempty(setdiff({'header'; 'timestamp'; 'value'}, fieldnames(currentValue)))
                % if it only contains these three fields, it is a time signal
                currentPaths = {currentParent, 'valueTimesignal'};
                
            elseif size(currentValue, 2) > 1
                % indexed data
                % loop over the indexes here
                for j = 1:size(currentValue, 2)
                    tmpPaths = getSignalPathsFromVariable(currentValue(j), [currentParent num2str(j)]);
                    if ~isempty(tmpPaths)
                        if isempty(currentPaths) % first time some data is added
                            currentPaths = tmpPaths;
                        else
                            currentPaths = [currentPaths; tmpPaths];
                        end
                    end
                end
            
            else % go to the next level in the struct
                currentPaths = getSignalPathsFromVariable(currentValue, currentParent);
            end
        % not a struct
        elseif numel(currentValue) > 1 % possible data signal 
            if (numel(fields) - 1) == numel(setdiff(fields, {'timestamp'})) ...
                    && ~strcmp(fields{i}, 'timestamp')
                % add signals that have multiple signals with the same
                % timestamp data
                currentPaths = {currentParent, 'sharedTimesignal'};
            end
        end
        
        % add the paths from this loop to the result variaable
        if ~isempty(currentPaths)
            paths = [paths; currentPaths];
        else
            % fprintf('Skipped /%s\n', cell2mat(join(currentParent, '/')));
        end
    end
end

