function UHFLIStreamParser(streamDirectory, outputDirectory, varargin)
% UHFLIStreamParser Parses stream files from the Zurich Instrumenst UHF
%               into designated signal files.
%
%   UHFLIStreamParser(streamDirectory, outputDirectory)
%   INPUT
%       streamDirectory     Directory path containing the stream files.
%       outputDirectory     Directory paht to the directory where the
%                           parsed files should be located.
%
%   Author: Ties Verschuren (c) 2020

    %% configuration
    clockbase = 1.8e9; % the clockbase of the device: /dev*/clockbase
    
    if nargin < 2
        % Set the output director if none given.
        outputDirectory = [streamDirectory '_parsed' ];
    end
    
    %% parse varargin
    n = 1; % number of arguments
    selectedSignal = [];
    while n < (nargin - 2)
        if strcmp(varargin{n}, 'signal') && numel(varargin) > n
            selectedSignal = varargin{n + 1};
            n = n + 2; % go to the next argument
        else
            error('UHFLIStreamParser:invalidArgument', ...
                'Argument %s is not expected.', varargin{n});
        end
    end
    
    %% start time tracking
    tic;
    
    fprintf('Started parsing %s...\n', streamDirectory); 
    
    %% check inputs
    streamFiles = dir([streamDirectory '/*.mat']);

    if size(streamFiles, 1) < 1
        error('UHFLIStreamParser:noStreamFilesFound', 'No stream files found in "%s"', streamDirectory);
    end
    
    % Check output directory
    if ~exist(outputDirectory, 'dir')
        mkdir(outputDirectory);
    end
    
    
    %% Read first file
    firstStreamFileData = load( fullfile(streamFiles(1).folder, streamFiles(1).name), 'dev*'); % only load the dev* variable
    deviceName = cell2mat( fieldnames(firstStreamFileData) );
    
    signalNames = getSignalPathsFromVariable(firstStreamFileData);
    
    % Get selected signal if any
    if ~isempty(selectedSignal)
        selectedSignalIndex = 0;
        for i = 1:numel(signalNames)
            if isempty(setdiff(signalNames{i, 1}, selectedSignal))
                % found selected signal
                selectedSignalIndex = i;
                break; % stop searching
            end
        end
        if selectedSignalIndex > 0
            signalNames = {signalNames{selectedSignalIndex, :}};
            fprintf('Selected signal "%s" to parse...\n', ['/' cell2mat(join(selectedSignal, '/'))]);
        else
            error('UHFLIStreamParser:selectedSignalNotFound', ...
                'Unable to find selected signal "%s" in the current stream.', ...
                ['/' cell2mat(join(selectedSignal, '/'))]);
        end
    end
    
    %% initialize some status vars
    numSignals = size(signalNames, 1); % number of signals
    t0 = zeros(numSignals, 1); % the value of the first timestamp
    dataIndex = t0;
    
    nbytes = fprintf('Processed %d of %d files...', 0, numel(streamFiles));
    
    %% initialize .mat file handles and add first data
    matFiles = cell(numSignals, 1);
    for i = 1:numSignals
        % create directory if not exists
        directoryName = fullfile(outputDirectory, signalNames{i, 1}{1:(end - 1)});
        if ~exist(directoryName, 'dir')
            mkdir(directoryName);
        end
        % create Mat file
        fileName = fullfile(directoryName,  signalNames{i, 1}{end});
        matFiles{i} = matfile(fileName,'Writable',true);
        
        % Write the data from the first file
        signalType = signalNames{i, 2};
        
        switch signalType
            case 'valueTimesignal'
                signalData = getDataByPath(firstStreamFileData, signalNames{i, 1}{:}, 'value');
                timeData = getDataByPath(firstStreamFileData, signalNames{i, 1}{:}, 'timestamp');
            case 'sharedTimesignal'
                signalData = getDataByPath(firstStreamFileData, signalNames{i, 1}{:});
                timeData = getDataByPath(firstStreamFileData, signalNames{i, 1}{1:(end -1)}, 'timestamp');
        end
        
        t0(i) = double(timeData(1)); % convert uint to double
        dataIndex(i) = numel(signalData); 
        % write it to file
        matFiles{i}.Data = signalData;
        matFiles{i}.Time = double(timeData - t0(i)) ./ clockbase;
        matFiles{i}.Name = ['/' cell2mat(join(signalNames{i, 1}, '/'))];
        matFiles{i}.Source = streamDirectory;
    end
    
    % clear the data of the first stream file as it is not needed anymore
    clear firstStreamFileData;
    
    %% loop through the stream files
    fprintf(repmat('\b',1,nbytes))
    nbytes = fprintf('Processed %d of %d files...', 1, numel(streamFiles)); % print status message

    for i = 2:numel(streamFiles)
        streamFileData = load( fullfile(streamFiles(i).folder, streamFiles(i).name), 'dev*');
        
        for j = 1:numSignals
            signalType = signalNames{j, 2};
            
            switch signalType
                case 'valueTimesignal'
                    signalData = getDataByPath(streamFileData, signalNames{j, 1}{:}, 'value');
                    timeData = getDataByPath(streamFileData, signalNames{j, 1}{:}, 'timestamp');
                case 'sharedTimesignal'
                    signalData = getDataByPath(streamFileData, signalNames{j, 1}{:});
                    timeData = getDataByPath(streamFileData, signalNames{j, 1}{1:(end -1)}, 'timestamp');
            end
            
            timeDouble = double(timeData - t0(j)) ./ clockbase;
            
            % skip adding if the same time and data is added
            if numel(signalData) > 1 ...
                    || matFiles{j}.Data(1, dataIndex(j)) ~= signalData(1) ...
                    || matFiles{j}.Time(1, dataIndex(j)) ~= timeDouble(1)
                % not the same value, so add it
                nData = numel(signalData);
                matFiles{j}.Data(1, (dataIndex(j) + 1):(dataIndex(j) + nData)) = signalData;
                matFiles{j}.Time(1, (dataIndex(j) + 1):(dataIndex(j) + nData)) = timeDouble;
                dataIndex(j) = dataIndex(j) + nData;
            end
        end
        
        % update status message
        fprintf(repmat('\b',1,nbytes))
        nbytes = fprintf('Processed %d of %d files...', i, numel(streamFiles));
    end
    fprintf('\nDone in %.2f seconds\n\n', toc);
end