% Runs a phase sweep for the simulation
% you need to load the parameters before running this
% Author: Ties Verschuren (c) 2019

if (0 == exist('omega_0', 'var'))
    error('Did you forget to load Duffing parameters?');
end
%% model parameters
model = 'PLL_duffing_with_thermal_noise_nonlinear_damping';
phase = pi/2;
minGateTime = 1 / ( (omega_0 / 100) / (2*pi));

% Simulation stop time
simulationTime = 1e6; % normalized time
% Data to discard due to setteling
skipTime = 1e5; % normalized time

% sweep variable parameters
varName = 'phase';
% varValues = deg2rad(10:10:180);%[1:0.1:10] * (2*pi / 10);
varValues = deg2rad(10:10:100); %[deg2rad(10:10:100) deg2rad(170:3:180)];

%% build simulation objects
load_system(model);

clear sims;
sims(1:numel(varValues)) = Simulink.SimulationInput(model);

for i = 1:numel(varValues)
    sims(i) = sims(i).setVariable(varName, varValues(i), 'Workspace', model);
%     y0=varValues(i) * Q / (omega_0^2);
%     sims(i) =  sims(i).setVariable('y0', y0, 'Workspace', model);
    sims(i) = sims(i).setModelParameter('StartTime', '0', 'StopTime', num2str(simulationTime));
end

%% run simluations

% clear big variables that not should be transfered to the simulations
clear simOut out

Simulink.sdi.setAutoArchiveMode(false); %prevent writing a lot of files
Simulink.sdi.setArchiveRunLimit(0);

simOut = parsim(sims, 'TransferBaseWorkspaceVariables', 'on');

%% Plot freq-amp and amp phase relation

if ~strcmp('phase', varName)
    warning('This one expects the phase to be the sweep variable');
end

amps = zeros(1, numel(varValues));
freqs = amps;
phases = varValues;
for i = 1:numel(varValues)
    last1percentDataPoints = round(numel(simOut(i).omega.Data)*0.01);
    amps(i) = mean(simOut(i).amp.Data((end-last1percentDataPoints):end));
    freqs(i) = mean(simOut(i).omega.Data((end-last1percentDataPoints):end));
end

plotAmpFreqGraph(amps, freqs, {['F = ' num2str(F) ]});
plotAmpPhaseGraph(amps, phases, {['F = ' num2str(F) ]});
plotFreqPhaseGraph(freqs,-1* phases, {['F = ' num2str(F) ]});

% return
%% plot allan deviation
figure;
minADVals = zeros(1, numel(varValues));
minTaus = zeros(1, numel(varValues));
for i = 1:numel(varValues)
    data = simOut(i).omega.Data(simOut(i).omega.Time >= skipTime);
    % sample frequency
    f_s = numel(simOut(i).omega.Time) / (simOut(i).omega.Time(end) - simOut(i).omega.Time(1));
    [adev, tau] = allandev(data, f_s, 'quiet', false, 'minGateTime', minGateTime, 'pointsPerDecade', 999999, 'minSamples', 10);
    [minADVals(i), idx] = min(adev);
    minTaus(i) = tau(idx);
    loglog(tau, adev, 'DisplayName', [varName ' = ' num2str(varValues(i))]);
    if i == 1
        hold on;
    end
end
legend('Location', 'best')
grid on;
xlabel('Time constant \tau (s)');
ylabel('Allan deviation');

%% Plot minimal AD

figure;
title('Minimal Allan Deviation for different PLL controlled phase');
yyaxis left; 
semilogy(rad2deg(varValues), minADVals, '-o');
xlabel('Phase (degree)');
ylabel('min(\sigma_y)');
yyaxis right;
semilogy(rad2deg(varValues), minTaus, '-o');
ylabel('\tau at min(\sigma_y) (seconds)')


% return
%% plot frequencies
figure;

for i = 1:numel(varValues)
    data = simOut(i).omega.Data(simOut(i).omega.Time >= skipTime);
    time = simOut(i).omega.Time(simOut(i).omega.Time >= skipTime);
    plot(time, data, 'DisplayName', [varName ' = ' num2str(varValues(i))]);
    if i == 1
        hold on;
    end
end
legend('Location', 'best')
grid on;
xlabel('Time (s)');
ylabel('Frequency (Hz)');