##########################################
# Landscapes of Trade data visualization #
##########################################

rm(list=ls())
library(tidyverse)
library(RColorBrewer)
library(sf)
library(ggpubr)
library(patchwork)

# read data
## read open data Landscapes of Trade
corridor <- st_read("datavisualization/data/corridor.gpkg") 
buildings <- read_csv("datavisualization/data/buildings2021NL_repo.csv") |>
  mutate(Size_cat = fct_relevel(Size_cat,"S","M","L","XL","XXL"))
## read open data CBS geometry corop regions
corop <- st_read("datavisualization/data/cbsgebiedsindelingen2022.gpkg", layer = "cbs_coropgebied_2022_gegeneraliseerd" )
  corop$statcode <- str_sub(corop$statcode, -2, -1)
  corop$statcode <- as.numeric(corop$statcode)
  corop$statcode <- as.character(corop$statcode)
## read processed data Landscapes of Trade
coropdata <- read_csv("datavisualization/data/coropdata.csv")
  coropdata$statcode <- as.character(coropdata$Corop_nr)
lq <- read_csv("datavisualization/data/lq.csv") #location quotient per region
multi1 <- read.csv2("datavisualization/data/multicriteria1.csv") |> rename(factor = ï..factor) #multicriteria weights
multi2 <- read.csv2("datavisualization/data/multicriteria2.csv") |> rename(factor = ï..factor) #multicriteria weights


# Figures Chapter 1
##Figure 1.7 indexed growth (trade and logistics only!)
### https://www.cbs.nl/nl-nl/longread/de-nederlandse-economie/2021/economische-groei-en-het-inkomen-van-nederlanders?onepage=true
### https://opendata.cbs.nl/statline/#/CBS/nl/dataset/84407NED/table?ts=1699196167197
### read index data
bbp <- read_csv2("datavisualization/data/CBS_bbp_cap_1980-2020.csv")
poptrade <- read_csv2("datavisualization/data/CBS_In_en_uitvoerwaarde_1980-2021.csv")
dc_area <- buildings |> filter(Function_n<3) %>% select(Constr_yea,Footprint) %>% 
  group_by(Constr_yea) %>% summarise(sum = sum(Footprint)) %>% mutate(area = cumsum(sum)) |>
  rename(year = Constr_yea)
### index values 1980
poptrade1980 <- poptrade |> filter(year == 1980) |> 
  mutate(trade1980 =`Invoerwaarde (mln euro)` + `Uitvoerwaarde (mln euro)`)
trade1980 <- poptrade1980$trade1980
pop1980 <- poptrade1980$population
bbpcap1980 <- bbp$`Bbp per hoofd (euro's van 2020)`[bbp$year == 1980]
area1980 <- dc_area$area[dc_area$year == 1980]

poptradebbp <- poptrade |> left_join(bbp) |> left_join(dc_area) |>
  rename(bbpcap = `Bbp per hoofd (euro's van 2020)`) |>
  mutate(trade = `Invoerwaarde (mln euro)`+`Uitvoerwaarde (mln euro)`,
         trade = trade/trade1980, #index 1980 = 100
         population = population/pop1980,
         bbpcap = bbpcap/bbpcap1980,
         area = area/area1980) |>
  select(year, population, bbpcap, trade, area)
figdata1.7 <- pivot_longer(poptradebbp, -year)
### plot index graph  
figdata1.7 |> ggplot(aes(x = year, y = value, color = name)) +
  geom_line() +
  theme_bw(base_size = 12) +
  scale_color_manual(name = "",
                     labels = c("DC footprint","GDP per capita (Euros of 2020)","Population",
                                "Trade volume (Euros of 2015)"),
                     values = c("black", "pink","red","blue")) +
  labs(x = "",y = "index 1980 = 1")
ggsave("datavisualization/export/1.7.pdf", height = 2000, width = 3000, units = "px", dpi = 300)


# Figures Chapter 2

## Figure 2.2 growth number
### calculate cumulative growth of buildings and building area
growth <- buildings |>
  group_by(Constr_yea) |> 
  summarise(number = n(), area = sum(Footprint)) |> 
  ungroup() |> #cumsum only works with ungrouped data!
  mutate(number = cumsum(number), area = cumsum(area))
### separate plots
a <- growth |> ggplot(aes(x = Constr_yea, y= number)) +
  geom_line() +
  theme_bw() +
  labs(x="", y="number of DCs") +
  xlim(1980,NA)
b  <- growth |> ggplot(aes(x = Constr_yea, y= area/1000000)) +
  geom_line() +
  theme_bw() +
  labs(x="",y="DC footprint (million sqm)") +
  xlim(1980,NA)
### combine plots
ggarrange(a,b, ncol = 2, nrow = 1)
ggsave("datavisualization/export/2.2.pdf", height = 2000, width = 3000, units = "px", dpi = 300)
ggsave("datavisualization/export/2.2.png", height = 2000, width = 3000, units = "px", dpi = 300)

## Figure 2.3 area chart footprint per size class per year
### treat data per size class
persize <- buildings |> filter(Constr_yea>1897) |>
  group_by(Constr_yea, Size_cat) |> 
  summarise(sum = sum(Footprint)) |>
  pivot_wider(names_from = Size_cat, values_from = sum) |> 
  ungroup()  #cumsum only works with ungrouped data!
figdata2 <- persize |> mutate(
  S = cumsum(replace_na(S, 0)),
  M = cumsum(replace_na(M, 0)),
  L = cumsum(replace_na(L, 0)),
  XL = cumsum(replace_na(XL, 0)),
  XXL = cumsum(replace_na(XXL, 0)),
  ) |> 
  pivot_longer(-Constr_yea) |>
  mutate(name = fct_relevel(name, "XXL","XL","L","M","S"))
### plot
figdata2 |> ggplot(aes(x= Constr_yea, y=value/1000000, fill = name)) +
  geom_area() +
  scale_fill_brewer(name = "footprint (sqm)", palette = "Blues", direction = -1, 
                    labels = c("XXL (> 40.000)","XL (20.000 - 40.000)","L (10.000 - 20.000)",
                               "M (2.500 - 10.000)","S (500 - 2.500)")) +
  xlim(1980,NA) +
  theme_bw(base_size = 12) +
  xlab("Construction year") + ylab("Footprint (million sqm)")
ggsave("datavisualization/export/2.3.pdf", height = 2000, width = 3000, units = "px", dpi = 300)
ggsave("datavisualization/export/2.3.png", height = 2000, width = 3000, units = "px", dpi = 300)

## Figure 2.10 function type per period
buildings |> filter(Constr_yea >= 1980) |>
  mutate(Function = fct_relevel(Function,"Trade, Import, Export", "Transport and logistic services",
                                "Logistic co-activity","XXL retail center")) |> 
  group_by(Year_cat, Function) |> summarise(sum = sum(Footprint)/1000000) |>
  ggplot(aes(x = Year_cat, y = sum, fill = Function)) +
  geom_bar(position = position_stack(reverse = TRUE), stat = "identity") +
  theme_bw(base_size = 12) +
  scale_fill_manual(name = "", breaks = c("Trade, Import, Export", "Transport and logistic services",
                                          "Logistic co-activity","XXL retail center"),
                    values = c("#FFD000","#FFA100","#87E1FF", "#59BADB")) +
  ylab("million sqm") + xlab("construction period")
ggsave("datavisualization/export/2.10.pdf", height = 2000, width = 3000, units = "px", dpi = 300)
ggsave("datavisualization/export/2.10.png", height = 2000, width = 3000, units = "px", dpi = 300)

## Figure 2.11 function per size
buildings |>
  mutate(Function = fct_relevel(Function,"Trade, Import, Export", "Transport and logistic services",
                                "Logistic co-activity","XXL retail center")) %>%
  group_by(Function, Size_cat) %>% summarise(sum = sum(Footprint)/1000000) %>%
  ggplot(aes(x = Function, y= sum, fill = Size_cat)) +
  geom_bar(position = position_stack(reverse = TRUE), stat = "identity") +
  theme_bw(base_size = 12) +
  scale_fill_brewer(name = "",palette = "Blues") +
  labs(y="million sqm",x="")
ggsave("datavisualization/export/2.11.pdf", height = 2000, width = 3000, units = "px", dpi = 300)
ggsave("datavisualization/export/2.11.png", height = 2000, width = 3000, units = "px", dpi = 300)

## Figure 2.12 average DC size
buildings |> group_by(Constr_yea) |> summarise(mean = mean(Footprint)) |>
  ggplot(aes(x= Constr_yea, y=mean)) +
  geom_line() + 
  theme_bw(base_size = 12) +
  xlab("Construction year") + ylab("Footprint (sqm)") +
  xlim(1980,NA)
ggsave("datavisualization/export/2.12.pdf", height = 2000, width = 3000, units = "px", dpi = 300)
ggsave("datavisualization/export/2.12.png", height = 2000, width = 3000, units = "px", dpi = 300)

## Figure 2.15 space quote per region
### define ESE corridor regions and treat data
ESE <- c('Arnhem/nijmegen', 'Groot-rijnmond', 'Midden-limburg','Midden-noord-brabant',
                    'Noord-limburg', 'Noordoost-noord-brabant','West-noord-brabant','Zuidoost-noord-brabant', 
                    'Zuidoost-zuid-holland', 'Zuidwest-gelderland')
coropdata <- coropdata |> mutate(corridor = case_when(Corop %in% ESE ~ "yes",
                                                         .default = "no")) 
coropdata$corridor <- as.factor(coropdata$corridor) 
coropdata <- coropdata |> mutate(corridor = fct_relevel(corridor,"yes","no"))
### plot
fig2.15 <- coropdata |> select(Corop,corridor, year,jobs_m) |> filter(year != 2020)
fig2.15 |> ggplot(aes(x = year, y = jobs_m,color=corridor, group = Corop)) +
  geom_path() +
  theme_bw(base_size = 12) +
  scale_color_manual(name ="",
                     labels = c("ESE-corridor regions","other regions"),
                    values = c("darkcyan","azure3")) +
  ylim(0,NA) +
  labs(x = "",y = "regional logistics jobs per sqm of warehouse")
ggsave("datavisualization/export/2.15.pdf", height = 2000, width = 3000, units = "px", dpi = 300)
ggsave("datavisualization/export/2.15.png", height = 2000, width = 3000, units = "px", dpi = 300)

## Figure 2.16 growth rate per region
### treat data
a <- buildings |> filter(Function_n <3,Year<=1980) |> group_by(Corop_nr) |> summarise(a = sum(Footprint))
b <- buildings |> filter(Function_n <3,Year<=2010) |> group_by(Corop_nr) |> summarise(b = sum(Footprint))
c <- buildings |> filter(Function_n <3) |> group_by(Corop_nr) |> summarise(c = sum(Footprint))
growth <- a |> left_join(b) |> left_join(c) |>
  mutate(growth1 = b-a, growth2 = c-b, statcode = as.character(Corop_nr)) |> select(statcode,growth1,growth2)
Fig2.16 <- corop |> left_join(growth)
### separate plots
x <- ggplot() +
  geom_sf(data = Fig2.16, aes(fill = growth1/1000000), color = "white") +
  scale_fill_gradient(guide = "none", low = "azure2", high = "cyan3") +
  geom_sf(data = corridor, color = "darkgrey", fill = NA) +  
  theme_classic(base_size = 12) +
  labs(title = "absolute growth 1980-2010") +  
  theme(axis.line = element_blank(),
        axis.text = element_blank(),
        axis.ticks = element_blank(),
        legend.position = "bottom"
  )
y <- ggplot() +
  geom_sf(data = Fig2.16, aes(fill = growth2/1000000), color = "white") +
  scale_fill_gradient(name="million sqm",low = "azure2", high = "cyan3") +
  geom_sf(data = corridor, color = "darkgrey", fill = NA) +  
  theme_classic(base_size = 12) +
  labs(title = "absolute growth 2011-2021") +  
  theme(axis.line = element_blank(),
        axis.text = element_blank(),
        axis.ticks = element_blank()
  )
### combine plots
combined <- x + y & theme(legend.position = "bottom")
combined + plot_layout(guides = "collect")
ggsave(filename = "datavisualization/export/2.16.png",
       dpi = 300, width = 3000, height = 2000, units = "px")
ggsave(filename = "datavisualization/export/2.16.pdf",
       dpi = 300, width = 3000, height = 2000, units = "px")

## Fig 2.17 LQ per region
### treat data
lq <- lq |> mutate(statcode = as.character(COROPC))
Fig2.17 <- corop |> left_join(lq)
Fig2.17_centr <- Fig2.17 |> st_centroid()
### plot
ggplot() +
  geom_sf(data = Fig2.17, aes(fill = LQ17), color = "white") +
  scale_fill_gradient(name = "location quotient",low = "azure2", high = "cyan3") +
  geom_sf(data = Fig2.17_centr, aes(size = reg_logis17),color = "darkcyan", shape = 16) +
  scale_size(name = "logistics employment", range = c(1,10)) +
  geom_sf(data = corridor, color = "darkgrey", fill = NA) +  
  theme_classic(base_size = 12) +
  theme(axis.line = element_blank(),
        axis.text = element_blank(),
        axis.ticks = element_blank(),
        legend.position = "right"
  )
ggsave(filename = "datavisualization/export/2.17.png",
       dpi = 300, width = 3000, height = 3000, units = "px")
ggsave(filename = "datavisualization/export/2.17.pdf",
       dpi = 300, width = 3000, height = 3000, units = "px")

## Fig 2.18 footprint (per capita)
### treat data
z <- coropdata |> filter(year == 2020) |> mutate(footprint_cap = footprint/population)
Fig2.18 <- corop |> left_join(z)
Fig2.18_centr <- Fig2.18 |> st_centroid()
### plot
ggplot() +
  geom_sf(data = Fig2.18, aes(fill = footprint_cap), color = "white") +
  scale_fill_gradient(name = "DC footprint per capita (sqm)",low = "azure2", high = "cyan3") +
  geom_sf(data = Fig2.18_centr, aes(size = footprint/1000000),color = "darkcyan", shape = 16) +
  scale_size(name = "DC footprint (million sqm)", range = c(1,10)) +
  geom_sf(data = corridor, color = "darkgrey", fill = NA) +  
  theme_classic(base_size = 12) +
  theme(axis.line = element_blank(),
        axis.text = element_blank(),
        axis.ticks = element_blank(),
        legend.position = "right"
  )
ggsave(filename = "datavisualization/export/2.18.png",
       dpi = 300, width = 3000, height = 3000, units = "px")
ggsave(filename = "datavisualization/export/2.18.pdf",
       dpi = 300, width = 3000, height = 3000, units = "px")

## Figure 2.22 logistics buildings in area with environmental category
### treat data
Env_cat <- buildings |> filter(Max_cat != "X") |> 
  mutate(Env_cat = as.numeric(Max_cat))
### plot
ggplot(Env_cat, aes(y = Env_cat, group = Size_cat)) +
  geom_bar(aes(fill = Size_cat)) +
  scale_fill_brewer(palette = "Blues", direction = 1, name = "building size class") +
  scale_y_continuous(breaks = seq(2,6, by = 1)) +
  theme_bw(base_size = 12) +
  labs(x = "number of logistics buildings",y = "maximum environmental category of business estate")
ggsave(filename = "datavisualization/export/2.22.png",
       dpi = 300, width = 3000, height = 2000, units = "px")
ggsave("datavisualization/export/2.22.pdf", height = 2000, width = 3000, units = "px", dpi = 300)

## Figure 2.25 (same as Chapter 7 fig 7.4) location factor heatmaps
### figure legend is graphically enhanced in Adobe Illustrator, see file copy in export folder
### treat data
Fig7.4a <- multi1 |> pivot_longer(-factor) |> 
  mutate(name = fct_relevel(name, c("global","material","remanufacturing","international","city")))
Fig7.4b <- multi2 |> pivot_longer(-factor)
### separate plots
a <- ggplot(Fig7.4a, aes(x=name,y=factor)) +
  geom_tile(aes(fill= as.factor(value))) +
  theme_bw(base_size = 12) +
  scale_fill_manual(name="weight", breaks = c(2,1,0.5,0,-1,-2),
                        values = c("darkgreen","chartreuse3","olivedrab1","yellow2", "darkorange","firebrick2")) +
  labs(x="",y="logistics location factor")
b <- ggplot(Fig7.4b, aes(x=name,y=factor)) +
  geom_tile(aes(fill= as.factor(value))) +
  theme_bw(base_size = 12) +
  scale_fill_manual(name="weight", breaks = c(-2,-4,-8,"no go"),
                    values = c("steelblue1", "steelblue3","steelblue4","lightgrey")) +
  labs(x="",y="environmental location factor")
### combine plots
combined2 <- a + b & theme(legend.position = "bottom")
combined2 + plot_layout(widths = c(5,1), guides = "collect")
ggsave(filename = "datavisualization/export/7.4.png",
       dpi = 300, width = 3000, height = 3000, units = "px")
ggsave(filename = "datavisualization/export/7.4.pdf",
       dpi = 300, width = 3000, height = 3000, units = "px")

