;; Code accompanying the paper "Agent-Based Simulation of Short-Term Peer-to-Peer Rentals: Evidence from the Amsterdam Housing Market"
;;
;; needs NetLogo 6.0.2 or later
;;
;; (c) 2014-2016 S. Picascia
;; (c) 2015-2017 N. Yorke-Smith
;; (c) 2018-     A. Overwater

extensions [gis table profiler csv]

globals [
  version save-directory save-directory-pictures file-name-results
  rnetlogo-param-regen     ; if nonzero, overwrite regeneration?according to this parameter
  city       ; city = those patches with instantiated 'city-zone' and 'neighbourhood'
  city-zones ; list of city zones of the city
  areas      ; list of areas of the city
  divisions1 ; raw GIS input 1
  divisions2 ; raw GIS input 2
  divisions-owned       ; raw GIS percentage of owned houses within quarter
  divisions-private-rent; raw GIS percentage of privately rented houses within quarter
  divisions-social      ; raw GIS percentage of social houses within quarter
  divisions-values      ; raw GIS percentage of valuation categories within quarter
  divisions-ethnicities ; raw GIS percentage of ethnicities within quarter
  divisions-space-tight     ; raw GIS percentage of tight home size category by concentration
  divisions-space-neutral   ; raw GIS percentage of normal home size category by concentration
  divisions-space-spacious1 ; raw GIS percentage of spacious home size category 1 by concentration
  divisions-space-spacious2 ; raw GIS percentage of spacious home size category 2 by concentration
  districts  ; = areas in this version of model (used to be different)
  valuation-bounds ; = list of bounds used for the valuation categories
  income-bounds    ; = list of bounds used for skewed incomes from CBS
  income-probabilities
  political-party-conditions ; estimated associated home conditions for each party
  total-owned        ; = total percentage of owned houses within city
  total-private-rent ; = total percentage of privately rented houses within city
  total-social       ; = total percentage of social houses within city
  total-values       ; = total percentage of valuation categories within city
  total-ethnicities  ; = total percentage of ethnicities within city
  total-space-tight     ; = total percentage of tight home size category within city
  total-space-neutral   ; = total percentage of normal home size category within city
  total-space-spacious1 ; = total percentage of spacious home size category 1 within city
  total-space-spacious2 ; = total percentage of spacious home size category 2 within city
  total-election-result ; = election result vector for all of Amsterdam
  allured-districts firstallure declining  ; metrics on district
  neighbourhoods-table  ; compiles a list of neighbourhoods at start for quick lookup when running
  occupancys-table      ; compiles a list of citizens in each neighbourhood, need to keep it updated
  election-table        ; list of election result vectors by area
  disp-freq  ; how many ticks between updates of entropy graph
  disp?      ; whether entropy graph should be displayed this tick
  ownersize  ; size of owner turtle
  city-allure av-income sd-income sd-income+ sd-income- occupation-rate gentrified-districts downfiltered-districts  ; reporters for analysis
  recolonisation degentrification recreation regentrification  ; more reporters
  housing-waiting-list  ; social housing waitlist
  immigrated-count emigrated-count  ; immigration and emigration counters
  inner-migrated-count ; inner-city migration counter
  stay-night-count ; Tourist nights counter
  tourist-revenue tourist-revenue-daily ; reporters for tourist economy
  entered-waiting-list entered-social-housing
  tax-rate ubi ; reporters for basic income
  total-levied
  median-Income medianPrices highestMedianPrice howmanySlums howmanyWellMaintained howmanyOvercrowded ; data lists for csv output
  dutch others  ; data lists for population-by-ethnicity-metagroup counting, for csv output
  extra-agent-creation-multiplier
]

breed [citizens citizen]  ; citizens are people in the city
breed [tourists tourist]
breed [people person]     ; people may or may not be citizens
links-own [time-lo]	      ; used to be called "time"; renamed for NetLogo 6
turtles-own [mobility-propensity rent-out-propensity months-here birthday culture income credit-mine dissonance tourists-as-neighbours place-changes time-in-a-slum owner? stay-length temp-income]
people-own [tried-populating?]
patches-own [
  city-zone city-zone-code         ; identifier from GIS
  condition price centre? cbd? dist local-dist premium price-gap months-empty
  area          ; = neighbourhood in this version of model (used to be different)
  neighbourhood neighbourhood-code ; identifier from GIS
  allure social? last-renovated
  available? tourist-price
  rent-out-history
  percentage-space-tight percentage-space-neutral percentage-space-spacious1 percentage-space-spacious2
  capacity
]

to setup
  ; clear-all will clear any globals that don't have an interface widget
  ; we use some such globals to drive netlogo from R
  ; so we save them to temp local variables (which clear-all doesn't clear)
  let my-rnetlogo-param-regen rnetlogo-param-regen
  clear-all
  set rnetlogo-param-regen my-rnetlogo-param-regen

  set version "0.7_AMS"
  set save-directory "results/"
  set save-directory-pictures (word save-directory "pics/")
  let gis-data-dir "GIS/"
  let gis-city-zone-file (word gis-data-dir "sel_NL_regio_wijken_2011/sel_NL_regio_wijken_2011.converted.shp")
  let gis-quarter-file (word gis-data-dir "sel_NL_regio_buurten_2011/sel_NL_regio_buurten_2011.converted.shp")
  let gis-owned-file (word gis-data-dir "0363BUURT_WONTEIGALL2011_Nge15/0363BUURT_WONTEIGALL2011_Nge15.converted.shp")
  let gis-private-rent-file (word gis-data-dir "0363BUURT_WONTPAHUALL2011_Nge15/0363BUURT_WONTPAHUALL2011_Nge15.converted.shp")
  let gis-social-file (word gis-data-dir "0363BUURT_WONTCORPALL2011_Nge15/0363BUURT_WONTCORPALL2011_Nge15.converted.shp")
  let gis-values-files (list
    (word gis-data-dir "0363BUURT_WOZLAAGALL2011_Nge15/0363BUURT_WOZLAAGALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_WOZMIDLAAGALL2011_Nge15/0363BUURT_WOZMIDLAAGALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_WOZMIDHOOGALL2011_Nge15/0363BUURT_WOZMIDHOOGALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_WOZHOOGALL2011_Nge15/0363BUURT_WOZHOOGALL2011_Nge15.converted.shp")
  ) ; Make sure its the same length as valuation-bounds
  let gis-ethnicities-files (list
    (word gis-data-dir "0363BUURT_ETNNLALL2011_Nge15/0363BUURT_ETNNLALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_ETNINDALL2011_Nge15/0363BUURT_ETNINDALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_ETNSURALL2011_Nge15/0363BUURT_ETNSURALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_ETNANTALL2011_Nge15/0363BUURT_ETNANTALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_ETNTURALL2011_Nge15/0363BUURT_ETNTURALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_ETNMARALL2011_Nge15/0363BUURT_ETNMARALL2011_Nge15.converted.shp")
    (word gis-data-dir "0363BUURT_ETNNINALL2011_Nge15/0363BUURT_ETNNINALL2011_Nge15.converted.shp")
  )
  let gis-space-tight-file (word gis-data-dir "0363CONC_WRMKRAPALL2011_Nge15/0363CONC_WRMKRAPALL2011_Nge15.converted.shp")
  let gis-space-neutral-file (word gis-data-dir "0363CONC_WRMNTRLALL2011_Nge15/0363CONC_WRMNTRLALL2011_Nge15.converted.shp")
  let gis-space-spacious1-file (word gis-data-dir "0363CONC_WRMRM1ALL2011_Nge15/0363CONC_WRMRM1ALL2011_Nge15.converted.shp")
  let gis-space-spacious2-file (word gis-data-dir "0363CONC_WRMRM2ALL2011_Nge15/0363CONC_WRMRM2ALL2011_Nge15.converted.shp")
  set valuation-bounds [[0 130000] [130000 225000] [225000 350000] [350000 525000]] ; Make sure its the same length as gis-values-files
  set valuation-bounds map [? -> map [?2 -> round (?2 / 180)] ?] valuation-bounds ; Correct for typical mortgage time (30 years) * 12 months * correction factor 2
  set income-bounds [[0 10000] [10000 20000] [20000 30000] [30000 40000] [40000 50000] [50000 100000] [100000 200000] [200000 800000]] ; Make sure its the same length as income-probabilities
  set income-probabilities [2608.5 2998.5 2366.6 1937.5 1206.7 1454.1 203.3 32] ; Make sure its the same length as income-bounds
  set income-probabilities map [? -> ? / 12807.3] income-probabilities ; From CBS
  set political-party-conditions [0.85 0.60 0.50 0.95 0.50 0.90 0.80 0.90 0.90 0.80 0.75 0.75 0.75 0.75 0.75 0.75 0.75]
  set total-owned 27.16
  set total-private-rent 25.36
  set total-social 47.48
  set total-values [7.86 46.94 29.49 13.05] ; From city monitor Amsterdam
  set total-ethnicities [49.71 15.29 8.84 1.51 5.26 9.05 10.34]
  set total-space-tight 14.99
  set total-space-neutral 32.52
  set total-space-spacious1 25.99
  set total-space-spacious2 12.21
  let normalize-spaces total-space-tight + total-space-neutral + total-space-spacious1 + total-space-spacious2
  set normalize-spaces 100 / normalize-spaces
  set total-space-tight total-space-tight * normalize-spaces
  set total-space-neutral total-space-neutral * normalize-spaces
  set total-space-spacious1 total-space-spacious1 * normalize-spaces
  set total-space-spacious2 total-space-spacious2 * normalize-spaces
  compile-election-results "CSV/2010_tweedekamer_tabel4.converted.csv" 17

  set disp-freq 5
  set disp? false
;  if record? [vid:reset-recorder]

  set firstallure 0
  set allured-districts table:make
  set gentrified-districts table:make
  set downfiltered-districts table:make
  set housing-waiting-list table:make
  set declining []
  set recolonisation []
  set degentrification []
  set regentrification []
  set recreation []

  set howmanyWellMaintained []
  set howmanyOvercrowded []
  set howmanySlums []
  set median-Income []
  set medianPrices []
  set highestMedianPrice []
  set dutch []
  set others []

  set extra-agent-creation-multiplier 4

  set tourist-revenue 0

  set city-allure n-values traits ["x"]
  ifelse enable-culture? [
    set mixing? true
    set pull? true
    set push? true
    set strong-neighbourhood? true
  ][ ; no culture
    set mixing? false
    set pull? false
    set push? false
    set strong-neighbourhood? false
  ]

  if non-slum-population >= slum-population [
    set slum-population non-slum-population + 1
    output-print word "Slum population limit must be greater than non-slum population limit, setting it to " non-slum-population
  ]

  ;; initialize patches
  set ownersize 0.5
  ask patches
    [ set pcolor blue ;white
      set social? false
      set months-empty 0
      set allure []
      set cbd? false
      ;set centreness 0.00001
      set condition random-float 1	  ; overwritten by GIS condition later
      set available? false
      set rent-out-history []
      set capacity 1
    ]

  ;; Amsterdam!
  set divisions1 gis:load-dataset gis-city-zone-file
  set divisions2 gis:load-dataset gis-quarter-file
  set divisions-owned gis:load-dataset gis-owned-file
  set divisions-private-rent gis:load-dataset gis-private-rent-file
  set divisions-social gis:load-dataset gis-social-file
  set divisions-values map [? -> gis:load-dataset ?] gis-values-files
  set divisions-ethnicities map [? -> gis:load-dataset ?] gis-ethnicities-files
  set divisions-space-tight gis:load-dataset gis-space-tight-file
  set divisions-space-neutral gis:load-dataset gis-space-neutral-file
  set divisions-space-spacious1 gis:load-dataset gis-space-spacious1-file
  set divisions-space-spacious2 gis:load-dataset gis-space-spacious2-file
  gis:set-world-envelope gis:envelope-of divisions2
  draw-city
  ;apply-prices  ; GIS prices and conditions
  gis:set-coverage-minimum-threshold 0.75
  gis:apply-coverage divisions1 "NAME" city-zone
  gis:apply-coverage divisions1 "CODE" city-zone-code
  gis:apply-coverage divisions2 "NAME" neighbourhood
  gis:apply-coverage divisions2 "CODE" neighbourhood-code
  ask patches [ifelse (is-string? city-zone) and (is-string? neighbourhood) [
    ;DEBUG output-print (word "set-area for " city-zone)
    set-area
  ][;output-print (word self " has no city-zone " city-zone)
  ]]
  apply-prices-by-area  ; GIS prices and conditions

  ;; city is those patches with instantiated 'city-zone', for them their 'area' = their 'city-zone'
  set city patches with [
    is-string? city-zone
  ]
  ;set districts remove-duplicates [neighbourhood] of patches  ; now done from areas
  compile-lists
  check-area-sanity  ; also sets districts = areas
  set neighbourhoods-table table:from-list map [? -> list ? (patches with [neighbourhood = ?]) ] districts  ; (needs NetLogo 6.0.2+)
  ;DEBUG print neighbourhoods-table

  set-centres-and-distances

  generate-population
  populate-city
  ask citizens [
    colour-agent
    shape-agent
  ]
  set-ownership-type

  ;;set-city-condition
  ask city [
    set-last-renovated
    colour-patches
  ]
  occupancy-table-update
  ;DEBUG print occupancys-table

  ;output-print "Ready!"
  print "Ready!"
  reset-ticks
;  if record? [vid:start-recorder]
  if profile? [profiler:start]
end



to compile-lists
  ;set districts remove-duplicates [ward] of city
  set city-zones remove-duplicates [city-zone] of city
  set areas remove-duplicates [area] of city
  output-print word (length city-zones) " city zones in city"
  output-print word (length areas) " areas in city"
end

to set-area
  set area neighbourhood
  ifelse city-zone = "Stadsdeel Binnenstad" [
    set cbd? true
    set plabel "CBD"
  ] [
    set cbd? false
  ]
end

to check-area-sanity
  set districts areas  ; districts = areas
end


to compile-election-results [csv-file num-parties]
  let raw-data csv:from-file csv-file
  set raw-data but-first raw-data ; Exclude heading row
  set total-election-result sublist (last raw-data) 4 (num-parties + 4)
  set raw-data but-last raw-data ; Exclude total row
  set election-table table:from-list map [? -> list first ? sublist ? 4 (num-parties + 4)] raw-data
end


to generate-population
  ;DEBUG output-print "Creating agents..."
  create-people N-Agents * extra-agent-creation-multiplier [render-human]  ; multiplied because we might have immigration
  set-default-shape turtles "circle"
  ;DEBUG output-print "Setting ethnicities..."
  ;create-ethnic-division ; Done after initial location
  ;DEBUG output-print "Setting economic statuses..."
  create-economic-status
end

to render-human
  set birthday 0
  ;if ticks = 0 [create-culture]
  create-culture
  reset-mobility-propensity
  reset-rent-out-propensity
  reset-stay-length
  set hidden? true
  set size 0.4
  set tried-populating? false
  set temp-income 0
end

to render-human-after-time-zero
  set birthday ticks
  create-culture-after-time-zero
  reset-mobility-propensity
  reset-rent-out-propensity
  reset-stay-length
  ;set hidden? true
  set size 0.4
  colour-agent
  shape-agent
  set temp-income 0
end

to render-tourist
  set birthday ticks
  create-culture-after-time-zero
  reset-stay-length
  set size 0.3
end

to reset-mobility-propensity
  set mobility-propensity (random-float prob-move) + 0.01
end

to reset-rent-out-propensity
  set rent-out-propensity (random-float prob-rent-out) + 0.01
end

to reset-stay-length
  set stay-length 1 + (random 10)
end

;; culture item 0: ethnicity
;;   0 => Dutch
;;   1 => Other Western
;;   2 => Surinamese
;;   3 => Antillean
;;   4 => Turkish
;;   5 => Moroccan
;;   6 => Other Non-Western
to create-culture
  set culture n-values traits [random values]
  set culture replace-item 0 culture (length divisions-ethnicities)  ; this means: ethnicity unknown/not allocated
end

to create-culture-after-time-zero
  set culture n-values traits [random values]
  set culture replace-item 0 culture (initial-ethnicity-after-time-zero)
end

to create-economic-status
  ifelse enable-economy?
  [ask people [set credit-mine credit]]
  [ask people [set credit-mine 1]]

  ifelse real-data? [ ; real numbers
    output-print "Setting Dutch incomes"
    ;; from CBS
    ;; https://opendata.cbs.nl/statline/#/CBS/nl/dataset/83931NED/table?dl=D4D1
    ;; set income
    ask people [set income 0]
    let totalpop count people
    (foreach income-probabilities income-bounds [[p b] ->
      ask n-of (round p * totalpop) people with [income = 0] [set income round (random-float-between (first b) (last b))]
    ])
    ask people with [income = 0] [set income round (random-float-between (first (first income-bounds)) (last (last income-bounds)))]
  ][ ; not real numbers
    ;; set income
    ask people [set income random-float 1]  ; initial random
  ]

  let totalpop count people
  ask people [set owner? false]
end


to populate-city
  output-print "Populating city..."
  while [count citizens < N-Agents and any? people with [not tried-populating?]] [
    ask one-of people with [not tried-populating?] [try-populate-one-person city]
  ]
end

to try-populate-one-person [where]
  ;DEBUG print (word "trying to populate person to " where)
  set where ensure-patchset-of-city where
  ;DEBUG output-print where
  ;let acceptable where with [not is-slum? and count citizens-here < non-slum-population and price <= ([income] of myself * [credit-mine] of myself)]  ; was: city with ...
  let acceptable where with [not is-slum? and count citizens-here < 1]
  ;DEBUG if (count acceptable = 0) [print (word "  acceptable 1 is " (count acceptable))]
  set acceptable acceptable with [price <= ([income] of myself * [credit-mine] of myself)]
  ;DEBUG if (count acceptable = 0) [print (word "  acceptable 2 is " (count acceptable))]  ; usually, if can't populate a person it's because he can't afford a place
  if owner? [set acceptable acceptable with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else
  ifelse any? acceptable [
    ;; there is somewhere I can afford
    let location one-of acceptable
    move-to-place-and-spread-out-turtles-and-no-update-occupancy-table location
    set months-here 0
    set breed citizens
    ask people [set tried-populating? false]
    ; owner flag is set according to ethnicity (elsewhere)
    set hidden? false
    ; Now that location is known set ethnicity
    set culture replace-item 0 culture (initial-ethnicity-of-district ([neighbourhood-code] of location))
  ] [
    ;DEBUG print (word "  failed to populate person to " ([area] of one-of where))
    set tried-populating? true
  ]
end

to-report initial-ethnicity-of-district [here]
  let percentages total-ethnicities
  let feature-vectors map [? -> gis:find-one-feature ? "CODE" here] divisions-ethnicities
  if not member? nobody feature-vectors [
    set percentages map [? -> gis:property-value ? "PERCENTAGE"] feature-vectors
  ]
  let randomized random-float 100
  (foreach percentages (n-values length divisions-ethnicities [i -> i]) [[p i] ->
    ifelse randomized < p [
      report i
    ] [
      set randomized randomized - p
    ]
  ])
  report length divisions-ethnicities - 1 ; Assume rounding error in percentages
end

to-report initial-ethnicity-after-time-zero
  let randomized random-float 100
  (foreach total-ethnicities (n-values length divisions-ethnicities [i -> i]) [[p i] ->
    ifelse randomized < p [
      report i
    ] [
      set randomized randomized - p
    ]
  ])
  report length total-ethnicities - 1 ; Assume rounding error in percentages
end


;; prices artificial, based on condition
to apply-prices
  ask patches [
    if condition > 1 [set condition 1]
    set price condition + 0.1
    if price > 1 [set price 1]
  ]
end

;; prices from actual data
to apply-prices-by-area
  ifelse real-data? [
    ask patches [
      let lb-prob (initial-maintenance-of-district area) * 0.8
      let ub-prob (initial-maintenance-of-district area) * 1.2
      set condition (random-float-between lb-prob ub-prob)
      if condition < 0.05 [set condition 0.05]
      if condition > 1 [set condition 1]
      ifelse is-string? area [
        let bounds initial-price-of-district neighbourhood-code
        set price random-float-between (item 0 bounds) (item 1 bounds)
        ;if price > 1 [set price 1]
        ;DEBUG output-print (word self " in area " area " has price " price)
      ] [
        set price 0
      ]
    ]
;    ask patches with [msoa11 = wherearewe] [set price random-poisson areaprice]
  ] [
    print "INFO: apply-prices-by-data -> apply-prices"
    apply-prices
  ]
end

;; initial prices, Amsterdam ~2011
to-report initial-price-of-district [here]
  let percentages total-values
  let feature-vectors map [? -> gis:find-one-feature ? "CODE" here] divisions-values
  if not member? nobody feature-vectors [
    set percentages map [? -> gis:property-value ? "PERCENTAGE"] feature-vectors
  ]
  let randomized random-float 100
  (foreach percentages valuation-bounds [[p v] ->
    ifelse randomized < p [
      report v
    ] [
      set randomized randomized - p
    ]
  ])
  report last valuation-bounds  ; assume rounding error in percentages
end

;; initial maintenance conditions, Amsterdam ~2011
to-report initial-maintenance-of-district [here]
  let election-results table:get-or-default election-table here total-election-result
  report (reduce + (map * election-results political-party-conditions)) / 100
end


to set-centres-and-distances
  if kind = "policentric" [
    foreach areas [ ? ->
      let x (([pxcor] of max-one-of city with [area = ?] [pxcor] - [pxcor] of min-one-of city with [area = ?] [pxcor]) / 2) + [pxcor] of min-one-of city with [area = ?] [pxcor]
      let y (([pycor] of max-one-of city with [area = ?] [pycor] - [pycor] of min-one-of city with [area = ?] [pycor]) / 2) + [pycor] of min-one-of city with [area = ?] [pycor]
      if member? patch x y city [
      ask patch x y [
        if not any? neighbors with [not member? self city] [
          set centre? true
          ;set pcolor blue
        ]
      ]
    ] ]
  ] ; endif policentric

  ;; compute dist, which records distance from me to the chosen centre, which in turn depends on type of city
  ask city [
    let centre min-one-of city with [city-zone = "Stadsdeel Binnenstad"] [distance myself]  ;; monocentric city
    if kind = "policentric" [
      set centre min-one-of city with [centre? = true] [distance myself]  ;; policentric city
    ]
    ifelse kind = "no centres" [set dist 1] [set dist distance centre]
  ]
end

to set-neighbourhood-nogis
  set cbd? false
  if pxcor >= -10 and pxcor < -2 and pycor > 2 and pycor <= 10 [set neighbourhood "nw"]
  if pxcor <= 10 and pxcor > 2 and pycor > 2 and pycor <= 10 [set neighbourhood "ne"]
  if pxcor >= -2 and pxcor <= 2 and pycor >= -2 and pycor <= 2 [set neighbourhood "c"]
  if pxcor >= -10 and pxcor < -2 and pycor < -2 and pycor >= -10 [set neighbourhood "sw"]
  if pxcor >= -10 and pxcor <= -3 and pycor > -4 and pycor < 4 [set neighbourhood "w"]
  if pxcor <= 10 and pxcor > 2 and pycor < -2 and pycor >= -10 [set neighbourhood "se"]
  if pxcor <= 10 and pxcor >= 3 and pycor > -4 and pycor < 4 [set neighbourhood "e"]
  if pxcor >= -3 and pxcor <= 3 and pycor < -2 and pycor >= -10 [set neighbourhood "s"]
  if pxcor >= -3 and pxcor <= 3 and pycor > 2 and pycor <= 10 [
    set neighbourhood "n"
    set cbd? true
    set plabel "CBD"
   ]
end

to set-ownership-type
  gis:apply-coverage divisions-space-tight "PERCENTAGE" percentage-space-tight
  gis:apply-coverage divisions-space-neutral "PERCENTAGE" percentage-space-neutral
  gis:apply-coverage divisions-space-spacious1 "PERCENTAGE" percentage-space-spacious1
  gis:apply-coverage divisions-space-spacious2 "PERCENTAGE" percentage-space-spacious2

  ask city [
    let percentage-owned total-owned
    let percentage-private-rent total-private-rent
    let feature-vector-owned gis:find-one-feature divisions-owned "CODE" neighbourhood-code
    let feature-vector-private-rent gis:find-one-feature divisions-private-rent "CODE" neighbourhood-code
    if feature-vector-owned != nobody and feature-vector-private-rent != nobody [
      set percentage-owned gis:property-value feature-vector-owned "PERCENTAGE"
      set percentage-private-rent gis:property-value feature-vector-private-rent "PERCENTAGE"
    ]
    let randomized random-float 100
    ifelse randomized < percentage-owned [
      ask citizens-here [
        set owner? true
        set size ownersize
      ]
      set social? false
    ] [
      ask citizens-here [set owner? false]
      ifelse randomized < percentage-private-rent + percentage-owned [
        set social? false
      ] [
        set social? true
      ]
    ]

    if not (percentage-space-tight <= 0 or percentage-space-tight >= 0) [set percentage-space-tight total-space-tight]
    if not (percentage-space-neutral <= 0 or percentage-space-neutral >= 0) [set percentage-space-neutral total-space-neutral]
    if not (percentage-space-spacious1 <= 0 or percentage-space-spacious1 >= 0) [set percentage-space-spacious1 total-space-spacious1]
    if not (percentage-space-spacious2 <= 0 or percentage-space-spacious2 >= 0) [set percentage-space-spacious2 total-space-spacious2]
    set randomized random-float 100
    ifelse randomized < percentage-space-tight [
      set capacity 0
    ] [
      ifelse randomized < percentage-space-tight + percentage-space-spacious1 [
        set capacity 2
      ] [
        set capacity 3
      ]
    ]
  ]
end

to set-last-renovated
  if condition <= 0.15 [set last-renovated 3600 + random 3600]
  if condition > 0.15 and condition <= 0.25 [set last-renovated 1800 + random 3600]
  if condition > 0.25 and condition <= 0.5 [set last-renovated 1800 + random 1800]
  if condition > 0.5 and condition <= 0.75 [set last-renovated 720 + random 1800]
  if condition > 0.75 [set last-renovated random 1440]
  ; Bugfix for both 0.6.5 Beirut and 0.7 Amsterdam: Initial last-renovated must be negative
  set last-renovated (- last-renovated)
end


;; ===== GO! =====

to go
  ;DEBUG type (word "[" ticks "]")
  ifelse ticks mod 30 != 0 [type "."] [print "."]
  ;if fixed-premium? = false [set-premia]
  ifelse ticks mod disp-freq = 0 [set disp? true] [set disp? false]

  ; optimization: compute and cache occupancys of neighbourhoods, to avoid their recomputation in next code block
  let current-occupancy-percentages table:from-list map [place -> list place (occupancy place "go")] table:keys neighbourhoods-table  ; (needs NetLogo 6.0.2+)
  ;print current-occupancy-percentages

  ask city [
    ifelse have-demand-premium [
      ;let occupancy-of-nhood (occupancy neighbourhood "go")
      let occupancy-of-nhood table:get current-occupancy-percentages neighbourhood
      ifelse occupancy-of-nhood >= 0.8   ;; NON DEVE ESSERE SOLO NEIGHBOURHOOD ! [should not be just neighbourhood]
        [set premium 1 + renovation-premium]
      [ifelse occupancy-of-nhood >= 0.5
        [set premium 1 + random-float renovation-premium]
        [set premium 1 + random-float (renovation-premium * 0.8)]
      ]
    ] [set premium 1 + renovation-premium]
    if enable-economy? [
      if gaps = "mean"    [set-gaps-mean]
      if gaps = "max"     [set-gaps-max]
      if gaps = "unified" [set-gaps-unified]
      if gaps = "new"     [set-gaps-new]
      if gaps = "lnd"     [set-gaps-lnd]
      decay
      if ticks mod 30 = 0 [update-price]
    ]
    if ticks mod 30 = 0 [update-emptiness]
    ;update-centreness
  ]

  ;; check-new-allure
  if ticks > 0 and (ticks = 180 or ticks mod 360 = 0) [
    ;DEBUG print (word "go: tick " ticks " starting allure update")
    set-city-allure
    ;ask city [set-allure-for-high-occupancy-districts]
    foreach districts [? -> set-allure-for-high-occupancy-district ? (table:get current-occupancy-percentages ?)]
    ;DEBUG print (word "go: tick " ticks " -- done allure update")
  ]
  ;if ticks < 1000 and any? patches with [allure = 0] [check-new-allure]
  ;if ticks mod 360 = 0 and table:length allured-districts > 0 [check-existing-allure]

  ;; +++++++ ADD SOMETHING HERE TO MONITOR GENTRIFICATION / SEGREGATION (not related to culture) +++++++++++++++

  if ticks mod 30 = 0 [
    update-links
    update-dissonance
    update-propensity

    ; Distribute Basic Income
    let gross-income sum [income] of citizens
    if basic-income = "Fixed Tax Rate" [
      let levied gross-income * fixed-tax-rate
      set ubi levied / count citizens
      print (word "Levied " levied " of gross income " gross-income " and distributed among " count citizens " citizens")
      ask citizens [set temp-income temp-income - income * fixed-tax-rate + ubi]
      set total-levied total-levied + levied
    ]
    if basic-income = "Fixed Income" [
      let levied count citizens * fixed-ubi
      set tax-rate levied / gross-income
      print (word "Levied " levied " of gross income " gross-income " and distributed among " count citizens " citizens")
      ask citizens [set temp-income temp-income - income * tax-rate + fixed-ubi]
      set total-levied total-levied + levied
    ]
    if mixing? [interact]
    ask citizens [
      set months-here months-here + 1
      if decide-moving [
        seek-place
        ; Evaluate results of seek-place for inner-city migration recording
        if not hidden? [set inner-migrated-count inner-migrated-count + 1]
      ]
      ; set size 0.01 * months-here
      ; if size >= 1.4 [set size 1.4]
      set temp-income 0
    ]
  ]

  if ticks > 0 and ticks mod 180 = 0 [  ;; every six months
    if enable-economy? [do-business]
    if immigration? [immigrate]
    if emigration?  [emigrate]
    if social-housing? [
      if (any? city with [social? and not any? citizens-here]) and (table:length housing-waiting-list > 0) [assign-social-housing]
      if any? city with [social?] [check-social-residents]
    ]
    if (regeneration? = "regular") and (random-float 1 > intervention-threshold) [regenerate]
    if (regeneration? = "EU-slums") [regenerate]
  ]

  if tourism? [
    tourists-update
    tourists-arrive
  ]

  ask city [colour-patches]
  ;cluster-cultures
  ;update-vacancy-rates
  set av-income mean [income] of citizens
  set sd-income standard-deviation [income] of citizens
  set sd-income+ standard-deviation [income] of citizens with [income >= av-income]
  set sd-income- standard-deviation [income] of citizens with [income <= av-income]
  set occupation-rate count city with [any? citizens-here] / count city
  check-prices

;  if record? [vid:record-view]
  tick
  if paper-stuff? and (ticks = 360 or ticks = 720 or ticks = 1440) [
    update-reports
;    save-screenshots
  ]
  if paper-stuff? and (ticks = end-tick - 1) [update-reports]  ; don't save screenshots at end-tick - 1

  if ticks = end-tick [
    if profile? [
      profiler:stop
      print profiler:report
      profiler:reset
    ]

;    if paper-stuff? [save-screenshots]
    if write-csv? or paper-stuff? [save-data]
    if write-csv? or paper-stuff? [save-price-deltas]
    report-population
    output-print word "Of people, final citizens: " (count citizens)
    ;if paper-stuff? [export-plot "Mean prices by neighbourhood" (word save-directory kapital)]
;    if record? [vid:save-recording "~/gentaxelling.mov"]
    print "Done"
    stop
  ]
end


to set-allure-for-high-occupancy-district [d occupancy-of-d]
  ;show "start set-allure-for-high-occupancy-district"
  ;if (occupancy d "safhod" > 0.3) [
  if (occupancy-of-d > 0.3) [
    ;print (word "go: set allure for " self " because occupancy " (occupancy self))
    let district-as-patchset one-of city with [neighbourhood = d]
    ask district-as-patchset [set-allure d]
    ;print (word "go: set allure for " self " -- done")
  ]
  ;show "done set-allure-for-high-occupancy-district"
end


;; ===== SOCIAL HOUSING =====

to enter-housing-list [agent place]
  ifelse social-housing? [ ; social housing exists: enter the waiting list
    ;; set housing-waiting-list lput (list(agent)(place)) housing-waiting-list
    table:put housing-waiting-list [who] of agent place
    ask agent [
      remove-turtle-from-occupancy-table
      set breed people
      set hidden? true
    ]
    set entered-waiting-list entered-waiting-list + 1
  ]
  [ ; else no social housing => leave city
    do-emigrate
  ]
end

;; time to leave social housing?
to check-social-residents
  ask citizens-on city with [social?] [
    if months-here >= max-time-in-social-housing * 12 [seek-place]
  ]
end

to assign-social-housing
  repeat min (list table:length housing-waiting-list count city with [social? and not any? citizens-here]) [
    let everybody []
    foreach table:keys housing-waiting-list [ ? -> set everybody lput turtle ? everybody ]
    let candidates everybody
    if length candidates > 4 [set candidates sublist candidates 0 4]
    let housedperson min-one-of turtle-set candidates [income]
    let locality table:get housing-waiting-list [who] of housedperson
    let dest one-of city with [social? and not any? citizens-here]
    if locality != "" and any? city with [social? and not any? citizens-here and neighbourhood = locality]
    [set dest one-of city with [social? and not any? citizens-here and neighbourhood = locality]]
    if housedperson != nobody and dest != nobody [ ; Should be the case, due to the loop repeat conditions, but bugs...
      move-to-social-housing housedperson dest
    ]
  ]
end

to move-to-social-housing [agent dest]
  table:remove housing-waiting-list [who] of agent
  ask agent [
      set breed citizens
      set months-here 0
      set hidden? false
      move-to-place-and-spread-out-turtles dest
  ]
  set entered-social-housing entered-social-housing + 1
end


;; ===== "BUSINESS" =====

to check-prices
  foreach districts [ ? ->
    ifelse member? ? declining
    [if median [price] of city with [neighbourhood = ?] >= 0.25 [set declining remove ? declining]]
    [if median [price] of city with [neighbourhood = ?] < 0.25 [set declining fput ? declining]]
    ]
end

;; Renovation happens twice a year according to available capital.
;; NYS where does this code do "twice a year"?
to do-business
  let howmany (Kapital * count city) / 2
  let goodgap city with [price-gap >= (price * profit-threshold) and not social? and condition <= 0.8];
  ; let goodgap city with [price-gap >= profit-threshold]; and condition <= 0.75];
  if count goodgap < howmany [set howmany count goodgap]
  let torenovate max-n-of howmany goodgap [price-gap]
  ask torenovate [renovate]
end

to renovate
  set price price + price-gap
  ;if price >= 1 [set price 0.98] ;FIXME-price
  set condition 0.95
  set last-renovated ticks
end

to decay
   let depr monthly-decay / 30
   if is-owned? [set depr depr / 2]   ; slower depreciation if owned
   ;ifelse (is-slum? and count citizens-here >= slum-population)
   ;  [set depr depr * 2]   ; faster depreciation if slum
   ;  [
       if (count citizens-here > non-slum-population) [set depr depr * 1.5]  ; faster depreciation if above normal population density
   ;]

   let time ticks - last-renovated
   if time < 1440 [set depr 0]
   if time >= 1440 and time <= 1800 [set depr depr / 2]
   if time >= 3600 and time <= 7200 [set depr depr * 2]
   if not any? citizens-here [set depr depr * 1.20]  ; faster depreciation if empty
   if count tourists-here > 1 [set depr depr * 2] ; faster depreciacion if some tourists
   if count tourists-here > capacity [set depr depr * 2] ; faster depreciacion if too much tourists
   ifelse condition - depr <= 0
      [set condition 0]
      [set condition condition - depr]

   if social? and condition <= intervention-threshold [set condition 0.55]  ; renovation of social housing by govt
   if cbd-price-control? and condition <= intervention-threshold and cbd? and random-float 1 < 0.2 [  ; investment in CBD by govt
     set condition 0.4 + random-float 0.35
     ;if price < 0.5 [set price (max list 0.75 condition)]  ; price control
   ]
end

to update-price
  ifelse social? [ ;; social housing
    set price (mean [price] of city) / 2]
  [ ;; not social housing
    let depr yearly-depreciation / 12   ; price naturally "decays" 2% per year
    let time (ticks - last-renovated) / 30
    if time <= 24 [set depr 0]
    if time > 24 and time <= 60 [set depr depr / 2]
    if time >= 120 and time <= 240 [set depr depr * 2]
    if months-empty > tolerable-vacancy [set depr depr * 2]
    if is-slum? [set depr depr * 2]  ; prices fall faster if above normal population density
    if length rent-out-history > 30 [set depr depr / 2] ; slower price deprecation for tourist profitability
    ifelse price - (price * depr) <= 0 [
      ifelse real-data? [set price 10] [set price 0.01]]
      [set price price - (price * depr)]

    if cbd-price-control? and cbd? [  ; CBD has minimum price set by govt
      ;if price < 0.5 [set price (max list 0.75 condition)]  ; price control FIXME-price
      if price < 0.5 * 10000 [set price (max list (0.75 * 10000) (condition * 10000))]  ; price control FIXME-price
    ]
  ]
end

to update-emptiness
  ifelse count citizens-here = 0
  [set months-empty months-empty + 1]
  [set months-empty 0]
end


;; ===== INDIVIDUALS' CHANGES =====

;; immigration happens twice a year
to immigrate
  let howmany 1 + ((count citizens * immigration-rate) / 2)
  ask n-of howmany people [do-immigrate]  ; immigrate some (any) people
end

to do-immigrate
  ;render-human-after-time-zero
  ;if table:has-key? housing-waiting-list who [table:remove housing-waiting-list who]
  ;set breed citizens
  ;set hidden? false
  ;seek-place

  let myprice income
  if enable-economy? [set myprice (income * credit-mine)]
  ifelse any? city with [price <= myprice and not any? citizens-here and not social?] [
    let whoami who
;      ; set income random-float 1
    set breed citizens
    set hidden? false
    seek-place
    render-human-after-time-zero
;        set birthday ticks
;        create-culture
;        colour-agent
;        reset-mobility-propensity
;      ]
    set immigrated-count immigrated-count + 1
  ] [
    show (word "Couldn't find a place to immigrate with income " myprice)
  ]
end


;; emigration happens twice a year
to emigrate
  let howmany 1 + ((count citizens * emigration-rate) / 2)
  ask n-of howmany citizens [do-emigrate]
end

to do-emigrate
  leave-city
  set emigrated-count emigrated-count + 1
end


to tourists-arrive
  set tourist-revenue-daily 0
  let howmany 1 + ((count citizens * tourist-arrival-rate) / 2)
  ask n-of howmany people [do-arrive]
  set tourist-revenue tourist-revenue + tourist-revenue-daily
end

to do-arrive
  render-tourist
  let mybudget income
  let available city with [available?]
  while [stay-length > 0] [
    let my-stay-length stay-length
    let options available with [tourist-price * my-stay-length <= mybudget]
    ifelse any? options [
      set breed tourists
      set hidden? false
      seek-accommodation available
      stop
    ] [
      set stay-length stay-length - 1
    ]
  ]
  if stay-length = 0 [
    show (word "Couldn't find a place to check in with budget " mybudget)
  ]
end

to tourists-update
  ask city [ ; Keep the history for enforcing maximum rent nights
    if (not empty? rent-out-history) and first rent-out-history <= ticks - 360 [
      set rent-out-history but-first rent-out-history ; Only useful to remember for at most a year
    ]
    if any? tourists-here [
      set rent-out-history lput ticks rent-out-history
      set price price * 1.0001 ^ (count tourists-here) ; Raise price if turns out to be more profitable
    ]
    if not any? citizens-here [ ; Currently no citizens to handle tourism, so manage it here as if by the "invisible market hand"
      ifelse available? [
        ifelse social? or length rent-out-history >= rent-out-maximum or count tourists-here >= capacity [
          set available? false
        ] [
          ifelse any? tourists-here [ ; Raise price if our current location is succesful
            set tourist-price tourist-price * 1.01 ^ (count tourists-here)
          ] [ ; Lower price if it's unsuccessful
            set tourist-price tourist-price * 0.90
          ]
        ]
      ] [
        if length rent-out-history < rent-out-maximum and count tourists-here < capacity and not social? [
          let avg price / 30 ; Bootstrap price is residental rental price per day
          if any? city with [available? and any? tourists-here] [set avg mean [tourist-price] of city with [available? and any? tourists-here]]
          if avg * 30 > price * 1.10 [ ; If profitable
            set available? true
            set tourist-price avg
          ]
        ]
      ]
    ]
  ]

  ask citizens [
    let myprice income * credit-mine
    ifelse any? tourists-on neighbors [
      set tourists-as-neighbours tourists-as-neighbours + 1
      if tourists-as-neighbours > 30 [set tourists-as-neighbours 30]
    ] [
      set tourists-as-neighbours tourists-as-neighbours - 1
      if tourists-as-neighbours < 0 [set tourists-as-neighbours 0]
    ]
    if tourists-as-neighbours > 5 [                      ; If neighbours are successful, then I want it too
      set rent-out-propensity rent-out-propensity * 1.10
    ]
    if myprice < ([price] of patch-here * 1.50) [        ; If I'm tight on budget, I need an extra income source
      set rent-out-propensity rent-out-propensity * 1.20
    ]
    if [available? and tourist-price < 0.001 * myprice] of patch-here [  ; If tourist earnings become neglectable consider quitting
      set rent-out-propensity rent-out-propensity * 0.90
    ]
    ifelse any? tourists-here [                          ; I'm successful. That tastes like more
      set rent-out-propensity rent-out-propensity * 1.20
      if count tourists-here > [capacity] of patch-here [
        set rent-out-propensity rent-out-propensity * 0.50 ; Too many tourists makes it less fun
      ]
    ] [                                                  ; I'm unsuccessful. Discouragement comes
      set rent-out-propensity rent-out-propensity * 0.95
    ]
    if rent-out-propensity > 1 [set rent-out-propensity 1]

    let my-rent-out-propensity rent-out-propensity
    if [social?] of patch-here [set my-rent-out-propensity my-rent-out-propensity * 0.5]
    if [length rent-out-history >= rent-out-maximum] of patch-here [
      set my-rent-out-propensity my-rent-out-propensity * 0.5 * 0.95 ^ (length rent-out-history - rent-out-maximum)
    ]
    ifelse random-float 1 < my-rent-out-propensity [
      ifelse [available?] of patch-here [ ; If I'm already advertising the house
        ifelse any? tourists-here [ ; Raise price if I'm succesful
          ask patch-here [set tourist-price tourist-price * 1.01 ^ (count tourists-here)]
        ] [ ; Lower price if I'm unsuccessful
          ask patch-here [set tourist-price tourist-price * 0.90]
	  if [tourist-price] of patch-here < 0.0005 * myprice [ ; But not too low
	    ask patch-here [set tourist-price 0.0005 * myprice]
	  ]
        ]
      ] [ ; I just made the decision to rent out the house
        set rent-out-propensity 1 ; Bonus to prevent too much decision switching
        ;if rent-out-propensity > 1 [set rent-out-propensity 1]
        let avg price / 30 ; Bootstrap price is residental rental price per day
        if any? city with [available? and any? tourists-here] [set avg mean [tourist-price] of city with [available? and any? tourists-here]]
        ask patch-here [
          set available? true
          ifelse avg < 0.001 * myprice [ ; Initially boost price if extremely low
            set tourist-price 0.001 * myprice
          ] [
            set tourist-price avg
          ]
        ]
      ]
    ] [
      if [available?] of patch-here [ ; If I was adversising the house
        set rent-out-propensity rent-out-propensity * 0.50 ; Bonus to prevent too much decision switching
        ask patch-here [set available? false]
      ]
    ]
  ]

  ask tourists [
    set stay-night-count stay-night-count + 1
    set stay-length stay-length - 1
    if stay-length = 0 [check-out]
  ]
end

;; shall I try to move somewhere else?
to-report decide-moving
  let param income * credit-mine + temp-income
  let my-mobility-propensity mobility-propensity
  if (owner? = true) [set my-mobility-propensity (mobility-propensity * 0.5)]  ; owners less likely to want to move
  if ([price] of patch-here > param and [social?] of patch-here = false and owner? = false) or  ; move if I can't afford the rent...
      (random-float 1 < my-mobility-propensity) [  ; ...or if I'm feeling like it
    set place-changes place-changes + 1
    report true
  ]
  report false
end

to leave-city
  remove-turtle-from-occupancy-table
  ask my-links [die]
  set breed people
  set hidden? true
end

to check-out
  ask my-links [die]
  set breed people
  set hidden? true
end


to update-links
 ask links [    ;; First we check existing links. If the agents are still neighbours we reinforce the relationship, if not we weaken it.
    let stillclose false
    ask one-of both-ends [if neighbourhood = [neighbourhood] of other-end or distance other-end <= 2 [
        set stillclose true] ]
    ifelse stillclose
    [if time-lo < 12 [set time-lo time-lo + 1]]
    [
      set time-lo time-lo - 1
      if time-lo <= 0 [die]
    ]
  ]
  ask citizens [  ;; Then we create new links for neighbours that still don't have one (i.e. new neighbours)
    let myneighbours other citizens with [distance myself <= 2 and link-neighbor? myself = false]
    let goodneighbours myneighbours with [(similarity self myself / traits) >= similarity-for-friendship]
    ask goodneighbours [
      create-link-with myself [
        set time-lo 1
        set hidden? true
      ]
    ]
  ]
end

to update-dissonance
  ask citizens [
    ifelse [condition] of patch-here <= slum-threshold or count citizens-here > non-slum-population or mean [condition] of other city in-radius 2 <= slum-threshold  ; slightly faster than old line below
    ;ifelse [condition] of patch-here <= slum-threshold or mean [condition] of other city in-radius 2 <= slum-threshold or count citizens-here > non-slum-population
      [set time-in-a-slum time-in-a-slum + 1]
      [set time-in-a-slum 0]
    if push? [
      let citizens-on-neighbors citizens-on neighbors
      ;if count citizens-on-neighbors > 0 [
      if any? citizens-on-neighbors [
        let alltraits count citizens-on-neighbors * traits
        let simil 0
        ; NYS: optimize by stopping the ask as soon as simil/alltraits > s-f-d
        ask citizens-on-neighbors [set simil simil + similarity self myself]
        ifelse (simil / alltraits) <= similarity-for-dissonance
          [set dissonance dissonance + 1]
          [set dissonance 0]
      ]
    ]
  ]
end

to update-propensity
  ask citizens [
    let myprice income * credit-mine + temp-income ; temp-income may or may not go here
    if time-in-a-slum = 0 and dissonance <= tolerable-dissonance [reset-mobility-propensity]
    if ((time-in-a-slum > 12) and (myprice > ([price] of patch-here * 1.20)))   ; If I can afford to, I leave the ghetto
      ;or (income >= ((median [condition] of neighbors) * 1.50))                ; This reflects the preference of middle class people towards status over convenience
      or (median [condition] of neighbors <= slum-threshold)
      [set mobility-propensity mobility-propensity * 1.50]
    if (dissonance > tolerable-dissonance) [
      set mobility-propensity mobility-propensity * 1.50
      if random-float 1 < 0.05 [mutate]
    ]
    if tourists-as-neighbours > 10 [                     ; Having tourists as neighbours is annoying
      set mobility-propensity mobility-propensity * 1.20
    ]
    if any? tourists-here [                              ; If I'm successful I am less inclines to move
      set mobility-propensity mobility-propensity * 0.80
    ]
    if mobility-propensity > 1 [set mobility-propensity 1]
  ]
end

;; The idea here is that prolonged co-location leads to cultural mixing.
;; We need each household to keep track of how long they have been neighbours with each other
to interact
   ask links with [time-lo > 6] [
    let a item 0 sort both-ends
    let b item 1 sort both-ends
    if similarity a b < traits [
      let whichone 1 + random (traits - 1)  ; trait 0 (ethnicity) never changes
      if item whichone [culture] of a != item whichone [culture] of b [
        ifelse random-float 1 <= 0.5
          [ask b [set culture replace-item whichone culture item whichone [culture] of a]]
          [ask a [set culture replace-item whichone culture item whichone [culture] of b]]
      ]
    ]
  ]
end

;to mutate-off
;  ask one-of citizens [
;    set culture replace-item (random traits) culture (random values)
;  ]
;end

to mutate
  let where [neighbourhood] of patch-here
  let trait (1 + random (traits - 1))   ; ethnicity doesn't change
  let most one-of modes [item trait culture] of citizens-on city with [neighbourhood = where]
  set culture replace-item trait culture most
end


;; ===== INTERVENTIONS =====

to build-social-housing [howmany]
  if social-housing? [
    let sofar 0
    let zerop min [price] of city
    let zeroc min [condition] of city
    let avg mean [price] of city with [count citizens-here > 0]
    let firstsocial nobody
    let worst city with [not any? citizens-here and price <= zerop and condition <= zeroc]
    ifelse any? worst
    [set firstsocial min-one-of worst [price-gap]]
    [set firstsocial max-one-of city [months-empty]]
    ask firstsocial [
      set social? true
      set price avg / 2
      set condition 0.95
      set sofar sofar + 1
      while [sofar < howmany] [
        ask one-of city in-radius 4 with [not social?] [
          if not any? citizens-here [
            set social? true
            set price avg / 2
            set condition 0.95
            set sofar sofar + 1
          ]
        ]
      ]
    ]
  ]
end

;; Regeneration is intended in the anglo-saxon, "small state but lets-be-compassionate-shall-we" way.
;; Money is put in the areas least desirable to investors (= those with the most narrow price-gap)
;; that are also empty and in run-down condition. These areas are brought to the maximum condition
;; and to the mean price of the city. The idea is to check whether this practice can trigger further private investment.
;;
;; Also, in case of targeted EU regeneration, focus is on (more frequent) small-scale improvement of worst areas.
to regenerate
  let zerop min [price] of city
  let zeroc min [condition] of city
  let avg mean [price] of city with [count citizens-here > 0]
  let worst city with [not any? citizens-here and price = zerop and condition = zeroc]
  if any? worst [
    ask min-one-of worst [price-gap] [
      ifelse regeneration? = "regular" [ ;; regular "anglo-saxon" intervention
        set price avg
        set condition 0.95
        ask neighbors with [not social?] [
          set price avg
          set condition 0.95
        ]]
      [if regeneration? = "EU-slums" [ ;; EU intervention focused on slums
        if zeroc <= max list slum-threshold intervention-threshold [
          set condition max list 0.5 zeroc
          set price avg
        ]
      ]]
    ]
  ]
end


; ===== RESIDENTIAL LOCATION PROCESS =====

;; When seeking a spot we consider vacant affordable places close to the origin (cbd) and with a pleasant cultural mix.
;; This is in line with Jackson et al. 2008, although they have a Schelling-like ethnic (not cultural) mix.
;; In this version agents only evaluate the CULTURAL ALLURE of a district, not the STATUS.
;; If we are to finegrain the model we could also include status in the decision process.

to seek-place
  let howmuch income * credit-mine
  ifelse pull? and table:length allured-districts > 0
  [ ;; alluring districts exist
    let where set-place
    ifelse where != "" [
      ifelse strong-neighbourhood? [relocate-to where howmuch] [weak-relocate-to where howmuch]
    ] [ ;; nowhere particular to relocate to, so go somewhere
      relocate howmuch
    ]
  ] [ ;; else no alluring districts
    relocate howmuch
  ]
end

to seek-accommodation [available]
  let mybudget income
  let my-stay-length stay-length
  let options available with [tourist-price * my-stay-length <= mybudget]
  let preferred options with [count tourists-here < capacity]
  let preferred2 preferred with [condition >= mean [condition] of available]
  ifelse any? preferred [
    ifelse any? preferred2 [
      check-in one-of preferred2 with-min [dist]
    ] [
      check-in one-of preferred with-min [dist]
    ]
  ] [
    check-in one-of options with-min [dist]
  ]
end

to-report set-place         ;;     ========= nononon questo si deve cambiare per rflettera la nuova composiziopne della allure che abbiamo appena fatto...
  ;let best_ftr 1
  let best_ftr traits * (similarity-threshold * 2)
  let bestdistrict ""
  foreach table:keys allured-districts [ ? ->
    let this_similarity similarity self one-of city with [neighbourhood = ?]  ;; FIXME this call to similarity has problem because its two inputs can be different lengths?
    if this_similarity >= best_ftr [
      set best_ftr this_similarity
      set bestdistrict ?
      ]
    ]
  report bestdistrict
end

to check-in [place]
  move-to place
  let my-stay-length stay-length
  ask citizens-here [ ; Receive payment
    set temp-income temp-income + [tourist-price] of place * my-stay-length
  ]
  set tourist-revenue-daily tourist-revenue-daily + [tourist-price] of place * my-stay-length
  ;; spread out myself instead of other turtles as I'm the guest
  setxy pxcor pycor  ; move to middle of patch
  set heading random 360  ; pick a direction
  fd random-float 0.3 + 0.05   ; move a bit
end

;; only point in the code where a turtle is assigned to a patch are the next pair of methods
to move-to-place-and-spread-out-turtles-and-no-update-occupancy-table [place]
  move-to place
  spread-out-other-turtles
end

to move-to-place-and-spread-out-turtles [place]
  if (pxcor != 0 and pycor != 0) [  ; (0,0) means no patch
    remove-turtle-from-occupancy-table
  ]
  move-to place
  add-turtle-to-occupancy-table
  spread-out-other-turtles
end

to spread-out-other-turtles
  if (any? citizens-here) [
    ask other turtles-here
      [ ;; spread out all other turtles
        setxy pxcor pycor  ; move to middle of patch
        set heading random 360  ; pick a direction
        fd random-float 0.3 + 0.05   ; move a bit
      ]
  ]
end

to relocate [will]  ;; will is dervied from how much willing to pay (i.e., as rent)
  set months-here 0

  ;let baseline city with [(count citizens-here <= acceptable-number-of-people-in-one-place) and (condition > min-condition) and (social? = false)]
  let baseline city with [(not is-slum?) and (social? = false) and (not any? citizens-here) and price <= will and not available?]
  if owner? [set baseline baseline with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else

  ifelse any? baseline [
;    let testbed n-of 6 city
    let number-of-places-to-benchmark-with min (list 20 (count city / 10))  ; at most look at 20 places to benchmark conditions
    let condi mean [condition] of n-of number-of-places-to-benchmark-with city
    ;let testbed n-of number-of-places-to-benchmark-with city
    ;;let testbed n-of (count city / 10) city
    ;let condi mean [condition] of testbed
;    let secondbest baseline with [(price <= [income] of myself) and (count citizens-here = 0) and (condition >= (mean [condition] of testbed - (mean [condition] of testbed * 0.15)))]  ; if we can't find a place we like then we move to one we can afford
    let secondbest baseline with [condition >= (condi - (condi * 0.15))]  ; if we can't find a place we like then we move to one we can afford
;    let secondbest baseline with [(count citizens-here <= acceptable-number-of-people-in-one-place) and (condition >= (condi - (condi * 0.15)))]  ; if we can't find a place we like then we move to one we can afford
;    if enable-economy? [
;       set secondbest secondbest with [price <= will]
;    ]
    ifelse any? secondbest
    [move-to-place-and-spread-out-turtles min-one-of secondbest [dist]]
    [move-to-place-and-spread-out-turtles min-one-of baseline [dist]]
  ]
  [enter-housing-list self ""]  ; if no place exists we apply for social housing
end

to relocate-to [place will]  ;; will is dervied from how much willing to pay
  set months-here 0

  let baseline city with [(count citizens-here < 1) and (not is-slum?) and (social? = false) and price <= will and not available?] ;Add to prevent people from moving to decrepit loc:; and (condition > 0)
  if owner? [set baseline baseline with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else

  ifelse any? baseline [
;    let testbed n-of 6 city
    let testbed n-of (count city / 10) city
    let condi mean [condition] of testbed
    let ideal baseline with [(neighbourhood = place) and (condition >= (condi - (condi * 0.15)))]
    ;let ideal baseline with [(neighbourhood = place) and (condition > 0.25)]
    ifelse any? ideal
      [move-to-place-and-spread-out-turtles min-one-of ideal [local-dist]]
      [ ;; no ideal, try second best
        let apex city with [neighbourhood = place]
        if kind = "policentric" [set apex apex with [centre? = true]]
        let acceptable baseline with [condition >= (condi - (condi / 2))]   ; TODO acceptable condition should vary across agents [i.e. each agent individual]
        let secondbest acceptable with [neighbourhood = place]               ; TODO if nothing is available in place look at the edges
        ifelse any? secondbest
        [move-to-place-and-spread-out-turtles min-one-of secondbest [local-dist]]
        [ ;; no second best, try somewhere acceptable
          ifelse any? acceptable
          [move-to-place-and-spread-out-turtles min-one-of acceptable [distance one-of apex]]             ; If we can't find anything in the desired area we try somewhere close
          [ ; nowhere acceptable, just pick from baseline
            move-to-place-and-spread-out-turtles min-one-of baseline [dist]]
        ]
     ]
  ]
  [enter-housing-list self place]  ; if no place exists we apply for social housing
end

;; this version is used if allure etc is switched off
to weak-relocate-to [place will]
  let ideal city with [(not any? citizens-here) and
    (social? = false) and (neighbourhood = place) and
    (condition >= (mean [condition] of city - (mean [condition] of city * 0.15))) and
    (price <= will) and
    (not available?)]
  if owner? [set ideal ideal with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else

  ifelse any? ideal [
    move-to-place-and-spread-out-turtles min-one-of ideal [dist]
  ] [ ;; no ideal place, try second best place
;    let testbed n-of 6 city
    let testbed n-of (count city / 10) city
    let secondbest city with [(not any? citizens-here) and (condition >= (mean [condition] of testbed - (mean [condition] of testbed * 0.15)))]  ;; if we can't find a place we like then we move to one we can afford
    if enable-economy? [
      set secondbest secondbest with [(price <= will)]
    ]

    ifelse any? secondbest [
      move-to-place-and-spread-out-turtles min-one-of secondbest [dist]
    ] [ ;; no second best place, try third best place
      let thirdbest city with [(not any? citizens-here)] ;; Uncomment the following to prevent people from moving in decrepit locations ;and (condition > 0)
      if enable-economy? [
        set thirdbest thirdbest with [(price <= will)]
      ]
      ;ifelse any? thirdbest [move-to min-one-of thirdbest [dist]] [enter-housing-list self place]  ;; if no place exists we leave the city.
      ifelse any? thirdbest [move-to-place-and-spread-out-turtles min-one-of thirdbest [dist]] [do-emigrate]  ;; if no place exists we leave the city.
    ]
  ]
end


; ===== PRICE GAPS =====

to set-gaps-lnd-off
  let sample max [price] of city in-radius 3
  let areaprice median [price] of city with [neighbourhood = [neighbourhood] of myself] * (1 + renovation-premium)
  let whichprice sample
  if areaprice > sample [set whichprice areaprice]
  ifelse whichprice > price
  [set price-gap (sample - price)]
  [set price-gap 0]
end

to set-gaps-lnd
  let whichprice 0
  let neigh-price 0
  let areaprice 0
  let sample city in-radius 2
  let sample-with-people sample with [count citizens-here > 0]
  ifelse any? sample-with-people [
    set areaprice mean [price] of sample-with-people
  ][
    set areaprice (mean [price] of sample) * 0.65
  ]
  set whichprice areaprice * (1 + renovation-premium)
  ifelse whichprice > price [
    let citizens-here-to-consider citizens-here with [owner? = false]
    ifelse any? citizens-here-to-consider with [(income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
    ;ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
;      [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
      [set price-gap (whichprice - (price + compute-resident-eviction-cost citizens-here-to-consider) )]  ; ...which will reduce prospective profit
    ;[set price-gap (whichprice - price)]
    [set price-gap (whichprice - price)]
  ]
  [set price-gap 0]
end

to-report compute-resident-eviction-cost [citizens-to-consider]
  let local-resident-removal-cost resident-removal-cost * price * (count citizens-to-consider with [income * credit-mine < price])
;  output-show local-resident-removal-cost
  report local-resident-removal-cost
end

;; FIXME code not updated cf Stefano's v0.4
to set-gaps-new   ;; Maximum of moore neighbourhood or district median
 let whichprice 0
 let neigh-price 0
 let area-price 0
 set neigh-price max [price] of neighbors
 set area-price median [price] of city with [neighbourhood = [neighbourhood] of myself] * premium
 ifelse neigh-price > area-price
 [set whichprice neigh-price]
 [set whichprice area-price]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end

;; FIXME code not updated cf Stefano's v0.4
to set-gaps-unified  ;;
 let whichprice 0
 let localprice mean [price] of neighbors * (1 + renovation-premium)
 if (occupancy neighbors "sg") >= 0.85 [set localprice max [price] of neighbors * (1 + renovation-premium) ]
 ifelse (occupancy neighbourhood "sg2") >= 0.85
 [set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * (1 + renovation-premium)]
 [set whichprice mean [price] of city with [neighbourhood = [neighbourhood] of myself] * (1 + renovation-premium)]
 if localprice > whichprice [set whichprice localprice]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end

;; FIXME code not updated cf Stefano's v0.4
to set-gaps-mean
 let whichprice 0
 let localprice mean [price] of neighbors * premium
 ;if count citizens-on neighbors / count neighbors >= 0.85 [set localprice max [price] of neighbors * premium ]
 ;ifelse occupancy neighbourhood >= 0.85
 ;[set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * premium]
 set whichprice mean [price] of city with [neighbourhood = [neighbourhood] of myself] * premium
 if localprice > whichprice [set whichprice localprice]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end

;; FIXME code not updated cf Stefano's v0.4
to set-gaps-max
 set premium 1.001
 let whichprice 0
 let localprice max [price] of neighbors * premium
 ;if count citizens-on neighbors / count neighbors >= 0.85 [set localprice max [price] of neighbors * premium ]
 ;ifelse occupancy neighbourhood >= 0.85
 ;[set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * premium]
 set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * premium
 if localprice > whichprice [set whichprice localprice]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end


;; ===== ALLURE =====

to-report update-city-culture
  let newallure n-values traits [0]
  let trt 0
  while [trt < traits] [
    let thistrait one-of modes [item trt culture] of citizens
    set newallure replace-item trt newallure thistrait
    set trt trt + 1
  ]
  report newallure
end

to set-city-allure
  let city-culture update-city-culture
  let pallure []
  let num-of-citizens (count citizens)
  let trait 0
  while [trait < traits] [
    set pallure lput ifelse-value (count citizens with
      [(item trait culture = item trait city-culture)] >= (num-of-citizens * 0.3))
    [item trait city-culture] ["x"] pallure
    set trait trait + 1
    ]
  set city-allure pallure
  ;show "done set-city-allure"
end

to-report localculture [district] ; FIXME: tidy up
  let people-here0 citizens-on city with [neighbourhood = district]
  ;print (word "localculture " district ": " (count people-here0) " people here: " people-here0)
  ;if any? people-here0 [show "yes, people here"]
  ifelse any? citizens-on city with [neighbourhood = district] [
    let people-here citizens-on city with [neighbourhood = district]  ; netlogo bug? this turtle set always is empty, even though there are people here!
    ;print (word "localculture " district ": " (count people-here) " people here: " people-here)
    ;if any? people-here0 [print (word "yes, people here: " people-here0)]
    let newallure n-values traits [0]
    let trt 0
    while [trt < traits] [
      let most-common-cultural-trait-here modes [item trt culture] of people-here0
      ifelse length most-common-cultural-trait-here > 0 [
        let thistrait one-of most-common-cultural-trait-here
        set newallure replace-item trt newallure thistrait
      ] [
        print (word "localculture " district ": none of " (count people-here) " people here with trait " trt)
      ]
      set trt trt + 1
    ]
    report newallure
   ; ask city with [neighbourhood = district] [set pculture newallure]
  ] [
    print (word "localculture " district ": no people here")
  ]
end

to set-allure [place]
  let ppl citizens-on city with [neighbourhood = place]
  ;print (word "set-allure " place ": " (count ppl) " people here")
  let areaculture (localculture place)
  ;print (word "set-allure " place ": " (count ppl) " people here and localculture " areaculture) ; ": " ppl)
  let pallure []
  let trait 0
  while [trait < traits] [
    set pallure lput ifelse-value
    (count ppl with [item trait culture = item trait areaculture] > count ppl * 0.3 and
      item trait areaculture != item trait city-allure and
      item trait city-allure != "x"
      )
    [item trait areaculture] ["x"] pallure
    set trait trait + 1
  ]
  let peculiar length filter [ ? -> ? != "x" ] pallure
  ;print (word "set-allure " place ": has peculiar " peculiar)

;  set-current-plot "peculiarity"
;  let placepen place
;  if place = "n" [set placepen "cbd"]  ; match the identifier in the allure graph
;  set-current-plot-pen placepen
;  plotxy ticks peculiar

  ifelse peculiar > peculiarity-for-allure [
    ask city with [neighbourhood = place] [set allure pallure]
    if not table:has-key? allured-districts place
    [table:put allured-districts place (list (median [income] of citizens-on city with [neighbourhood = place]) (median [price] of city with [neighbourhood = place]) (occupancy place "sa") (ticks))]
  ]
  [if table:has-key? allured-districts place [
      let longevity ticks - item 3 table:get allured-districts place
      if longevity > 60 [table:remove allured-districts place]]
  ]

  ;print (word "set-allure " place " -- done")
end


;; ===== INTERNAL REPORTING AND PHENOMENA CALCULATION =====

;to-report neighz
;  report neighbors with [member? self city]
;end

to-report median-price-of [place]
  let district-as-patchset place
  if not is-patch-set? place [
    set district-as-patchset table:get neighbourhoods-table place
  ]
  report median [price] of district-as-patchset
end

to-report gini [group]
  let sorted-wealths sort [income] of group
  ;; algorithm from http://shlegeris.com/2016/12/29/gini
  let sum-of-absolute-differences 0
  let subsum 0
  let i 0
  let N count group
  while [i < N] [
    let x item i sorted-wealths
    set sum-of-absolute-differences (sum-of-absolute-differences + i * x - subsum)
    set subsum (subsum + x)
    set i (i + 1)
  ]
  report sum-of-absolute-differences / subsum / N
end

to report-population
  let totalpop count people
  output-print (word "Population " totalpop ": "
    (count people with [item 0 culture = 0]) " Dutch, "
    (count people with [item 0 culture = 1]) " Other Western, "
    (count people with [item 0 culture = 2]) " Surinamese, "
    (count people with [item 0 culture = 3]) " Antillean, "
    (count people with [item 0 culture = 4]) " Turkish, "
    (count people with [item 0 culture = 5]) " Moroccan, "
    (count people with [item 0 culture = 6]) " Other Non-Western")
end

to-report is-owned?
  if any? citizens-here with [owner?] [report true]
  report false
end

;; place is a slum if condition too low or occupancy to high
to-report is-slum?
  if (condition <= slum-threshold) or (count citizens-here > non-slum-population) [report true]
  report false
end

to-report is-cbd? [place]
  if any? place with [cbd?] [report true]
  report false
end

to-report occupancy [place debug-tag]
  let district-name place
  let district-patches place
  ifelse is-patch-set? place [
    ;print (word "setting total as nhood " place)
    set district-name [neighbourhood] of one-of place
  ] [
    ;print (word "setting total via table " place)
    set district-patches occupancy-helper place
  ]

  let number-patches-occupied count district-patches with [count citizens-here > 0]
  let total-number-patches count district-patches
  ;DEBUG if (debug-tag != "go") [print (word debug-tag " " district-name " " number-patches-occupied " / " total-number-patches)]
  report safe-division number-patches-occupied total-number-patches

  ;print (word total " >> " one-of total " >> " [neighbourhood] of one-of total)
  ;let occupied table:get occupancys-table district-name

;  let total place
;;  if not is-patch-set? place [set total city with [neighbourhood = place]]
;  if not is-patch-set? place [
;    ;print (word "setting total via table " place)
;    set total occupancy-helper place]
;  let occupied count total with [count citizens-here > 0]
;  ifelse count total = 0 [
;    report 0  ; FIXME
;  ] [
;    report occupied / count total
;  ]
end

to-report occupancy-helper [place]
  report table:get neighbourhoods-table place  ; much faster than line below: since neighbourhoods never change we can use lookup table
  ;report city with [neighbourhood = place]
end

;; overwrite occupancys table based on current neighbourhood of citizens (needs NetLogo 6.0.2+)
to occupancy-table-update
  ;print "occupancy-table-update"
  let occupancys-table-agents table:group-agents citizens [neighbourhood]
  ;print occupancys-table-agents
  set occupancys-table table:from-list map [place -> list place (count table:get-or-default occupancys-table-agents place no-turtles)] table:keys neighbourhoods-table ;occupancys-table-agents
  ;print occupancys-table
  ;print "occupancy-table-update -- done"
end

to remove-turtle-from-occupancy-table
;  let current-population-of-old-patch table:get occupancys-table neighbourhood
;  table:put occupancys-table neighbourhood (current-population-of-old-patch - 1)
;  if (current-population-of-old-patch - 1 < 0) [
;    print (word neighbourhood " " current-population-of-old-patch "-=>" (current-population-of-old-patch - 1))
;  ]
end

to add-turtle-to-occupancy-table
;  let current-population-of-new-patch table:get occupancys-table neighbourhood
;  table:put occupancys-table neighbourhood (current-population-of-new-patch + 1)
;  if (current-population-of-new-patch + 1 > count citizens) [
;    print (word neighbourhood " " current-population-of-new-patch "+=>" (current-population-of-new-patch + 1))
;  ]
end

to-report adjacentarea [place]
  let where min-one-of city with [count city with [area = [area] of myself] >= 5] [distance one-of city with [area = place]]
  report [area] of where
end

;; if place is not a patchset, but a string (district name), return the subset of city patches which are in the district place
to-report ensure-patchset-of-city [place]
  if not is-patch-set? place [
    ;DEBUG print (word place " is not a patchset")
    report city with [area = place]  ; HACK works because areas = districts
  ]
  ;DEBUG print (word place " is already a patchset")
  report place
end


to determine-phenomenon [place]
  ifelse median [income] of citizens-on city with [neighbourhood = place] > item 0 table:get allured-districts place
  [
    table:put gentrified-districts place (list median [income] of citizens-on city with [neighbourhood = place] median [price] of city with [neighbourhood = place] occupancy place "DP")
    output-show word "Here is a gentrified neighbourhood: " place
    if write-csv? [export-view (word save-directory-pictures "GENTRIFIED_DISTRICT -" "K" Kapital "-t" ticks "-" place ".png")]
    ]
  [
    table:put downfiltered-districts place (list median [income] of citizens-on city with [neighbourhood = place] median [price] of city with [neighbourhood = place] occupancy place "DP")
    output-show word "Here is a downfiltered neighbourhood: " place
    if write-csv? [export-view (word save-directory-pictures "DOWNFILTERED_DISTRICT -" "K" Kapital "-t" ticks "-" place ".png")]
    ]
end

to determine-super-phenomenon [district case]  ;; when a place lost than regained uniformity. What's happening???
  ifelse case = 0  [   ;; in this case originally gentrification dissolved uniformity
    ifelse median [income] of citizens-on city with [neighbourhood = district] >= (item 0 table:get gentrified-districts district - 0.1) and median [price] of city with [neighbourhood = district] >= (item 1 table:get gentrified-districts district - 0.1)
    [
      output-show word "Here is a recolonised neighbourhood: " district
      if not member? district recolonisation [set recolonisation fput district recolonisation]
      ;set recolonisation recolonisation + 1
      if write-csv? [export-view (word save-directory-pictures "RECOLONISED_DISTRICT -" "K" Kapital "-t" ticks "-" district ".png")]
      ]
    [
      if not member? district degentrification [set degentrification fput district degentrification]
      output-show word "Here is a DEGENTRIFIED neighbourhood: " district
      ;set degentrification degentrification + 1
      ]
  ] [ ; here originally downfiltering dissolved uniformity
  ifelse mean [income] of citizens-on city with [neighbourhood = district] <= (item 0 table:get downfiltered-districts district + 0.1)
  [set recreation fput district recreation]
  [set regentrification fput district regentrification]
  ]
end

to-report entropy [district]
  let common 0
  let thispeople citizens-on city with [neighbourhood = district]
  let pairs (count thispeople * (count thispeople - 1)) / 2
  ask n-of (count thispeople / 2) thispeople [
    ask other thispeople [
      set common common + (similarity self myself / traits)
    ]
  ]
  report safe-division common pairs
end

to-report entropy-old [district]
  let common 0
  let thispeople citizens-on city with [neighbourhood = district]
  let pairs (count thispeople * (count thispeople - 1)) / 2
  ask thispeople [
    ask other thispeople [
      set common common + (similarity self myself / traits)
    ]
  ]
  report safe-division (common / 2) pairs
end


;; ===== DISPLAY ======

to draw-city
  gis:set-drawing-color red
  ;gis:draw prices 0.2
  gis:draw divisions2 0.4
end

to colour-agent
  if item 0 culture = 0 [set color pink + 4]
  if item 0 culture = 1 [set color pink + 3]
  if item 0 culture = 2 [set color brown - 3]
  if item 0 culture = 3 [set color brown - 4]
  if item 0 culture = 4 [set color brown]
  if item 0 culture = 5 [set color brown + 1]
  if item 0 culture = 6 [set color gray]
  if owner? = true [set size ownersize]
end

to shape-agent
  ifelse real-data? [
    let median-income-of-citizens median [income] of citizens
    if income >= median-income-of-citizens * 2 [set shape "face happy"]
    if income < median-income-of-citizens * 2 and income >= median-income-of-citizens / 2 [set shape "face neutral"]
    if income < median-income-of-citizens / 2 [set shape "face sad"]
  ][
    if income >= 0.80 [set shape "face happy"]
    if income < 0.80 and income >= 0.25 [set shape "face neutral"]
    if income < 0.25 [set shape "face sad"]
  ]
end

to colour-patches
  set pcolor condition * 10
  if pcolor > white [set pcolor white]
  if not social? and available? [set pcolor pcolor + lime - 5]
  if social? and not available? [set pcolor pcolor + red - 5]
  if social? and available? [set pcolor pcolor + yellow - 5]
  if condition <= slum-threshold [set pcolor pcolor + magenta - 3]
end


;; ===== SUPPORTING =====

to-report safe-division [a b]
  if a = 0 or b = 0 [report 0]
  report a / b
end

to-report similarity [a b]
  report similarity-of ([culture] of a) (ifelse-value is-turtle? b [[culture] of b] [[allure] of b])
end

; called a lot: how to speed it up? ideas: https://stackoverflow.com/questions/42877203/netlogo-how-to-filter-a-list-using-a-criterion-which-refers-on-a-corresponding
to-report similarity-of [ls1 ls2]                    ;;;;;; DA QUA DOBBIAMO PRENDERE IL CODICE PER FARE IL NUOVO ENTROPY
  ifelse length ls1 != length ls2 [
      output-print (word ls1 "!=" ls2)
      report 0
  ] [
    ; the imperative is the fastest so far
    let N length ls1
    let i 0
    let mysum 0
    while [i < N] [
      if (item i ls1) = (item i ls2) [set mysum mysum + 1]
      set i i + 1
    ]
    report mysum
    ;report sum (map [ [?1 ?2] -> ifelse-value (?1 = ?2) [1] [0] ] ls1 ls2)  ; faster than the older line below
    ;report length filter [ [?1] -> ?1 ] (map [ [?1 ?2] -> ?1 = ?2 ] ls1 ls2)
  ]
end

to-report add-underscores-to-string [s]
  let i -1
  while [i != false] [
    set i position " " s
    if i != false [set s replace-item i s "_"]
  ]
  report s
end

to-report closest-integer-to [x]
  report round x
end

to-report random-float-between [a b]
  if a > b [
    let tempc b
    set b a
    set a tempc
  ]
  report a + random-float (b - a)
end


;; ===== PLOTTING =====

to plot-ent [dis]
  if disp? [repeat disp-freq [plot entropy dis]]
end

to-report medianincome [place]
  ifelse any? citizens-on city with [neighbourhood = place] [
    report median [income] of citizens-on city with [neighbourhood = place]
  ][report 0]
end


;; ===== DATA OUTPUT =====

to update-reports
  set medianPrices lput median [price] of city with [condition >= slum-threshold] medianPrices
  set median-Income lput median [income] of citizens median-Income
  ;set howmanySlums lput (count city with [condition <= slum-threshold] / count city) howmanySlums
  set howmanySlums lput (count city with [is-slum?] / count city) howmanySlums  ; since is-slum? combines patch-condition and patch-population criteria, there's a case to use instead just condition <= slum-threshold here
  set howmanyWellMaintained lput (count city with [condition > 0.5] / count city) howmanyWellMaintained
  set howmanyOvercrowded lput (count city with [count citizens-here > non-slum-population] / count city) howmanyOvercrowded
  set highestMedianPrice lput add-underscores-to-string item 0 sort-by [ [?1 ?2] -> median-price-of ?1 > median-price-of ?2 ] districts highestMedianPrice  ; neighbourhood with highest price
  set dutch lput (count citizens with [item 0 culture = 0] / count citizens) dutch
  set others lput (count citizens with [item 0 culture > 0] / count citizens) others
end

to save-screenshots
  ;export-view (word save-directory-pictures Kapital "-t" ticks ".png")
  ;export-view (word save-directory-pictures "" "-t" ticks ".png")  ; commented out for running from rnetlogo, because we'll drown in the number of screenshots! -- and nlexperiment saves the final view anyway
  ;print "(not saving screenshots)"
  type "(no png)"
  if ticks = end-tick [
      ;export-interface (word save-directory-pictures "gentax-" version "-interface.png")
  ]
end

to save-data
  ;export-all-plots (word save-directory "gentax-" version "-plots.csv")  ; save all the plots
  set file-name-results (word save-directory "gentax-" version ".csv")   ; save one big csv file with all the measures we want
  let run-number 1  ; NYS was 0
  if behaviorspace-run-number != 0 [set run-number behaviorspace-run-number]
  let should-write-header not (file-exists? file-name-results)
  file-open file-name-results
  if run-number = 0 or should-write-header
    [file-print (word "run#;immigrated-count;emigrated-count;inner-migrated-count;stay-night-count;tourist-revenue;total-levied;gini;Kapital;gaps;regeneration?;howmanyOvercrowded;medianPrices;median-Income;highestMedianPrice;dutch;others")]
  file-print (word run-number ";" immigrated-count ";" emigrated-count ";" inner-migrated-count ";" stay-night-count ";" tourist-revenue ";" total-levied ";" gini citizens ";" Kapital ";" gaps ";" regeneration? ";" howmanyOvercrowded ";" medianPrices ";" median-Income ";" highestMedianPrice ";" dutch ";" others)
  file-close-all
end

to save-price-deltas
  file-open (word file-name-results "-district-prices.csv")
  foreach districts [ district ->
    let initialPrice initial-price-of-district district
    let finalPrice median-price-of district
;    let deltaPricePercentage ( finalPrice - initialPrice) / initialPrice
    file-print (word "\"" (add-underscores-to-string district) "\", " (first initialPrice) ", " (last initialPrice) ", " finalPrice)
  ]
  file-close
end
@#$#@#$#@
GRAPHICS-WINDOW
4
10
1052
1059
-1
-1
16.0
1
9
1
1
1
0
0
0
1
-32
32
-32
32
1
1
1
Days
30.0

SLIDER
1065
217
1239
250
monthly-decay
monthly-decay
0
0.1
0.0015
0.0001
1
NIL
HORIZONTAL

SLIDER
1065
252
1240
285
tolerable-vacancy
tolerable-vacancy
0
24
8.0
1
1
months
HORIZONTAL

SLIDER
1065
557
1201
590
prob-move
prob-move
0
0.01
0.002
0.0001
1
NIL
HORIZONTAL

SLIDER
1217
631
1309
664
traits
traits
2
10
7.0
1
1
NIL
HORIZONTAL

SLIDER
1313
631
1408
664
values
values
5
10
5.0
1
1
NIL
HORIZONTAL

TEXTBOX
1224
505
1285
523
CULTURE
12
0.0
1

SLIDER
1065
669
1200
702
immigration-rate
immigration-rate
0
0.2
0.02
0.0001
1
NIL
HORIZONTAL

SLIDER
1063
522
1201
555
N-Agents
N-Agents
0
count patches
1000.0
1
1
NIL
HORIZONTAL

SWITCH
1239
357
1374
390
random-income?
random-income?
1
1
-1000

SLIDER
1239
182
1374
215
Kapital
Kapital
0
0.1
0.035
0.001
1
NIL
HORIZONTAL

PLOT
1731
16
2046
231
Basic income
NIL
NIL
0.0
2520.0
0.0
10000.0
true
true
"" ""
PENS
"UBI" 1.0 0 -12087248 true "" "plot ubi"
"Tax" 1.0 0 -2674135 true "" "plot tax-rate"

SWITCH
1066
634
1200
667
immigration?
immigration?
0
1
-1000

SLIDER
1217
667
1407
700
tolerable-dissonance
tolerable-dissonance
0
24
5.0
1
1
months
HORIZONTAL

SLIDER
1217
700
1407
733
similarity-for-dissonance
similarity-for-dissonance
0
0.5
0.2
0.01
1
NIL
HORIZONTAL

MONITOR
1067
864
1125
909
Pop
count citizens
0
1
11

SLIDER
1065
182
1237
215
profit-threshold
profit-threshold
0
1
0.125
0.001
1
NIL
HORIZONTAL

PLOT
2046
16
2366
231
Tourist GDP
NIL
NIL
0.0
2520.0
0.0
10000.0
true
true
"" ""
PENS
"EUR" 1.0 0 -16777216 true "" "plot tourist-revenue-daily"

SWITCH
1217
596
1308
629
push?
push?
0
1
-1000

SWITCH
1682
982
1776
1015
record?
record?
1
1
-1000

SWITCH
1313
596
1408
629
strong-neighbourhood?
strong-neighbourhood?
0
1
-1000

SWITCH
1313
560
1408
593
pull?
pull?
0
1
-1000

SWITCH
1217
561
1311
594
mixing?
mixing?
0
1
-1000

SWITCH
1682
1055
1775
1088
write-csv?
write-csv?
1
1
-1000

BUTTON
1566
987
1651
1020
Show allure
ask city with [allure != 0][set pcolor yellow]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1566
952
1651
985
Undisplay
ask city [colour-patches]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
1217
128
1281
146
ECONOMY
12
0.0
1

TEXTBOX
1075
504
1225
522
POPULATION
12
0.0
1

TEXTBOX
1683
957
1747
976
OUTPUT
12
0.0
1

CHOOSER
1420
522
1582
567
kind
kind
"monocentric" "policentric" "no centres"
0

BUTTON
1566
1020
1651
1053
Show centres
ask city with [centre? = true][set pcolor blue]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
1425
506
1575
524
CITY
12
0.0
1

SLIDER
1420
570
1651
603
peculiarity-for-allure
peculiarity-for-allure
0
traits
3.0
1
1
/ 10 traits
HORIZONTAL

SLIDER
1217
736
1407
769
similarity-for-friendship
similarity-for-friendship
0
1
0.2
0.01
1
NIL
HORIZONTAL

SLIDER
1064
287
1237
320
resident-removal-cost
resident-removal-cost
0
0.25
0.05
0.001
1
NIL
HORIZONTAL

SLIDER
1239
252
1375
285
renovation-premium
renovation-premium
0
1
0.2
0.01
1
NIL
HORIZONTAL

SWITCH
1682
1090
1775
1123
paper-stuff?
paper-stuff?
0
1
-1000

MONITOR
1540
864
1597
909
Mobility
sum [place-changes] of turtles / count turtles
2
1
11

BUTTON
1566
1090
1651
1123
Regeneration
regenerate
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SWITCH
1239
217
1374
250
fixed-premium?
fixed-premium?
0
1
-1000

PLOT
2003
676
2264
826
Income Distribution
Income
Freq
0.0
10000.0
0.0
1.0
true
false
"set-histogram-num-bars 20" "histogram [income] of citizens"
PENS
"default" 1.0 1 -16777216 true "" ""

MONITOR
1733
864
1793
909
Gini
gini citizens
2
1
11

MONITOR
1479
864
1537
909
Occ. R.
occupation-rate
2
1
11

PLOT
2002
523
2263
673
Price Distribution
Price
Freq
0.0
10000.0
0.0
1.0
true
false
"set-histogram-num-bars 20" "histogram [price] of patches with [count citizens-here > 0]"
PENS
"default" 1.0 1 -16777216 true "" ""

SWITCH
1217
523
1407
556
enable-culture?
enable-culture?
0
1
-1000

MONITOR
1603
864
1665
909
% in Slums
100 * (count citizens-on patches with [condition <= slum-threshold or count citizens-here > non-slum-population] / count turtles)
4
1
11

BUTTON
1566
1055
1651
1088
Build SH
build-social-housing 4
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
1512
611
1651
644
max-time-in-social-housing
max-time-in-social-housing
0
100
50.0
1
1
NIL
HORIZONTAL

MONITOR
1590
648
1646
693
Waiting list
table:length housing-waiting-list
17
1
11

SLIDER
1065
357
1238
390
yearly-depreciation
yearly-depreciation
0
0.1
0.02
0.001
1
NIL
HORIZONTAL

BUTTON
1066
78
1178
111
RUN!
setup\nrepeat 1440 [go]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1066
42
1121
75
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1122
42
1177
75
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1246
864
1303
909
Centre
median [income] of citizens-on patches with [dist < 4]
4
1
11

MONITOR
1302
864
1359
909
Semi
median [income] of citizens-on patches with [dist >= 4 and dist < 8]
4
1
11

MONITOR
1356
864
1413
909
Far
median [income] of citizens-on patches with [dist >= 8]
4
1
11

PLOT
2046
230
2366
430
Population and Tourists
NIL
NIL
0.0
2520.0
0.0
1000.0
true
true
"" ""
PENS
"Pop" 1.0 0 -16777216 true "" "plot count citizens"
"Tou" 1.0 0 -2674135 true "" "plot count tourists"

PLOT
1415
16
1730
231
Prices
NIL
NIL
0.0
2520.0
0.0
10000.0
true
true
"" ""
PENS
"V" 1.0 0 -16777216 true "" "plot median [price] of city"

PLOT
1415
230
1730
430
Income
NIL
NIL
0.0
2520.0
0.0
10000.0
true
true
"" ""
PENS
"I" 1.0 0 -2674135 true "" "plot median [income] of citizens"

MONITOR
1592
522
1649
567
Allure
table:length allured-districts
17
1
11

CHOOSER
1065
132
1200
177
gaps
gaps
"mean" "max" "unified" "new" "lnd"
4

SWITCH
1240
392
1373
425
have-demand-premium
have-demand-premium
0
1
-1000

SWITCH
1211
146
1375
179
enable-economy?
enable-economy?
0
1
-1000

SLIDER
1065
392
1236
425
credit
credit
0
15
3.0
1
1
NIL
HORIZONTAL

SLIDER
1217
772
1407
805
similarity-threshold
similarity-threshold
0
1
0.3
0.01
1
NIL
HORIZONTAL

SWITCH
1420
611
1510
644
social-housing?
social-housing?
0
1
-1000

OUTPUT
1066
920
1416
1016
8

PLOT
1670
523
1991
673
Ethnicity
NIL
Freq
0.0
7.0
0.0
10.0
true
false
"set-histogram-num-bars 7" "histogram [item 0 culture] of citizens"
PENS
"default" 1.0 1 -16777216 true "" ""

PLOT
1670
676
1991
825
Population by ethnicity
NIL
NIL
0.0
2520.0
0.0
10.0
true
false
"" ""
PENS
"Dutch" 1.0 0 -465430 true "" "plot count citizens with [item 0 culture = 0]"
"Other Western" 1.0 0 -865067 true "" "plot count citizens with [item 0 culture = 1]"
"Surinamese" 1.0 0 -12440034 true "" "plot count citizens with [item 0 culture = 2]"
"Antillean" 1.0 0 -14477296 true "" "plot count citizens with [item 0 culture = 3]"
"Turkish" 1.0 0 -6459832 true "" "plot count citizens with [item 0 culture = 4]"
"Moroccan" 1.0 0 -5207188 true "" "plot count citizens with [item 0 culture = 5]"
"Other Non-Western" 1.0 0 -7500403 true "" "plot count citizens with [item 0 culture = 6]"

SLIDER
1420
716
1542
749
slum-population
slum-population
2
10
6.0
1
1
NIL
HORIZONTAL

SLIDER
1420
751
1543
784
non-slum-population
non-slum-population
1
slum-population - 1
3.0
1
1
NIL
HORIZONTAL

SLIDER
1420
784
1543
817
cbd-population
cbd-population
1
10
1.0
1
1
NIL
HORIZONTAL

SWITCH
1239
287
1374
320
have-owners?
have-owners?
0
1
-1000

SLIDER
1239
322
1374
355
ownership
ownership
0
1
0.25
0.01
1
NIL
HORIZONTAL

MONITOR
1666
864
1730
909
% Owners
100 * (count citizens with [owner? = true] / count citizens)
3
1
11

MONITOR
1421
864
1476
909
Avg. Occ.
count citizens-on patches / count patches
4
1
11

SWITCH
1065
706
1200
739
emigration?
emigration?
0
1
-1000

SLIDER
1065
742
1200
775
emigration-rate
emigration-rate
0
0.2
0.019
0.0001
1
NIL
HORIZONTAL

SWITCH
1420
822
1540
855
cbd-price-control?
cbd-price-control?
0
1
-1000

SLIDER
1420
682
1580
715
slum-threshold
slum-threshold
0
1
0.15
0.01
1
NIL
HORIZONTAL

SLIDER
1420
646
1579
679
intervention-threshold
intervention-threshold
0
1
0.2
0.01
1
NIL
HORIZONTAL

MONITOR
1590
697
1648
742
% Slums
count patches with [is-slum?] / count patches
4
1
11

MONITOR
1206
1028
1352
1073
Total Immigrated
immigrated-count
17
1
11

MONITOR
1206
1078
1352
1123
Total Emigrated
emigrated-count
17
1
11

TEXTBOX
1072
22
1222
40
SIMULATION
11
0.0
1

INPUTBOX
1183
44
1293
104
end-tick
2520.0
1
0
Number

PLOT
1731
230
2046
430
Conditions
NIL
NIL
0.0
2520.0
0.0
1.0
false
true
"" ""
PENS
"C" 1.0 0 -16777216 true "" "plot mean [condition] of city"

CHOOSER
1553
794
1648
839
regeneration?
regeneration?
"none" "regular" "EU-slums"
0

MONITOR
1190
864
1247
909
CBD
median [income] of citizens-on patches with [cbd?]
4
1
11

MONITOR
1590
744
1648
789
% Derelict
count patches with [condition = 0] / count patches
4
1
11

SWITCH
1682
1018
1776
1051
profile?
profile?
1
1
-1000

SWITCH
1424
990
1558
1023
real-data?
real-data?
0
1
-1000

SLIDER
1065
594
1201
627
prob-rent-out
prob-rent-out
0
1
0.002
0.0001
1
NIL
HORIZONTAL

SWITCH
1065
782
1200
815
tourism?
tourism?
0
1
-1000

SLIDER
1066
818
1200
851
tourist-arrival-rate
tourist-arrival-rate
0
0.1
0.1
0.0005
1
NIL
HORIZONTAL

SLIDER
1217
810
1407
843
rent-out-maximum
rent-out-maximum
0
360
60.0
1
1
nights
HORIZONTAL

CHOOSER
1065
430
1236
475
basic-income
basic-income
"None" "Fixed Tax Rate" "Fixed Income"
0

SLIDER
1241
429
1374
462
fixed-tax-rate
fixed-tax-rate
0
0.5
0.05
0.005
1
NIL
HORIZONTAL

SLIDER
1242
467
1374
500
fixed-ubi
fixed-ubi
0
50000
10000.0
100
1
NIL
HORIZONTAL

MONITOR
1358
1028
1468
1073
Total nights
stay-night-count
17
1
11

MONITOR
1065
1028
1203
1073
Total inner-migrated
inner-migrated-count
17
1
11

MONITOR
1358
1078
1475
1123
Tourist revenue
tourist-revenue
17
1
11

PLOT
1814
858
2149
1063
Tourist price
NIL
NIL
0.0
2520.0
0.0
100.0
true
false
"" ""
PENS
"No cit" 1.0 0 -15390905 true "" "plot mean [tourist-price] of city with [available? and not any? citizens-here]"
"Cit" 1.0 0 -10873583 true "" "plot mean [tourist-price] of city with [available? and any? citizens-here]"

PLOT
2152
857
2466
1063
Tourist availablility
NIL
NIL
0.0
2520.0
0.0
50.0
true
false
"" ""
PENS
"No cit" 1.0 0 -15390905 true "" "plot count city with [available? and not any? citizens-here]"
"Cit" 1.0 0 -10873583 true "" "plot count city with [available? and any? citizens-here]"

PLOT
1815
1073
2153
1275
Tourist success
NIL
NIL
0.0
2520.0
0.0
1.0
true
false
"" ""
PENS
"No cit" 1.0 0 -15390905 true "" "plot count city with [any? tourists-here and not any? citizens-here] / count city with [available? and not any? citizens-here]"
"Cit" 1.0 0 -10873583 true "" "plot count city with [any? tourists-here and any? citizens-here] / count city with [available? and any? citizens-here]"

MONITOR
1205
1129
1354
1174
Total entered waiting list
entered-waiting-list
17
1
11

MONITOR
1205
1178
1372
1223
Total entered social housing
entered-social-housing
17
1
11

@#$#@#$#@
# Introduction
*Gentrification meets Axelrod meets Schelling.*

This is a city-scale residential mobiliy model. It couples residential choice with cultural dynamics and investment/disinvestment cycles, modelled, the latter, according to Neil Smith's (RIP) rent-gap theory.
Dwellings (individual patches) have a price and a mainteniance condition. They progressively decay in their condition and, accordingly, in their asking price.
If sufficient Kapital is available, renovation is carried out on those locations that present the wider "price-gap" with the neighbouring properties, as proposed in most computational implementations of the rent-gap theory.
After renovation a property is reset to the highest possible condition and is able to charge a price equal to the average of neighbouring properties + a 15% premium.

Agents are created with a wealth level, a mobility propensity and a n-th dimensional string representing their culture; they mix traits with neighbours and have a homophily preference when selecting a place of residence.

The aim of the model is to explore the effects of different levels of capital available for redevelopment on price dynamics, residential dynamics and cultural diversity.

# Model detail
## Agent model
The agent's **culture** is modelled as a n-th dimensional multi-value string (currently n=10) of "traits", as in the great tradition of "string culture" (Axelrod, etc.)
The agent's **income level** is set at random, normalized to the interval 0-1. No "social mobility" exists: income is a fixed attribute and never changes.
Agents are also created with a **mobility-propensity** parameter, which is very low in the beginning (the initial probability to move is poisson distributed in the population centred on 0.001/month).

### Micro-level cultural dynamics
Long time neighbouring agents with at least one common trait are more likely to interact and exchange traits, thus rendering the respective cultural strings more similar. A **cultural cognitive dissonance** parameter implements a concept proposed by Portugali (1996): this is, roughly, the frustration of being surrounded by too many culturally distant agents. Yes, it's Schelling in other terms.

### Residential mobility
One agent's mobility propensity attribute is increased when:

* Excessive time is spent in a dwelling in very bad condition (slum)
* The cultural cognitive dissonance level is high for too long (cultural push).
* The price of the dwelling currently occupied exceeds the agent's income (in this case the agent is automatically put in "seek new place" mode)

A new dwelling has to be:

* affordable
* in relatively good condition
* as close as possible to the centre of the city
* located in a culturally appealing neighbourhood (cultural pull).

## Land dynamics
The city is divided in 8 neighbourhoods + the CBD. 441 patches total

Dwellings' price and condition are initially set at a random value normalized in the 0-1 interval, with price being set at 0.25 above condition. Decay happens at every step by a fixed factor (currently 0.0016 per month, meaning that a property decays from 1 to 0 in 50 years) which is increased by 25% when the dwelling is empty. The price of the dwelling is adjusted every year and is decreased if the dwelling has been empty.

If the cultural makeup of the residents is sufficiently homogeneous a neighbourhood can develop an "**allure**", or reputation, based on the average cultural configuration of its inhabitants. This attribute is visible to perspective movers and tends to be "sticky", i.e. is updated seldom and not always reflects the actual composition of the neighbourhood.

The allure of a district, in other words, is not imposed from the beginning, instead it is an emergent feature. Allure is initially blank (meaning that the area has no particular connotation), when cultural uniformity reaches a threshold (see update-allure function) the allure is set. This is to reflect the fact that not every neighbourhood has a special connotation in the mind of agents, but only those with a recognizable population (e.g. WOW! HIPPIES LIVE THERE, I WANT IN!!)


### Residential movement process
When an agent is set to relocate, or first enters the system, compares its culture string to the allure of each neighbourhood that have one, and finds the empty spots within the neighbourhood most similar to himself. Among the empty spots, the affordable ones are detected and among these, the agent moves to the one in the best condition and closest to the centre. Failing to find a suitable spot results in the agent trying in a different neighbourhood, then lowering its requirements and ultimately leaving the city.

# Results
## THE PROBLEM OF SETTING A LOCATION TO THE "HIGHEST AND BEST USE"

Setting the price to the maximum or the average makes a huge difference, regardless of the scope of the area considered.
Only gaps set to average (local or area based) give rise to the typical "uneven development" scenario, where one area attracts all the investment where the rest rot to hell. Interestingly the premium and the amount of available capital only determine the speed and the scope of the process, whereas ultimately average or maximum determine the shape of the dynamic.


## Effects of investment levels on house prices and distribution of agents
### The role of Kapital
In this model (and in the real world) Kapital has a dual role. A sufficient amount of K is necessary to ensure that every property in the city is mainteined and inhabitable, but the nomadic nature of K, which travels across the city in pursuit of the highest profit, generates shocks in the form of abrupt spikes in prices, which affect the ability of (especially least well off) agents to stay or move to the spot of choice. From this duality arise, ultimately, all the dynamics that we see occurring in the model.

The model runs for 1200 ticks = 100 years.

For low levels of investment (**Kapital < 15** = ~3.4% of dwellings receiving investment each year) prices collapse in every neighbourhood and no clear differences in maintenance levels emerge: refurbished patches are scattered across the city. In this condition the population increases to its maximum and the income is average, since very low-priced housing is accessible to everybody.
**Version 0.2 Update** In the 0.2 version (with a threshold on investments being introduced, see above) after a while the city is no longer able to attract Kapital, because no patches present a sufficiently wide price-gap and all patches end up in the slum condition.

At **Kapital > 15** a pattern is visible: the centre of the city and the immediate surrounding area are constantly being maintained and achieving high prices, while the remaining neighbourhoods tend to very low prices and maintenance levels. Investments concentrate in the central area, where price gaps are systematically higher.

When Kapital reaches **K=25** (~6% of dwellings being refurbished each year) two or three entire neighbourhoods are able to attract all the investments. In this case the city tends to divide into two distinct areas, roughly of the same size: one with high price / high repair condition and one of slums.

Around this value the most interesting effects emerge. Gentrification can be spotted often, with neighbourhoods steadily increasing the mean income while the population decreases and increase abruptly, often in several waves, signalling that the poor go and the rich come.

A **Kapital > 35** (refurbishing 8% per year) is able to trigger high prices/ high mainteniance in the whole city. The population is very low because only the richest immigrants can afford to enter the city.
Interestingly, dissonance levels tend to be higher with higher investments. Even though there is no relation between income levels and culture, a high priced / highly selective city makes it difficult for agents to find compatible neighbours. Because the low population doesn't add enough diversity to the mix, a sort of "boring little village" effect or "boring white flight suburb" effect emerges.

### UPDATE 0.2.3
Changing the mechanism for setting the price-gaps (see changelog), somehow, made capital more effective end efficient in the way it distributed the benefit of renovation across the city. Without being bounded by the immediate Moore neighbourhood it was even more effective.
No. It was the constraint towards repairing locations with condition < 0.75 that generated this effect.


Now less capital is capable of spreading the renovation effect to a wider area. As little as K=12 (2.7%) is capable of generating a neighbourhood in good state of repair for 1400 ticks and K=14 generates two neighbourhoods raising to the highest price levels.

## Cultural dynamics
### The emergence and sustainment of culturally homogeneous neighbourhoods

The initial emergence of a recognizable, culturally homogeneous, neighbourhood, ultimately, depends on the availability of decent housing at a medium/low price. Long periods of stable or decreasing prices allow the agents to stay put and interact, becoming more and more similar. This is the only way for a neighbourhood to emerge in the first place. Because of the random initial distribution of prices and repair conditions (and therefore price gaps), in the initial steps of the simulation the locations being renovated are scattered throughout the city and a couple of hundreds of steps are needed before the clustering of investments happens. In this interval the mean prices of individual neighbourhoods tend to decrease and the first neighbourhood emerges, usually the CBD. This is because the agents have a preference for living towards the centre of the city, therefore CBD is the first district to fill and the first localtion where many bagents start to interact.

The fate of the early uniform neighbourhoods depends on the trajectories of Kapital. If the prices keep falling and the dwellings keep decaying eventually the community dissolves, because agents have limited tolerance towards living in a slum... If prices start to rise, as a consequence of Kapital flowing in, the place is bound to genrtify and lose its cultural uniformity. **The fate of many a working class neighbourhood is accurately reproduced in the model!**

Gentrification doesn't always dissolve cultural homogeneity, though. At this stage much also depends on the processes going on in the rest of the city. If other neighbourhoods in the city are decaying, for example, an outflow of agents is to be expected, and since there is one "allured" neighbourhood recognizable, some agents can relocate to a location that reflects better their cultural makeup, reinforcing the homogeneity. Correlation between decreasing prices in one area and increasing uniformity in another is frequent, signalling that this is a recurring dynamic.

In general, abrupt shifts in prices seem to always have a disruptive effect on cultural homogeneity. A high-prices+high-uniformity district where prices start to fall sees an influx of "parvenues" which dilute the uniformity. A low-price+high-uniformity district where prices start to rise displaces some of the residents.



# Changelog

## Version 0.2.15
### NEW ALLURE CODE

We define allure as *locally over-represented minority cultural traits*: traits that are very frequent in a certain area and infrequent at city level.

#### Implementation

We build a city-level allure which contains traits present in at least 40% of the population. This represents the majority cultural configuration.

We then check if there are traits that are represented in at least the 20% of the population in certain districts. If at least one over-represented trait exists and it is different from the corresponding trait in the majority population, we say that the area has an allure.


## Version 0.2.7
We added a "regeneration" button to test the effects of state sponsored regeneration programmes.
Regeneration is intended in the anglo-saxon, "small state" way. Extra money (outside of the existing Kapital stock = i.e. coming from the public purse) is put in the areas least desirable to investors (= those with the most narrow price-gap) that are also empty and in run-down condition. These areas are brought to the maximum condition and to the mean price of the city. The idea is to check whether this practice can trigger further private investment.

## Version 0.2.6-test
We introduce gaps based on the local maximum price instead of local average

## Version 0.2.5-test
We introduce automatic detection of cultural dynamics. Spotting gentrification, recolonisation, etc is now automatic

## Version 0.2.5-be
### Faster
Parts of the code (similarity, set-gaps) have been made much faster.  Also plot frequency of entropy reduced to every 5 ticks as this was slowing the simulation.

### No centres option
New option of "no centres" added in addiction to mono- policentric.  For comparison with those with centres.

### Better (?) income distribution
Tried a different method of generating a distribution with different gini, but same average income - basically moving one up and one correspondingly down.  Tried a few variations of this - this can take a few seconds to do but seems to work.  More thought on this is needed to make "realistic" income distributions.

### Hisotgrams
Histograms of the distributions of incomes and prices replace previous simple line graph.

### New monitors...
...for occupancy rate and current gini index added.

### similarity for allure slider
now goes up to 1.01!  :-) so that at this setting effectively the allure mechanisms is cut out, for comparison.

## Version 0.2.5
### EMERGING ALLURE
In this version the allure of a district is not imposed from the beginning, instead it is an emergent feature. Allure is initially blank (meaning that the area has no particular connotation), when cultural uniformity reaches a threshold (see update-allure function) the allure is set. This is to reflect the fact that not every neighbourhood has a special connotation in the mind of agents, but only those with a recognizable population (e.g. ONLY HIPPIES LIVE THERE, DON'T GO!)

### Other
In this version we have better code to implement multidimensional culture. Every trait can now have as many values as we want, instead of 2.

### Allure update
Allure is updated every 24 months only if uniformity is high, otherwise the old allure stays.

## Version 0.2.2 - 0.2.3
### Six months interval
Previous change reverted. Investment happens every 6 ticks (for performance pourposes).

### Location location location!
Price gap setting mechanism changed! Now the comparison is made *EITHER* with the Moore neighbourhood as in the original version *OR* with the entire district. The assumption is that, when renovating, the investor will be able to maximize profit charging the mean price of the neighbourhood, if the district is expensive, or the more restrictive Moore neighbourhood. The change has a huge impact, see description.

## Version 0.2.1
### Continous investment.
Unlike the previous version, now property investment happens at every tick. The result is a smoother dynamic.

### Strong neighbourhood preference
Agent value more moving to an alluring neighbourhood. They will accept lower repair state in their preferred location.

## Version 0.2-big
### City size
The city is 1692 patches and 12 neighbourhoods ***** This is now in a separate file ****

### Residential choice

* Residents won't move to a location with condition = 0
* Locations with condition = 0 are given price = 0.0001

## Version 0.2
### Threshold for investment
We now have a price-gap threshold for investment (set in procedure go). The rationale is that the Kapital available is spent in the city only if enough profit can be extracted.

Note that the conservative implementetion of version 0.1 is not as implausible as it seems: there can be cases when the investment is carried out, however little the profit may be. For example if nothing else in the economy produces superior profit. Or for money laundering purposes ;-)

### Neighbourhood selecting process
This is slightly changed to make the agents less choosy when selecting the neighbourhood. They now accept as second best a place 15% below the average condition.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="abmus" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <enumeratedValueSet variable="Kapital">
      <value value="0.015"/>
      <value value="0.025"/>
      <value value="0.035"/>
      <value value="0.045"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="refugee-start-tick">
      <value value="90"/>
      <value value="1400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="regeneration?">
      <value value="&quot;none&quot;"/>
      <value value="&quot;regular&quot;"/>
      <value value="&quot;EU-slums&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="pnas" repetitions="40" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1400"/>
    <metric>median [price] of patches</metric>
    <metric>median [income] of turtles</metric>
    <enumeratedValueSet variable="Kapital">
      <value value="0.015"/>
      <value value="0.02"/>
      <value value="0.025"/>
      <value value="0.03"/>
      <value value="0.04"/>
      <value value="0.05"/>
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="profit-threshold">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="immigration-rate">
      <value value="0.012"/>
      <value value="0.045"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="beirut" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <enumeratedValueSet variable="Kapital">
      <value value="0.015"/>
      <value value="0.025"/>
      <value value="0.035"/>
      <value value="0.045"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="refugee-start-tick">
      <value value="90"/>
      <value value="1400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="refugee-end-tick">
      <value value="180"/>
      <value value="1400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="regeneration?">
      <value value="&quot;none&quot;"/>
      <value value="&quot;regular&quot;"/>
      <value value="&quot;EU-slums&quot;"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
