% =========================================================================
% MATLAB script to reproduce results from:
% E. Sulollari, K. N. van Dalen, and A. Cabboi
% "Vibration-induced friction modulation for a general frequency of excitation"
% Journal of Sound and Vibration, Vol. 573, 118200, 2024.
%
% This paper corresponds to Chapter 4 of the dissertation
% "Vibration-induced friction modulation"
%
% This script generates the results shown in the figures of the paper.
%
% Author: Enxhi Sulollari
% Date: 20-08-2025
% =========================================================================

%% This script generates the results shown in Figure 2 of the paper.

% close all
% clear all
% clc

syms v_r             % Define relative velocity (symbolic variable)

% Case 1: Excitation intensity a*omega ~0
mu_s = 0.4;          % Static friction coefficient
omega = 10;          % Excitation frequency
a = 0.0001;          % Excitation amplitude (~0)
mu1 = piecewise(abs(v_r) <= a*omega, ...
                mu_s*((2/pi)*asin(v_r/(a*omega))), ...
                abs(v_r) > a*omega, mu_s*sign(v_r)); % Average friction 
x1 = -0.8:0.001:0.8; % Range of belt velocity
y1 = subs(mu1, v_r, x1);

% Case 2: Excitation intensity a*omega = 0.1 (a = 0.01, omega = 10)
omega = 10;
a = 0.01;
mu2 = piecewise(abs(v_r) <= a*omega, ...
                mu_s*((2/pi)*asin(v_r/(a*omega))), ...
                abs(v_r) > a*omega, mu_s*sign(v_r));
x2 = -0.8:0.001:0.8;
y2 = subs(mu2, v_r, x2);

% Case 3: Excitation intensity a*omega = 0.3 (a = 0.01, omega = 30)
omega = 30;
a = 0.01;
mu3 = piecewise(abs(v_r) <= a*omega, ...
                mu_s*(1-(2/pi)*acos(v_r/(a*omega))), ...
                abs(v_r) > a*omega, mu_s*sign(v_r));
x3 = -0.8:0.001:0.8;
y3 = subs(mu3, v_r, x3);

% Case 4: Excitation intensity a*omega = 0.5 (a = 0.01, omega = 50)
omega = 50;
a = 0.01;
mu4 = piecewise(abs(v_r) <= a*omega, ...
                mu_s*(1-(2/pi)*acos(v_r/(a*omega))), ...
                abs(v_r) > a*omega, mu_s*sign(v_r));
x4 = -0.8:0.001:0.8;
y4 = subs(mu4, v_r, x4);

% Plotting the results
figure
plot(x1, y1, 'DisplayName', '$\alpha\Omega=0$')     % Case 1
hold on
plot(x2, y2, 'DisplayName', '$\alpha\Omega=0.1$')   % Case 2
plot(x3, y3, 'DisplayName', '$\alpha\Omega=0.3$')   % Case 3
plot(x4, y4, 'DisplayName', '$\alpha\Omega=0.5$')   % Case 4

% Add vertical lines for reference velocities
xline(0.1, "--");
xline(0.3, "--");
xline(0.5, "--");

% Axis labels and legend
xlabel('$v_b$','Interpreter','Latex','FontSize',14)          % Belt velocity
ylabel('$\bar{\mu}(v_b)$','Interpreter','Latex','FontSize',14) % Averaged friction coefficient
legend(" $\alpha\Omega=0$"," $\alpha\Omega=0.1$","$ \alpha\Omega=0.3$","$ \alpha\Omega=0.5$",'Interpreter','Latex','Location','southeast','FontSize',11)

ylim([-0.5 0.5])  % Set y-axis limits

%% This script generates the results shown in Figure 4 of the paper.

syms v_r             % Define relative velocity (symbolic variable)

omega = 30;          % Excitation frequency
a = 0.01;            % Excitation amplitude
mu_s = 0.4;          % Static friction coefficient

% MDSM model
mu5 = piecewise(abs(v_r) <= a*omega, ...
                mu_s*(1-(2/pi)*acos(v_r/(a*omega))), ...
                abs(v_r) > a*omega, mu_s*sign(v_r));
x5 = -0.8:0.001:0.8;   % Range of belt velocity
y5 = subs(mu5, v_r, x5);

% Kumar & Hutchings model
mu6 = piecewise(abs(v_r) <= a*omega, ...
                mu_s*((2/pi)*asin(v_r/(a*omega))), ...
                abs(v_r) > a*omega, mu_s*sign(v_r));
x6 = -0.8:0.001:0.8;
y6 = subs(mu6, v_r, x6);

% Plotting the results
figure
plot(x5, y5, 'LineWidth', 1, 'DisplayName', 'MDSM')
hold on
plot(x6, y6, '--', 'LineWidth', 1, 'DisplayName', 'Model of Kumar and Hutchings')

xlabel('$v_b$', 'Interpreter','Latex','FontSize',14)            % Belt velocity
ylabel('$\bar{\mu}(v_b)$', 'Interpreter','Latex','FontSize',14) % Averaged friction coefficient
legend('Interpreter','Latex','FontSize',11)
ylim([-0.5 0.5])  % Set y-axis limits

%% This script generates the results shown in Figures 5 of the paper.

beta = 0.1;                          % Damping coefficient
omega = 0:0.001:10;                  % Excitation frequency range
a = 10;                              % Excitation amplitude

% Compute velocity modulation amplitude (V_kappa = \hat{V} in the paper)
C = a*omega.^3 ./ sqrt(4*beta^2.*omega.^2 + (1 - omega.^2).^2);

% Compute phase angle θ
angle = atan2(2*beta.*omega, (omega.^2 - 1));

% Figure 5(a): Amplitude of velocity response
figure
plot(omega, C, 'LineWidth', 1, 'DisplayName', '$\hat{V}$')      % Velocity response amplitude
hold on
plot(omega, a*omega, 'LineWidth', 1, 'DisplayName', '$\alpha\Omega$') % Excitation intensity
xline(5, "--")  % Reference vertical line
xlabel('$\Omega$', 'Interpreter','Latex','FontSize',14) 
ylabel('$\hat{V}$', 'Interpreter','Latex','FontSize',14)
title('$(a)$', 'Interpreter','Latex','FontSize',11, 'Position', [5, 125, 0])
legend(" $\hat{V}$"," $\alpha\Omega$",'Interpreter','Latex','FontSize',11)

% Figure 5(b): Phase angle
figure
plot(omega, angle, 'LineWidth', 1)
xlabel('$\Omega$', 'Interpreter','Latex','FontSize',14)
ylabel('$\theta$', 'Interpreter','Latex','FontSize',14)
title('$(b)$', 'Interpreter','Latex','FontSize',11, 'Position', [5, 3.645, 0])

%% This script generates the results shown in Figures 6 of the paper.
% Parameters
beta = 0.1;          % Damping coefficient
mu_s = 0.4;          % Static friction coefficient
a = 10;              % Excitation amplitude

% -----------------------------
% Part (a): Velocity amplitude 
% -----------------------------
omega = 0:0.001:5;   % Excitation frequency range
C = a*omega.^3 ./ sqrt(4*beta^2.*omega.^2 + (1 - omega.^2).^2);  % V_kappa

figure
plot(omega, C, "k", 'LineWidth', 1.2)  % Amplitude curve
hold on

% Highlighted operating points (Ω, V_hat)
plot(0.8, 12.9964, "x", 'LineWidth', 1.1)
plot(1.022, 51, "x", 'LineWidth', 1.1)
plot(1.2, 34.4774, "x", 'LineWidth', 1.1)
plot(4, 42.606, "x", 'LineWidth', 1.1)

xlabel('$\Omega$', 'Interpreter','Latex','FontSize',14)
ylabel('$\hat{V}$', 'Interpreter','Latex','FontSize',14)
title('$(a)$','Interpreter','Latex','FontSize',11, 'Position', [2.5, 62.5, 0])

% -----------------------------
% Part (b): Average friction 
% -----------------------------
omega_vals = [0.8, 1.0, 1.2, 4.0];   % Selected excitation frequencies
x_range = -55:0.01:55;               % Belt velocity range

figure
hold on
for om = omega_vals
    % Compute C at given frequency
    C = a*om^3 / sqrt(4*beta^2*om^2 + (1 - om^2)^2);
    
    % Define average friction 
    syms v_r
    mu = piecewise(abs(v_r) <= C, ...
                   mu_s*(1-(2/pi)*acos(v_r/C)), ...
                   abs(v_r) > C, mu_s*sign(v_r));
    
    % Evaluate on belt velocity range
    y_vals = subs(mu, v_r, x_range);
    
    % Plot curve
    plot(x_range, y_vals, 'LineWidth', 1, 'DisplayName', ['$\Omega=', num2str(om), '$'])
end

% Add vertical lines for the operating points
xline(12.9964,"--")
xline(51,"--")
xline(34.4774,"--")
xline(42.606,"--")

xlabel('$v_b$', 'Interpreter','Latex','FontSize',14)
ylabel('$\bar{\mu}(v_b)$', 'Interpreter','Latex','FontSize',14)
legend(" $\Omega=0.8$"," $\Omega=1.0$","$\Omega=1.2$"," $\Omega=4.0$",'Interpreter','Latex','FontSize',11,'Location','southeast')
title('$(b)$','Interpreter','Latex','FontSize',11, 'Position', [0, 0.54167, 0])

xlim([-55 55])
ylim([-0.5 0.5])

%% This script generates the results shown in Figures 8 of the paper.
% Note:
% To understand the parameters used in this code, see:
% J. Den Hartog, 
% "Forced vibrations with combined viscous and Coulomb damping",
% London, Edinburgh, Dublin Philosophical Magazine and Journal of Science,
% Vol. 9 (59), pp. 801–817, 1930.

% -----------------------------
% Part (a): Stick-slip boundary for a damped SDOF system subjected to harmonic excitation
% -----------------------------

% den Hartog  + viscous damping make beta reallyyy small ~0
r_v = 0.01:0.01:5; 
tau_1 = 0:0.01:pi;

omega_n=1;     % natural frequency
k=1;           % stiffness
m=1;           % mass
g=9.81;        % gravity
N=9.81*m;      % normal force
beta=0.000001; % damping coefficient
c=2*beta*omega_n*m; 
V = 1./(1-r_v.^2);
mu=0.4;        % friction coefficient
xf=mu*N/k;
a=xf/0.2;      % excitation amplitude (choose per Fig. 7)

c_c=2*m*omega_n;

p=omega_n*sqrt(1-(c/c_c)^2);
q=sqrt((1./V).^2+(2*c.*r_v./c_c).^2);
G=(sinh(pi*c./(c_c*r_v))-(c/c_c)/sqrt(1-(c/c_c)^2) *sin(pi*sqrt(1-(c/c_c)^2)./r_v))./(cosh(pi*c./(c_c.*r_v))+ cos(pi*sqrt(1-(c/c_c).^2)./r_v));
H=(sin(pi*sqrt(1-(c/c_c)^2)./r_v)./(cosh(pi*c./(c_c*r_v))+cos(pi*sqrt(1-(c/c_c)^2)./r_v))).*(1./(r_v*sqrt(1-(c/c_c)^2)));
x0=(-G.*(xf./a)+sqrt(1./q.^2 - H.^2*(xf./a)^2)).*a; 

for it = 1:length(tau_1)
    tau = tau_1(it);

    lim16(it,:)= (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)).*((p./(r_v.*omega_n) +c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin(p.*(tau./(r_v*omega_n))) +H.*(c.*sin(p.*(tau./(r_v*omega_n)))./(2*p*m)-cos(p.*(tau./(r_v*omega_n))))) +H.*cot(tau)-G;

end

lim_16=max(lim16);

lim_17=-G+2*H*c./(c_c*r_v)+(1+G)./(r_v.^2);
S1=lim_16./lim_17;
I=2*H*c./(c_c.*r_v) +(1+G)./r_v.^2;
lim_18=1./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); % xf/a <
lim_19=S1.*(1-G)./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); % x0/a >

hold on
plot(r_v,lim_18)

% den Hartog  + viscous damping; case of beta = 0.1
r_v = 0.01:0.01:5; 
tau_1 = 0:0.01:pi;

omega_n=1;     % natural frequency
k=1;           % stiffness
m=1;           % mass
beta=0.1;      % damping coefficient
c=2*beta*omega_n*m;
V = 1./(1-r_v.^2);
mu=0.4;        % friction coefficient
xf=mu*N/k;
a=xf/0.2;      % excitation amplitude (choose per Fig. 7)

c_c=2*m*omega_n;

p=omega_n*sqrt(1-(c/c_c)^2);
q=sqrt((1./V).^2+(2*c.*r_v./c_c).^2);
G=(sinh(pi*c./(c_c*r_v))-(c/c_c)/sqrt(1-(c/c_c)^2) *sin(pi*sqrt(1-(c/c_c)^2)./r_v))./(cosh(pi*c./(c_c.*r_v))+ cos(pi*sqrt(1-(c/c_c).^2)./r_v));
H=(sin(pi*sqrt(1-(c/c_c)^2)./r_v)./(cosh(pi*c./(c_c*r_v))+cos(pi*sqrt(1-(c/c_c).^2)./r_v))).*(1./(r_v*sqrt(1-(c/c_c)^2)));
x0=(-G.*(xf./a)+sqrt(1./q.^2 - H.^2*(xf./a)^2)).*a; 

for it = 1:length(tau_1)
    tau = tau_1(it);

    lim16(it,:)= (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)).*((p./(r_v.*omega_n) +c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin(p.*(tau./(r_v*omega_n))) +H.*(c.*sin(p.*(tau./(r_v*omega_n)))./(2*p*m)-cos(p.*(tau./(r_v*omega_n))))) +H.*cot(tau)-G;

end

lim_16=max(lim16);

lim_17=-G+2*H*c./(c_c*r_v)+(1+G)./(r_v.^2);
S1=lim_16./lim_17;
I=2*H*c./(c_c.*r_v) +(1+G)./r_v.^2;
lim_18=1./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); % xf/a <
lim_19=S1.*(1-G)./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); % x0/a >

hold on
plot(r_v,lim_18)

% den Hartog  + viscous damping; case of beta = 0.2
r_v = 0.01:0.01:5; 
tau_1 = 0:0.01:pi;

omega_n=1;     % natural frequency
m=1;           % mass
beta=0.2;      % damping coefficient
c=2*beta*omega_n*m;
V = 1./(1-r_v.^2);
c_c=2*m*omega_n;

p=omega_n*sqrt(1-(c/c_c)^2);
q=sqrt((1./V).^2+(2*c.*r_v./c_c).^2);
G=(sinh(pi*c./(c_c*r_v))-(c/c_c)/sqrt(1-(c/c_c)^2) *sin(pi*sqrt(1-(c/c_c)^2)./r_v))./(cosh(pi*c./(c_c.*r_v))+ cos(pi*sqrt(1-(c/c_c).^2)./r_v));
H=(sin(pi*sqrt(1-(c/c_c)^2)./r_v)./(cosh(pi*c./(c_c*r_v))+cos(pi*sqrt(1-(c/c_c)^2)./r_v))).*(1./(r_v*sqrt(1-(c/c_c)^2)));

for it = 1:length(tau_1)
    tau = tau_1(it);

    lim16(it,:)= (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)).*((p./(r_v.*omega_n) +c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin(p.*(tau./(r_v*omega_n))) +H.*(c.*sin(p.*(tau./(r_v*omega_n)))./(2*p*m)-cos(p.*(tau./(r_v*omega_n))))) +H.*cot(tau)-G;

end

lim_16=max(lim16);

lim_17=-G+2*H*c./(c_c*r_v)+(1+G)./(r_v.^2);
S1=lim_16./lim_17;
I=2*H*c./(c_c.*r_v) +(1+G)./r_v.^2;
lim_18=1./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); % xf/a <
lim_19=S1.*(1-G)./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); % x0/a >

hold on
plot(r_v,lim_18)

xlabel('$\Omega$','Interpreter','Latex','FontSize',14)
ylabel('$\frac{\mu_sN}{mr\Omega_e^2}$','Interpreter','Latex','FontSize',14)
legend("$\beta=0$","$\beta=0.1$","$\beta=0.2$",'Interpreter','Latex','FontSize',11)
title('$(a)$','Interpreter','Latex','FontSize',14, 'Position', [2.5,0.9375, 0])

%% Figure 8(b) for 3 vb values

%clc; clear; close all;

% Global parameters
global beta mu_s omega_e gamma A0 vb omega_n m g

% Time vector
tau = 0:0.001:110;

% System parameters
beta = 0.1;
mu_s = 0.4;
omega_e = 0.4; % excitation frequency
g = 9.81;
m = 1;
k = 1;
omega_n = 1; % natural frequency
omega = omega_e/omega_n;
gamma = g/omega_n^2; % consider L=1
a1 = 10;
A0 = a1*omega_e^2;

% Initial conditions
initial_position = 0;
initial_speed = 0;
y0 = [initial_position initial_speed];

% vb values to sweep
vb_values = [0.6 0.7 0.8];
colors = {'k-','b-','r-'}; % line styles/colors

figure; hold on
for i = 1:length(vb_values)
    vb = vb_values(i);
    [~, y] = ode23(@rhs_sdof_num, tau, y0);
    plot(tau, y(:,2), colors{i}, 'LineWidth', 1.2)
end

% Labels, legend, and title with LaTeX
xlabel('$\tau$', 'Interpreter', 'Latex', 'FontSize', 14)
ylabel('$\frac{dx}{d\tau}$', 'Interpreter', 'Latex', 'FontSize', 14)
legend(sprintf('$v_b = %.1f$', vb_values(1)), ...
       sprintf('$v_b = %.1f$', vb_values(2)), ...
       sprintf('$v_b = %.1f$', vb_values(3)), ...
       'Interpreter', 'Latex', 'FontSize', 11)
title('$(b)$', 'Interpreter', 'Latex', 'FontSize', 14, 'Position', [50, 1.5, 0])
xlim([70 105])

%uncomment the function when running this section
% RHS function
% function ydot = rhs_sdof_num(tau, y)
%     global beta mu_s omega_e A0 vb omega_n m g
%     ydot_1 = y(2); % dx/dtau
%     ydot_2 = -2*beta*omega_n*y(2) ...
%              - omega_n^2*y(1) ...
%              - g*(mu_s*sign(y(2)-vb)) ...
%              + A0*sin(omega_e*tau)/m ...
%              - mu_s*g;
%     ydot = [ydot_1; ydot_2];
% end
%%
%% This script generates the results shown in Figures 10 of the paper.
% Stick-slip boundary for 2-DOF system (α = 10, μs = 0.4)

r_v = 0.01:0.01:5;          
tau_1 = 0:0.01:pi;         

omega_n = 1;
k       = 1;
m       = 1;
beta    = 0.1;
c       = 2*beta*omega_n*m;

c_c     = 2*m*omega_n;
p       = omega_n*sqrt(1-(c/c_c)^2);

% Preallocate arrays
lim16 = zeros(length(tau_1), length(r_v));

V = 1./(1-r_v.^2);

q = sqrt((1./V).^2 + (2*c.*r_v./c_c).^2);
G = (sinh(pi*c./(c_c*r_v)) - (c/c_c)/sqrt(1-(c/c_c)^2) * sin(pi*sqrt(1-(c/c_c)^2)./r_v)) ...
    ./ (cosh(pi*c./(c_c.*r_v)) + cos(pi*sqrt(1-(c/c_c)^2)./r_v));
H = (sin(pi*sqrt(1-(c/c_c)^2)./r_v) ./ (cosh(pi*c./(c_c*r_v)) + cos(pi*sqrt(1-(c/c_c)^2)./r_v))) ...
    .* (1./(r_v*sqrt(1-(c/c_c)^2)));

for it = 1:length(tau_1)
    tau = tau_1(it);
    lim16(it,:) = (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)) ...
        .* ((p./(r_v.*omega_n) + c^2./(4*m^2*p.*(r_v.*omega_n))) .* (1+G) .* sin(p.*(tau./(r_v*omega_n))) ...
        + H .* (c.*sin(p.*(tau./(r_v.*omega_n)))./(2*p*m) - cos(p.*(tau./(r_v.*omega_n))))) ...
        + H.*cot(tau) - G;
end

lim_16 = max(lim16);
lim_17 = -G + 2*H*c./(c_c*r_v) + (1+G)./(r_v.^2);
S1 = lim_16 ./ lim_17;
I  = 2*H*c./(c_c.*r_v) + (1+G)./r_v.^2;
lim_18 = 1 ./ (q .* sqrt(H.^2 + (S1.*I + (1-S1).*G).^2));

% Tangential loading
mu_s = 0.4;
alpha = 10;
g = 9.81;

ampl_t = m*alpha*r_v.^2;
lim_2dof_t = mu_s*m*g ./ ampl_t;

% Plot results
figure
plot(r_v, lim_18, 'LineWidth',1.5)
hold on
plot(r_v, lim_2dof_t ,'LineWidth',1.5)

xlim([0 5])             % Limit x-axis from 0 to 5
ylim([0 0.8])           % Limit y-axis
xlabel('$\Omega$','Interpreter','Latex','FontSize',14)
ylabel('$\mu_s g / P_t$','Interpreter','Latex','FontSize',14)
legend("stick-slip boundary", "$\mu_s g / P_t$", 'Interpreter','Latex','Location','northeast','FontSize',11)
title('$(a)$','Interpreter','Latex','FontSize',11,'Position',[2.5,0.834])
% grid on

%% This script generates the results shown in Figures 11 of the paper.

omega_2 = 0.5;
beta_2  = 0.15;
g       = 9.81;

a       = 0.1:1:11;
omega_e = 0.1:0.1:5;

x = zeros(length(a), length(omega_e));

% Compute amplitude
for j = 1:length(omega_e)
    for i = 1:length(a)
        Xc = -2*a(i)*beta_2*omega_e(j)^3*omega_2 / ((omega_2^2 - omega_e(j)^2)^2 + 4*beta_2^2*omega_e(j)^2*omega_2^2);
        Xs =  a(i)*omega_e(j)^2 * (omega_2^2 - omega_e(j)^2) / ((omega_2^2 - omega_e(j)^2)^2 + 4*beta_2^2*omega_e(j)^2*omega_2^2);
        x(i,j) = sqrt(Xc^2 + Xs^2);
    end
end

% Reference limit
lim = (g / omega_2^2) * ones(size(x));

% Plot
figure
surf(omega_e, a, x, 'FaceAlpha',0.8);  % slightly transparent
hold on
surf(omega_e, a, lim, 'FaceAlpha',0.5)

xlabel('$\Omega_e$', 'Interpreter','Latex','FontSize',14)
ylabel('$\alpha$', 'Interpreter','Latex','FontSize',14)
zlabel('$|X_2|$', 'Interpreter','Latex','FontSize',14)

legend("$|X_2|$", "$X_{2,0}$", 'Interpreter','Latex','Location','southeast','FontSize',11)
grid on
box on
view(45,30)  % adjust view for clarity

%% This script generates the results shown in Figures 12 of the paper.
% clear all
% close all
% clc
% Parameters
r_v = 0.01:0.01:5;
tau_1 = 0:0.01:pi;

omega_n = 1;
m = 1;
beta = 0.1;
c = 2*beta*omega_n*m;

c_c = 2*m*omega_n;

V = 1 ./ (1 - r_v.^2);
p = omega_n * sqrt(1 - (c/c_c)^2);
q = sqrt((1./V).^2 + (2*c.*r_v./c_c).^2);

G = (sinh(pi*c./(c_c*r_v)) - (c/c_c)/sqrt(1-(c/c_c)^2) * sin(pi*sqrt(1-(c/c_c)^2)./r_v)) ...
    ./ (cosh(pi*c./(c_c.*r_v)) + cos(pi*sqrt(1-(c/c_c)^2)./r_v));

H = (sin(pi*sqrt(1-(c/c_c)^2)./r_v) ...
    ./ (cosh(pi*c./(c_c*r_v)) + cos(pi*sqrt(1-(c/c_c)^2)./r_v))) ...
    .* (1 ./ (r_v * sqrt(1-(c/c_c)^2)));

lim16 = zeros(length(tau_1), length(r_v));
for it = 1:length(tau_1)
    tau = tau_1(it);
    exp_term = exp(-c.*(tau./(r_v.*omega_n)) / (2*m)) ./ sin(tau);
    sin_term = sin(p.*(tau./(r_v*omega_n)));
    cos_term = cos(p.*(tau./(r_v*omega_n)));

    lim16(it,:) = exp_term .* ((p./(r_v.*omega_n) + c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin_term ...
                    + H .* (c.*sin_term./(2*p*m) - cos_term)) ...
                    + H.*cot(tau) - G;
end

lim_16 = max(lim16);

% Compute stick-slip boundary
lim_17 = -G + 2*H*c./(c_c*r_v) + (1+G)./(r_v.^2);
S1 = lim_16 ./ lim_17;
I = 2*H*c./(c_c.*r_v) + (1+G)./r_v.^2;

lim_18 = 1 ./ (q .* sqrt(H.^2 + (S1.*I + (1-S1).*G).^2));  % xf/a<
lim_19 = S1.*(1-G) ./ (q .* sqrt(H.^2 + (S1.*I + (1-S1).*G).^2));  % x0/a>

% Plot stick-slip boundary
figure
plot(r_v, lim_18)
hold on

% 2-DOF system with tangential and normal loading
mu_s = 0.4;
omega_2 = 0.5;
beta_2 = 0.15;
alpha = 10;
g = 9.81;

tau_max = 55*pi;
dt = 0.01;
tau_vec = 0:dt:tau_max;
forcing_term = zeros(length(tau_vec), length(r_v));

for j = 1:length(r_v)
    rv = r_v(j);
    theta = atan2(2*beta_2*omega_2*rv, rv^2 - omega_2^2);
    for i = 1:length(tau_vec)
        tau = tau_vec(i);
        forcing_term(i,j) = -(mu_s*alpha*rv^2 / sqrt(4*beta_2^2*omega_2^2*rv^2 + (omega_2^2 - rv^2)^2)) ...
                            * (omega_2^2*sin(rv*tau + theta) + 2*beta_2*omega_2*rv*cos(rv*tau + theta)) ...
                            + alpha*rv^2*sin(rv*tau);
    end
end

ampl_tn = max(abs(forcing_term));
lim_2dof_tn = mu_s * g ./ ampl_tn;

% Plot 
plot(r_v, lim_2dof_tn)
ylim([0, 0.8]);
xlabel('$\Omega$', 'Interpreter','Latex','FontSize',14)
ylabel('$\mu_s g / P_{tn}$', 'Interpreter','Latex','FontSize',14)
legend("stick-slip boundary","$\mu_s g / P_{tn}$", 'Interpreter','Latex','Location','northeast','FontSize',11)
title('$(a)$','Interpreter','Latex','FontSize',11,'Position',[2.5,0.834])

%% Figure 12(b)

% --- Parameters ---
r_v = 0.01:0.01:5;
tau_1 = 0:0.01:pi;

omega_n=1;
k=1;
m=1;
beta=0.1;
c=2*beta*omega_n*m;
V = 1./(1-r_v.^2);

c_c=2*m*omega_n;

p=omega_n*sqrt(1-(c/c_c)^2);
q=sqrt((1./V).^2+(2*c.*r_v./c_c).^2);
G=(sinh(pi*c./(c_c*r_v))-(c/c_c)/sqrt(1-(c/c_c)^2) *sin(pi*sqrt(1-(c/c_c)^2)./r_v))./(cosh(pi*c./(c_c.*r_v))+ cos(pi*sqrt(1-(c/c_c).^2)./r_v));
H=(sin(pi*sqrt(1-(c/c_c)^2)./r_v)./(cosh(pi*c./(c_c*r_v))+cos(pi*sqrt(1-(c/c_c)^2)./r_v))).*(1./(r_v*sqrt(1-(c/c_c)^2)));

for it = 1:length(tau_1)
    tau = tau_1(it);

    lim16(it,:)= (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)).*((p./(r_v.*omega_n) +c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin(p.*(tau./(r_v*omega_n))) +H.*(c.*sin(p.*(tau./(r_v*omega_n)))./(2*p*m)-cos(p.*(tau./(r_v*omega_n))))) +H.*cot(tau)-G;

end

lim_16=max(lim16);
lim_17=-G+2*H*c./(c_c*r_v)+(1+G)./(r_v.^2);
S1=lim_16./lim_17;
I=2*H*c./(c_c.*r_v) +(1+G)./r_v.^2;
lim_18=1./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %xf/a<
lim_19=S1.*(1-G)./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %x0/a>

% getting 3D plot
m=1;
omega_2=0.5;
beta_2=0.15;
alpha=10;
g=9.81;

lim_2dof_tn_array = [];

for mu_s=0.1:0.01:1
    i=0;
    j=0;
    for r_v= 0.01:0.01:5
       j=1+j;
       i=0;
        for tau = 0:0.01:55*pi
            i=1+i;
            theta=atan2(2*beta_2*omega_2*r_v,r_v.^2-omega_2^2);
            forcing_term(i,j)= -(mu_s*alpha*r_v.^2./sqrt(4*beta_2^2*omega_2^2.*r_v.^2+(omega_2^2-r_v.^2).^2)).*(omega_2^2.*sin(r_v.*tau+theta)+2*beta_2*omega_2*r_v.*cos(r_v*tau+theta)) +alpha*r_v.^2.*sin(r_v*tau);
        end
    end

    ampl_tn=max(abs(forcing_term)); 
    r_v = 0.01:0.01:5;
    lim_2dof_tn=mu_s*g./ampl_tn;
    % append the calculated values to the arrays
    lim_2dof_tn_array = [lim_2dof_tn_array; lim_2dof_tn];
    
end


result = zeros(size(lim_2dof_tn_array, 1), 1); % Initialize the result vector for fig 14

for i = 1:size( lim_2dof_tn_array, 1)
    [~, idx] = min(abs(lim_18 -  lim_2dof_tn_array(i,:))); % Find the index of the closest element
    idx_all(i,:) = idx; % Save the indices to the matrix
    result(i) = lim_18(idx); % Save the closest element to the result vector
end
omegas = r_v(idx_all);

mu_s=0.1:0.01:1;
beta_1 = 0.1;
beta_2 = 0.15;
omega_1 = 1;
omega_2 = 0.5;
a = 10;

[omega_ss, mu_s_ss] = meshgrid(0:0.01:5, 0.1:0.01:1);


C_tn_ss = a * omega_ss.^3 .* sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s_ss+1).^2 ...
    + (omega_2^2-omega_ss.^2).^2 + mu_s_ss.*omega_2^2.*(mu_s_ss*omega_2^2-2*omega_ss.^2+2*omega_2^2)) ...
    ./ sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2) ...
    .* (4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));

% Initialize color_map to all red
color_map = ones(size(C_tn_ss, 1), size(C_tn_ss, 2), 3);
color_map(:,:,2) = color_map(:,:,2) * 0; % Green channel to 0
color_map(:,:,3) = color_map(:,:,3) * 0; % Blue channel to 0

% Loop through rows of mu_s_ss
for i = 1:size(mu_s_ss, 1)
    % Find corresponding elements in omega_ss and color_map
    omega_m = omegas(i);
    omega_indices = omega_ss(i, :) <= omega_m;
    
    % Set color values to magenta for corresponding elements
    color_map(i, omega_indices, :) = repmat([1 0 1], sum(omega_indices), 1);
end

[omega_total,mu_s_total] = meshgrid(0:0.05:5,0.1:0.01:1);


C_fast=(1).*a.*omega_total;



% Create 3D plot
surf(omega_ss, mu_s_ss, C_tn_ss, 'CData', color_map,'EdgeAlpha',0.5);
hold on
s3=surf(omega_total,mu_s_total,C_fast);
s3.FaceColor='b';
xlabel('$\Omega_e$','Interpreter','Latex','FontSize',14)
ylabel('$\mu_s$','Interpreter','Latex','FontSize',14)
zlabel('$\hat{V}_{tn}$','Interpreter','Latex','FontSize',14)

%% This script generates the results shown in Figures 13 of the paper.

% clear all
% close all
% clc
% Parameters
r_v = 0.01:0.01:5;
tau_1 = 0:0.01:pi;

omega_n=1;
k=1;
m=1;
beta=0.1;
c=2*beta*omega_n*m;
V = 1./(1-r_v.^2);
c_c=2*m*omega_n;
p=omega_n*sqrt(1-(c/c_c)^2);
q=sqrt((1./V).^2+(2*c.*r_v./c_c).^2);
G=(sinh(pi*c./(c_c*r_v))-(c/c_c)/sqrt(1-(c/c_c)^2) *sin(pi*sqrt(1-(c/c_c)^2)./r_v))./(cosh(pi*c./(c_c.*r_v))+ cos(pi*sqrt(1-(c/c_c).^2)./r_v));
H=(sin(pi*sqrt(1-(c/c_c)^2)./r_v)./(cosh(pi*c./(c_c*r_v))+cos(pi*sqrt(1-(c/c_c)^2)./r_v))).*(1./(r_v*sqrt(1-(c/c_c)^2)));
% x0=(-G.*(xf./a)+sqrt(1./q.^2 - H.^2*(xf./a)^2)).*a; 
    

for it = 1:length(tau_1)
    tau = tau_1(it);

    lim16(it,:)= (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)).*((p./(r_v.*omega_n) +c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin(p.*(tau./(r_v*omega_n))) +H.*(c.*sin(p.*(tau./(r_v*omega_n)))./(2*p*m)-cos(p.*(tau./(r_v*omega_n))))) +H.*cot(tau)-G;

end
lim_16=max(lim16);


lim_17=-G+2*H*c./(c_c*r_v)+(1+G)./(r_v.^2);
S1=lim_16./lim_17;
I=2*H*c./(c_c.*r_v) +(1+G)./r_v.^2;
lim_18=1./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %xf/a<
lim_19=S1.*(1-G)./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %x0/a>

figure
plot(r_v,lim_18)


% 
% considering tangential (-) and normal loading in 2 dof

m=1;
mu_s=0.4;
omega_2=0.5;
beta_2=0.15;
alpha=10;
g=9.81;



i=0;
j=0;
for r_v= 0.01:0.01:5
   j=1+j;
   i=0;
    for tau = 0:0.01:55*pi
        i=1+i;
        theta=atan2(2*beta_2*omega_2*r_v,r_v.^2-omega_2^2);
% for tangential forcing in -x
        forcing_term2(i,j)= -(mu_s*alpha*r_v.^2./sqrt(4*beta_2^2*omega_2^2.*r_v.^2+(omega_2^2-r_v.^2).^2)).*(omega_2^2.*sin(r_v.*tau+theta)+2*beta_2*omega_2*r_v.*cos(r_v*tau+theta))-alpha*r_v.^2.*sin(r_v*tau);
    end
end


ampl_tn2=max(abs(forcing_term2));

r_v = 0.01:0.01:5;


ampl_tn2_19=(abs(alpha*r_v.^2*(-mu_s-1)));


lim_2dof_tn2=mu_s*g./ampl_tn2;

lim_2dof_tn2_19=mu_s*g./ampl_tn2_19;
hold on
plot(r_v, lim_2dof_tn2_19)
ylim([0, 0.8]);
xlabel('$\Omega$','Interpreter','Latex','FontSize',14)
ylabel('$\mu_sg/P_{tn}$','Interpreter','Latex','FontSize',14)
% legend(" stick-slip boundary "," $\mu_sg/P_{tn}$",'Interpreter','Latex','Location','northeast','FontSize',11)
legend(" stick-slip boundary "," $\mu_sg/P_{tn}(-)$"," $\mu_sg/P_{tn}$",'Interpreter','Latex','Location','northeast','FontSize',11)

%% Figure 13(b)

% --- Parameters ---
r_v = 0.01:0.01:5;
tau_1 = 0:0.01:pi;

omega_n=1;
k=1;
m=1;
beta=0.1;
c=2*beta*omega_n*m;
V = 1./(1-r_v.^2);

c_c=2*m*omega_n;

p=omega_n*sqrt(1-(c/c_c)^2);
q=sqrt((1./V).^2+(2*c.*r_v./c_c).^2);
G=(sinh(pi*c./(c_c*r_v))-(c/c_c)/sqrt(1-(c/c_c)^2) *sin(pi*sqrt(1-(c/c_c)^2)./r_v))./(cosh(pi*c./(c_c.*r_v))+ cos(pi*sqrt(1-(c/c_c).^2)./r_v));
H=(sin(pi*sqrt(1-(c/c_c)^2)./r_v)./(cosh(pi*c./(c_c*r_v))+cos(pi*sqrt(1-(c/c_c)^2)./r_v))).*(1./(r_v*sqrt(1-(c/c_c)^2)));

for it = 1:length(tau_1)
    tau = tau_1(it);

    lim16(it,:)= (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)).*((p./(r_v.*omega_n) +c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin(p.*(tau./(r_v*omega_n))) +H.*(c.*sin(p.*(tau./(r_v*omega_n)))./(2*p*m)-cos(p.*(tau./(r_v*omega_n))))) +H.*cot(tau)-G;

end

lim_16=max(lim16);


lim_17=-G+2*H*c./(c_c*r_v)+(1+G)./(r_v.^2);
S1=lim_16./lim_17;
I=2*H*c./(c_c.*r_v) +(1+G)./r_v.^2;
lim_18=1./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %xf/a<
lim_19=S1.*(1-G)./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %x0/a>

% getting 3D plot
m=1;
omega_2=0.5;
beta_2=0.15;
alpha=10;
g=9.81;

lim_2dof_tn_array = [];
lim_2dof_tn2_array = [];

for mu_s=0.1:0.01:1
    i=0;
    j=0;
    for r_v= 0.01:0.01:5
       j=1+j;
       i=0;
        for tau = 0:0.01:55*pi
            i=1+i;
            theta=atan2(2*beta_2*omega_2*r_v,r_v.^2-omega_2^2);
    % for tangential forcing in -x
            forcing_term2(i,j)= -(mu_s*alpha*r_v.^2./sqrt(4*beta_2^2*omega_2^2.*r_v.^2+(omega_2^2-r_v.^2).^2)).*(omega_2^2.*sin(r_v.*tau+theta)+2*beta_2*omega_2*r_v.*cos(r_v*tau+theta))-alpha*r_v.^2.*sin(r_v*tau);
        end
    end


    ampl_tn2=max((forcing_term2));
    r_v = 0.01:0.01:5;
    lim_2dof_tn2=mu_s*g./ampl_tn2;
    
    % append the calculated values to the arrays
    lim_2dof_tn2_array = [lim_2dof_tn2_array; lim_2dof_tn2];
end



result = zeros(size(lim_2dof_tn2_array, 1), 1); % Initialize the result vectorfor fig 16

for i = 1:size( lim_2dof_tn2_array, 1)
    [~, idx] = min(abs(lim_18 -  lim_2dof_tn2_array(i,:))); % Find the index of the closest element
    idx_all(i,:) = idx; % Save the indices to the matrix
    result(i) = lim_18(idx); % Save the closest element to the result vector
end
omegas = r_v(idx_all);
mu_s=0.1:0.01:1;

figure
plot(omegas,mu_s)


beta_1 = 0.1;
beta_2 = 0.15;
omega_1 = 1;
omega_2 = 0.5;
a = 10;

[omega_ss, mu_s_ss] = meshgrid(0:0.01:5, 0.1:0.01:1);

C_tn_ss = a * omega_ss.^3 .* sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s_ss-1).^2 ...
    + (omega_2^2-omega_ss.^2).^2 + mu_s_ss.*omega_2^2.*(mu_s_ss*omega_2^2+2*omega_ss.^2-2*omega_2^2)) ...
    ./ sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2) ...
    .* (4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));



% Initialize color_map to all red
color_map = ones(size(C_tn_ss, 1), size(C_tn_ss, 2), 3);
color_map(:,:,2) = color_map(:,:,2) * 0; % Green channel to 0
color_map(:,:,3) = color_map(:,:,3) * 0; % Blue channel to 0

% Loop through rows of mu_s_ss
for i = 1:size(mu_s_ss, 1)
    % Find corresponding elements in omega_ss and color_map
    omega_m = omegas(i);
    omega_indices = omega_ss(i, :) <= omega_m;
    
    % Set color values to magenta for corresponding elements
    color_map(i, omega_indices, :) = repmat([1 0 1], sum(omega_indices), 1);
end

[omega_total,mu_s_total] = meshgrid(0:0.05:5,0.1:0.01:1);


C_fast=(1).*a.*omega_total;


% Create 3D plot
surf(omega_ss, mu_s_ss, C_tn_ss, 'CData', color_map,'EdgeAlpha',0.5);
hold on
s3=surf(omega_total,mu_s_total,C_fast);
s3.FaceColor='b';
xlabel('$\Omega_e$','Interpreter','Latex','FontSize',14)
ylabel('$\mu_s$','Interpreter','Latex','FontSize',14)
zlabel('$\hat{V}_{tn}$','Interpreter','Latex','FontSize',14)

%% This script generates the results shown in Figures 14 of the paper.

 beta_1 =0.1;
 beta_2=0.15;
 omega_1=1;
 omega_2=0.5;
 a=10;
 mu_s=0.4;
 
% tangential only
beta =0.1;
omega_ss1=0:0.001:0.78;
C_ss=a*omega_ss1.^3./sqrt(4*beta^2.*omega_ss1.^2+(1-omega_ss1.^2).^2);
omega1=0.78:0.001:5;
C_t=a*omega1.^3./sqrt(4*beta^2.*omega1.^2+(1-omega1.^2).^2);

%tangential and normal loading
omega_ss2=0:0.001:0.84;

C_tn_ss=a*omega_ss2.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss2.^2.*(mu_s+1).^2 +(omega_2^2-omega_ss2.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega_ss2.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss2.^2 +(omega_1^2-omega_ss2.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss2.^2 +(omega_2^2-omega_ss2.^2).^2));

omega2=0.84:0.001:5;

C_tn=a*omega2.^3.*sqrt(4*beta_2^2*omega_2^2.*omega2.^2.*(mu_s+1).^2 +(omega_2^2-omega2.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega2.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega2.^2 +(omega_1^2-omega2.^2).^2).*(4*beta_2^2*omega_2^2.*omega2.^2 +(omega_2^2-omega2.^2).^2));

%tangential(-) and normal loading
omega_ss=0:0.001:0.72;
C_ttn_ss=a*omega_ss.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s-1).^2 +(omega_2^2-omega_ss.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega_ss.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));

omega=0.72:0.001:5;
C_ttn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s-1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

figure
plot(omega_ss1,C_ss,"m",'LineWidth',1.5)
hold on
plot(omega1,C_t)
hold on
plot(omega_ss2,C_tn_ss,"m",'LineWidth',1.5)
hold on
plot(omega2,C_tn)
hold on
plot(omega_ss,C_ttn_ss,"m",'LineWidth',1.5)
hold on
plot(omega,C_ttn)
xlabel('$\Omega_e$','Interpreter','Latex','FontSize',14)
ylabel('$\hat{V}$','Interpreter','Latex','FontSize',14)
legend(" stick-slip"," $\hat{V}_{t}$"," stick-slip"," $\hat{V}_{tn}$"," stick-slip"," $\hat{V}_{tn}(-)$", 'Interpreter','Latex','Location','southeast','FontSize',11)
% 
title('$(a)$','Interpreter','Latex','FontSize',11, 'Position', [2.5,62.5, 0])
%% Figure14(b)

beta =0.1;     % damping coeff x1 direction
 beta_1=beta;  % damping coeff x1 direction
 beta_2=0.15;  % damping coeff x2 direction
 mu_s=0.4;     % friction coefficient
 omega_e=1;    % excitation freq
 omega_1=1;    % nat freq x1 direction
 omega_2=0.5;  % nat freq x2 direction
 omega_n=1;
 omega=omega_e/omega_n;
 a=10;


% average friction tangential loading only
C=a*omega.^3./sqrt(4*beta^2.*omega.^2+(1-omega.^2).^2);

syms v_r
mu5 = piecewise(abs(v_r) <= C,mu_s*(1-(2/pi)*acos(v_r/C)),abs(v_r)>C,mu_s*sign(v_r));
x5=-85:0.01:85;
y5=subs(mu5,v_r,x5);    

% average friction tangential and normal loading
C_tn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s+1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

syms v_r
mu7 = piecewise(abs(v_r) <= C_tn,mu_s*(1-(2/pi)*acos(v_r/C_tn)),abs(v_r)>C_tn,mu_s*sign(v_r));
x7=-85:0.01:85;
y7=subs(mu7,v_r,x7); 

% average friction tangential (-) and normal loading
C_ttn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s-1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

syms v_r
mu8 = piecewise(abs(v_r) <= C_ttn,mu_s*(1-(2/pi)*acos(v_r/C_ttn)),abs(v_r)>C_ttn,mu_s*sign(v_r));
x8=-85:0.01:85;
y8=subs(mu8,v_r,x8); 
    
    
figure
plot(x5,y5)
hold on
plot(x7,y7)
hold on
plot(x8,y8)
xlabel('$V_b$','Interpreter','Latex','FontSize',14)
ylabel('$\bar{\mu}(V_b)$','Interpreter','Latex','FontSize',14)
legend(" $\bar{\mu}_t$","$\bar{\mu}_{tn}$"," $\bar{\mu}_{tn}(-)$",'Interpreter','Latex','FontSize',11,'Location','southeast')
title('$(b)$','Interpreter','Latex','FontSize',11, 'Position', [0,0.54167])

xlim([-65 65])
ylim([-0.5 0.5])

%% This script generates the results shown in Figures 15 of the paper.

  beta_1 =0.1;
 beta_2=0.15;
 omega_1=1;
 omega_2=2;
 a=1;
 mu_s=0.1;

 omega_ss=0:0.001:1.15;
% omega_ss=0:0.001:1.4; %for fig 19(a)

C_tn_ss=a*omega_ss.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s+1).^2 +(omega_2^2-omega_ss.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega_ss.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));
% C_tn_ss=a*omega_ss.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s-1).^2 +(omega_2^2-omega_ss.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega_ss.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));

omega=1.15:0.001:5;
% omega=1.4:0.001:5;

C_tn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s+1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));
% C_tn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s-1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

omega_total=0:0.001:5;


figure
plot(omega_ss,C_tn_ss,"m",'LineWidth',1.1)
hold on
plot(omega,C_tn,"r")
hold on
plot(omega,C_tn,"k--")
hold on
plot(omega_total,(1)*a*omega_total,"b")
xlabel('$\Omega_e$','Interpreter','Latex','FontSize',14)
ylabel('$\hat{V}_{tn}$','Interpreter','Latex','FontSize',14)
legend(" stick-slip"," $\hat{V}_{tn}$","$|\hat{Y}_1(\Omega_e) |$" ,"$\alpha\Omega_e$",'Interpreter','Latex','Location','southeast','FontSize',11)

title('$(a)$','Interpreter','Latex','FontSize',11, 'Position', [2.5,6.25, 0])

%% Figure 15(b)

% Parameters
r_v = 0.01:0.01:5;
tau_1 = 0:0.01:pi;

omega_n=1;
k=1;
m=1;
beta=0.1;
c=2*beta*omega_n*m;
V = 1./(1-r_v.^2);

c_c=2*m*omega_n;

p=omega_n*sqrt(1-(c/c_c)^2);
q=sqrt((1./V).^2+(2*c.*r_v./c_c).^2);
G=(sinh(pi*c./(c_c*r_v))-(c/c_c)/sqrt(1-(c/c_c)^2) *sin(pi*sqrt(1-(c/c_c)^2)./r_v))./(cosh(pi*c./(c_c.*r_v))+ cos(pi*sqrt(1-(c/c_c).^2)./r_v));
H=(sin(pi*sqrt(1-(c/c_c)^2)./r_v)./(cosh(pi*c./(c_c*r_v))+cos(pi*sqrt(1-(c/c_c)^2)./r_v))).*(1./(r_v*sqrt(1-(c/c_c)^2)));

for it = 1:length(tau_1)
    tau = tau_1(it);

    lim16(it,:)= (exp(-c.*(tau./(r_v.*omega_n))/(2*m))./sin(tau)).*((p./(r_v.*omega_n) +c^2./(4*m^2*p.*(r_v.*omega_n))).*(1+G).*sin(p.*(tau./(r_v*omega_n))) +H.*(c.*sin(p.*(tau./(r_v*omega_n)))./(2*p*m)-cos(p.*(tau./(r_v*omega_n))))) +H.*cot(tau)-G;

end

lim_16=max(lim16);


lim_17=-G+2*H*c./(c_c*r_v)+(1+G)./(r_v.^2);
S1=lim_16./lim_17;
I=2*H*c./(c_c.*r_v) +(1+G)./r_v.^2;
lim_18=1./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %xf/a<
lim_19=S1.*(1-G)./(q.*sqrt(H.^2+(S1.*I+(1-S1).*G).^2)); %x0/a>

figure
plot(r_v,lim_18)


% considering tangential and normal loading in 2 dof

m=1;
omega_2=2;
beta_2=0.15;
alpha=1;
g=9.81;

lim_2dof_tn_array = [];

for mu_s=0.1:0.05:1
    i=0;
    j=0;
    for r_v= 0.01:0.01:5
       j=1+j;
       i=0;
        for tau = 0:0.01:55*pi
            i=1+i;
            theta=atan2(2*beta_2*omega_2*r_v,r_v.^2-omega_2^2);
            forcing_term(i,j)= -(mu_s*alpha*r_v.^2./sqrt(4*beta_2^2*omega_2^2.*r_v.^2+(omega_2^2-r_v.^2).^2)).*(omega_2^2.*sin(r_v.*tau+theta)+2*beta_2*omega_2*r_v.*cos(r_v*tau+theta)) +alpha*r_v.^2.*sin(r_v*tau);
        end
    end

    ampl_tn=max(abs(forcing_term)); 
    r_v = 0.01:0.01:5;
    lim_2dof_tn=2*mu_s*g./ampl_tn;
    % append the calculated values to the arrays
    lim_2dof_tn_array = [lim_2dof_tn_array; lim_2dof_tn];
   
end

result = zeros(size(lim_2dof_tn_array, 1), 1); % Initialize the result vector for fig 14

for i = 1:size( lim_2dof_tn_array, 1)
    [~, idx] = min(abs(lim_18 -  lim_2dof_tn_array(i,:))); % Find the index of the closest element
    idx_all(i,:) = idx; % Save the indices to the matrix
    result(i) = lim_18(idx); % Save the closest element to the result vector
end
omegas = r_v(idx_all);
mu_s=0.1:0.05:1;
beta_1 = 0.1;
beta_2 = 0.15;
omega_1 = 1;
omega_2 = 2;
a = 1;

[omega_ss, mu_s_ss] = meshgrid(0:0.01:5, 0.1:0.05:1);

%figure 15b
C_tn_ss = a * omega_ss.^3 .* sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s_ss+1).^2 ...
    + (omega_2^2-omega_ss.^2).^2 + mu_s_ss.*omega_2^2.*(mu_s_ss*omega_2^2-2*omega_ss.^2+2*omega_2^2)) ...
    ./ sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2) ...
    .* (4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));


% Initialize color_map to all red
color_map = ones(size(C_tn_ss, 1), size(C_tn_ss, 2), 3);
color_map(:,:,2) = color_map(:,:,2) * 0; % Green channel to 0
color_map(:,:,3) = color_map(:,:,3) * 0; % Blue channel to 0

% Loop through rows of mu_s_ss
for i = 1:size(mu_s_ss, 1)
    % Find corresponding elements in omega_ss and color_map
    omega_m = omegas(i);
    omega_indices = omega_ss(i, :) <= omega_m;
    
    % Set color values to magenta for corresponding elements
    color_map(i, omega_indices, :) = repmat([1 0 1], sum(omega_indices), 1);
end

[omega_total,mu_s_total] = meshgrid(0:0.05:5,0.1:0.05:1);


C_fast=(1).*a.*omega_total;



% Create 3D plot
surf(omega_ss, mu_s_ss, C_tn_ss, 'CData', color_map,'EdgeAlpha',0.5);
hold on
s3=surf(omega_total,mu_s_total,C_fast);
s3.FaceColor='b';
xlabel('$\Omega_e$','Interpreter','Latex','FontSize',14)
ylabel('$\mu_s$','Interpreter','Latex','FontSize',14)
zlabel('$\hat{V}_{tn}$','Interpreter','Latex','FontSize',14)
% title('$(b)$','Interpreter','Latex','FontSize',11, 'Position', [0.5,2.5,65])

%% This script generates the results shown in Figures 16 of the paper.


 beta_1 =0.1;
 beta_2=0.15;
 omega_1=1;
 omega_2=2;
 a=1;
 mu_s=0.1;
 
% tangential only
beta =0.1;
omega_ss1=0:0.001:0.78;
C_ss=a*omega_ss1.^3./sqrt(4*beta^2.*omega_ss1.^2+(1-omega_ss1.^2).^2);
omega1=0.78:0.001:5;
C_t=a*omega1.^3./sqrt(4*beta^2.*omega1.^2+(1-omega1.^2).^2);

%tangential and normal loading
omega_ss2=0:0.001:1.15;

C_tn_ss=a*omega_ss2.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss2.^2.*(mu_s+1).^2 +(omega_2^2-omega_ss2.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega_ss2.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss2.^2 +(omega_1^2-omega_ss2.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss2.^2 +(omega_2^2-omega_ss2.^2).^2));

omega2=1.15:0.001:5;

C_tn=a*omega2.^3.*sqrt(4*beta_2^2*omega_2^2.*omega2.^2.*(mu_s+1).^2 +(omega_2^2-omega2.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega2.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega2.^2 +(omega_1^2-omega2.^2).^2).*(4*beta_2^2*omega_2^2.*omega2.^2 +(omega_2^2-omega2.^2).^2));

%tangential(-) and normal loading
omega_ss=0:0.001:1.4;
C_ttn_ss=a*omega_ss.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s-1).^2 +(omega_2^2-omega_ss.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega_ss.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));

omega=1.4:0.001:5;
C_ttn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s-1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

figure
plot(omega_ss1,C_ss,"m",'LineWidth',1.5)
hold on
plot(omega1,C_t)
hold on
plot(omega_ss2,C_tn_ss,"m",'LineWidth',1.5)
hold on
plot(omega2,C_tn)
hold on
plot(omega_ss,C_ttn_ss,"m",'LineWidth',1.5)
hold on
plot(omega,C_ttn)
xlabel('$\Omega_e$','Interpreter','Latex','FontSize',14)
ylabel('$\hat{V}$','Interpreter','Latex','FontSize',14)
legend(" stick-slip"," $\hat{V}_{t}$"," stick-slip"," $\hat{V}_{tn}$"," stick-slip"," $\hat{V}_{tn}(-)$", 'Interpreter','Latex','Location','southeast','FontSize',11)
% 
title('$(a)$','Interpreter','Latex','FontSize',11, 'Position', [2.5,62.5, 0])
%% Figure16(b)

 beta =0.1;     % damping coeff x1 direction
 beta_1=beta;  % damping coeff x1 direction
 beta_2=0.15;  % damping coeff x2 direction
 mu_s=0.1;     % friction coefficient
 omega_e=2;    % excitation freq
 omega_1=1;    % nat freq x1 direction
 omega_2=2;  % nat freq x2 direction
 omega_n=1;
 omega=omega_e/omega_n;
 a=1;


% average friction tangential loading only
C=a*omega.^3./sqrt(4*beta^2.*omega.^2+(1-omega.^2).^2);

syms v_r
mu5 = piecewise(abs(v_r) <= C,mu_s*(1-(2/pi)*acos(v_r/C)),abs(v_r)>C,mu_s*sign(v_r));
x5=-85:0.01:85;
y5=subs(mu5,v_r,x5);    

% average friction tangential and normal loading
C_tn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s+1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

syms v_r
mu7 = piecewise(abs(v_r) <= C_tn,mu_s*(1-(2/pi)*acos(v_r/C_tn)),abs(v_r)>C_tn,mu_s*sign(v_r));
x7=-85:0.01:85;
y7=subs(mu7,v_r,x7); 

% average friction tangential (-) and normal loading
C_ttn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s-1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

syms v_r
mu8 = piecewise(abs(v_r) <= C_ttn,mu_s*(1-(2/pi)*acos(v_r/C_ttn)),abs(v_r)>C_ttn,mu_s*sign(v_r));
x8=-85:0.01:85;
y8=subs(mu8,v_r,x8); 
    
    
figure
plot(x5,y5)
hold on
plot(x7,y7)
hold on
plot(x8,y8)
xlabel('$V_b$','Interpreter','Latex','FontSize',14)
ylabel('$\bar{\mu}(V_b)$','Interpreter','Latex','FontSize',14)
legend(" $\bar{\mu}_t$","$\bar{\mu}_{tn}$"," $\bar{\mu}_{tn}(-)$",'Interpreter','Latex','FontSize',11,'Location','southeast')
title('$(b)$','Interpreter','Latex','FontSize',11, 'Position', [0,0.54167])

xlim([-3.5 3.5])
ylim([-0.11 0.11])

%% This script generates the results shown in Figures 17 of the paper.
 beta =0.1;
 beta_1=beta;
 beta_2=0.15;
 mu_s=0.4;
 omega_1=1;
 omega_2=0.5;
 omega_n=1;
 a=10;
 
%tangential and normal loading
omega_ss2=0:0.001:1.15;

C_tn_ss=a*omega_ss2.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss2.^2.*(mu_s+1).^2 +(omega_2^2-omega_ss2.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega_ss2.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss2.^2 +(omega_1^2-omega_ss2.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss2.^2 +(omega_2^2-omega_ss2.^2).^2));

omega2=1.15:0.001:5;

C_tn=a*omega2.^3.*sqrt(4*beta_2^2*omega_2^2.*omega2.^2.*(mu_s+1).^2 +(omega_2^2-omega2.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega2.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega2.^2 +(omega_1^2-omega2.^2).^2).*(4*beta_2^2*omega_2^2.*omega2.^2 +(omega_2^2-omega2.^2).^2));

%tangential(-) and normal loading
omega_ss=0:0.001:1.4;
C_ttn_ss=a*omega_ss.^3.*sqrt(4*beta_2^2*omega_2^2.*omega_ss.^2.*(mu_s-1).^2 +(omega_2^2-omega_ss.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega_ss.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega_ss.^2 +(omega_1^2-omega_ss.^2).^2).*(4*beta_2^2*omega_2^2.*omega_ss.^2 +(omega_2^2-omega_ss.^2).^2));

omega=1.4:0.001:5;
C_ttn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s-1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

figure

plot(omega_ss2,C_tn_ss,"m",'LineWidth',1.5)
hold on
plot(omega2,C_tn)
hold on
plot(omega_ss,C_ttn_ss,"m",'LineWidth',1.5)
hold on
plot(omega,C_ttn) 

omega_ss=0:0.001:0.99;

C_ttn_ss=sqrt(a^2*omega_ss.^6*(mu_s-1).^2)./sqrt(4*beta^2.*omega_1^2*omega_ss.^2+(omega_1^2-omega_ss.^2).^2);

omega=0.99:0.001:5;
C_ttn=sqrt(a^2*omega.^6*(mu_s-1).^2)./sqrt(4*beta^2.*omega_1^2*omega.^2+(omega_1^2-omega.^2).^2);

omega_total=0:0.001:5;


hold on
plot(omega_ss,C_ttn_ss,"m",'LineWidth',1.5)
hold on
plot(omega,C_ttn)

omega_ss=0:0.001:0.7;
C_ttn_ss=sqrt(a^2*omega_ss.^6*(mu_s+1).^2)./sqrt(4*beta^2.*omega_1^2*omega_ss.^2+(omega_1^2-omega_ss.^2).^2);

omega=0.7:0.001:5;
C_ttn=sqrt(a^2*omega.^6*(mu_s+1).^2)./sqrt(4*beta^2.*omega_1^2*omega.^2+(omega_1^2-omega.^2).^2);

omega_total=0:0.001:5;


hold on
plot(omega_ss,C_ttn_ss,"m",'LineWidth',1.5)
hold on
plot(omega,C_ttn)
xlabel('$\Omega_e$','Interpreter','Latex','FontSize',14)
ylabel('$\hat{V}_{tn}$','Interpreter','Latex','FontSize',14)

%% Figure17(b)

% Parameters
 beta =0.1;     % damping coeff x1 direction
 beta_1=beta;  % damping coeff x1 direction
 beta_2=0.15;  % damping coeff x2 direction
 mu_s=0.4;     % friction coefficient
 omega_e=5;    % excitation freq
 omega_1=1;    % nat freq x1 direction
 omega_2=0.5;  % nat freq x2 direction
 omega_n=1;
 omega=omega_e/omega_n;
 a=10;


% first the results as in figure 14b(b)
% average friction tangential and normal loading
C_tn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s+1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2-2*omega.^2+2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

syms v_r
mu7 = piecewise(abs(v_r) <= C_tn,mu_s*(1-(2/pi)*acos(v_r/C_tn)),abs(v_r)>C_tn,mu_s*sign(v_r));
x7=-85:0.01:85;
y7=subs(mu7,v_r,x7); 

% average friction tangential (-) and normal loading
C_ttn=a*omega.^3.*sqrt(4*beta_2^2*omega_2^2.*omega.^2.*(mu_s-1).^2 +(omega_2^2-omega.^2).^2+mu_s.*omega_2^2.*(mu_s*omega_2^2+2*omega.^2-2*omega_2^2))./sqrt((4*beta_1^2*omega_1^2.*omega.^2 +(omega_1^2-omega.^2).^2).*(4*beta_2^2*omega_2^2.*omega.^2 +(omega_2^2-omega.^2).^2));

syms v_r
mu8 = piecewise(abs(v_r) <= C_ttn,mu_s*(1-(2/pi)*acos(v_r/C_ttn)),abs(v_r)>C_ttn,mu_s*sign(v_r));
x8=-85:0.01:85;
y8=subs(mu8,v_r,x8); 
figure
plot(x7,y7)
hold on
plot(x8,y8)

%  the results assuming inertial contact force
C_intertia_plus=sqrt(a^2*omega.^6*(mu_s-1).^2)./sqrt(4*beta^2.*omega_1^2*omega.^2+(omega_1^2-omega.^2).^2);
syms v_r
mu9 = piecewise(abs(v_r) <= C_intertia_plus,mu_s*(1-(2/pi)*acos(v_r/C_intertia_plus)),abs(v_r)>C_intertia_plus,mu_s*sign(v_r));
x9=-85:0.01:85;
y9=subs(mu9,v_r,x9); 

C_intertia_minus=sqrt(a^2*omega.^6*(mu_s+1).^2)./sqrt(4*beta^2.*omega_1^2*omega.^2+(omega_1^2-omega.^2).^2);
syms v_r
mu10 = piecewise(abs(v_r) <= C_intertia_minus,mu_s*(1-(2/pi)*acos(v_r/C_intertia_minus)),abs(v_r)>C_intertia_minus,mu_s*sign(v_r));
x10=-85:0.01:85;
y10=subs(mu10,v_r,x10); 

plot(x9,y9)
hold on
plot(x10,y10)

% 
xlabel('$V_b$','Interpreter','Latex','FontSize',14)
ylabel('$\bar{\mu}(V_b)$','Interpreter','Latex','FontSize',14)
legend(" $\bar{\mu}_t$","$\bar{\mu}_{tn}$"," $\bar{\mu}_{tn}(-)$"," $\bar{\mu}_{tn}(-)$",'Interpreter','Latex','FontSize',11,'Location','southeast')
title('$(b)$','Interpreter','Latex','FontSize',11, 'Position', [0,0.54167])

xlim([-85 85])
ylim([-0.5 0.5])
%% This script generates the results shown in the appendix of Chapter 3

% Numerical solution, SDOF mass-spring-damper
% -------------------------------------------------
% Solves a system using ode23 with Coulomb law
% -------------------------------------------------

close all; clear; clc;

global beta_1 beta_2 omega_1 omega_2 omega a mu_s

% Time vector
tau = 0:0.001:100;

% Parameters
beta_1 = 0.1;
beta_2 = 0.15;
omega_1 = 1;   % natural frequency
omega_2 = 0.5;
omega   = 2;   % excitation frequency
a       = 10;
mu_s    = 0.4;

% Initial conditions
y0 = [0 0 0 0 0 0 0 0];

% Solve ODE
[tau,y] = ode23(@rhs_abst,tau,y0);

% -------------------------------------------------
% Plot results
% -------------------------------------------------

figure
plot(tau,y(:,1)) 
xlabel('$\tau$','Interpreter','Latex','FontSize',11)
ylabel('$B_{11}$','Interpreter','Latex','FontSize',11)
title('$(a)$','Interpreter','Latex','FontSize',11)

figure
plot(tau,y(:,3))
xlabel('$\tau$','Interpreter','Latex','FontSize',11)
ylabel('$B_{12}$','Interpreter','Latex','FontSize',11)
title('$(b)$','Interpreter','Latex','FontSize',11)

figure
plot(tau,y(:,5))
xlabel('$\tau$','Interpreter','Latex','FontSize',11)
ylabel('$B_{21}$','Interpreter','Latex','FontSize',11)
title('$(c)$','Interpreter','Latex','FontSize',11)

figure
plot(tau,y(:,7))
xlabel('$\tau$','Interpreter','Latex','FontSize',11)
ylabel('$B_{22}$','Interpreter','Latex','FontSize',11)
title('$(d)$','Interpreter','Latex','FontSize',11)


%% -------------------------------------------------
% RHS function (Coulomb law, numerical solution)
% -------------------------------------------------
function ydot = rhs_abst(tau,y)

    global beta_1 beta_2 omega_1 omega_2 omega a mu_s

    % State derivatives
    ydot_1 = y(2); % velocity
    ydot_2 = -(2*beta_1*omega_1*y(2) - 2*omega*y(4) + 2*beta_2*mu_s*omega_2*y(6) ...
              + (omega_1^2-omega^2)*y(1) - 2*beta_2*omega*mu_s*omega_2*y(7) ...
              - a*omega^3 - 2*beta_1*omega_1*omega*y(3) + mu_s*omega_2^2*y(5));

    ydot_3 = y(4); % velocity
    ydot_4 = -(2*omega*y(2) + 2*beta_1*omega_1*y(4) + 2*beta_2*mu_s*omega_2*y(8) ...
              + (omega_1^2-omega^2)*y(3) + 2*beta_2*omega*mu_s*omega_2*y(5) ...
              + mu_s*omega_2^2*y(7) + 2*beta_1*omega_1*omega*y(1));

    ydot_5 = y(6); % velocity
    ydot_6 = -(2*beta_2*omega_2*y(6) - 2*omega*y(8) + (omega_2^2-omega^2)*y(5) ...
              - a*omega^3 - 2*beta_2*omega_2*omega*y(7));

    ydot_7 = y(8); % velocity
    ydot_8 = -(2*omega*y(6) + 2*beta_2*omega_2*y(8) + (omega_2^2-omega^2)*y(7) ...
              + 2*beta_2*omega_2*omega*y(5));

    % Return derivatives
    ydot = [ydot_1 ; ydot_2 ; ydot_3 ; ydot_4 ; ...
            ydot_5 ; ydot_6 ; ydot_7 ; ydot_8];
end
