%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Author:   Jelle Snieder
% Date:     17-11-2024
% 
% Script to plot the normalized fluorescence spectra
% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clearvars
close all
clc

% load spectra
load('dataset_fluorescein_eosinY.mat');

% load data LED spectrum
load('dataset_LED_spectrum.mat')

% plotsettings
run('plotsettings.m')   % general settings for plotting
royal_blue = [0, 35, 102]./255;

% 475 nm Edmund Optics longpass filter spectrum (high performance)
filename                = '475nm_longpass_curv_84737.xlsx';
dataRaw                 = readmatrix(filename,'sheet',2);
wavelength_475nm_EO     = dataRaw(:,1);
signal_475nm_EO         = dataRaw(:,2)./100;
indices_nan             = isnan(wavelength_475nm_EO);
wavelength_475nm_EO(indices_nan) = [];
signal_475nm_EO(indices_nan)     = [];
% signal_475nm_EO         = interp1(wavelength_475nm_EO,signal_475nm_EO,wavelengths);

% parameters
cwl_filter_A    = 520; % [nm]
cwl_filter_B    = 540; % [nm]
band_filter_A   = 10;  % [nm]
band_filter_B   = 10;  % [nm]
LED             = 448; % [nm]

%% plot normalized spectra
figure(1)
hold on
grid on

% fluorescence
h1 = plot(wavelengths, fluoresceinExcitation, '--', 'linewidth', linewidth, 'color', color1);
h2 = plot(wavelengths, fluoresceinEmission, 'linewidth', linewidth, 'color', color1);
h3 = plot(wavelengths, eosinYExcitation, '--', 'linewidth', linewidth, 'color', color3);
h4 = plot(wavelengths, eosinYEmission, 'linewidth', linewidth, 'color', color3);

hd1 = plot(-1,  -1, '--',    'color', 'black', 'linewidth', linewidth);
hd2 = plot(-1,  -1, '-',   'color', 'black', 'linewidth', linewidth);

xlim([400 700])
ylim([0 1])
xlabel('Wavelength (nm)','fontsize',fontsize)
ylabel('Normalized magnitude (-)','fontsize',fontsize)

lgdStr1 = 'Color';
lgd1 = legendflex([h2 h4],...
    {'Fluorescein','Eosin Y'},...
    'padding', padding,...
    'xscale', xscale,...
    'title', lgdStr1,...
    'ref', gca, 'anchor', {'ne', 'ne'}, 'buffer', [-offset -offset],... 
    'fontsize', fontsizeLegend); 
set(findall(lgd1, 'string', lgdStr1), 'fontweight', 'bold');

lgdStr2 = 'Lines';
lgd2 = legendflex([hd1, hd2],...
    {'Excitation','Emission'},...
    'padding', padding,...
    'xscale', xscale,...
    'title', lgdStr2,...
    'ref', lgd1, 'anchor', {'se', 'ne'}, 'buffer', [0 -offset],...    
    'fontsize', fontsizeLegend); 
set(findall(lgd2, 'string', lgdStr2), 'fontweight', 'bold'); 

ax = gca;
ax.FontSize = fontsize;

fig = gcf;
width_plot = fig.Position(3);
fig.Position(3:4) = [width_plot, 0.5*width_plot]; % default is height/width ratio of 0.75

% exportgraphics(gcf,'normalized_spectra.pdf','BackgroundColor','none')

%% separate plots

% Plot spectra 
figure(2)
subplot(2,1,1)
hold on
h1a = plot(wavelengths, spectrum_LED, 'linewidth', linewidth, 'color', royal_blue);
h1b = plot(wavelengths, fluoresceinExcitation, '--', 'linewidth', linewidth, 'color', color1);
h1c = plot(wavelengths, eosinYExcitation, '--', 'linewidth', linewidth, 'color', color3);

hf1a = fill([wavelengths';  flipud(wavelengths')],...
           [spectrum_LED; zeros(length(wavelengths),1)],...
           royal_blue,...
           'linestyle','none');
set(hf1a,'facealpha', 0.2)

grid on;
xlabel('Wavelength (nm)','fontsize',fontsize); 
ylabel('Normalized magnitude (-)','fontsize',fontsize); 
title('Excitation')

lgd1 = legendflex([h1a h1b h1c],...
    {'Light source', 'Fluorescein','Eosin Y'},...
    'padding', padding,...
    'xscale', xscale,...
    'ref', gca, 'anchor', {'ne', 'ne'}, 'buffer', [-offset -2*offset],...    
    'fontsize', fontsizeLegend); 

ax = gca;
ax.FontSize = fontsize;

subplot(2,1,2)
hold on
grid on
h2a = plot(wavelengths, fluoresceinEmission, '-', 'linewidth', linewidth, 'color', color1);
h2b = plot(wavelengths, eosinYEmission, '-', 'linewidth', linewidth, 'color', color3);

% filters
hf1 = fill([cwl_filter_A-band_filter_A/2 cwl_filter_A-band_filter_A/2 cwl_filter_A+band_filter_A/2 cwl_filter_A+band_filter_A/2],...
           [0 1 1 0],...
          'black',...
          'linestyle','none');
set(hf1,'facealpha', 0.1)
hf2 = fill([cwl_filter_B-band_filter_B/2 cwl_filter_B-band_filter_B/2 cwl_filter_B+band_filter_B/2 cwl_filter_B+band_filter_B/2],...
           [0 1 1 0],...
          'black',...
          'linestyle','none');
set(hf2,'facealpha', 0.1)
hf3 = plot([cwl_filter_A cwl_filter_A],[0 1],':','color','black','linewidth',linewidth);
hf4 = plot([cwl_filter_B cwl_filter_B],[0 1],':','color','black','linewidth',linewidth);

% arrow filter A
delta_x = 25;
delta_y = 0;
x_start = cwl_filter_A-2-delta_x;
y_start = 0.6;
anArrow = annotation('textarrow') ;
anArrow.Parent = gca;
anArrow.Position = [x_start, y_start, delta_x, delta_y] ;
anArrow.String = {'Filter ','520 nm '};
anArrow.FontSize = fontsize;
anArrow.HeadStyle = 'vback3';
anArrow.HeadLength = 5;

% arrow filter B
delta_x = -25;
delta_y = 0;
x_start = cwl_filter_B+2-delta_x;
y_start = 0.6;
anArrow = annotation('textarrow') ;
anArrow.Parent = gca;
anArrow.Position = [x_start, y_start, delta_x, delta_y] ;
anArrow.String = {' Filter',' 540 nm'};
anArrow.FontSize = fontsize;
anArrow.HeadStyle = 'vback3';
anArrow.HeadLength = 5;

xlabel('Wavelength (nm)','fontsize',fontsize); 
ylabel('Normalized magnitude (-)','fontsize',fontsize); 
title('Emission')

lgd2 = legendflex([h2a h2b],...
    {'Fluorescein','Eosin Y'},...
    'padding', padding,...
    'xscale', xscale,...
    'ref', gca, 'anchor', {'ne', 'ne'}, 'buffer', [-offset -2*offset],...    
    'fontsize', fontsizeLegend); 

ax = gca;
ax.FontSize = fontsize;

fig = gcf;
% exportgraphics(fig,'normalized_spectra_subplots_ch7.pdf','BackgroundColor','none')


%% separate plots experiment

% Plot spectra 
figure(3)
subplot(2,1,1)
hold on
h1a = plot(wavelengths, spectrum_LED, 'linewidth', linewidth, 'color', royal_blue);
h1b = plot(wavelengths, fluoresceinExcitation, '--', 'linewidth', linewidth, 'color', color1);
h1c = plot(wavelengths, eosinYExcitation, '--', 'linewidth', linewidth, 'color', color3);

hf1a = fill([wavelengths';  flipud(wavelengths')],...
           [spectrum_LED; zeros(length(wavelengths),1)],...
           royal_blue,...
           'linestyle','none');
set(hf1a,'facealpha', 0.2)

grid on;
xlabel('Wavelength (nm)','fontsize',fontsize); 
ylabel('Normalized magnitude (-)','fontsize',fontsize); 
xlim([400 700])
ylim([0 1])

lgd1 = legendflex([h1a h1b h1c],...
    {'Light source', 'Fluorescein','Eosin Y'},...
    'padding', padding,...
    'xscale', xscale,...
    'ref', gca, 'anchor', {'ne', 'ne'}, 'buffer', [-offset -2*offset],...    
    'fontsize', fontsizeLegend); 

title('Excitation')
ax = gca;
ax.FontSize = fontsize;

subplot(2,1,2)
hold on
grid on
h2a = plot(wavelength_475nm_EO, signal_475nm_EO, '-', 'linewidth', linewidth, 'color', grey);
h2b = plot(wavelengths, fluoresceinEmission, '-', 'linewidth', linewidth, 'color', color1);
h2c = plot(wavelengths, eosinYEmission, '-', 'linewidth', linewidth, 'color', color3);

xlabel('Wavelength (nm)','fontsize',fontsize); 
ylabel('Normalized magnitude (-)','fontsize',fontsize); 
xlim([400 700])
ylim([0 1])

lgd2 = legendflex([h2a h2b h2c],...
    {'Long pass filter','Fluorescein','Eosin Y'},...
    'padding', padding,...
    'xscale', xscale,...
    'ref', gca, 'anchor', {'ne', 'ne'}, 'buffer', [-offset -5*offset],...    
    'fontsize', fontsizeLegend); 

title('Emission')
ax = gca;
ax.FontSize = fontsize;

fig = gcf;
% exportgraphics(fig,'normalized_spectra_subplots_ch5.pdf','BackgroundColor','none')

%% plot Eosin Y spectrum for introduction

figure(4)
hold on
grid on

% fluorescence
h1 = plot(wavelengths, eosinYExcitation, '--', 'linewidth', linewidth, 'color', color3);
h2 = plot(wavelengths, eosinYEmission, 'linewidth', linewidth, 'color', color3);

xlim([400 700])
ylim([0 1])
xlabel('Wavelength (nm)','fontsize',fontsize)
ylabel('Normalized magnitude (-)','fontsize',fontsize)

lgd1 = legendflex([h1, h2],...
    {'Molar absorptivity','Emission'},...
    'padding', padding,...
    'xscale', xscale,...
    'ref', gca, 'anchor', {'ne', 'ne'}, 'buffer', [-offset -offset],...    
    'fontsize', fontsizeLegend); 

ax = gca;
ax.FontSize = fontsize;

fig = gcf;
width_plot = fig.Position(3);
fig.Position(3:4) = [width_plot, 0.5*width_plot]; % default is height/width ratio of 0.75

% exportgraphics(gcf,'normalized_spectra_EosinY.pdf','BackgroundColor','none')


