%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Author:   Jelle Snieder
% Date:     10-12-2024
%  
% This m-file imports the absorption spectra at the specified
% concentrations for both Fluorescein and Eosin Y. Next, it calculates the
% molar absorption coefficient, based on Beer-Lambert absorption law:
% 
% absorption = path length * concentration * extinction coefficient
% 
% NOTE: the absorbance values from the dataset are determined with the
% decimal logarithm. To convert the values to the absorbance with the
% natural logarithm, it is multiplied by a factor ln(10) 
% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% initialize
clearvars
close all
clc

% data location
folder          = 'Absorption_spectra';

% parameters
pathLength = 10e-3; % m

concentrationValues =  [0
                        0.0005 
                        0.001 
                        0.003
                        0.005
                        0.01].*1e-3; % M

concentrationLabels = [{'00005'}
                        {'0001'}
                        {'0003'}
                        {'0005'} 
                        {'001'}];

nMeasurements = 3;
nConcentrationLabels = length(concentrationLabels);
nConcentrationValues = length(concentrationValues);
nWavelengths = 401; % 300-700 nm

peakWavelengthFluorescein = 491; % nm
peakWavelengthEosinY = 515; % nm
nSTD = 3;

% import data
absorptionFluorescein = zeros(nWavelengths,nConcentrationLabels*nMeasurements);
absorptionEosinY = zeros(nWavelengths,nConcentrationLabels*nMeasurements);
transmissionFluorescein = zeros(nWavelengths,nConcentrationLabels*nMeasurements);
transmissionEosinY = zeros(nWavelengths,nConcentrationLabels*nMeasurements);

for iConcentrations = 1:nConcentrationLabels
    for iMeasurements = 1:nMeasurements

        concentration   = cell2mat(concentrationLabels(iConcentrations));

        % fluorescein
        file            = append('fluorescein_',concentration,'_mM_m', int2str(iMeasurements),'.csv');
        filename        = fullfile(folder,file);
        dataRaw         = readmatrix(filename);
        absorptionFluorescein(:,(iConcentrations-1)*nMeasurements+iMeasurements) = dataRaw(:,2);
        transmissionFluorescein(:,(iConcentrations-1)*nMeasurements+iMeasurements) = dataRaw(:,3);

        % eosin Y
        file            = append('eosinY_',concentration,'_mM_m', int2str(iMeasurements),'.csv');
        filename        = fullfile(folder,file);
        dataRaw         = readmatrix(filename);
        absorptionEosinY(:,(iConcentrations-1)*nMeasurements+iMeasurements) = dataRaw(:,2);
        transmissionEosinY(:,(iConcentrations-1)*nMeasurements+iMeasurements) = dataRaw(:,3);

    end
end
wavelengthsAbsorption = dataRaw(:,1);

% NOTE: the absorbance values from the dataset are determined with the
% decimal logarithm. To convert the values to the absorbance with the
% natural logarithm, it is multiplied by a factor ln(10)

absorptionFluorescein = absorptionFluorescein.*log(10);
absorptionEosinY      = absorptionEosinY.*log(10);

% determine mean and standard deviation
absorptionFluoresceinMean = zeros(nWavelengths,nConcentrationLabels);
absorptionFluoresceinStd = zeros(nWavelengths,nConcentrationLabels);
absorptionEosinYMean = zeros(nWavelengths,nConcentrationLabels);
absorptionEosinYStd = zeros(nWavelengths,nConcentrationLabels);

transmissionFluoresceinMean = zeros(nWavelengths,nConcentrationLabels);
transmissionFluoresceinStd = zeros(nWavelengths,nConcentrationLabels);
transmissionEosinYMean = zeros(nWavelengths,nConcentrationLabels);
transmissionEosinYStd = zeros(nWavelengths,nConcentrationLabels);

for iConcentrations = 1:nConcentrationLabels

    startColumn = (iConcentrations-1)*nMeasurements+1;
    endColumn = (iConcentrations-1)*nMeasurements+nMeasurements;

    absorptionFluoresceinMean(:,iConcentrations) = mean(absorptionFluorescein(:,startColumn:endColumn),2);
    absorptionFluoresceinStd(:,iConcentrations) = std(absorptionFluorescein(:,startColumn:endColumn),0,2);
    absorptionEosinYMean(:,iConcentrations) = mean(absorptionEosinY(:,startColumn:endColumn),2);
    absorptionEosinYStd(:,iConcentrations) = std(absorptionEosinY(:,startColumn:endColumn),0,2);

    transmissionFluoresceinMean(:,iConcentrations) = mean(transmissionFluorescein(:,startColumn:endColumn),2);
    transmissionFluoresceinStd(:,iConcentrations) = std(transmissionFluorescein(:,startColumn:endColumn),0,2);
    transmissionEosinYMean(:,iConcentrations) = mean(transmissionEosinY(:,startColumn:endColumn),2);
    transmissionEosinYStd(:,iConcentrations) = std(transmissionEosinY(:,startColumn:endColumn),0,2);

end

% take the absorption mean and std at peak wavelengths
indexPeakWavelengthFluorescein = find(wavelengthsAbsorption==peakWavelengthFluorescein);
indexPeakWavelengthEosinY = find(wavelengthsAbsorption==peakWavelengthEosinY);

absorptionFluoresceinMean_peakWavelength    = absorptionFluoresceinMean(indexPeakWavelengthFluorescein,:);
absorptionFluoresceinStd_peakWavelength     = absorptionFluoresceinStd(indexPeakWavelengthFluorescein,:);
absorptionEosinYMean_peakWavelength         = absorptionEosinYMean(indexPeakWavelengthEosinY,:);
absorptionEosinYStd_peakWavelength          = absorptionEosinYStd(indexPeakWavelengthEosinY,:);

transmissionFluoresceinMean_peakWavelength  = transmissionFluoresceinMean(indexPeakWavelengthFluorescein,:);
transmissionFluoresceinStd_peakWavelength   = transmissionFluoresceinStd(indexPeakWavelengthFluorescein,:);
transmissionEosinYMean_peakWavelength       = transmissionEosinYMean(indexPeakWavelengthEosinY,:);
transmissionEosinYStd_peakWavelength        = transmissionEosinYStd(indexPeakWavelengthEosinY,:);

% zero padding
absorptionFluoresceinMean_peakWavelength    = [0 absorptionFluoresceinMean_peakWavelength];
absorptionFluoresceinStd_peakWavelength     = [0 absorptionFluoresceinStd_peakWavelength];
absorptionEosinYMean_peakWavelength         = [0 absorptionEosinYMean_peakWavelength];
absorptionEosinYStd_peakWavelength          = [0 absorptionEosinYStd_peakWavelength];

transmissionFluoresceinMean_peakWavelength  = [100 transmissionFluoresceinMean_peakWavelength];
transmissionFluoresceinStd_peakWavelength   = [0 transmissionFluoresceinStd_peakWavelength];
transmissionEosinYMean_peakWavelength       = [100 transmissionEosinYMean_peakWavelength];
transmissionEosinYStd_peakWavelength        = [0 transmissionEosinYStd_peakWavelength];

% linear fit
% fluorescein
nFluorescein                        = nConcentrationValues; % determine extinction coefficient over first results
X1                                  = [ones(length(concentrationValues(1:nFluorescein)),1) concentrationValues(1:nFluorescein)];
fit1                                = X1\absorptionFluoresceinMean_peakWavelength(1:nFluorescein)';
absorptionCoefficientFluorescein    = fit1(2)/pathLength;
absorptionFluoresceinLinearFit      = concentrationValues*pathLength*absorptionCoefficientFluorescein + fit1(1);

% eosin y
nEosinY                             = nConcentrationValues; % determine extinction coefficient over first results
X2                                  = [ones(length(concentrationValues(1:nEosinY)),1) concentrationValues(1:nEosinY)];
fit2                                = X2\absorptionEosinYMean_peakWavelength(1:nEosinY)';
absorptionCoefficientEosinY         = fit2(2)/pathLength;
absorptionEosinYLinearFit           = concentrationValues*pathLength*absorptionCoefficientEosinY + fit2(1);

%% plot
run('plotsettings.m')

figure
hold on
plot(1e6*concentrationValues,absorptionFluoresceinMean_peakWavelength,'-o','linewidth', linewidth,'color',color1)
errorbar(1e6*concentrationValues,absorptionFluoresceinMean_peakWavelength,3.*absorptionFluoresceinStd_peakWavelength,'-')
plot(1e6*concentrationValues,absorptionFluoresceinLinearFit,'--','linewidth', linewidth,'color',color4)
grid on
xlabel('Concentration (\muM)')
ylabel('Absorption (-)')
titleFluorescein = append('Absorption of Fluorescein (at ', int2str(peakWavelengthFluorescein),' nm)');
title(titleFluorescein)

figure
hold on
plot(1e6*concentrationValues,absorptionEosinYMean_peakWavelength,'-o','linewidth', linewidth,'color',color1)
plot(1e6*concentrationValues,absorptionEosinYLinearFit,'--','linewidth', linewidth,'color',color4)
grid on
xlabel('Concentration (\muM)')
ylabel('Absorption (-)')
titleEosinY = append('Absorption of Eosin Y (at ', int2str(peakWavelengthEosinY),' nm)');
title(titleEosinY)

figure
ax = gca;
ax.FontSize = fontsize;
hold on
grid on
h1 = errorbar(1e6*concentrationValues,absorptionFluoresceinMean_peakWavelength,nSTD*absorptionFluoresceinStd_peakWavelength,...
         '+','linewidth', linewidth,'color',color1);
h2 = errorbar(1e6*concentrationValues,absorptionEosinYMean_peakWavelength,nSTD*absorptionEosinYStd_peakWavelength,...
         '+','linewidth', linewidth,'color',color3);
h3 = plot(1e6*concentrationValues,absorptionFluoresceinLinearFit,'--','linewidth', linewidth,'color',color1);
h4 = plot(1e6*concentrationValues,absorptionEosinYLinearFit,'--','linewidth', linewidth,'color',color3);
xlabel('Concentration (\muM)','fontsize',fontsize)
ylabel('Absorption (-)','fontsize',fontsize)
xlim([0 10])
ylim([0 2])

hd1 = plot(-1,-1,'-','linewidth', linewidth,'color',color1);
hd2 = plot(-1,-1,'-','linewidth', linewidth,'color',color3);
hd3 = errorbar(-1,-1,1,'+','linewidth', linewidth,'color','black');
hd4 = plot(-1,-1,'--','linewidth', linewidth,'color','black');

fig = gcf;
width_plot = fig.Position(3);
fig.Position(3:4) = [width_plot, 0.5*width_plot]; % default is height/width ratio of 0.75

legendFluorescein = append('Fluorescein (\lambda = ', int2str(peakWavelengthFluorescein),' nm)');
legendEosinY = append('Eosin Y (\lambda = ', int2str(peakWavelengthEosinY),' nm)');

lgdStr1 = 'Color';
lgd1 = legendflex([hd1 hd2],...
    {legendFluorescein,legendEosinY},...
    'padding', padding,...
    'xscale', xscale,...
    'title', lgdStr1,...   
    'ref', gca, 'anchor', {'nw', 'nw'}, 'buffer', [+offset -offset],...    
    'fontsize', fontsizeLegend); 
set(findall(lgd1, 'string', lgdStr1), 'fontweight', 'bold'); 

lgdStr2 = 'Lines';
lgd2 = legendflex([hd3, hd4],...
    {'Measurement \pm3\sigma','Linear fit'},...
    'padding', padding,...
    'xscale', xscale,...
    'title', lgdStr2,...
    'ref', lgd1, 'anchor', {'sw', 'nw'}, 'buffer', [0 -offset],...     
    'fontsize', fontsizeLegend); 
set(findall(lgd2, 'string', lgdStr2), 'fontweight', 'bold');

% exportgraphics(fig,'absorption_coefficients.pdf','BackgroundColor','none')

%% Transmission


concentrationVector = linspace(0,concentrationValues(end),1e3);
transmissionFluoresceinFit = exp(-absorptionCoefficientFluorescein*concentrationVector*pathLength);
transmissionEosinYFit = exp(-absorptionCoefficientEosinY*concentrationVector*pathLength);

figure
hold on
grid on
h1 = plot(1e6.*concentrationValues,transmissionFluoresceinMean_peakWavelength./100,'o','linewidth', linewidth,'color',color1);
h2 = plot(1e6.*concentrationValues,transmissionEosinYMean_peakWavelength./100,'o','linewidth', linewidth,'color',color3);
h3 = plot(1e6.*concentrationVector,transmissionFluoresceinFit,'--','linewidth', linewidth,'color',color1);
h4 = plot(1e6.*concentrationVector,transmissionEosinYFit,'--','linewidth', linewidth,'color',color3);

hd1 = plot(-1,-1,'-','linewidth', linewidth,'color',color1);
hd2 = plot(-1,-1,'-','linewidth', linewidth,'color',color3);
hd3 = errorbar(-1,-1,1,'o','linewidth', linewidth,'color','black');
hd4 = plot(-1,-1,'--','linewidth', linewidth,'color','black');

fig = gcf;
width_plot = fig.Position(3);
fig.Position(3:4) = [width_plot, 0.5*width_plot]; % default is height/width ratio of 0.75
ax = gca;
ax.FontSize = fontsize;

xlabel('Concentration (\muM)','fontsize',fontsize)
ylabel('$I_\mathrm{out}/I_0$ (-)','fontsize',14,'interpreter','latex')
xlim([0 10])
ylim([0 1])

legendFluorescein = append('Fluorescein (\lambda = ', int2str(peakWavelengthFluorescein),' nm)');
legendEosinY = append('Eosin Y (\lambda = ', int2str(peakWavelengthEosinY),' nm)');

lgdStr1 = 'Color';
lgd1 = legendflex([hd1 hd2],...
    {legendFluorescein,legendEosinY},...
    'padding', padding,...
    'xscale', xscale,...
    'title', lgdStr1,...   
    'ref', gca, 'anchor', {'ne', 'ne'}, 'buffer', [-offset -offset],...    
    'fontsize', fontsizeLegend); 
set(findall(lgd1, 'string', lgdStr1), 'fontweight', 'bold'); 

lgdStr2 = 'Lines';
lgd2 = legendflex([hd3, hd4],...
    {'Measurement','Fit'},...
    'padding', padding,...
    'xscale', xscale,...
    'title', lgdStr2,...
    'ref', lgd1, 'anchor', {'se', 'ne'}, 'buffer', [0 -offset],...     
    'fontsize', fontsizeLegend); 
set(findall(lgd2, 'string', lgdStr2), 'fontweight', 'bold');

% exportgraphics(fig,'transmission_fluorophores.pdf','BackgroundColor','none')




