%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Author:   Jelle Snieder
% Date:     18-3-2024 
% 
% This m-file imports the measure excitation and emission spectra of 
% Fluorescein and Eosin Y. The spectra are process in the following way:
% 
% - averaging 
% - mapping and interpolation on desired wavelength range
% - multiplying with correction curves
% - Gaussian smoothing
% - normalising
% 
% The spectra are saved as .mat file for future use.
% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clearvars
close all
clc

% parameters
folder          = 'Excitation_and_emission_spectra';
concentration   = '00005'; % [mM]
wavelengths     = 400:1:700; % [nm] desired wavelength range for mapping and interpolation

run('plotsettings.m')

%% import measured spectra

% extract data - excitation Fluorescein
file                            = append('Fluorescein_excitation_',concentration,'mM.csv');
filename                        = fullfile(folder,file);
dataRaw                         = readmatrix(filename);
dataRaw                         = rmmissing(dataRaw);

wavelengths_fluorescein_exc     = dataRaw(:,1:2:end-1);
wavelengths_fluorescein_exc     = mean(wavelengths_fluorescein_exc,2); % averaging
fluorescein_exc                 = dataRaw(:,2:2:end);
fluorescein_exc                 = mean(fluorescein_exc,2); % averaging

% extract data - emission Fluorescein
file                            = append('Fluorescein_emission_',concentration,'mM.csv');
filename                        = fullfile(folder,file);
dataRaw                         = readmatrix(filename); 
dataRaw                         = rmmissing(dataRaw);

wavelengths_fluorescein_em      = dataRaw(:,1:2:end-1);
wavelengths_fluorescein_em      = mean(wavelengths_fluorescein_em,2); % averaging
fluorescein_em                  = dataRaw(:,2:2:end);
fluorescein_em                  = mean(fluorescein_em,2); % averaging

% extract data - excitation Eosin Y
file                            = append('EosinY_excitation_',concentration,'mM.csv');
filename                        = fullfile(folder,file);
dataRaw                         = readmatrix(filename);
dataRaw                         = rmmissing(dataRaw);

wavelengths_eosinY_exc          = dataRaw(:,1:2:end-1);
wavelengths_eosinY_exc          = mean(wavelengths_eosinY_exc,2); % averaging
eosinY_exc                      = dataRaw(:,2:2:end);
eosinY_exc                      = mean(eosinY_exc,2); % averaging

% extract data - emission Eosin Y
file                            = append('EosinY_emission_',concentration,'mM.csv');
filename                        = fullfile(folder,file);
dataRaw                         = readmatrix(filename);
dataRaw                         = rmmissing(dataRaw);

wavelengths_eosinY_em           = dataRaw(:,1:2:end-1);
wavelengths_eosinY_em           = mean(wavelengths_eosinY_em,2); % averaging
eosinY_em                       = dataRaw(:,2:2:end);
eosinY_em                       = mean(eosinY_em,2); % averaging

%% zeropadding and interpolation
% Data is not necessarily measured on the desired wavelenghts. Zero padding 
% in front or at the end is performed to match the desired wavelengthexcitation
% range. Next, the data is interpolated on the desired wavelengths.

fluoresceinExcitation   = interpolate(wavelengths_fluorescein_exc, fluorescein_exc, wavelengths);
fluoresceinEmission     = interpolate(wavelengths_fluorescein_em, fluorescein_em, wavelengths);
eosinYExcitation        = interpolate(wavelengths_eosinY_exc, eosinY_exc, wavelengths);
eosinYEmission          = interpolate(wavelengths_eosinY_em, eosinY_em, wavelengths);

% save not corrected spectra
gaussian = 5;
fluoresceinExcitation  = smoothdata(fluoresceinExcitation, 'gaussian',gaussian);
fluoresceinEmission    = smoothdata(fluoresceinEmission , 'gaussian',gaussian);
eosinYExcitation       = smoothdata(eosinYExcitation , 'gaussian',gaussian);
eosinYEmission         = smoothdata(eosinYEmission, 'gaussian',gaussian);

fluoresceinExcitationNotCorrected  = normalize(fluoresceinExcitation, 'range');
fluoresceinEmissionNotCorrected    = normalize(fluoresceinEmission, 'range');
eosinYExcitationNotCorrected       = normalize(eosinYExcitation, 'range');
eosinYEmissionNotCorrected         = normalize(eosinYEmission, 'range');


%% correct spectra with spectra correction curves

load('dataset_correction_spectra.mat');

fluoresceinExcitation  = fluoresceinExcitation.*correctionSpectrumExcitation;
fluoresceinEmission    = fluoresceinEmission.*correctionSpectrumEmission;
eosinYExcitation       = eosinYExcitation.*correctionSpectrumExcitation;
eosinYEmission         = eosinYEmission.*correctionSpectrumEmission;


%% smooth and normalize

fluoresceinExcitation  = smoothdata(fluoresceinExcitation, 'gaussian',gaussian);
fluoresceinEmission    = smoothdata(fluoresceinEmission , 'gaussian',gaussian);
eosinYExcitation       = smoothdata(eosinYExcitation , 'gaussian',gaussian);
eosinYEmission         = smoothdata(eosinYEmission, 'gaussian',gaussian);

fluoresceinExcitation  = normalize(fluoresceinExcitation, 'range');
fluoresceinEmission    = normalize(fluoresceinEmission, 'range');
eosinYExcitation       = normalize(eosinYExcitation, 'range');
eosinYEmission         = normalize(eosinYEmission, 'range');

%% save spectra

% save spectra to .mat file
save("dataset_fluorescein_eosinY.mat",...
    "wavelengths",...
    "fluoresceinExcitation",...
    "fluoresceinEmission",...
    "eosinYExcitation",...
    "eosinYEmission",...
    "fluoresceinExcitationNotCorrected",...
    "fluoresceinEmissionNotCorrected",...
    "eosinYExcitationNotCorrected",...
    "eosinYEmissionNotCorrected");

%% plot

% single raw spectra
figure
hold on
plot(wavelengths_fluorescein_exc, normalize(fluorescein_exc, 'range'), 'linewidth', linewidth,'color',color1)
plot(wavelengths_fluorescein_em, normalize(fluorescein_em, 'range'), 'linewidth', linewidth,'color',color2)
grid on
xlabel('wavelength (nm)')
ylabel('amplitude (-)')
title(append('Fluorescein ', concentration, ' mM'))

figure
hold on
plot(wavelengths_eosinY_exc, normalize(eosinY_exc, 'range'), 'linewidth', linewidth,'color',color1)
plot(wavelengths_eosinY_em, normalize(eosinY_em, 'range'), 'linewidth', linewidth,'color',color2)
grid on
xlabel('wavelength (nm)')
ylabel('amplitude (-)')
title(append('Eosin Y ', concentration, ' mM'))

% combined raw spectra
figure
hold on
plot(wavelengths_fluorescein_exc, normalize(fluorescein_exc, 'range'), '--','linewidth', linewidth,'color',color1)
plot(wavelengths_fluorescein_em, normalize(fluorescein_em, 'range'), 'linewidth', linewidth,'color',color1)
plot(wavelengths_eosinY_exc, normalize(eosinY_exc, 'range'), '--','linewidth', linewidth,'color',color3)
plot(wavelengths_eosinY_em, normalize(eosinY_em, 'range'), 'linewidth', linewidth,'color',color3)
grid on
xlim([400 700])
xlabel('wavelength (nm)')
ylabel('amplitude (-)')
legend('Excitation Fluorescein','Emission Fluorescein',...
    'Excitation Eosin Y','Emission Eosin Y','Location','northeast')

%% functions
function [signalNew] = interpolate(wavelengthsOld, signalOld, wavelengths)
    
    % add zeros in front, if neccessary.
    n1 = round(min(wavelengthsOld))-min(wavelengths); % number of wavelengths to be added in front
    wavelengthsFront = min(wavelengths):1:(min(wavelengths)+n1-1);
    wavelengthsNew = [wavelengthsFront'; wavelengthsOld];
    vectorZeros = zeros(n1,1);
    signalNew = [vectorZeros; signalOld];

    % add zeros in the end, if necessary.
    n2 = max(wavelengths)-round(max(wavelengthsNew)); % number of wavelengths to be added in the end
    wavelengthsEnd = (max(wavelengths)-n2+1):1:(max(wavelengths));
    wavelengthsNew = [wavelengthsNew; wavelengthsEnd'];
    vectorZeros = zeros(n2,1); 
    signalNew = [signalNew; vectorZeros];

    % interpolate at desrired wavelength range
    signalNew = interp1(wavelengthsNew, signalNew, wavelengths)'; 

end


















