%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Author:   Jelle Snieder
% Date:     15-3-2024
%  
% This m-file imports the correction excitation and emission spectrum
% (imported from Agilent Cary Eclipse Fluorescence Spectrophotometer) and
% maps and interpolates the spectra onto the desired wavelengh range. 
% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clearvars
close all
clc

%% load correction spectra

% parameters
folder      = 'Excitation_and_emission_spectra';
subfolder   = 'Correction_curves';

% extract excitation correction curve
file                            = 'Excorr.csv';
filename                        = fullfile(folder,subfolder,file);
dataRaw                         = readmatrix(filename);
indicesNaN                      = isnan(dataRaw);
dataRaw(all(indicesNaN,2),:)    = [];
wavelengths_correction_exc      = dataRaw(:,1);
signal_correction_exc           = dataRaw(:,2);

% extract emission correction curve
file                            = 'Emcorr_220_to_600.csv';
filename                        = fullfile(folder,subfolder,file);
dataRaw                         = readmatrix(filename);
indicesNaN                      = isnan(dataRaw);
dataRaw(all(indicesNaN,2),:)    = [];
wavelengths_correction_em       = dataRaw(:,1);
signal_correction_em            = dataRaw(:,2);

%% padding and interpolation
wavelengths = 400:1:700; % nm

% The correction spectra are not necessarily given for the desired 
% wavelengths. The correction spectra are padded in front (with the first 
% value) or behind (with the last value) if neccessary. Next, the data is 
% interpolated on the desired wavelengths.

signal_correction_exc_modified = interpolate(wavelengths_correction_exc, signal_correction_exc, wavelengths);
signal_correction_em_modified = interpolate(wavelengths_correction_em, signal_correction_em, wavelengths);

%% plot

% excitation
figure
hold on
plot(wavelengths_correction_exc,signal_correction_exc)
plot(wavelengths,signal_correction_exc_modified,'--')
grid on
xlabel('wavelength (nm)')
ylabel('amplitude (-)')
title('Excitation correction')

% emission
figure
hold on
plot(wavelengths_correction_em,signal_correction_em)
plot(wavelengths,signal_correction_em_modified,'--')
grid on
xlabel('wavelength (nm)')
ylabel('amplitude (-)')
title('Emission correction')

%% export spectra
correctionSpectrumExcitation = signal_correction_exc_modified;
correctionSpectrumEmission = signal_correction_em_modified;
save('dataset_correction_spectra.mat',...
    'correctionSpectrumExcitation',...
    'correctionSpectrumEmission');

%% functions
function [signalNew] = interpolate(wavelengthsOld, signalOld, wavelengths)
    
    % add vector with first value in front, if neccessary.
    n1 = round(min(wavelengthsOld))-min(wavelengths); % number of wavelengths to be added in front
    wavelengthsFront = min(wavelengths):1:(min(wavelengths)+n1-1);
    wavelengthsNew = [wavelengthsFront'; wavelengthsOld];
    vectorOnes = ones(n1,1);
    signalNew = [signalOld(1)*vectorOnes; signalOld];

    % add vector with last value in the end, if necessary.
    n2 = max(wavelengths)-round(max(wavelengthsNew)); % number of wavelengths to be added in the end
    wavelengthsEnd = (max(wavelengths)-n2+1):1:(max(wavelengths));
    wavelengthsNew = [wavelengthsNew; wavelengthsEnd'];
    vectorOnes = ones(n2,1); 
    signalNew = [signalNew; signalOld(end)*vectorOnes];

    % interpolate at desrired wavelength range
    signalNew = interp1(wavelengthsNew, signalNew, wavelengths)'; 

end









