############################################################
### User-defined functions                               ###
###                                                      ###
### by Lisa Scholten, l.scholten@tudelft.nl              ### 
############################################################


# ==== 1 Prepare R environment, load packages ====

require(tmvtnorm) # Due to S3 incompatibility, functions need to be called as tmvtnorm::function 

# ==== 2 Value and choice functions ====

## If need to normalize x and y?
## xi = (xi-xmin)/(xmax-xmin)

# EV of gamble
ev.gamble <- function(x,p,y){
  x = x
  y = y
  p = p
  v <- p*x + (1-p)*y
  return(v)
}

# EUT with CRRA - Constant relative risk attitude
v.crra <- function(x,p,y,r){
  x = abs(x) # ifelse(x > y, x, y)
  y = abs(y) # ifelse(x > y, y, x)
  p = p 
  r = r
  if(r == 1) {
    v <- p*log(x) + (1-p)*log(y)
  } else {
    v <- p*(x^(1-r))/(1-r) + (1-p)*(y^(1-r))/(1-r)
  }
  if (is.na(any(v))) return(Inf)
  return(v)
}

# DEPRECATED - issue if logical value passed to ifelse is 
# different lenght than vector elements to
# v.crra <- function(x,p,y,r){
#   x = abs(x) # ifelse(x > y, x, y)
#   y = abs(y) # ifelse(x > y, y, x)
#   p = p 
#   r = r
#   v.x <- ifelse(r == 1, 
#                 p*log(x),
#                 p*(x^(1-r))/(1-r))
#   v.y <- ifelse(r == 1,
#                 (1-p)*log(y),
#                 (1-p)*(y^(1-r))/(1-r))
#   v <- v.x + v.y
#   if (is.na(v)) return(Inf)
#   return(v)
# }


# CPT - Value and probability weighting functions

## Prelec's (1998) probability weighting function
### I par version
w.prelec <- function(p,a){
  w.p <- 1/(exp(log(1/p))^a)
  return(w.p)
} 

### II par version
w.prelec2 <- function(p,d,g){
  w.p <- 1/(exp(log(1/p))^a)
  return(w.p)
}

## Sign-dependent CPT and Prelec I probability weighting  

### Probability-weighted value function
v.pw <- function(x,p,a,s,l){
  v <- w.prelec(p,a)*v.power(x,s,l)
}  

### Power value function
v.power <- function(x,s,l) {
  v <- ifelse(x >= 0, 
              x^s, # for gains
              -l*(-x)^s) # for losses
  return(v)
}

### Value of a gamble under CPT
v.gamble <- function(x,p,y,a,s,l){
  x = ifelse(x > y, x, y)
  y = ifelse(x > y, y, x)
  p = p
  v <- ifelse((x < 0)&(0 < y),
              # mixed gamble
              w.prelec(p,a)*v.power(x,s,l)+ 
                w.prelec(1-p,a)*v.power(y,s,l),
              # pure gain or loss gamble
              v.power(y,s,l)+
                w.prelec(p,a)*(v.power(x,s,l)-v.power(y,s,l)))
  return(v)
}


# Logit choice function as in (Stott, 2006), which was also used by (Pedroni, 2017)
# - returns probability of chosing A over B
# - 'eps' is a noise parameter: the lower, the less people are able to choose between A and B 


## Choice data in matrix format
c.logit <- function (data = data, pars = pars){ # A, B, pars are named vectors
  eps <- pars["eps"]
  
  A <- list(x=data[,"x.A"], p=data[,"px.A"], y=data[,"y.A"], pars["a"], pars["s"], pars["l"])
  B <- list(x=data[,"x.B"], p=data[,"px.B"], y=data[,"y.B"], pars["a"], pars["s"], pars["l"])
  p.A <- 1/(1+exp(-1*eps*(do.call(v.gamble, args = A) -
                          do.call(v.gamble, args = B))))
  # p.A <- 1/(1+exp(eps*(do.call(v.gamble, args = B) -  # Luce choice model
  #                           do.call(v.gamble, args = A))))
  return(unname(p.A, force = TRUE))
} 

c.logit.crra <- function (data=data, pars=pars){ # A, B, pars are named vectors
  eps <- pars["eps"]
  A <- list(x = data[,"x.A"], p = data[,"px.A"], y = data[,"y.A"], r = pars["r"])
  B <- list(x = data[,"x.B"], p = data[,"px.B"], y = data[,"y.B"], r = pars["r"])
  p.A <- 1/(1+exp(-1*eps*(do.call(v.crra, args = A) - 
                          do.call(v.crra, args = B))))
  return(unname(p.A, force = TRUE))
} 

### Make sure it does as intended
# - Column 1 indicates 1 if A chosen, otherwise B = 0 --> WHAT IF A == B?
# dat <- matrix(NA, nrow = 2, ncol = 7)
# colnames(dat) <- c("choice", "x.A", "px.A", "y.A", "x.B", "y.B", "px.B")
# dat[1,] <- c(0, 10, 0.75, 0, 50, 10, 0.9)
# dat[2,] <- c(1, 80, 0.6, 10, 50, 10, 0.2)
# pars <- c(a=1, s=1, l=, eps=0.5)

# test: c.logit(data=dat, pars = pars)
# test: c.logit.crra(data=dat, pars = c(eps = 1, r = 0.99))

# tofy <- data.frame(
# "c.logit" = c.logit(data = pedroni.AL, pars = c(a=1,s=1,l=1,eps=1)),
#  ">0.5"  = ifelse(c.logit(data = pedroni.AL, pars = c(a=1,s=1,l=1,eps=1)) > 0.5, 1, 0),
#  "v.A" = v.gamble(x = pedroni.AL$x.A, p = pedroni.AL$px.A, y = pedroni.AL$y.A, a=1, s= 1, l = 1),
#  "v.B" = v.gamble(x = pedroni.AL$x.B, p = pedroni.AL$px.B, y = pedroni.AL$y.B, a=1, s= 1, l = 1)
# )

# DEPRECATED VERSION
# c.logit <- function (eps, A, B){ # A and B are named vectors
#   eps <- eps
#   A <- list(A["x"], A["p"], A["y"], A["a"], A["s"], A["l"])
#   B <- list(B["x"], B["p"], B["y"], B["a"], B["s"], B["l"])
#   p.A <- 1/(1+exp(-eps*(do.call(v.gamble,A) - do.call(v.gamble,B))))
#   return(p.A)
# }  # works but issues if called within other function
# 
# ## Example to test whether it does as intended
# # A <- c(x=50,p=0.1,y=10,a=1,s=1,l=1)
# # B <- c(x=50,p=0.9,y=10,a=1,s=1,l=1)
# # 
# # c.logit(eps = 0.1, A = A , B = B)

# # Newer version
# c.logit.1 <- function (A=A, B=B, pars=pars){ # A, B, pars are named vectors
#   eps <- pars["eps"]
#   A <- list(A["x"], A["p"], A["y"], pars["a"], pars["s"], pars["l"])
#   B <- list(B["x"], B["p"], B["y"], pars["a"], pars["s"], pars["l"])
#   p.A <- 1/(1+exp(-eps*(do.call(v.gamble, args = A) - 
#                           do.call(v.gamble, args = B))))
#   return(p.A)
# }
# 
# c.logit.1(A=c(x=50,p=0.1,y=10), B=c(x=50,p=0.9,y=10), pars = c(eps=0.1, a=1,s=1,l=1))
# # 0.03916572

# ==== 3 Likelihood functions ====

## For CPT
### Likelihood function
l.logit <- function(data=data,pars=pars){
  c <- data[,"choice"]
  pars <- pars
  data <- data
  l.a <- prod(do.call(c.logit, args = list(data, pars))^c * 
                (1 - do.call(c.logit, args= list(data, pars)))^(1 - c)
  )
  if (is.na(l.a)) return(-Inf)
  return(l.a)
} # test: l.logit(data=dat, pars = pars)

### Log likelihood function with safeguards against log(0)
ll.logit <- function(data=data,pars=pars){
  c <- data[,"choice"]
  pars <- pars
  data <- data
  ll.a <- sum(c*ifelse(do.call(c.logit, args = list(data, pars)) == 0, 0, 
                       log(do.call(c.logit, args = list(data, pars)))) + 
                (1-c)*ifelse(do.call(c.logit, args= list(data, pars)) == 1, 1,
                             log(1-do.call(c.logit, args= list(data, pars)))),
              na.rm = TRUE)
  
  if (is.na(ll.a)) return(-Inf)
  return(ll.a)
} # test: ll.logit(data=dat, pars = pars)

## DEPRECATED: (negative) Log likelihood function
# ll.logit <- function(data=data,pars=pars){
#   y <- data[,"choice"]
#   pars <- pars
#   data <- data
#   ll.a <- sum(y*log(do.call(c.logit, args = list(data, pars))) +
#                 (1-y)*log(1-do.call(c.logit, args= list(data, pars)))
#               )
#   if (is.na(ll.a)) return(Inf)
#   return(-ll.a)
# }

## For EUT
### Likelihood function EUT-CRRA
l.logit.crra <- function(data=data,pars=pars){
  c <- data[,"choice"]
  pars <- pars
  data <- data
  l.a <- prod(do.call(c.logit.crra, args = list(data, pars))^c * 
                (1-do.call(c.logit.crra, args= list(data, pars)))^(1-c)
  )
  if (is.na(l.a)) return(-Inf)
  return(l.a)
} # test: l.logit.crra(data=dat, pars = pars)

### Log-likelihood function EUT-CRRA
ll.logit.crra <- function(data=data,pars=pars){
  c <- data[,"choice"]
  pars <- pars
  data <- data
  ll.a <- sum(c*ifelse(do.call(c.logit.crra, args = list(data, pars)) == 0, 0, 
                       log(do.call(c.logit.crra, args = list(data, pars)))) + 
                (1-c)*ifelse(do.call(c.logit.crra, args= list(data, pars)) == 1, 1,
                             log(1-do.call(c.logit.crra, args= list(data, pars)))),
              na.rm = TRUE)
  
  if (is.na(ll.a)) return(-Inf)
  return(ll.a)
} # test: ll.logit.crra(data=dat, pars = pars)

# ==== 4 Bayesian prior and posterior functions ====

##  Bayesian prior for multivariate normal distribution
# -  pars to value function, prior.mean and prior.cov tbd
f.pri <- function(pars, prior.mean, prior.cov) {
  pars = pars
  prior.mean = prior.mean
  prior.cov = prior.cov
  pri <- tmvtnorm:::dtmvnorm(x= pars, mean = prior.mean, sigma= prior.cov,
                  lower = prior.l,
                  upper = prior.u,
                  log=TRUE)
  if (is.na(pri)) return(-Inf)
  return (pri)
} # test: f.pri(pars = c(a=1, s=1, l=1, eps=1), prior.mean = prior.mean, prior.cov = prior.cov)

## Posterior for multivariate normal distribution
f.post <- function(pars, data, prior.mean, prior.cov, ...) f.pri(pars, prior.mean, prior.cov) + ll.logit(data, pars, ...)
# test: f.post(data = dat, pars = c(a=1, s=1, l=1, eps=1), prior.mean = prior.mean, prior.cov = prior.cov)



## Bayesian prior for uniform distribution
# -  pars to value function, prior.l and prior.u are lower and upper bound
f.pri.u <- function(pars, prior.l, prior.u) { 
  pars = pars
  prior.l = prior.l
  prior.u = prior.u
  pri <- sum(dunif(x= pars, min = prior.l, max = prior.u,
                   log=TRUE), na.rm = TRUE)
  if (is.na(pri)) return(-Inf)
  return (pri)
} # test: f.pri.u(pars = c(a=1, s=1, l=2, eps=2), prior.l = prior.l, prior.u = prior.u)

## Posterior for uniform distribution
f.post.u <- function(pars, data, prior.l, prior.u, ...) f.pri.u(pars, prior.l, prior.u) + ll.logit(data, pars, ...)
# test: f.post.u(data = dat, pars = c(a=1, s=1, l=5, eps=2), prior.l = prior.l, prior.u = prior.u)


# ==== 5 Hierarchical Bayes prior and posterior functions ====

# Group-level parameters (hyper-priors)
hprior.mean <- function(pars){
  x = pars
  mu.a <- dnorm(x["mu.a"], mean = 0, sd = 1)
  mu.s <- dnorm(x["mu.s"], mean = 0, sd = 1)
  mu.l <- dunif(x["mu.l"], min = -2.3, max = 1.61) 
  mu.eps <- dunif(x["mu.eps"], min = -2.3, max = 1.61)   
  return(c(mu.a, mu.s, mu.l, mu.eps))
}

hprior.sd <- function(pars){
  x = pars
  sd.a <- dunif(x["sd.a"], min = 0, max = 10)
  sd.s <- dunif(x["sd.s"], min = 0, max = 10)
  sd.l <- dunif(x["sd.l"], min = 0, max = 1.13)
  sd.eps <- dunif(x["sd.eps"], min = 0, max = 1.13)   
  return(c(sd.a, sd.s, sd.l, sd.eps))
}

# test:
# start.par.h <- c(
#   mu.a = 0.7, sd.a = 1, 
#   mu.s = 0.7, sd.s = 1, 
#   mu.l = 0, sd.l = 0.5, 
#   mu.eps = 0, sd.eps = 0.5
# )
# hprior.mean(start.par.h)
# hprior.sd(start.par.h)


# Individual-level parameters (per participant)
# - j: respondents
# - i: gambles

i.pars <- function(pars) {
  x = pars
  a <- rtnorm(1, hprior.mean(x)["mu.a"], hprior.sd(x)["sd.a"], lower = -3, upper = 3) # Nilsson etal 2011: gamma
  s <- rtnorm(1, hprior.mean(x)["mu.s"], hprior.sd(x)["sd.s"], lower = -3, upper = 3) # ibid: alpha
  l <- rnorm(1, hprior.mean(x)["mu.l"], hprior.sd(x)["sd.l"]) # ibid: lambda
  eps <- rnorm(1, hprior.mean(x)["mu.eps"], hprior.sd(x)["sd.eps"])  # ibid: luce
  return(c("a" = unname(a), "s" = unname(s), "l" = unname(l), "eps" = unname(eps)))
} 
# test: i.pars(start.par.h)


# Prior distribution for individual
f.pri.h <- function(pars, ipars) {
  x = pars
  ipars = ipars
  mean = hprior.mean(x)
  sd = hprior.sd(x)
  hprior.cor <- matrix(c(1, 0, 0, 0,
                         0, 1, 0, 0,
                         0, 0, 1, 0,
                         0, 0, 0, 1),
                       ncol = 4)
  hprior.cov <- diag(hprior.sd(x))*hprior.cor*diag(hprior.sd(x))
  pri <- tmvtnorm:::dtmvnorm(x = ipars, mean = mean, sigma = hprior.cov, log = TRUE)
  if (is.na(pri)) return(NA)
  return (pri)
} 
# test: f.pri.h(pars = start.par.h, ipars = i.pars(start.par.h))

# Specify posterior
f.post.h <- function(pars, data, ...) {
  pars = pars
  ipars <- i.pars(pars)
  f.pri.h(pars, ipars) + ll.logit(data, pars = ipars, ...)}

# test: f.post.h(data = dat, pars = start.par.h)

# ==== 5 Data analysis functions ====

# Adapted *summaryfunction* of Conelly (2014) as R base package's *summary* function is not particularly plotting-friendly.
# https://www.r-bloggers.com/summary-function-that-is-compatible-with-xtable/) 

summaryfunction = function (x){
  if( is.numeric(x)!=TRUE) {stop("Supplied X is not numeric")}
  mysummary = data.frame(
    "n"       = length(x),
    "min."    = min(x),
    "Q.25"    = quantile(x)[2],
    "median"  = median(x),
    "mean"    = mean(x),
    "Q.75"    = quantile(x)[4],
    "max."    = max(x),
    "sd"      = sd(x, na.rm = T),
    "kurtosis" = kurtosis(x),
    "skewness" = skewness(x),
    row.names = ""
  )
  names(mysummary) = c("n", "min","Q.25","median","mean","Q.75","max", "sd", "kurtosis", "skewness")
  return(mysummary)
}


# for glmer overdispersion check
# source:  https://ase.tufts.edu/gsc/gradresources/guidetomixedmodelsinr/mixed%20model%20guide.html, 2019-11-27
overdisp_fun <- function(model) {
  ## number of variance parameters in an n-by-n variance-covariance matrix
  vpars <- function(m) {
    nrow(m) * (nrow(m) + 1)/2
  }
  # The next two lines calculate the residual degrees of freedom
  model.df <- sum(sapply(VarCorr(model), vpars)) + length(fixef(model))
  rdf <- nrow(model.frame(model)) - model.df
  # extracts the Pearson residuals
  rp <- residuals(model, type = "pearson")
  Pearson.chisq <- sum(rp^2)
  prat <- Pearson.chisq/rdf
  # Generates a p-value. If less than 0.05, the data are overdispersed.
  pval <- pchisq(Pearson.chisq, df = rdf, lower.tail = FALSE)
  c(chisq = Pearson.chisq, ratio = prat, rdf = rdf, p = pval)
}


